/**
 * ************************************************************************
 * * The contents of this file are subject to the MRPL 1.2
 * * (the  "License"),  being   the  Mozilla   Public  License
 * * Version 1.1  with a permitted attribution clause; you may not  use this
 * * file except in compliance with the License. You  may  obtain  a copy of
 * * the License at http://www.floreantpos.org/license.html
 * * Software distributed under the License  is  distributed  on  an "AS IS"
 * * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * * License for the specific  language  governing  rights  and  limitations
 * * under the License.
 * * The Original Code is FLOREANT POS.
 * * The Initial Developer of the Original Code is OROCUBE LLC
 * * All portions are Copyright (C) 2015 OROCUBE LLC
 * * All Rights Reserved.
 * ************************************************************************
 */
package com.floreantpos.model;

import java.util.ArrayList;
import java.util.List;

import javax.print.PrintService;
import javax.print.PrintServiceLookup;
import javax.xml.bind.annotation.XmlRootElement;

import com.floreantpos.PosLog;
import com.floreantpos.constants.AppConstants;
import com.floreantpos.main.Application;
import com.floreantpos.model.dao.TerminalPrintersDAO;
import com.floreantpos.model.util.DataProvider;
import com.floreantpos.print.PrinterType;
import com.floreantpos.util.POSUtil;

@XmlRootElement(name = "printers")
public class PosPrinters {
	private String receiptPrinter;

	private Printer defaultKitchenPrinter;

	private List<Printer> kitchenPrinters = new ArrayList<>(1);
	private List<Printer> reportPrinters = new ArrayList<>(1);
	private List<Printer> receiptPrinters = new ArrayList<>(1);
	private List<Printer> stickerPrinters = new ArrayList<>(1);
	private List<Printer> labelPrinters = new ArrayList<>(1);
	private List<Printer> packingPrinters = new ArrayList<>(1);
	private boolean printToKds;

	public String getReceiptPrinter() {
		return receiptPrinter;
	}

	public List<Printer> getKitchenPrinters() {
		return kitchenPrinters;
	}

	public List<Printer> getPackingPrinters() {
		return packingPrinters;
	}

	@Deprecated
	public Printer getDefaultKitchenPrinter() {
		if (getKitchenPrinters().size() > 0) {
			defaultKitchenPrinter = kitchenPrinters.get(0);

			for (Printer printer : kitchenPrinters) {
				if (printer.isDefaultPrinter()) {
					defaultKitchenPrinter = printer;
					break;
				}
			}
		}

		return defaultKitchenPrinter;
	}

	public static PosPrinters load() {
		return load(Application.getInstance().getTerminal());
	}

	public static PosPrinters load(Terminal terminal) {
		PosPrinters printers = new PosPrinters();
		try {
			List<TerminalPrinters> terminalPrinters = TerminalPrintersDAO.getInstance().findTerminalPrinters(terminal);

			List<Printer> terminalActivePrinters = new ArrayList<Printer>();
			for (TerminalPrinters terminalPrinter : terminalPrinters) {
				int printerType = terminalPrinter.getVirtualPrinter().getType();
				if (printerType == VirtualPrinter.REPORT) {
					printers.reportPrinters.add(createPrinter(terminalPrinter));
				}
				else if (printerType == VirtualPrinter.LABEL) {
					printers.labelPrinters.add(createPrinter(terminalPrinter));
				}
				else if (printerType == VirtualPrinter.RECEIPT) {
					printers.receiptPrinter = terminalPrinter.getPrinterName();
					printers.receiptPrinters.add(createPrinter(terminalPrinter));
				}
				else if (printerType == VirtualPrinter.KITCHEN) {
					printers.kitchenPrinters.add(createPrinter(terminalPrinter));
				}
				else if (printerType == VirtualPrinter.STICKER) {
					printers.stickerPrinters.add(createPrinter(terminalPrinter));
				}
				else if (printerType == VirtualPrinter.PACKING) {
					printers.packingPrinters.add(createPrinter(terminalPrinter));
				}
				else {
					Printer printer = createPrinter(terminalPrinter);
					terminalActivePrinters.add(printer);
				}
			}
			if (printers.receiptPrinter == null) {
				printers.receiptPrinter = getDefaultPrinterName();
			}
			printers.printToKds = DataProvider.get().getStore().isKDSenabled();

		} catch (Exception e) {
			PosLog.error(PosPrinters.class, e);
		}

		return printers;
	}

	public static Printer createPrinter(TerminalPrinters terminalPrinter) {
		Printer printer = new Printer(terminalPrinter.getVirtualPrinter(), terminalPrinter.getPrinterName());
		printer.setPrintSystem(terminalPrinter.getProperty(AppConstants.RECEIPT_PRINT_SYSTEM, AppConstants.RECEIPT_JASPER));
		printer.setPrinterType(terminalPrinter.getProperty(TerminalPrinters.PRINTER_TYPE, PrinterType.LOCAL.getName()));
		printer.setIpAddress(terminalPrinter.getProperty(TerminalPrinters.IP_ADDRESS));
		printer.setIpPort(terminalPrinter.getProperty(TerminalPrinters.IP_PORT));
		printer.setSerialPort(terminalPrinter.getProperty(TerminalPrinters.SERIAL_PORT));
		printer.setTextLength(POSUtil.parseInteger(terminalPrinter.getProperty(TerminalPrinters.TEXT_LENGTH)));
		return printer;
	}

	public static String getDefaultPrinterName() {
		PrintService defaultPrintService = PrintServiceLookup.lookupDefaultPrintService();
		if (defaultPrintService != null) {
			return defaultPrintService.getName();
		}

		PrintService[] services = PrintServiceLookup.lookupPrintServices(null, null);
		if (services.length > 0) {
			return services[0].getName();
		}
		return null;
	}

	public boolean isPrintToKds() {
		return printToKds;
	}

	public List<Printer> getReportPrinters() {
		return reportPrinters;
	}

	public List<Printer> getReceiptPrinters() {
		return receiptPrinters;
	}

	public List<Printer> getStickerPrinters() {
		return stickerPrinters;
	}

	public List<Printer> getLabelPrinters() {
		return labelPrinters;
	}

	public String getLabelPrinter() {
		if (labelPrinters.size() == 0) {
			return null;
		}
		return labelPrinters.get(0).getDeviceName();
	}

}
