/**
 * ************************************************************************
 * * The contents of this file are subject to the MRPL 1.2
 * * (the  "License"),  being   the  Mozilla   Public  License
 * * Version 1.1  with a permitted attribution clause; you may not  use this
 * * file except in compliance with the License. You  may  obtain  a copy of
 * * the License at http://www.floreantpos.org/license.html
 * * Software distributed under the License  is  distributed  on  an "AS IS"
 * * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * * License for the specific  language  governing  rights  and  limitations
 * * under the License.
 * * The Original Code is FLOREANT POS.
 * * The Initial Developer of the Original Code is OROCUBE LLC
 * * All portions are Copyright (C) 2015 OROCUBE LLC
 * * All Rights Reserved.
 * ************************************************************************
 */
package com.floreantpos.model.dao;

import java.io.Serializable;
import java.util.Iterator;
import java.util.List;

import org.apache.commons.beanutils.PropertyUtils;
import org.hibernate.Criteria;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.hibernate.criterion.Order;

import com.floreantpos.Messages;
import com.floreantpos.PosException;
import com.floreantpos.PosLog;
import com.floreantpos.model.ShopTable;
import com.floreantpos.model.ShopTableType;
import com.orocube.rest.service.server.BaseDataServiceDao;

public class ShopTableTypeDAO extends BaseShopTableTypeDAO {

	/**
	 * Default constructor.  Can be used in place of getInstance()
	 */
	public ShopTableTypeDAO() {
	}

	@Override
	protected Serializable save(Object obj, Session s) {
		updateTime(obj);
		return super.save(obj, s);
	}

	@Override
	protected void update(Object obj, Session s) {
		updateTime(obj);
		super.update(obj, s);
	}

	@Override
	protected void saveOrUpdate(Object obj, Session s) {
		updateTime(obj);
		super.saveOrUpdate(obj, s);
	}

	@Override
	protected void delete(Object obj, Session session) {
		ShopTableType bean = (ShopTableType) obj;
		if (bean == null) {
			throw new PosException(Messages.getString("ShopTableTypeDAO.0")); //$NON-NLS-1$
		}
		checkForeignRelation(bean);
		bean.setDeleted(Boolean.TRUE);
		update(bean, session);
	}

	private void checkForeignRelation(ShopTableType bean) throws PosException {
		String id = bean.getId();
		try (Session session = createNewSession()) {
			List foreignItems = GenericDAO.getInstance().getForeignDataList(session, ShopTable.class, "types", //$NON-NLS-1$
					ShopTableType.PROP_ID, id, ShopTable.PROP_ID);
			if (foreignItems != null && foreignItems.size() > 0) {
				String details = Messages.getString("ShopTableTypeDAO.2") + (foreignItems.size() > 1 ? Messages.getString("ShopTableTypeDAO.3") : ":"); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
				int count = 1;
				for (Iterator iterator = foreignItems.iterator(); iterator.hasNext();) {
					Integer tableNumber = (Integer) iterator.next();
					details += "\n" + count + ". " + tableNumber; //$NON-NLS-1$ //$NON-NLS-2$
					count++;
				}
				throw new PosException(Messages.getString("ShopTableTypeDAO.7"), details); //$NON-NLS-1$
			}
		}
	}

	@Override
	public List<ShopTableType> findAll() {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			addDeletedFilter(criteria);
			return criteria.list();
		}
	}

	public void saveOrUpdateShopTableType(List<ShopTableType> shopTableTypeList, boolean updateLastUpdateTime, boolean updateSyncTime) throws Exception {
		if (shopTableTypeList == null)
			return;

		Transaction tx = null;
		try (Session session = createNewSession()) {
			tx = session.beginTransaction();
			for (Iterator<ShopTableType> iterator = shopTableTypeList.iterator(); iterator.hasNext();) {
				ShopTableType item = (ShopTableType) iterator.next();
				ShopTableType existingItem = get(item.getId());
				if (existingItem != null) {
					if (!BaseDataServiceDao.get().shouldSave(item.getLastUpdateTime(), existingItem.getLastUpdateTime())) {
						PosLog.info(getClass(), item.getName() + " already updated"); //$NON-NLS-1$
						continue;
					}
					long version = existingItem.getVersion();
					PropertyUtils.copyProperties(existingItem, item);
					existingItem.setVersion(version);
					existingItem.setUpdateLastUpdateTime(updateLastUpdateTime);
					existingItem.setUpdateSyncTime(updateSyncTime);
					update(existingItem);
				}
				else {
					item.setVersion(0);
					item.setUpdateLastUpdateTime(updateLastUpdateTime);
					item.setUpdateSyncTime(updateSyncTime);
					save(item);
				}
			}
			tx.commit();
		} catch (Exception e) {
			tx.rollback();
			throw e;
		}
	}

	@Override
	public Order getDefaultOrder() {
		return Order.asc(ShopTableType.PROP_ID);
	}

}