package com.floreantpos.model.util;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.Iterator;
import java.util.List;

import org.apache.commons.lang.StringUtils;
import org.json.JSONArray;
import org.json.JSONObject;

import com.floreantpos.PosLog;
import com.floreantpos.model.MenuItem;
import com.floreantpos.model.OneTimePrice;
import com.floreantpos.model.TicketItem;
import com.floreantpos.util.OrgJsonUtil;

public class MenuItemPriceUtil {

	public static List<OneTimePrice> getPrices(MenuItem menuItem) {
		List<OneTimePrice> prices = new ArrayList<OneTimePrice>();

		if (menuItem != null && menuItem.getProperties() != null && menuItem.getProperties().has(OneTimePrice.REF)) {
			try {
				JSONArray jsonArray = menuItem.getProperties().getJSONArray(OneTimePrice.REF);
				for (int i = 0; i < jsonArray.length(); i++) {
					JSONObject jsonObject = (JSONObject) jsonArray.get(i);
					OneTimePrice oneTimePrice = new OneTimePrice();
					oneTimePrice.setDuration(OrgJsonUtil.getString(jsonObject, OneTimePrice.PROP_DURATION));
					oneTimePrice.setDurationType(OrgJsonUtil.getString(jsonObject, OneTimePrice.PROP_DURATION_TYPE));
					oneTimePrice.setPrice(OrgJsonUtil.getDouble(jsonObject, OneTimePrice.PROP_PRICE));
					oneTimePrice.setSubscription(OrgJsonUtil.getBoolean(jsonObject, OneTimePrice.PROP_SUBSCRIPTION));
					oneTimePrice.setPriceId(OrgJsonUtil.getString(jsonObject, OneTimePrice.PROP_PRICE_ID));
					oneTimePrice.setPriceSaveText(OrgJsonUtil.getString(jsonObject, OneTimePrice.PROP_PRICE_SAVE_TEXT));
					oneTimePrice.setPriceSaveTextInFrontPage(OrgJsonUtil.getString(jsonObject, OneTimePrice.PROP_PRICE_SAVE_TEXT_IN_FRONT_PAGE));
					oneTimePrice.setTitle(OrgJsonUtil.getString(jsonObject, OneTimePrice.PROP_TITLE));
					oneTimePrice.setDefaultPrice(OrgJsonUtil.getBoolean(jsonObject, OneTimePrice.PROP_DEFAULT_PRICE));
					prices.add(oneTimePrice);
				}
				Collections.sort(prices, Comparator.comparing(OneTimePrice::isSubscription).reversed().thenComparing(OneTimePrice::getDurationAsInt));
			} catch (Exception e) {
				PosLog.error(MenuItemPriceUtil.class, e);
			}
		}

		return prices;
	}

	public static List<OneTimePrice> getRecurringPrices(MenuItem menuItem) {
		List<OneTimePrice> prices = getPrices(menuItem);
		for (Iterator iterator = prices.iterator(); iterator.hasNext();) {
			OneTimePrice oneTimePrice = (OneTimePrice) iterator.next();
			if (!oneTimePrice.isSubscription()) {
				iterator.remove();
			}
		}

		return prices;
	}

	public static List<OneTimePrice> getOneTimePrices(MenuItem menuItem) {
		List<OneTimePrice> prices = getPrices(menuItem);
		for (Iterator iterator = prices.iterator(); iterator.hasNext();) {
			OneTimePrice oneTimePrice = (OneTimePrice) iterator.next();
			if (oneTimePrice.isSubscription()) {
				iterator.remove();
			}
		}

		return prices;
	}

	public static List<OneTimePrice> getLongTermPrices(MenuItem menuItem) {
		List<OneTimePrice> prices = getPrices(menuItem);
		for (Iterator iterator = prices.iterator(); iterator.hasNext();) {
			OneTimePrice oneTimePrice = (OneTimePrice) iterator.next();
			if (isNotOneTime(oneTimePrice)) {
				iterator.remove();
			}
		}

		return prices;
	}

	public static boolean isNotOneTime(OneTimePrice oneTimePrice) {
		return oneTimePrice.isSubscription() || Integer.parseInt(oneTimePrice.getDuration()) >= 20;
	}

	public static List<OneTimePrice> getForeverPrices(MenuItem menuItem) {
		List<OneTimePrice> prices = getPrices(menuItem);
		for (Iterator iterator = prices.iterator(); iterator.hasNext();) {
			OneTimePrice oneTimePrice = (OneTimePrice) iterator.next();
			if (isNotForever(oneTimePrice)) {
				iterator.remove();
			}
		}

		return prices;
	}

	public static boolean isNotForever(OneTimePrice oneTimePrice) {
		return oneTimePrice.isSubscription() || Integer.parseInt(oneTimePrice.getDuration()) < 20;
	}

	public static OneTimePrice getMonthlyPrice(MenuItem menuItem) {
		List<OneTimePrice> prices = getPrices(menuItem);
		for (Iterator iterator = prices.iterator(); iterator.hasNext();) {
			OneTimePrice oneTimePrice = (OneTimePrice) iterator.next();
			if (OneTimePrice.MONTH.equalsIgnoreCase(oneTimePrice.getDurationType())) {
				return oneTimePrice;
			}
		}

		return null;
	}

	public static OneTimePrice getYearlyPrice(MenuItem menuItem) {
		List<OneTimePrice> prices = getPrices(menuItem);
		for (Iterator iterator = prices.iterator(); iterator.hasNext();) {
			OneTimePrice oneTimePrice = (OneTimePrice) iterator.next();
			if (OneTimePrice.YEAR.equalsIgnoreCase(oneTimePrice.getDurationType())) {
				return oneTimePrice;
			}
		}

		return null;
	}

	public static OneTimePrice getPrice(MenuItem menuItem, String priceId) {
		if (priceId == null) {
			return null;
		}
		List<OneTimePrice> prices = getPrices(menuItem);
		for (Iterator iterator = prices.iterator(); iterator.hasNext();) {
			OneTimePrice oneTimePrice = (OneTimePrice) iterator.next();
			if (oneTimePrice.getPriceId() != null && priceId.equals(oneTimePrice.getPriceId())) {
				return oneTimePrice;
			}
		}
		return null;
	}

	public static List<OneTimePrice> getPrices(MenuItem menuItem, String interval) {
		if (interval == null) {
			return getPrices(menuItem);
		}
		switch (interval) {
			case OneTimePrice.MONTH:
				OneTimePrice monthlyPrice = MenuItemPriceUtil.getMonthlyPrice(menuItem);
				if (monthlyPrice == null) {
					return null;
				}
				return Arrays.asList(monthlyPrice);
			case OneTimePrice.YEAR:
				OneTimePrice yearlyPrice = MenuItemPriceUtil.getYearlyPrice(menuItem);
				if (yearlyPrice == null) {
					return null;
				}
				return Arrays.asList(yearlyPrice);
			case OneTimePrice.ONE_TIME:
				return MenuItemPriceUtil.getLongTermPrices(menuItem);
			case OneTimePrice.FOREVER:
				return MenuItemPriceUtil.getForeverPrices(menuItem);
			default:
				break;
		}
		return null;
	}

	public static OneTimePrice getStartingPrice(MenuItem item) {
		List<OneTimePrice> oneTimePrices = getPrices(item, null);
		if (oneTimePrices == null || oneTimePrices.isEmpty()) {
			return null;
		}
		if (oneTimePrices.size() == 1) {
			return oneTimePrices.get(0);
		}
		for (OneTimePrice oneTimePrice : oneTimePrices) {
			if (oneTimePrice.isDefaultPrice()) {
				return oneTimePrice;
			}
		}
		Collections.sort(oneTimePrices, Comparator.comparing(OneTimePrice::getMonthlyPrice));
		return oneTimePrices.get(0);
	}

	public static OneTimePrice getDefaultPrice(MenuItem item, String interval) {
		List<OneTimePrice> oneTimePrices = getPrices(item, interval);
		if (oneTimePrices == null || oneTimePrices.isEmpty()) {
			return null;
		}
		if (oneTimePrices.size() == 1) {
			return oneTimePrices.get(0);
		}
		Collections.sort(oneTimePrices, Comparator.comparing(OneTimePrice::getMonthlyPrice));
		return oneTimePrices.get(0);
	}

	public static OneTimePrice getPriceByInterval(MenuItem item, String interval, Integer duration) {
		if (StringUtils.isBlank(interval)) {
			return null;
		}
		List<OneTimePrice> oneTimePrices = getPrices(item, interval);
		if (oneTimePrices == null || oneTimePrices.isEmpty()) {
			return null;
		}
		if (duration != null) {
			for (OneTimePrice oneTimePrice : oneTimePrices) {
				if (oneTimePrice.getDurationAsInt() == duration.intValue()) {
					return oneTimePrice;
				}
			}
		}
		return oneTimePrices.get(0);
	}

	public static String getInterval(OneTimePrice oneTimePrice) {
		if (oneTimePrice == null) {
			return null;
		}

		return oneTimePrice.isSubscription() ? oneTimePrice.getDurationType().toLowerCase()
				: (isNotOneTime(oneTimePrice) ? OneTimePrice.FOREVER : OneTimePrice.ONE_TIME);
	}

	public static String getStripeInterval(OneTimePrice oneTimePrice) {
		return oneTimePrice == null || !oneTimePrice.isSubscription() ? OneTimePrice.ONE_TIME : oneTimePrice.getDurationType().toLowerCase();
	}

	public static boolean isSameIntervalPrice(OneTimePrice price, OneTimePrice otherPrice) {
		if (price == null || otherPrice == null) {
			return false;
		}
		return price.getDuration().equals(otherPrice.getDuration()) && price.isSubscription() == otherPrice.isSubscription();
	}

	public static void updateStripePriceProperties(TicketItem ticketItem, OneTimePrice oneTimePrice) {
		if (oneTimePrice == null) {
			return;
		}
		ticketItem.addProperty("product_id", ticketItem.getMenuItemId()); //$NON-NLS-1$
		ticketItem.addProperty("price_id", oneTimePrice.getPriceId()); //$NON-NLS-1$
		ticketItem.addProperty(OneTimePrice.PROP_TITLE, oneTimePrice.getTitle()); //$NON-NLS-1$
		String interval = MenuItemPriceUtil.getInterval(oneTimePrice);
		ticketItem.addProperty("price_interval", interval == null ? "" : interval.toLowerCase()); //$NON-NLS-1$
		ticketItem.addProperty("price_interval_count", oneTimePrice.getDuration()); //$NON-NLS-1$
		ticketItem.addProperty("subscription", String.valueOf(oneTimePrice.isSubscription())); //$NON-NLS-1$
		if (!oneTimePrice.isSubscription()) {
			ticketItem.addProperty("one_time_price_interval", oneTimePrice.getDurationType()); //$NON-NLS-1$
			ticketItem.addProperty("one_time_price_interval_count", oneTimePrice.getDuration()); //$NON-NLS-1$
		}
	}

}