package com.floreantpos.extension.dejavoo;

import java.io.IOException;
import java.io.InputStream;
import java.io.StringReader;
import java.io.UnsupportedEncodingException;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLConnection;
import java.net.URLDecoder;
import java.net.URLEncoder;
import java.text.NumberFormat;
import java.util.List;
import java.util.Locale;

import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jdom2.Document;
import org.jdom2.Element;
import org.jdom2.JDOMException;
import org.jdom2.input.SAXBuilder;

import com.floreantpos.PosException;
import com.floreantpos.PosLog;
import com.floreantpos.model.CardReader;
import com.floreantpos.model.DebitCardTransaction;
import com.floreantpos.model.PosTransaction;
import com.floreantpos.model.Terminal;
import com.floreantpos.model.Ticket;
import com.floreantpos.model.User;
import com.floreantpos.model.dao.BasePosTransactionDAO;
import com.floreantpos.model.dao.PosTransactionDAO;
import com.floreantpos.model.dao.TicketDAO;
import com.floreantpos.model.util.DataProvider;
import com.floreantpos.model.util.PaymentMethodJsonExtractor;
import com.floreantpos.ui.views.payment.CardProcessor;
import com.floreantpos.util.AESencrp;
import com.floreantpos.util.GlobalIdGenerator;
import com.floreantpos.util.NumberUtil;
import com.floreantpos.util.POSUtil;
import com.google.gson.JsonObject;

public class DejavooProcessor implements CardProcessor {
	private Log logger = LogFactory.getLog(DejavooProcessor.class);

	public static final String PRE_AUTHENTICATION = "Auth";
	public static final String TIP_ADJUST = "TipAdjust";
	public static final String RRETURN = "Return";
	public static final String VOID = "Void";
	public static final String SALE = "Sale";
	public static final String CAPTURE = "Capture";
	public static final String BATCH_CLOSE = "Settle";

	private static NumberFormat numberFormatter;

	public DejavooProcessor() {
		numberFormatter = NumberFormat.getInstance(Locale.US);
		numberFormatter.setGroupingUsed(false);
		numberFormatter.setMinimumFractionDigits(2);
		numberFormatter.setMaximumFractionDigits(2);
	}

	public void initTransaction(PosTransaction transaction) {
		if (StringUtils.isBlank(transaction.getId())) {
			transaction.setId(GlobalIdGenerator.generateGlobalId());
		}
	}

	@Override
	public void preAuth(PosTransaction transaction) throws Exception {
		InputStream inputStream = null;
		try {
			initTransaction(transaction);

			DejavooConnectionProperty connectionProperty = createOutletConnectionProperty(transaction);

			Ticket ticket = transaction.getTicket();
			String transactionType = DejavooProcessor.SALE;
			inputStream = sendTransactionRequest(transaction, ticket, transactionType, connectionProperty);
			String responseString = IOUtils.toString(inputStream);

			logger.debug("dejavoo response: " + responseString);

			populateResult(responseString, transaction, transactionType, connectionProperty);

			transaction.setAuthorizable(true);
			transaction.setCaptured(false);
			saveTransaction(transaction);
		} finally {
			if (inputStream != null) {
				inputStream.close();
			}
		}
	}

	@Override
	public void captureAuthAmount(PosTransaction transaction) throws Exception {
		InputStream inputStream = null;
		Double amount = transaction.getAmount();

		try {
			initTransaction(transaction);

			DejavooConnectionProperty connectionProperty = createOutletConnectionProperty(transaction);

			double netAmount = amount;
			try {
				netAmount = NumberUtil.parse(transaction.getProperty("Amount")).doubleValue();
			} catch (Exception e) {
			}

			transaction.setAmount(netAmount);

			Ticket ticket = transaction.getTicket();
			String transactionType = DejavooProcessor.TIP_ADJUST;
			inputStream = sendTransactionRequest(transaction, ticket, transactionType, connectionProperty);
			String responseString = IOUtils.toString(inputStream);

			logger.debug("dejavoo response: " + responseString);

			populateResult(responseString, transaction, transactionType, connectionProperty);

			transaction.setCaptured(true);
			saveTransaction(transaction);
		} finally {
			if (inputStream != null) {
				inputStream.close();
			}
			transaction.setAmount(amount);
		}
	}

	@Override
	public void chargeAmount(PosTransaction transaction) throws Exception {
		InputStream inputStream = null;
		try {
			initTransaction(transaction);
			DejavooConnectionProperty connectionProperty = createOutletConnectionProperty(transaction);

			Ticket ticket = transaction.getTicket();
			String transactionType = DejavooProcessor.SALE;
			inputStream = sendTransactionRequest(transaction, ticket, transactionType, connectionProperty);
			String responseString = IOUtils.toString(inputStream);

			logger.debug("dejavoo response: " + responseString);

			populateResult(responseString, transaction, transactionType, connectionProperty);
			transaction.setCaptured(true);
			saveTransaction(transaction);
		} finally {
			if (inputStream != null) {
				inputStream.close();
			}
		}
	}

	private String createRequestParam(Ticket ticket, PosTransaction transaction, String transactionType, DejavooConnectionProperty connectionProperty)
			throws UnsupportedEncodingException {

		String paymentType = getPaymentType(transaction);
		if (StringUtils.isBlank(paymentType)) {
			throw new PosException("Payment type is not selected");
		}
		String frequency = "OneTime";
		String invoiceNumber = transaction.getId();
		String refId = transaction.getId();

		String clerkID = String.valueOf(ticket.getOwner().getId());
		String tableNum = ticket.getTableNumbers() != null ? ticket.getTableNumbers().toString() : "";
		String ticketNum = ticket.getId() != null ? ticket.getId().toString() : "";
		String printReceipt = "No";
		String alertNo = String.valueOf(3);
		String signature = "No";

		double gratuityAmount = ticket.getGratuityAmount();
		double tipsAmount = 0;
		if (transaction.getTenderAmount() >= ticket.getTotalAmount()) {
			tipsAmount = gratuityAmount;
		}
		if (transaction.isAuthorizable()) {
			tipsAmount = gratuityAmount;
		}
		if (tipsAmount > 0) {
			signature = "Yes";
		}

		StringBuilder stringBuilder = new StringBuilder();
		stringBuilder.append("<request>");
		stringBuilder.append("<PaymentType>" + paymentType + "</PaymentType>");
		stringBuilder.append("<TransType>" + transactionType + "</TransType>");
		stringBuilder.append("<Amount>" + numberFormatter.format(transaction.getAmount()) + "</Amount>");
		stringBuilder.append("<Tip>" + numberFormatter.format(tipsAmount) + "</Tip>");
		stringBuilder.append("<InvNum>" + invoiceNumber + "</InvNum>");
		stringBuilder.append("<RefId>" + refId + "</RefId>");
		stringBuilder.append("<RegisterId>" + connectionProperty.registerID + "</RegisterId>");
		stringBuilder.append("<AuthKey>" + connectionProperty.authenticationKey + "</AuthKey>");
		stringBuilder.append("<ClerkId>" + clerkID + "</ClerkId>");
		if (TIP_ADJUST.equals(transactionType) || RRETURN.equals(transactionType)) {
			if (StringUtils.isNotEmpty(transaction.getProperty("AcntLast4"))) {
				stringBuilder.append("<AcntLast4>" + transaction.getProperty("AcntLast4") + "</AcntLast4>");
			}
		}
		else {
			stringBuilder.append("<Frequency>" + frequency + "</Frequency>");
			stringBuilder.append("<TicketNum>" + ticketNum + "</TicketNum>");
			stringBuilder.append("<TableNum>" + tableNum + "</TableNum>");
			stringBuilder.append("<PrintReceipt>" + printReceipt + "</PrintReceipt>");
			stringBuilder.append("<Alert>" + alertNo + "</Alert>");
			stringBuilder.append("<SigCapture>" + signature + "</SigCapture>");
			if (StringUtils.isNotEmpty(transaction.getCardAuthCode())) {
				stringBuilder.append("<AuthCode>" + transaction.getCardAuthCode() + "</AuthCode>");
			}
		}
		stringBuilder.append("</request>");
		String requstString = stringBuilder.toString();
		logger.debug("dejavoo request: " + requstString);
		return requstString;
	}

	private InputStream sendTransactionRequest(PosTransaction transaction, Ticket ticket, String transactionType, DejavooConnectionProperty connectionProperty)
			throws UnsupportedEncodingException, MalformedURLException, IOException {

		Terminal terminal = DataProvider.get().getCurrentTerminal();
		switch (transactionType) {
			case DejavooProcessor.CAPTURE:
			case DejavooProcessor.VOID:
			case DejavooProcessor.RRETURN:
			case DejavooProcessor.TIP_ADJUST:

				terminal = transaction.getTerminal();
				if (terminal == null) {
					terminal = DataProvider.get().getCurrentTerminal();
				}

				break;

			default:
				break;
		}

		String encodedRequest = createRequestParam(ticket, transaction, transactionType, connectionProperty);
		URLConnection urlConnection = createDejavooConnection(connectionProperty, encodedRequest);
		urlConnection.connect();
		return urlConnection.getInputStream();
	}

	private void populateResult(String resposseString, PosTransaction posTransaction, String transactionType, DejavooConnectionProperty connectionProperty)
			throws IOException, JDOMException {
		String resultCode;
		String responseMessage;
		String resultMessage;
		String authenticationCode;
		String serianlNumber;
		String extraData;
		String emvData;
		String authenticationId = "";
		String PNRef;

		PosLog.debug(DejavooProcessor.class, "found respose: " + resposseString);

		SAXBuilder saxBuilder = new SAXBuilder();
		Document doc = saxBuilder.build(new StringReader(resposseString));
		List<Element> elementList = doc.getRootElement().getChildren("response");

		resultCode = elementList.get(0).getChildText("ResultCode");
		responseMessage = elementList.get(0).getChildText("RespMSG");
		if (StringUtils.isNotEmpty(responseMessage)) {
			responseMessage = URLDecoder.decode(responseMessage, "utf-8");
		}
		PNRef = elementList.get(0).getChildText("PNRef");
		resultMessage = elementList.get(0).getChildText("Message");
		authenticationCode = elementList.get(0).getChildText("AuthCode");
		serianlNumber = elementList.get(0).getChildText("SN");
		extraData = elementList.get(0).getChildText("ExtData");
		emvData = elementList.get(0).getChildText("EMVData");

		double transactionAmount = posTransaction.getAmount();
		double tips = posTransaction.getTipsAmount();
		double fee = 0;
		double total = 0;
		int result;

		try {
			result = Integer.parseInt(resultCode);
		} catch (Exception e) {
			throw new PosException("Can not parse result.");
		}

		if (result != 0) {
			if (StringUtils.isNotBlank(responseMessage)) {
				throw new PosException(responseMessage);
			}
			else {
				throw new PosException(resultMessage);
			}
		}
		if (StringUtils.isNotEmpty(emvData)) {
			authenticationId = emvData.substring(4, 18);
			String[] emvDataArray = emvData.split(",");
			for (int i = 0; i < emvDataArray.length; i++) {
				String[] property = emvDataArray[i].split("=");
				if (property.length < 2) {
					continue;
				}
				posTransaction.addProperty(property[0], property[1]);
			}
		}

		String[] extraDateArray = extraData.split(",");
		for (int i = 0; i < extraDateArray.length; i++) {
			String[] property = extraDateArray[i].split("=");
			if (property.length < 2) {
				continue;
			}
			if (property[0].equals("CardType")) {
				posTransaction.setCardType(property[1]);
			}
			else if (property[0].equals("AcntLast4")) {
				posTransaction.setCardNumber(property[1]);
			}
			else if (property[0].equals("Amount")) {
				try {
					transactionAmount = Double.parseDouble(property[1]);
				} catch (Exception e) {
				}
			}
			else if (property[0].equals("Tip")) {
				try {
					tips = Double.parseDouble(property[1]);
				} catch (Exception e) {

				}
			}
			else if (property[0].equals("Fee")) {
				try {
					fee = Double.parseDouble(property[1]);
				} catch (Exception e) {

				}
			}
			else if (property[0].equals("TotalAmt")) {
				try {
					total = Double.parseDouble(property[1]);
				} catch (Exception e) {

				}
			}
			posTransaction.addProperty(property[0], property[1]);
		}

		Ticket ticket = posTransaction.getTicket();

		if (!DejavooProcessor.PRE_AUTHENTICATION.equals(transactionType) && !DejavooProcessor.VOID.equals(transactionType)) {
			TicketDAO.getInstance().loadFullTicket(ticket);

			double transactionFee = POSUtil.parseDouble(posTransaction.getProperty(PosTransaction.TRANSACTION_FEE, "0"));
			posTransaction.addExtraProperty(PosTransaction.TRANSACTION_FEE, String.valueOf(transactionFee + fee));
			if (tips > 0) {
				ticket.setGratuityAmount(tips);
			}
			ticket.setFeeAmount(ticket.getFeeAmount() + fee);
			ticket.calculatePrice();
		}
		posTransaction.setAmount(total);
		posTransaction.setTenderAmount(total);
		/*if ((transactionAmount + tips + fee) == (total)) {
			posTransaction.setTenderAmount(posTransaction.getTenderAmount() + tips + fee);
		}
		else {
			posTransaction.setTenderAmount(posTransaction.getTenderAmount() + fee);
		}*/
		posTransaction.setTipsAmount(tips);
		posTransaction.setCardTransactionId(posTransaction.getId());
		posTransaction.setCardReader(CardReader.CHIP.name());
		posTransaction.setCardAuthCode(authenticationCode);
		posTransaction.addProperty("ResultCode", resultCode);
		posTransaction.addProperty("ResponseMessage", responseMessage);
		posTransaction.addProperty("ResultMessage", resultMessage);
		posTransaction.addProperty("AuthenticationCode", authenticationCode);
		posTransaction.addProperty("SerialNumber", serianlNumber);
		posTransaction.addProperty("AuthenticationID", authenticationId);
		posTransaction.addProperty("PNRef", PNRef);

		setConnectionProperty(connectionProperty, posTransaction);
	}

	public void saveTransaction(PosTransaction transaction) {
		PosTransactionDAO transactionDAO = BasePosTransactionDAO.getInstance();
		PosTransaction posTransaction = transactionDAO.get(transaction.getId());
		if (posTransaction == null) {
			BasePosTransactionDAO.getInstance().save(transaction);
		}
		else {
			BasePosTransactionDAO.getInstance().update(transaction);
		}
	}

	@Override
	public void voidTransaction(PosTransaction transaction) throws Exception {
		InputStream inputStream = null;
		try {
			initTransaction(transaction);
			DejavooConnectionProperty connectionProperty = createOutletConnectionProperty(transaction);

			Ticket ticket = transaction.getTicket();
			String transactionType = DejavooProcessor.VOID;
			inputStream = sendTransactionRequest(transaction, ticket, transactionType, connectionProperty);
			String responseString = IOUtils.toString(inputStream);

			logger.debug("dejavoo response: " + responseString);

			populateResult(responseString, transaction, transactionType, connectionProperty);
			transaction.setAuthorizable(false);
			transaction.setVoided(true);
			ticket.setFeeAmount(ticket.getFeeAmount() - transaction.getFeeAmount());
			ticket.calculatePrice();
			BasePosTransactionDAO.getInstance().saveOrUpdate(transaction);
		} finally {
			if (inputStream != null) {
				inputStream.close();
			}
		}

	}

	public void refundTransaction(PosTransaction transaction, double refundAmount) throws Exception {
		InputStream inputStream = null;
		try {
			initTransaction(transaction);
			DejavooConnectionProperty connectionProperty = createOutletConnectionProperty(transaction);

			Ticket ticket = transaction.getTicket();
			String transactionType = DejavooProcessor.RRETURN;
			inputStream = sendTransactionRequest(transaction, ticket, transactionType, connectionProperty);
			String responseString = IOUtils.toString(inputStream);

			logger.debug("dejavoo response: " + responseString);

			populateResult(responseString, transaction, transactionType, connectionProperty);
			transaction.setRefunded(true);
		} finally {
			if (inputStream != null) {
				inputStream.close();
			}
		}

	}

	@Override
	public String getCardInformationForReceipt(PosTransaction transaction) {
		return null;
	}

	@Override
	public void cancelTransaction() {

	}

	public boolean supportTipsAdjustMent() {
		return true;
	}

	public void adjustTips(PosTransaction transaction) throws Exception {
		InputStream inputStream = null;
		try {
			initTransaction(transaction);
			DejavooConnectionProperty connectionProperty = createOutletConnectionProperty(transaction);

			Ticket ticket = transaction.getTicket();
			String transactionType = DejavooProcessor.TIP_ADJUST;
			inputStream = sendTransactionRequest(transaction, ticket, transactionType, connectionProperty);
			String responseString = IOUtils.toString(inputStream);

			logger.debug("dejavoo response: " + responseString);

			populateResult(responseString, transaction, transactionType, connectionProperty);
			transaction.setCaptured(true);
			saveTransaction(transaction);
		} finally {
			if (inputStream != null) {
				inputStream.close();
			}
		}
	}

	//	@Override
	//	public void closeBatch(Terminal terminal) {
	//		closeBatch(terminal, Currents);
	//	}

	@Override
	public void closeBatch(Terminal terminal, User performer) {
		try {
			DejavooConnectionProperty connectionProperty = createOutletConnectionProperty();
			if (connectionProperty == null) {
				throw new PosException("Terminal " + terminal.getName() + " not configured to use Dejavoo.");
			}

			StringBuilder stringBuilder = new StringBuilder();
			stringBuilder.append("<request>");
			stringBuilder.append("<TransType>Settle</TransType>");
			stringBuilder.append("<Param>Force</Param>");

			stringBuilder.append("<RegisterId>" + connectionProperty.registerID + "</RegisterId>");
			stringBuilder.append("<AuthKey>" + connectionProperty.authenticationKey + "</AuthKey>");

			stringBuilder.append("</request>");
			String requstString = stringBuilder.toString();
			logger.debug("dejavoo request: " + requstString);
			URLConnection urlConnection = createDejavooConnection(connectionProperty, requstString);
			urlConnection.connect();
			try (InputStream inputStream = urlConnection.getInputStream()) {
				String responseString = IOUtils.toString(inputStream);

				//				System.out.println(responseString);

				SAXBuilder saxBuilder = new SAXBuilder();
				Document doc = saxBuilder.build(new StringReader(responseString));

				String respMessage = doc.getRootElement().getChild("response").getChildText("RespMSG");
				String message = doc.getRootElement().getChild("response").getChildText("Message");
				if (StringUtils.isNotBlank(respMessage)) {
					throw new PosException(respMessage);
				}

				String extraData = doc.getRootElement().getChild("response").getChildText("ExtData");
				String[] extraDateArray = extraData.split(",");
				for (int i = 0; i < extraDateArray.length; i++) {
					String[] property = extraDateArray[i].split("=");
					if (property.length < 2) {
						continue;
					}
					if (property[0].trim().equals("Rsp")) {
						throw new PosException(message + " (" + property[1] + ")");
					}
				}
			}
		} catch (PosException e) {
			throw e;
		} catch (Exception e) {
			throw new PosException("Failed (" + e.getMessage() + ")");
		}

	}

	private String getPaymentType(PosTransaction transaction) {
		try {
			Terminal terminal = DataProvider.get().getCurrentTerminal();

			if (!terminal.getBooleanProperty(DejavooConstants.ENABLE_PAYMENT_SELECTION, false)) {
				if (transaction instanceof DebitCardTransaction) {
					return "Debit";
				}
				return "Credit";
			}

			DejavooConnectionProperty connectionProperty = createOutletConnectionProperty();
			if (connectionProperty == null) {
				throw new PosException("Terminal " + terminal.getName() + " not configured to use Dejavoo.");
			}

			StringBuilder stringBuilder = new StringBuilder();
			stringBuilder.append("<request>");
			stringBuilder.append("<RegisterId>" + connectionProperty.registerID + "</RegisterId>");
			stringBuilder.append("<AuthKey>" + connectionProperty.authenticationKey + "</AuthKey>");
			stringBuilder.append("<UserChoice title=\"Choose Payment Type:\" count=\"3\">");
			stringBuilder.append("<Choice>Credit</Choice>");
			stringBuilder.append("<Choice>Debit</Choice>");
			stringBuilder.append("<Choice>EBT</Choice>");
			stringBuilder.append("</UserChoice>");
			stringBuilder.append("</request>");
			String requstString = stringBuilder.toString();
			logger.debug("dejavoo request: " + requstString);

			URLConnection urlConnection = createDejavooConnection(connectionProperty, requstString);
			urlConnection.connect();
			try (InputStream inputStream = urlConnection.getInputStream()) {
				String responseString = IOUtils.toString(inputStream);

				SAXBuilder saxBuilder = new SAXBuilder();
				Document doc = saxBuilder.build(new StringReader(responseString));

				String message = doc.getRootElement().getChild("response").getChildText("Message");
				if ("Cancel".equalsIgnoreCase(message)) {
					throw new PosException("Payment type selection canceled");
				}
				return doc.getRootElement().getChild("response").getChildText("UserChoice");
			}
		} catch (PosException e) {
			throw e;
		} catch (Exception e) {
			throw new PosException("Failed (" + e.getMessage() + ")");
		}
	}

	private URLConnection createDejavooConnection(DejavooConnectionProperty connectionProperty, String requstString)
			throws UnsupportedEncodingException, MalformedURLException, IOException {
		String requestString = URLEncoder.encode(requstString, "utf-8");

		String urlString = connectionProperty.protocol.toLowerCase() + "://" + connectionProperty.terminalIpAddress + ":" + connectionProperty.cgiPort
				+ "/spin/cgi.html?TerminalTransaction=";

		URL url = new URL(urlString + requestString);
		URLConnection urlConnection = url.openConnection();
		urlConnection.setConnectTimeout(30 * 1000);
		urlConnection.setReadTimeout(2 * 60 * 1000);
		return urlConnection;
	}

	private DejavooConnectionProperty createOutletConnectionProperty(PosTransaction transaction) {
		DejavooConnectionProperty connectionProperty = createTransactionConnectionProperty(transaction);
		if (connectionProperty == null) {
			connectionProperty = createOutletConnectionProperty();
		}
		return connectionProperty;
	}

	private DejavooConnectionProperty createOutletConnectionProperty() {
		JsonObject paymentConfiguration = PaymentMethodJsonExtractor.getPaymentConfiguration(DejavooPaymentPlugin.ID);
		DejavooConnectionProperty connectionProperty = new DejavooConnectionProperty();
		connectionProperty.protocol = paymentConfiguration.get(DejavooConstants.PROTOCOL).getAsString();
		connectionProperty.terminalIpAddress = paymentConfiguration.get(DejavooConstants.TERMINAL_IP_ADDRESS).getAsString();
		connectionProperty.cgiPort = paymentConfiguration.get(DejavooConstants.CGI_PORT).getAsString();
		connectionProperty.authenticationKey = paymentConfiguration.get(DejavooConstants.AUTHENTICATION_KEY).getAsString();
		connectionProperty.registerID = paymentConfiguration.get(DejavooConstants.REGISTER_ID).getAsString();

		if (StringUtils.isBlank(connectionProperty.protocol) || StringUtils.isBlank(connectionProperty.terminalIpAddress)
				|| StringUtils.isBlank(connectionProperty.cgiPort) || StringUtils.isBlank(connectionProperty.authenticationKey)
				|| StringUtils.isBlank(connectionProperty.registerID)) {
			return null;
		}

		return connectionProperty;
	}

	private DejavooConnectionProperty createTransactionConnectionProperty(PosTransaction transaction) {
		if (transaction == null) {
			return null;
		}

		if (!transaction.hasProperty(DejavooConstants.AUTHENTICATION_KEY) || !transaction.hasProperty(DejavooConstants.REGISTER_ID)) {
			return null;
		}

		DejavooConnectionProperty connectionProperty = new DejavooConnectionProperty();
		connectionProperty.protocol = transaction.getProperty(DejavooConstants.PROTOCOL);
		connectionProperty.terminalIpAddress = transaction.getProperty(DejavooConstants.TERMINAL_IP_ADDRESS);
		connectionProperty.cgiPort = transaction.getProperty(DejavooConstants.CGI_PORT);
		connectionProperty.registerID = transaction.getProperty(DejavooConstants.REGISTER_ID);

		try {
			connectionProperty.authenticationKey = AESencrp.decrypt(transaction.getProperty(DejavooConstants.AUTHENTICATION_KEY));
		} catch (Exception e) {
			connectionProperty.authenticationKey = transaction.getProperty(DejavooConstants.AUTHENTICATION_KEY);
			PosLog.error(getClass(), e);
		}

		if (StringUtils.isBlank(connectionProperty.registerID) || StringUtils.isBlank(connectionProperty.authenticationKey)) {
			return null;
		}

		return connectionProperty;
	}

	private void setConnectionProperty(DejavooConnectionProperty connectionProp, PosTransaction posTransaction) {
		posTransaction.addProperty(DejavooConstants.PROTOCOL, connectionProp.protocol);
		posTransaction.addProperty(DejavooConstants.TERMINAL_IP_ADDRESS, connectionProp.terminalIpAddress);
		posTransaction.addProperty(DejavooConstants.CGI_PORT, connectionProp.cgiPort);
		posTransaction.addProperty(DejavooConstants.REGISTER_ID, connectionProp.registerID);

		try {
			posTransaction.addProperty(DejavooConstants.AUTHENTICATION_KEY, AESencrp.encrypt(connectionProp.authenticationKey));
		} catch (Exception e) {
			posTransaction.addProperty(DejavooConstants.AUTHENTICATION_KEY, connectionProp.authenticationKey);
			PosLog.error(getClass(), e);
		}
	}

	private class DejavooConnectionProperty {
		private String protocol;
		private String terminalIpAddress;
		private String cgiPort;
		private String authenticationKey;
		private String registerID;
	}
}
