package com.floreantpos.model.dao;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.apache.commons.beanutils.PropertyUtils;
import org.apache.commons.lang.StringUtils;
import org.hibernate.Criteria;
import org.hibernate.HibernateException;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.hibernate.criterion.Criterion;
import org.hibernate.criterion.Projections;
import org.hibernate.criterion.Restrictions;

import com.floreantpos.Messages;
import com.floreantpos.PosException;
import com.floreantpos.PosLog;
import com.floreantpos.model.MenuItem;
import com.floreantpos.model.MenuModifier;
import com.floreantpos.model.Tax;
import com.floreantpos.model.TaxGroup;
import com.floreantpos.util.NameBaseIdGenerator;
import com.orocube.rest.service.server.BaseDataServiceDao;

public class TaxGroupDAO extends BaseTaxGroupDAO {

	/**
	 * Default constructor.  Can be used in place of getInstance()
	 */
	public TaxGroupDAO() {
	}

	public void saveOrUpdate(TaxGroup taxGroup) throws HibernateException {
		if (StringUtils.isBlank(taxGroup.getId())) {
			taxGroup.setId(NameBaseIdGenerator.generateId(taxGroup.getName()));
		}
		super.saveOrUpdate(taxGroup);
	}

	@Override
	protected void delete(Object obj, Session session) {
		TaxGroup taxGroup = (TaxGroup) obj;
		if (taxGroup == null) {
			throw new PosException(Messages.getString("TaxGroupDAO.0")); //$NON-NLS-1$
		}

		checkTaxGroupCanbeDeleted(session, taxGroup);

		taxGroup.setDeleted(true);
		update(taxGroup, session);
	}

	private void checkTaxGroupCanbeDeleted(Session session, TaxGroup taxGroup) {
		String details = ""; //$NON-NLS-1$

		List<MenuItem> menuItemList = MenuItemDAO.getInstance().findMenuItemByTaxGroupId(taxGroup.getId());
		if (menuItemList != null && !menuItemList.isEmpty()) {
			if (StringUtils.isNotBlank(details)) {
				details += "\n"; //$NON-NLS-1$
			}
			details += Messages.getString("TaxGroupDAO.1") + " " + taxGroup.getName() + Messages.getString("TaxGroupDAO.3"); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
			int count = 1;
			for (MenuItem menuItem : menuItemList) {
				details += "\n" + count + ". " + menuItem.getName(); //$NON-NLS-1$ //$NON-NLS-2$
				count++;
			}
		}
		List<MenuModifier> modifierList = MenuModifierDAO.getInstance().findMenuModifierByTaxGroupId(taxGroup.getId());
		if (modifierList != null && !modifierList.isEmpty()) {
			if (StringUtils.isNotBlank(details)) {
				details += "\n\n"; //$NON-NLS-1$
			}
			details += Messages.getString("TaxGroupDAO.1") + " " + taxGroup.getName() + Messages.getString("TaxGroupDAO.6"); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
			int count = 1;
			for (MenuModifier menuModifier : modifierList) {
				details += "\n" + count + ". " + menuModifier.getName(); //$NON-NLS-1$ //$NON-NLS-2$
				count++;
			}
		}
		if (StringUtils.isNotBlank(details)) {
			throw new PosException(Messages.getString("TaxGroupDAO.1") + " <b>" + taxGroup.getName() + "</b> " + Messages.getString("TaxGroupDAO.10"), details); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$
		}

		List<Tax> taxes = taxGroup.getTaxes();
		if (taxes != null && !taxes.isEmpty()) {
			details = Messages.getString("TaxGroupDAO.1") + " " + taxGroup.getName() + Messages.getString("TaxGroupDAO.13"); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
			int count = 1;
			for (Tax tax : taxes) {
				details += "\n" + count + ". " + tax.getName(); //$NON-NLS-1$ //$NON-NLS-2$
				count++;
			}
		}
		if (StringUtils.isNotBlank(details)) {
			throw new PosException(Messages.getString("TaxGroupDAO.1") + " <b>" + taxGroup.getName() + "</b> " + Messages.getString("TaxGroupDAO.17"), details); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$
		}
	}

	@Override
	public List<TaxGroup> findAll() {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			addDeletedFilter(criteria);
			return criteria.list();
		}
	}

	public boolean nameExists(TaxGroup taxGroup, String name) {
		Session session = null;
		try {
			session = getSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.eq(TaxGroup.PROP_NAME, name).ignoreCase());
			if (taxGroup.getId() != null)
				criteria.add(Restrictions.ne(TaxGroup.PROP_ID, taxGroup.getId()));

			criteria.setProjection(Projections.rowCount());
			Number rowCount = (Number) criteria.uniqueResult();

			if (rowCount == null)
				return false;

			return (rowCount.intValue() > 0) ? true : false;

		} finally {
			if (session != null) {
				closeSession(session);
			}
		}
	}

	public List<TaxGroup> findAllUnSyncTaxGroup() {
		Session session = null;
		try {
			session = createNewSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			Criterion falseCriteria = Restrictions.eq(TaxGroup.PROP_CLOUD_SYNCED, Boolean.FALSE);
			Criterion nullCriteria = Restrictions.isNull(TaxGroup.PROP_CLOUD_SYNCED);
			criteria.add(Restrictions.or(falseCriteria, nullCriteria));
			return criteria.list();
		} finally {
			if (session != null) {
				closeSession(session);
			}
		}
	}

	@Override
	protected Serializable save(Object obj, Session s) {
		updateTime(obj);
		return super.save(obj, s);
	}

	@Override
	protected void update(Object obj, Session s) {
		updateTime(obj);
		super.update(obj, s);
	}

	@Override
	protected void saveOrUpdate(Object obj, Session s) {
		updateTime(obj);
		super.saveOrUpdate(obj, s);
	}

	public void saveOrUpdateTaxGroups(List<TaxGroup> dataList, boolean updateLastUpdateTime, boolean updateSyncTime) throws Exception {
		if (dataList == null)
			return;

		Transaction tx = null;
		Session session = null;
		try {
			session = createNewSession();
			tx = session.beginTransaction();
			for (Iterator iterator = dataList.iterator(); iterator.hasNext();) {
				TaxGroup item = (TaxGroup) iterator.next();
				List<Tax> existingsTax = new ArrayList<>();
				if (item.getTaxes() != null) {
					for (Tax tax : item.getTaxes()) {
						Tax existingTax = TaxDAO.getInstance().get(tax.getId(), tax.getOutletId());
						if (existingsTax == null)
							continue;
						existingsTax.add(existingTax);
					}
				}

				item.setTaxes(existingsTax);
				TaxGroup existingItem = get(item.getId(), item.getOutletId());
				if (existingItem != null) {
					if (!BaseDataServiceDao.get().shouldSave(item.getLastUpdateTime(), existingItem.getLastUpdateTime())) {
						PosLog.info(getClass(), item.getName() + " already updated"); //$NON-NLS-1$
						continue;
					}
					//					final String id = existingItem.getId();
					long version = existingItem.getVersion();
					PropertyUtils.copyProperties(existingItem, item);
					//					existingItem.setId(id);
					existingItem.setVersion(version);
					existingItem.setUpdateLastUpdateTime(updateLastUpdateTime);
					existingItem.setUpdateSyncTime(updateSyncTime);
					update(existingItem, session);
				}
				else {
					item.setUpdateLastUpdateTime(updateLastUpdateTime);
					item.setUpdateSyncTime(updateSyncTime);
					save(item, session);
				}
			}
			tx.commit();
		} catch (Exception e) {
			tx.rollback();
			throw e;
		} finally {
			closeSession(session);
		}
	}

	public TaxGroup get(String taxGroupId, String outletId) {
		return get(new TaxGroup(taxGroupId, outletId));
	}

	public TaxGroup findById(String taxGroupId) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.eq(TaxGroup.PROP_ID, taxGroupId));
			criteria.setMaxResults(1);
			return (TaxGroup) criteria.uniqueResult();
		}
	}

	@SuppressWarnings("unchecked")
	public List<TaxGroup> findByOutletId(String outletId) {
		if (StringUtils.isBlank(outletId)) {
			return null;
		}
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.eq(TaxGroup.PROP_OUTLET_ID, outletId));
			addDeletedFilter(criteria);
			return criteria.list();
		}
	}
}