package com.floreantpos.model.util;

import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.lang.StringUtils;

import com.floreantpos.PosLog;
import com.floreantpos.model.Customer;
import com.floreantpos.model.MenuItem;
import com.floreantpos.model.OrderHistory;
import com.floreantpos.model.Ticket;
import com.floreantpos.model.TicketItem;
import com.floreantpos.model.dao.CustomerDAO;
import com.floreantpos.model.dao.InvoiceDAO;
import com.floreantpos.util.CurrencyUtil;
import com.floreantpos.util.NumberUtil;
import com.stripe.model.Charge;
import com.stripe.model.Charge.PaymentMethodDetails;
import com.stripe.model.Charge.PaymentMethodDetails.Card;
import com.stripe.model.Invoice;
import com.stripe.model.Invoice.DiscountAmount;
import com.stripe.model.InvoiceLineItem;
import com.stripe.model.Subscription;

public class InvoiceUtil {

	public static Map<String, Object> buildStripeInvoicePaymentContextMap(Invoice invoice, Charge charge, Ticket ticket) {
		Map<String, Object> invoiceMap = new HashMap<String, Object>();

		invoiceMap.put("invoiceNo", invoice.getNumber()); //$NON-NLS-1$
		if (StringUtils.isNotBlank(ticket.getId())) {
			invoiceMap.put("orderId", ticket.getId()); //$NON-NLS-1$
		}
		invoiceMap.put("issueDate", DateUtil.formatDate(new Date((charge == null ? invoice.getCreated() : charge.getCreated()) * 1000L))); //$NON-NLS-1$
		invoiceMap.put("customerName", invoice.getCustomerName()); //$NON-NLS-1$
		invoiceMap.put("customerEmail", invoice.getCustomerEmail()); //$NON-NLS-1$
		invoiceMap.put("customerPhone", ticket.getCustomerMobileNo()); //$NON-NLS-1$1

		String currencySymbol = "$"; //$NON-NLS-1$1
		double chargeAmount = (charge == null ? invoice.getAmountPaid().doubleValue() : charge.getAmount().doubleValue()) / 100D;

		invoiceMap.put("cardPayment", true); //$NON-NLS-1$
		invoiceMap.put("cardAmountNumber", NumberUtil.round(chargeAmount)); //$NON-NLS-1$
		invoiceMap.put("cardAmount", currencySymbol + NumberUtil.formatNumber(chargeAmount)); //$NON-NLS-1$
		invoiceMap.put("paidAmount", currencySymbol + NumberUtil.formatNumber(chargeAmount)); //$NON-NLS-1$
		invoiceMap.put("paymentInformation", getCardPaymentInformation(charge)); //$NON-NLS-1$
		invoiceMap.put("paymentStatus", "Paid"); //$NON-NLS-1$ //$NON-NLS-2$
		invoiceMap.put("paymentType", "Credit card"); //$NON-NLS-1$ //$NON-NLS-2$
		invoiceMap.put("orderItemsFromTemplate", getItemsInformation(invoice, ticket, currencySymbol)); //$NON-NLS-1$
		invoiceMap.put("paid", invoice.getPaid()); //$NON-NLS-1$
		loadResellerPointsIfExists(ticket, invoiceMap);
		double totalAmount = invoice.getTotal().doubleValue() / 100D;
		double paidAmount = invoice.getPaid() ? totalAmount : (invoice.getAmountPaid().doubleValue() / 100D);
		double discountAmount = 0D;
		List<DiscountAmount> discountAmounts = invoice.getTotalDiscountAmounts();
		if (discountAmounts != null && discountAmounts.size() > 0) {
			for (DiscountAmount dAmount : discountAmounts) {
				if (dAmount.getAmount() == null) {
					continue;
				}
				double amount = dAmount.getAmount().doubleValue() / 100;
				discountAmount += amount;
			}
		}
		double dueAmount = invoice.getPaid() ? 0D : (totalAmount - paidAmount);
		invoiceMap.put("dueAmount", currencySymbol + NumberUtil.format(dueAmount)); //$NON-NLS-1$
		invoiceMap.put("subTotal", currencySymbol + NumberUtil.format(totalAmount + discountAmount)); //$NON-NLS-1$
		loadDiscountInfo(ticket, invoiceMap, currencySymbol, discountAmount);
		invoiceMap.put("total", currencySymbol + NumberUtil.format(totalAmount)); //$NON-NLS-1$
		invoiceMap.put("amountDue", currencySymbol + NumberUtil.format(dueAmount)); //$NON-NLS-1$
		return invoiceMap;
	}

	public static Map<String, Object> buildWireTransferInvoiceContextMap(OrderHistory orderHistory, Ticket ticket) {
		String invoiceNo = ticket.getProperty("invoiceNo"); //$NON-NLS-1$
		if (orderHistory != null && StringUtils.isBlank(invoiceNo)) {
			return null;
		}
		Map<String, Object> invoiceMap = new HashMap<String, Object>();
		String currencySymbol = CurrencyUtil.getCurrencySymbol();
		if (!invoiceNo.startsWith("WTIN") && StringUtils.isNotBlank(ticket.getProperty("invoiceId"))) { //$NON-NLS-1$ //$NON-NLS-2$
			com.floreantpos.model.Invoice invoice = InvoiceDAO.getInstance().findByInvoiceId(ticket.getProperty("invoiceId")); //$NON-NLS-1$
			if (invoice != null) {
				invoiceNo = invoice.getInvoiceNumberDisplayString();
			}
		}

		invoiceMap.put("invoiceNo", invoiceNo); //$NON-NLS-1$
		ticket.addProperty("invoiceNumberDisplay", invoiceNo);
		String paymentDate = orderHistory.getProperty("paymentDate"); //$NON-NLS-1$
		if (StringUtils.isNotBlank(paymentDate)) {
			invoiceMap.put("issueDate", paymentDate); //$NON-NLS-1$
		}
		else {
			invoiceMap.put("issueDate", DateUtil.formatDate(ticket.getCreateDate())); //$NON-NLS-1$
		}
		if (StringUtils.isNotBlank(ticket.getId())) {
			invoiceMap.put("orderId", ticket.getId()); //$NON-NLS-1$
		}
		loadCustomerInformation(ticket, invoiceMap);
		invoiceMap.put("dueAmount", currencySymbol + NumberUtil.format(ticket.getDueAmount())); //$NON-NLS-1$
		invoiceMap.put("bankPayment", true); //$NON-NLS-1$

		String paymentInfo = getBankPaymentInformation(orderHistory);
		if (StringUtils.isNotBlank(paymentInfo)) {
			invoiceMap.put("paymentInformation", paymentInfo); //$NON-NLS-1$
		}
		loadResellerPointsIfExists(ticket, invoiceMap);
		invoiceMap.put("paid", ticket.isPaid()); //$NON-NLS-1$
		if (!ticket.isPaid()) {
			invoiceMap.put("paymentVerificationStage", true); //$NON-NLS-1$
		}
		invoiceMap.put("paidAmount", currencySymbol + NumberUtil.format(ticket.getPaidAmount())); //$NON-NLS-1$
		invoiceMap.put("orderItemsFromTemplate", getItemsInformation(ticket, currencySymbol)); //$NON-NLS-1$
		invoiceMap.put("subTotal", currencySymbol + NumberUtil.format(ticket.getSubtotalAmount())); //$NON-NLS-1$
		loadDiscountInfo(ticket, invoiceMap, currencySymbol);
		invoiceMap.put("total", currencySymbol + NumberUtil.format(ticket.getTotalAmount())); //$NON-NLS-1$
		invoiceMap.put("amountDue", currencySymbol + NumberUtil.format(ticket.getDueAmount())); //$NON-NLS-1$
		invoiceMap.put("hasDue", ticket.getDueAmount() > 0); //$NON-NLS-1$
		return invoiceMap;
	}

	private static void loadCustomerInformation(Ticket ticket, Map<String, Object> invoiceMap) {
		if (ticket == null) {
			return;
		}
		invoiceMap.put("customerName", ticket.getCustomerName()); //$NON-NLS-1$
		invoiceMap.put("customerEmail", ticket.getCustomerEmail()); //$NON-NLS-1$
		invoiceMap.put("customerPhone", ticket.getCustomerMobileNo()); //$NON-NLS-1$

		String customerCountry = ticket.getProperty(Ticket.JSON_PROP_CUSTOMER_COUNTRY);
		String customerState = ticket.getProperty(Ticket.JSON_PROP_CUSTOMER_STATE);
		String customerCity = ticket.getProperty(Ticket.JSON_PROP_CUSTOMER_CITY);
		String customerZip = ticket.getProperty(Ticket.JSON_PROP_CUSTOMER_ZIP);
		String businesAddressLine1 = null;
		String businesAddressLine2 = ""; //$NON-NLS-1$
		businesAddressLine1 = ticket.getProperty(Ticket.JSON_PROP_CUSTOMER_ADDRESS); //$NON-NLS-1$
		if (businesAddressLine1 == null) {
			businesAddressLine1 = ""; //$NON-NLS-1$
		}
		businesAddressLine2 += StringUtils.isBlank(customerCity) ? "" : " " + customerCity; //$NON-NLS-1$
		businesAddressLine2 += StringUtils.isBlank(customerState) ? "" : " " + customerState; //$NON-NLS-1$
		businesAddressLine2 += StringUtils.isBlank(customerZip) ? "" : " " + customerZip; //$NON-NLS-1$
		invoiceMap.put("customerAddressLine1", businesAddressLine1); //$NON-NLS-1$
		invoiceMap.put("customerAddressLine2", businesAddressLine2); //$NON-NLS-1$
		invoiceMap.put("customerCountry", StringUtils.isBlank(customerCountry) ? "" : " " + customerCountry); //$NON-NLS-1$
	}

	public static void loadResellerPointsIfExists(Ticket ticket, Map<String, Object> invoiceMap) {
		if (!ticket.isPaid() || StringUtils.isBlank(ticket.getResellerId()) || ticket.getCustomerId().equals(ticket.getResellerId())) {
			return;
		}
		Customer reseller = CustomerDAO.getInstance().findById(ticket.getResellerId());
		if (reseller == null) {
			return;
		}
		invoiceMap.put("resellerPoints", ticket.getResellerPoints()); //$NON-NLS-1$
		invoiceMap.put("totalResellerPoints", reseller.getResellerPoints()); //$NON-NLS-1$
	}

	public static String getItemsInformation(Invoice invoice, Ticket ticket, String currencySymbol) {
		if (invoice == null || invoice.getLines() == null) {
			return ""; //$NON-NLS-1$
		}
		Map<String, String> itemNameMap = new HashMap<>();
		Map<String, String> intervalMap = new HashMap<>();
		for (TicketItem ticketItem : ticket.getTicketItems()) {
			MenuItem menuItem = ticketItem.getMenuItem();
			String name = ticketItem.getNameDisplay(ticketItem.getName(), false, false, false);
			if (menuItem != null && menuItem.isVariant() && menuItem.getParentMenuItemId() != null) {
				itemNameMap.put(menuItem.getParentMenuItemId(), name);
				continue;
			}
			itemNameMap.put(ticketItem.getMenuItemId(), name);
			intervalMap.put(ticketItem.getMenuItemId(), ticketItem.getShortInterval());
		}
		List<InvoiceLineItem> orderItems = invoice.getLines().getData();
		final StringBuilder itemsBuilder = new StringBuilder(""); //$NON-NLS-1$
		if (orderItems != null && orderItems.size() > 0) {
			int count = 1;
			for (InvoiceLineItem orderItem : orderItems) {
				double quantity = orderItem.getQuantity().doubleValue();
				if (quantity <= 0) {
					continue;
				}
				double totalAmount = orderItem.getAmount().doubleValue() / 100D;
				if (totalAmount <= 0) {
					continue;
				}
				StringBuilder itemRow = new StringBuilder("<tr style=\"border-right:none !important;border-left: none !important;\">"); //$NON-NLS-1$
				itemRow.append("<td style=\"width: 70px;border-right: none !important;border-left: none !important;\">").append(String.valueOf(count)) //$NON-NLS-1$
						.append("</td>"); //$NON-NLS-1$
				String productName = orderItem.getDescription();
				String shortInterval = "";
				if (orderItem.getPrice() != null) {
					String product = orderItem.getPrice().getProduct();
					String name = itemNameMap.get(product);
					if (StringUtils.isNotBlank(name)) {
						productName = name;
					}
					shortInterval = intervalMap.get(product);
				}
				if (shortInterval == null) {
					shortInterval = ""; //$NON-NLS-1$
				}
				double subtotalAmount = totalAmount / quantity;
				itemRow.append("<td style=\"width: 358px;border-right: none !important;border-left: none !important;\">") //$NON-NLS-1$
						.append(productName).append("</td>"); //$NON-NLS-1$
				itemRow.append("<td style=\"width: 100px;border-right: none !important;border-left: none !important;\">") //$NON-NLS-1$
						.append(NumberUtil.formatNumberIfNeeded(quantity)).append("</td>"); //$NON-NLS-1$
				itemRow.append("<td style=\"width: 120px;text-align: right;border-right: none !important;border-left: none !important;\">") //$NON-NLS-1$
						.append(currencySymbol + NumberUtil.format(subtotalAmount) + shortInterval).append("</td>"); //$NON-NLS-1$
				itemRow.append("<td style=\"width: 200px;text-align: right;border-right: none !important;border-left: none !important;\">") //$NON-NLS-1$
						.append(currencySymbol + NumberUtil.format(totalAmount)).append("</td>"); //$NON-NLS-1$
				itemRow.append("</tr>"); //$NON-NLS-1$

				itemsBuilder.append(itemRow.toString());
				count++;
			}
		}
		return itemsBuilder.toString();
	}

	public static String getItemsInformation(Ticket ticket, String currencySymbol) {
		List<TicketItem> orderItems = ticket.getTicketItems();
		final StringBuilder itemsBuilder = new StringBuilder(""); //$NON-NLS-1$
		if (orderItems != null && orderItems.size() > 0) {
			int count = 1;
			for (TicketItem orderItem : orderItems) {
				StringBuilder itemRow = new StringBuilder("<tr style=\"border-right:none !important;border-left: none !important;\">"); //$NON-NLS-1$
				itemRow.append("<td style=\"width: 70px;border-right: none !important;border-left: none !important;\">").append(String.valueOf(count)) //$NON-NLS-1$
						.append("</td>"); //$NON-NLS-1$
				itemRow.append("<td style=\"width: 358px;border-right: none !important;border-left: none !important;\">") //$NON-NLS-1$
						.append(orderItem.getNameDisplay(orderItem.getName(), false, false, false)).append("</td>"); //$NON-NLS-1$
				itemRow.append("<td style=\"width: 100px;border-right: none !important;border-left: none !important;\">") //$NON-NLS-1$
						.append(NumberUtil.formatNumberIfNeeded(orderItem.getQuantity())).append("</td>"); //$NON-NLS-1$
				itemRow.append("<td style=\"width: 120px;text-align: right;border-right: none !important;border-left: none !important;\">") //$NON-NLS-1$
						.append(currencySymbol + NumberUtil.format(orderItem.getUnitPrice()) + orderItem.getShortInterval()).append("</td>"); //$NON-NLS-1$
				itemRow.append("<td style=\"width: 200px;text-align: right;border-right: none !important;border-left: none !important;\">") //$NON-NLS-1$
						.append(currencySymbol + NumberUtil.format(orderItem.getTotalAmount())).append("</td>"); //$NON-NLS-1$
				itemRow.append("</tr>"); //$NON-NLS-1$

				itemsBuilder.append(itemRow.toString());
				count++;
			}
		}
		return itemsBuilder.toString();
	}

	private static String getBankPaymentInformation(OrderHistory orderHistory) {
		if (orderHistory == null) {
			return null;
		}
		String br = "<br>"; //$NON-NLS-1$
		String paymentInfo = ""; //$NON-NLS-1$
		try {
			String paymentDate = orderHistory.getProperty("paymentDate"); //$NON-NLS-1$
			if (StringUtils.isNotBlank(paymentDate)) {
				paymentInfo += br + "Payment date: " + paymentDate; //$NON-NLS-1$
			}
			String reference = orderHistory.getProperty("reference"); //$NON-NLS-1$
			if (StringUtils.isNotBlank(reference)) {
				paymentInfo += br + "Reference: " + reference; //$NON-NLS-1$
			}
			String notes = orderHistory.getProperty("notes"); //$NON-NLS-1$
			if (StringUtils.isNotBlank(notes)) {
				paymentInfo += br + "Notes: " + notes; //$NON-NLS-1$
			}
			if (StringUtils.isBlank(paymentInfo)) {
				return null;
			}
		} catch (Exception e) {
			PosLog.error(InvoiceUtil.class, e);
		}
		return "<b>Wire transfer information:</b>" + paymentInfo; //$NON-NLS-1$
	}

	private static String getCardPaymentInformation(Charge transaction) {
		if (transaction == null) {
			return "<br><br>"; //$NON-NLS-1$
		}
		String br = "<br>"; //$NON-NLS-1$
		String string = "<br><br><b>Payment information:</b>"; //$NON-NLS-1$
		try {
			PaymentMethodDetails paymentMethodDetails = transaction.getPaymentMethodDetails();
			Card card = paymentMethodDetails.getCard();
			string += br + "Card type: " + card.getBrand(); //$NON-NLS-1$
			String cardNumber = card.getLast4();
			if (cardNumber != null) {
				string += br + "Acct: " + "*** *** ***" + cardNumber; //$NON-NLS-1$ //$NON-NLS-2$
			}
			if (card.getIssuer() != null) {
				string += br + "Card holder: " + card.getIssuer(); //$NON-NLS-1$
			}
			if (transaction.getId() != null) {
				string += br + "Trans id: " + transaction.getId(); //$NON-NLS-1$
			}
			if (transaction.getAuthorizationCode() != null) {
				string += br + "Approval: " + transaction.getAuthorizationCode(); //$NON-NLS-1$
			}
		} catch (Exception e) {
			PosLog.error(InvoiceUtil.class, e);
		}
		return string;
	}

	public static Map<String, Object> buildInvoiceContextMap(com.floreantpos.model.Invoice invoice, Ticket ticket) {
		Map<String, Object> invoiceMap = new HashMap<String, Object>();
		String currencySymbol = CurrencyUtil.getCurrencySymbol();

		String invoiceString = invoice.getInvoiceNumberDisplayString();
		invoiceMap.put("invoiceNo", invoiceString); //$NON-NLS-1$
		if (StringUtils.isNotBlank(ticket.getId())) {
			invoiceMap.put("orderId", ticket.getId()); //$NON-NLS-1$
		}
		invoiceMap.put("dueDate", invoice.getDueDate() == null ? "" : DateUtil.formatDate(invoice.getDueDate())); //$NON-NLS-1$
		invoiceMap.put("issueDate", DateUtil.formatDate(ticket.getCreateDate())); //$NON-NLS-1$
		invoiceMap.put("resellerName", invoice.getProperty("resellerName")); //$NON-NLS-1$
		loadCustomerInformation(ticket, invoiceMap);
		invoiceMap.put("dueAmount", currencySymbol + NumberUtil.format(ticket.getDueAmount())); //$NON-NLS-1$
		invoiceMap.put("bankPayment", true); //$NON-NLS-1$
		invoiceMap.put("paid", ticket.isPaid()); //$NON-NLS-1$
		invoiceMap.put("paidAmount", currencySymbol + NumberUtil.format(ticket.getPaidAmount())); //$NON-NLS-1$
		invoiceMap.put("orderItemsFromTemplate", getItemsInformation(ticket, currencySymbol)); //$NON-NLS-1$
		invoiceMap.put("subTotal", currencySymbol + NumberUtil.format(ticket.getSubtotalAmount())); //$NON-NLS-1$
		loadDiscountInfo(ticket, invoiceMap, currencySymbol);
		invoiceMap.put("total", currencySymbol + NumberUtil.format(ticket.getTotalAmount())); //$NON-NLS-1$
		invoiceMap.put("amountDue", currencySymbol + NumberUtil.format(ticket.getDueAmount())); //$NON-NLS-1$
		invoiceMap.put("hasDue", ticket.getDueAmount() > 0); //$NON-NLS-1$
		return invoiceMap;
	}

	public static void loadDiscountInfo(Ticket ticket, Map<String, Object> map, String currencySymbol) {
		loadDiscountInfo(ticket, map, currencySymbol, ticket.getDiscountAmount());
	}

	private static void loadDiscountInfo(Ticket ticket, Map<String, Object> map, String currencySymbol, Double discountAmount) {
		map.put("discountCode", ticket == null ? "" : ticket.getDiscountCodeDisplay()); //$NON-NLS-1$ //$NON-NLS-2$
		map.put("discountAmount", currencySymbol + NumberUtil.format(discountAmount)); //$NON-NLS-1$
		map.put("hasDiscount", discountAmount > 0); //$NON-NLS-1$
	}

	public static void loadResellerProperties(Map<String, Object> contextMap, Ticket ticket, Customer reseller, String consoleUrl) {
		String businessName = null;
		String businessPhone = null;
		String businesEmail = null;
		String businesAddressLine1 = null;
		String businesAddressLine2 = ""; //$NON-NLS-1$
		String logoId = null;
		if (reseller != null) {
			businessName = reseller.getProperty("reseller.organization"); //$NON-NLS-1$
			if (StringUtils.isBlank(businessName)) {
				businessName = reseller.getName();
			}
			logoId = reseller.getImageId();
			businessPhone = reseller.getProperty("reseller.business.phone"); //$NON-NLS-1$
			businesEmail = reseller.getProperty("reseller.business.email"); //$NON-NLS-1$
			if (StringUtils.isBlank(businesEmail)) {
				businesEmail = reseller.getEmail();
			}
			businesAddressLine1 = reseller.getProperty("reseller.business.address"); //$NON-NLS-1$
			if (businesAddressLine1 == null) {
				businesAddressLine1 = ""; //$NON-NLS-1$
			}
			String businesCity = reseller.getProperty("reseller.business.city"); //$NON-NLS-1$
			businesAddressLine2 += StringUtils.isBlank(businesCity) ? "" : " " + businesCity; //$NON-NLS-1$

			String businesState = reseller.getProperty("reseller.business.state"); //$NON-NLS-1$
			businesAddressLine2 += StringUtils.isBlank(businesState) ? "" : " " + businesState; //$NON-NLS-1$

			String businesZipCode = reseller.getProperty("reseller.business.zip"); //$NON-NLS-1$
			businesAddressLine2 += StringUtils.isBlank(businesZipCode) ? "" : " " + businesZipCode; //$NON-NLS-1$
		}
		contextMap.put("hasResellerLogo", StringUtils.isNotEmpty(logoId)); //$NON-NLS-1$
		contextMap.put("resellerBusinessName", businessName == null ? "" : businessName); //$NON-NLS-1$ //$NON-NLS-2$
		contextMap.put("resellerEmail", businesEmail == null ? "" : businesEmail); //$NON-NLS-1$ //$NON-NLS-2$
		contextMap.put("resellerAddressLine1", businesAddressLine1 == null ? "" : businesAddressLine1); //$NON-NLS-1$ //$NON-NLS-2$
		contextMap.put("resellerAddressLine2", businesAddressLine2 == null ? "" : businesAddressLine2); //$NON-NLS-1$ //$NON-NLS-2$
		contextMap.put("resellerPhone", businessPhone == null ? "" : businessPhone); //$NON-NLS-1$ //$NON-NLS-2$
		contextMap.put("resellerLogoUrl", consoleUrl + "/imgresource/public/" + logoId + ".png"); //$NON-NLS-1$ //$NON-NLS-2$
		if (reseller != null && ticket.isPaid()) {
			Integer resellerPoints = reseller.getResellerPoints();
			contextMap.put("resellerPoints", ticket.getResellerPoints()); //$NON-NLS-1$
			contextMap.put("totalResellerPoints", resellerPoints + ticket.getResellerPoints()); //$NON-NLS-1$
		}
		String invoiceText = ticket.isPaid() ? "Invoice" : "Estimate"; //$NON-NLS-1$ //$NON-NLS-2$
		contextMap.put("invoiceText", invoiceText); //$NON-NLS-1$ 
		contextMap.put("invoiceTextLower", invoiceText.toLowerCase()); //$NON-NLS-1$ 
	}

	public static Map<String, Object> buildStripeRecurringInvoicePaymentContextMap(Subscription subscription, Invoice invoice, Charge charge, Ticket ticket) {
		Map<String, Object> invoiceMap = new HashMap<String, Object>();

		invoiceMap.put("orderId", ticket.getId()); //$NON-NLS-1$
		invoiceMap.put("paymentDate", DateUtil.formatDate(new Date((charge == null ? invoice.getCreated() : charge.getCreated()) * 1000L))); //$NON-NLS-1$
		loadCustomerInformation(ticket, invoiceMap);
		invoiceMap.put("customerName", invoice.getCustomerName()); //$NON-NLS-1$
		invoiceMap.put("customerEmail", invoice.getCustomerEmail()); //$NON-NLS-1$
		invoiceMap.put("customerPhone", ticket.getCustomerMobileNo()); //$NON-NLS-1$1
		invoiceMap.put("nextPaymentDate", DateUtil.formatDate(new Date(subscription.getCurrentPeriodEnd() * 1000L)));

		String currencySymbol = "$"; //$NON-NLS-1$1
		double chargeAmount = (charge == null ? invoice.getAmountPaid().doubleValue() : charge.getAmount().doubleValue()) / 100D;

		invoiceMap.put("cardPayment", true); //$NON-NLS-1$
		invoiceMap.put("cardAmountNumber", NumberUtil.round(chargeAmount)); //$NON-NLS-1$
		invoiceMap.put("cardAmount", currencySymbol + NumberUtil.formatNumber(chargeAmount)); //$NON-NLS-1$
		invoiceMap.put("paidAmount", currencySymbol + NumberUtil.formatNumber(chargeAmount)); //$NON-NLS-1$
		invoiceMap.put("paymentInformation", getCardPaymentInformation(charge)); //$NON-NLS-1$
		invoiceMap.put("paymentStatus", "Paid"); //$NON-NLS-1$ //$NON-NLS-2$
		invoiceMap.put("paymentMethod", "Credit card"); //$NON-NLS-1$ //$NON-NLS-2$
		invoiceMap.put("orderItemsFromTemplate", getItemsInformation(invoice, ticket, currencySymbol)); //$NON-NLS-1$
		invoiceMap.put("paid", invoice.getPaid()); //$NON-NLS-1$

		double totalAmount = invoice.getTotal().doubleValue() / 100D;
		double paidAmount = invoice.getPaid() ? totalAmount : (invoice.getAmountPaid().doubleValue() / 100D);
		double discountAmount = 0D;
		List<DiscountAmount> discountAmounts = invoice.getTotalDiscountAmounts();
		if (discountAmounts != null && discountAmounts.size() > 0) {
			for (DiscountAmount dAmount : discountAmounts) {
				if (dAmount.getAmount() == null) {
					continue;
				}
				double amount = dAmount.getAmount().doubleValue() / 100;
				discountAmount += amount;
			}
		}
		double dueAmount = invoice.getPaid() ? 0D : (totalAmount - paidAmount);
		invoiceMap.put("dueAmount", currencySymbol + NumberUtil.format(dueAmount)); //$NON-NLS-1$
		invoiceMap.put("subTotal", currencySymbol + NumberUtil.format(totalAmount + discountAmount)); //$NON-NLS-1$
		loadDiscountInfo(ticket, invoiceMap, currencySymbol);
		invoiceMap.put("total", currencySymbol + NumberUtil.format(totalAmount)); //$NON-NLS-1$
		invoiceMap.put("amountDue", currencySymbol + NumberUtil.format(dueAmount)); //$NON-NLS-1$
		return invoiceMap;
	}
}
