package com.floreantpos.model.dao;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import org.apache.commons.lang.StringUtils;
import org.hibernate.Criteria;
import org.hibernate.Session;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.Projections;
import org.hibernate.criterion.Restrictions;

import com.floreantpos.PosException;
import com.floreantpos.model.Customer;
import com.floreantpos.model.Invoice;
import com.floreantpos.model.InvoiceType;
import com.floreantpos.model.Pagination;
import com.floreantpos.util.POSUtil;

public class InvoiceDAO extends BaseInvoiceDAO {

	/**
	 * Default constructor.  Can be used in place of getInstance()
	 */
	public InvoiceDAO() {
	}

	public void loadData(Pagination<Invoice> tableModel, String invoiceId, Date startDate, Date endDate, List<String> customerIds, String resellerId) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(Invoice.class);
			criteria.setProjection(Projections.rowCount());
			criteria.add(Restrictions.eq(Invoice.PROP_OWNER_ID, resellerId));
			criteria.add(Restrictions.isNotNull(Invoice.PROP_TICKET_JSON));
			if (customerIds != null && customerIds.size() > 0) {
				criteria.add(Restrictions.in(Invoice.PROP_CUSTOMER_ID, customerIds));
			}
			if (StringUtils.isNotBlank(invoiceId)) {
				criteria.add(Restrictions.eq(Invoice.PROP_INVOICE_ID, POSUtil.parseInteger(invoiceId)));
			}
			criteria.add(Restrictions.or(Restrictions.isNull(Invoice.PROP_ORDER_TYPE), Restrictions.eq(Invoice.PROP_ORDER_TYPE, InvoiceType.INVOICE.name())));
			if (startDate != null) {
				criteria.add(Restrictions.ge(Invoice.PROP_ORDER_DATE, startDate));
			}
			if (endDate != null) {
				criteria.add(Restrictions.lt(Invoice.PROP_ORDER_DATE, endDate));
			}
			addDeletedFilter(criteria);
			Number result = (Number) criteria.uniqueResult();
			int rowCount = result == null ? 0 : result.intValue();
			tableModel.setNumRows(rowCount);
			if (rowCount == 0) {
				tableModel.setRows(new ArrayList<>());
				return;
			}
			criteria.setProjection(null);
			criteria.addOrder(Order.desc(Invoice.PROP_ORDER_DATE));
			criteria.setFirstResult(tableModel.getCurrentRowIndex());
			criteria.setMaxResults(tableModel.getPageSize());
			tableModel.setRows(criteria.list());
		}
	}

	public Invoice findByInvoiceId(String invoiceId) {
		if (StringUtils.isBlank(invoiceId)) {
			return null;
		}
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(Invoice.class);
			criteria.add(Restrictions.isNotNull(Invoice.PROP_TICKET_JSON));
			criteria.add(Restrictions.eq(Invoice.PROP_INVOICE_ID, POSUtil.parseInteger(invoiceId)));
			criteria.setMaxResults(1);
			return (Invoice) criteria.uniqueResult();
		}
	}

	public int getNextInvoiceNumber() {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(Invoice.class);
			criteria.setProjection(Projections.max(Invoice.PROP_INVOICE_ID));
			Number result = (Number) criteria.uniqueResult();
			int maxInvoiceId = result == null ? 0 : result.intValue();
			return maxInvoiceId + 1;
		}
	}

	@Override
	protected void delete(Object obj, Session session) {
		Invoice invoice = (Invoice) obj;
		if (invoice == null) {
			throw new PosException("Invoice not found!");
		}
		invoice.setDeleted(true);
		update(invoice, session);
	}

	public List<Customer> findCustomersByReseller(Customer reseller) {
		if (reseller == null) {
			return null;
		}
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(Invoice.class);
			criteria.setProjection(Projections.property(Invoice.PROP_CUSTOMER_ID));
			criteria.add(Restrictions.eq(Invoice.PROP_OWNER_ID, reseller.getId()));
			List<String> customerIds = criteria.list();
			if (customerIds == null || customerIds.isEmpty()) {
				return null;
			}
			criteria = session.createCriteria(Customer.class);
			criteria.add(Restrictions.in(Customer.PROP_ID, customerIds));
			addDeletedFilter(criteria);
			return criteria.list();
		}
	}

}