/**
 * ************************************************************************
 * * The contents of this file are subject to the MRPL 1.2
 * * (the  "License"),  being   the  Mozilla   Public  License
 * * Version 1.1  with a permitted attribution clause; you may not  use this
 * * file except in compliance with the License. You  may  obtain  a copy of
 * * the License at http://www.floreantpos.org/license.html
 * * Software distributed under the License  is  distributed  on  an "AS IS"
 * * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * * License for the specific  language  governing  rights  and  limitations
 * * under the License.
 * * The Original Code is FLOREANT POS.
 * * The Initial Developer of the Original Code is OROCUBE LLC
 * * All portions are Copyright (C) 2015 OROCUBE LLC
 * * All Rights Reserved.
 * ************************************************************************
 */
package com.floreantpos.model.dao;

import java.io.Serializable;
import java.util.Iterator;
import java.util.List;

import org.apache.commons.beanutils.PropertyUtils;
import org.apache.commons.lang.StringUtils;
import org.hibernate.Criteria;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.Restrictions;

import com.floreantpos.Messages;
import com.floreantpos.PosException;
import com.floreantpos.PosLog;
import com.floreantpos.constants.AppConstants;
import com.floreantpos.model.Currency;
import com.floreantpos.util.NameBaseIdGenerator;
import com.orocube.rest.service.server.BaseDataServiceDao;

public class CurrencyDAO extends BaseCurrencyDAO {

	/**
	 * Default constructor. Can be used in place of getInstance()
	 */
	public CurrencyDAO() {
	}

	@Override
	protected Serializable save(Object obj, Session s) {
		updateTime(obj);
		return super.save(obj, s);
	}

	@Override
	protected void update(Object obj, Session s) {
		updateTime(obj);
		super.update(obj, s);
	}

	@Override
	protected void saveOrUpdate(Object obj, Session s) {
		updateTime(obj);
		super.saveOrUpdate(obj, s);
	}

	public void saveOrUpdate(Currency currency) {
		Session session = null;
		Transaction transaction = null;
		try {
			session = createNewSession();
			transaction = session.beginTransaction();

			if (currency.isMain()) {
				List<Currency> mainCurrencies = getMainCurrencies(session, currency.getId(), currency.getOutletId());
				if (mainCurrencies != null && mainCurrencies.size() > 0) {
					for (Currency currency2 : mainCurrencies) {
						currency2.setMain(false);
						update(currency2, session);
					}
				}
			}

			saveOrUpdate(currency, session);
			transaction.commit();
		} finally {
			session.close();
		}
	}

	public void saveOrUpdateCurrency(List<Currency> dataList, boolean updateLastUpdateTime, boolean updateSyncTime) throws Exception {
		saveOrUpdateCurrency(dataList, updateLastUpdateTime, updateSyncTime, true);
	}

	public void saveOrUpdateCurrency(List<Currency> dataList, boolean updateLastUpdateTime, boolean updateSyncTime, boolean forceSave) throws Exception {

		if (dataList == null)
			return;

		Transaction tx = null;
		Session session = null;
		try {
			session = createNewSession();
			tx = session.beginTransaction();

			for (Iterator<Currency> iterator = dataList.iterator(); iterator.hasNext();) {
				Currency item = (Currency) iterator.next();
				Currency existingItem = get(item.getId(), item.getOutletId(), session);
				boolean mainCurr = item.isMain();

				if (existingItem != null) {
					if (!forceSave && !BaseDataServiceDao.get().shouldSave(item.getLastUpdateTime(), existingItem.getLastUpdateTime())) {
						PosLog.info(getClass(), item.getName() + " already updated"); //$NON-NLS-1$
						continue;
					}

					long version = existingItem.getVersion();
					boolean deleted = item.isDeleted();
					PropertyUtils.copyProperties(existingItem, item);
					existingItem.setVersion(version);
					// mainCurr = doCheckMainCurrencyAndUpdate(session, item, mainCurr);
					// existingItem.setMain(mainCurr);
					existingItem.setDeleted(deleted);
					existingItem.setUpdateLastUpdateTime(updateLastUpdateTime);
					existingItem.setUpdateSyncTime(updateSyncTime);
					update(existingItem, session);
				}
				else {
					// mainCurr = doCheckMainCurrencyAndUpdate(session, item, mainCurr);
					// item.setMain(mainCurr);
					item.setUpdateLastUpdateTime(updateLastUpdateTime);
					item.setUpdateSyncTime(updateSyncTime);
					save(item, session);
				}
			}
			tx.commit();
		} catch (Exception e) {
			tx.rollback();
			throw e;
		} finally {
			closeSession(session);
		}

	}

	private Currency get(String id, String outletId, Session session) {
		return get(new Currency(id, outletId), session);
	}

	private boolean doCheckMainCurrencyAndUpdate(Session session, Currency item, boolean mainCurr) {
		if (mainCurr) {
			List<Currency> mainCurrencies = getMainCurrencies(session, item.getId(), item.getOutletId());
			if (mainCurrencies != null && mainCurrencies.size() > 0) {
				for (Currency currency2 : mainCurrencies) {

					if (item.getLastUpdateTime() == null) {
						return false;
					}

					if (currency2.getLastUpdateTime() == null || item.getLastUpdateTime().after(currency2.getLastUpdateTime())) {
						currency2.setMain(false);
						update(currency2, session);
					}
					else {
						mainCurr = false;
					}
				}
			}
		}
		return mainCurr;
	}

	@Override
	protected void delete(Object obj, Session session) {
		Currency currency = (Currency) obj;
		if (currency == null) {
			throw new PosException(Messages.getString("CurrencyDAO.0")); //$NON-NLS-1$
		}
		if (currency.isMain()) {
			throw new PosException(Messages.getString("CurrencyDAO.1")); //$NON-NLS-1$
		}
		List<Currency> allCurrencies = findAll();
		if (allCurrencies.size() == 1) {
			throw new PosException(Messages.getString("CanNotDeleteLastOne"));//$NON-NLS-1$
		}
		currency.setDeleted(true);
		update(currency, session);
	}

	@Override
	public List<Currency> findAll() {
		return findByOutletId(null);
	}

	public List<Currency> findByOutletId(String outletId) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			if (StringUtils.isNotBlank(outletId)) {
				criteria.add(Restrictions.eq(Currency.PROP_OUTLET_ID, outletId));
			}
			addDeletedFilter(criteria);
			return criteria.list();
		}
	}

	public List<Currency> getMainCurrencies(Session session, String currencyId, String outletId) {
		Criteria criteria = session.createCriteria(getReferenceClass());
		criteria.add(Restrictions.eq(Currency.PROP_MAIN, Boolean.TRUE));
		if (currencyId != null) {
			criteria.add(Restrictions.ne(Currency.PROP_ID, currencyId));
		}
		if (outletId != null) {
			criteria.add(Restrictions.eq(Currency.PROP_OUTLET_ID, outletId));
		}
		criteria.add(Restrictions.or(Restrictions.isNull(AppConstants.PROP_DELETED), Restrictions.eq(AppConstants.PROP_DELETED, Boolean.FALSE)));
		return criteria.list();
	}

	public Currency get(String id, String outletId) {
		return get(new Currency(id, outletId));
	}

	public List<Currency> getMainCurrencies(Session session, String id) {
		// TODO:
		return getMainCurrencies(session, id, null);
	}

	public Currency getMainCurrencyByOutletId(String outletId) {
		try (Session session = createNewSession()) {
			{
				Criteria criteria = session.createCriteria(Currency.class);
				if (StringUtils.isNotBlank(outletId)) {
					criteria.add(Restrictions.eq(Currency.PROP_OUTLET_ID, outletId));
				}
				criteria.add(Restrictions.eq(Currency.PROP_MAIN, Boolean.TRUE));
				addDeletedFilter(criteria);
				criteria.addOrder(Order.asc(Currency.PROP_NAME));
				criteria.setMaxResults(1);
				Currency uniqueResult = (Currency) criteria.uniqueResult();
				if (uniqueResult != null) {
					return uniqueResult;
				}
			}
			{
				Criteria criteria = session.createCriteria(Currency.class);
				if (outletId != null) {
					criteria.add(Restrictions.eq(Currency.PROP_OUTLET_ID, outletId));
				}
				addDeletedFilter(criteria);
				criteria.addOrder(Order.asc(Currency.PROP_NAME));
				criteria.setMaxResults(1);
				Currency uniqueCurrency = (Currency) criteria.uniqueResult();

				if (uniqueCurrency == null) {
					String generateId = NameBaseIdGenerator.generateId("US Dollar");
					Currency deletedCurrency = get(generateId, outletId);
					if (deletedCurrency == null) {
						Currency currency = new Currency();
						currency.setName("US Dollar"); //$NON-NLS-1$
						currency.setCode("USD"); //$NON-NLS-1$
						currency.setSymbol("$"); //$NON-NLS-1$
						currency.setExchangeRate(1.0);
						if (StringUtils.isNotBlank(outletId)) {
							currency.setId(generateId);
							currency.setOutletId(outletId);
						}
						currency.setMain(true);
						uniqueCurrency = currency;
						CurrencyDAO.getInstance().save(currency);
					}
					else {
						deletedCurrency.setDeleted(false);
						deletedCurrency.setMain(true);
						uniqueCurrency = deletedCurrency;
						CurrencyDAO.getInstance().update(deletedCurrency);
					}
				}
				else {
					uniqueCurrency.setMain(true);
					CurrencyDAO.getInstance().update(uniqueCurrency);
				}
				return uniqueCurrency;
			}
		}
	}

}