/**
 * ************************************************************************
 * * The contents of this file are subject to the MRPL 1.2
 * * (the  "License"),  being   the  Mozilla   Public  License
 * * Version 1.1  with a permitted attribution clause; you may not  use this
 * * file except in compliance with the License. You  may  obtain  a copy of
 * * the License at http://www.floreantpos.org/license.html
 * * Software distributed under the License  is  distributed  on  an "AS IS"
 * * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * * License for the specific  language  governing  rights  and  limitations
 * * under the License.
 * * The Original Code is FLOREANT POS.
 * * The Initial Developer of the Original Code is OROCUBE LLC
 * * All portions are Copyright (C) 2015 OROCUBE LLC
 * * All Rights Reserved.
 * ************************************************************************
 */
package com.floreantpos.model.dao;

import java.io.Serializable;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import org.apache.commons.beanutils.PropertyUtils;
import org.hibernate.Criteria;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.hibernate.criterion.ProjectionList;
import org.hibernate.criterion.Projections;
import org.hibernate.criterion.Restrictions;

import com.floreantpos.PosException;
import com.floreantpos.PosLog;
import com.floreantpos.model.CashTransaction;
import com.floreantpos.model.DeclaredTips;
import com.floreantpos.model.Gratuity;
import com.floreantpos.model.GratuityPaymentHistory;
import com.floreantpos.model.PosTransaction;
import com.floreantpos.model.StoreSession;
import com.floreantpos.model.Ticket;
import com.floreantpos.model.TipsCashoutReport;
import com.floreantpos.model.TipsCashoutReportData;
import com.floreantpos.model.User;
import com.floreantpos.model.util.DateUtil;
import com.floreantpos.util.StoreUtil;
import com.orocube.rest.service.server.BaseDataServiceDao;

public class GratuityDAO extends BaseGratuityDAO {

	/**
	 * Default constructor.  Can be used in place of getInstance()
	 */
	public GratuityDAO() {
	}

	@Override
	protected Serializable save(Object obj, Session s) {
		updateTime(obj);
		return super.save(obj, s);
	}

	@Override
	protected void update(Object obj, Session s) {
		updateTime(obj);
		super.update(obj, s);
	}

	@Override
	protected void saveOrUpdate(Object obj, Session s) {
		updateTime(obj);
		super.saveOrUpdate(obj, s);
	}

	public List<Gratuity> findByUser(User user) throws PosException {
		return findByUser(user, false);
	}

	public List<Gratuity> findByUser(User user, boolean includePaid) throws PosException {
		Session session = null;

		try {
			session = getSession();

			Criteria criteria = session.createCriteria(getReferenceClass());

			if (user != null)
				criteria.add(Restrictions.eq(Gratuity.PROP_OWNER_ID, user.getId()));

			if (!includePaid)
				criteria.add(Restrictions.eq(Gratuity.PROP_PAID, Boolean.FALSE));

			criteria.add(Restrictions.eq(Gratuity.PROP_REFUNDED, Boolean.FALSE));
			return criteria.list();
		} catch (Exception e) {
			throw new PosException("" + user.getFirstName() + " " + user.getLastName()); //$NON-NLS-1$ //$NON-NLS-2$
		} finally {
			closeSession(session);
		}
	}

	public void payGratuities(List<Gratuity> gratuities, List<GratuityPaymentHistory> gratuityPaymentList) {
		Session session = null;
		Transaction tx = null;

		try {
			session = createNewSession();
			tx = session.beginTransaction();
			for (Gratuity gratuity : gratuities) {
				saveOrUpdate(gratuity, session);
			}
			for (GratuityPaymentHistory gratuityPayment : gratuityPaymentList) {
				GratuityPaymentHistoryDAO.getInstance().saveOrUpdate(gratuityPayment, session);
			}
			tx.commit();
		} catch (Exception e) {
			if (tx != null) {
				tx.rollback();
			}
			throw new PosException(""); //$NON-NLS-1$
		} finally {
			closeSession(session);
		}
	}

	public TipsCashoutReport createReport(Date fromDate, Date toDate, User user) {
		return createReport(fromDate, toDate, user, false);
	}

	public TipsCashoutReport createReport(Date fromDate, Date toDate, User user, boolean showCurrentSessionHistory) {
		Session session = null;

		try {
			session = getSession();

			if (fromDate != null)
				fromDate = DateUtil.startOfDay(fromDate);
			if (toDate != null)
				toDate = DateUtil.endOfDay(toDate);

			Criteria criteria = session.createCriteria(Ticket.class);
			if (user != null)
				criteria.add(Restrictions.eq(Ticket.PROP_OWNER_ID, user.getId()));

			TipsCashoutReport report = new TipsCashoutReport();
			StoreSession currentSession = StoreUtil.getCurrentStoreOperation().getCurrentData();
			criteria.createAlias(Ticket.PROP_GRATUITY, "g");
			criteria.add(Restrictions.isNotNull(Ticket.PROP_GRATUITY));
			criteria.add(Restrictions.gt("g." + Gratuity.PROP_AMOUNT, new Double(0)));

			if (showCurrentSessionHistory) {
				criteria.add(Restrictions.eq(Ticket.PROP_STORE_SESSION_ID, currentSession.getId()));
				report.setFromDate(currentSession.getOpenTime());
				report.setToDate(new Date());
			}
			else {
				if (fromDate == null && toDate == null) {
					criteria.add(Restrictions.eq("g." + Gratuity.PROP_PAID, Boolean.FALSE));
					criteria.add(Restrictions.eq("g." + Gratuity.PROP_REFUNDED, Boolean.FALSE));
				}
				else {
					criteria.add(Restrictions.ge(Ticket.PROP_CREATE_DATE, fromDate));
					criteria.add(Restrictions.le(Ticket.PROP_CREATE_DATE, toDate));
				}
			}

			List list = criteria.list();
			report.setServer(user == null ? "ALL" : user.getId() + "/" + user.toString()); //$NON-NLS-1
			report.setReportTime(new Date());

			if (!showCurrentSessionHistory) {
				report.setFromDate(fromDate);
				report.setToDate(toDate);
			}
			for (Iterator iter = list.iterator(); iter.hasNext();) {
				Ticket ticket = (Ticket) iter.next();
				Gratuity gratuity = ticket.getGratuity();
				TipsCashoutReportData data = new TipsCashoutReportData();
				data.setTicketId(String.valueOf(ticket.getTokenNo()));
				data.setTicketTotal(ticket.getTotalAmountWithTips());
				if (gratuity != null)
					data.setDeclareTipsAmount(gratuity.getDeclareTipsAmount());

				ticket = TicketDAO.getInstance().loadCouponsAndTransactions(ticket.getId(), ticket.getOutletId());
				if (ticket != null) {
					Set<PosTransaction> transactions = ticket.getTransactions();
					if (transactions != null && transactions.size() > 0) {
						for (PosTransaction posTransaction : transactions) {
							if (posTransaction instanceof CashTransaction)
								data.setCashTips(data.getCashTips() + posTransaction.getTipsAmount());
							else
								data.setChargedTips(data.getChargedTips() + posTransaction.getTipsAmount());
						}
					}
				}
				if (gratuity != null && !gratuity.isRefunded()) {
					if (fromDate == null && toDate == null) {
						if ((data.getCashTips() + data.getChargedTips() + data.getDeclareTipsAmount()) <= 0) {
							iter.remove();
							continue;
						}
					}
					if (gratuity.getAmount() > 0)
						data.setSaleType("");
					data.setPaid(gratuity.isPaid().booleanValue());
					data.setTipsPaidAmount(gratuity.getTipsPaidAmount());
				}
				report.addReportData(data);
			}
			criteria = session.createCriteria(DeclaredTips.class);
			criteria.setProjection(Projections.sum(DeclaredTips.PROP_AMOUNT));
			if (fromDate == null && toDate == null) {
				criteria.add(Restrictions.eq(DeclaredTips.PROP_SESSION_ID, currentSession.getId()));
			}
			else {
				criteria.add(Restrictions.ge(DeclaredTips.PROP_DECLARED_TIME, fromDate));
				criteria.add(Restrictions.le(DeclaredTips.PROP_DECLARED_TIME, toDate));
			}
			if (user != null)
				criteria.add(Restrictions.eq(DeclaredTips.PROP_OWNER_ID, user.getId()));

			Object uniqueResult = criteria.uniqueResult();
			if (uniqueResult != null)
				report.setDeclaredTipsAmount((double) uniqueResult);
			report.calculateOthers();
			return report;
		} catch (Exception e) {
			if (user != null)
				throw new PosException("" + user.getFirstName() + " " + user.getLastName(), e); //$NON-NLS-1$ //$NON-NLS-2$
			throw e;
		} finally {
			closeSession(session);
		}

	}

	public double findTotalGratuityAmount(Date fromDate, Date toDate, User user) throws PosException {
		Session session = null;
		try {
			session = createNewSession();
			//DetachedCriteria detachedCriteria = DetachedCriteria.forClass(Ticket.class);
			Criteria criteria = session.createCriteria(getReferenceClass());
			//criteria.createAlias(Gratuity.PROP_TICKET_ID, alias)
			//			criteria.add(Restrictions.between(DeclaredTips.PROP_DECLARED_TIME, fromDate, toDate));
			if (user != null) {
				criteria.add(Restrictions.eq(Gratuity.PROP_OWNER_ID, user.getId()));
			}
			criteria.add(Restrictions.eq(Gratuity.PROP_REFUNDED, Boolean.FALSE));
			ProjectionList projectionList = Projections.projectionList();
			projectionList.add(Projections.sum(Gratuity.PROP_AMOUNT), Gratuity.PROP_AMOUNT);

			criteria.setProjection(projectionList);
			Number totalAmount = (Number) criteria.uniqueResult();
			if (totalAmount != null) {
				return totalAmount.doubleValue();

			}
		} finally {
			closeSession(session);
		}
		return 0.0;
	}

	public void saveOrUpdateGratuityList(List<Gratuity> dataList, boolean updateLastUpdateTime, boolean updateSyncTime) throws Exception {
		if (dataList == null)
			return;

		Transaction tx = null;
		try (Session session = createNewSession()) {
			tx = session.beginTransaction();

			for (Iterator<Gratuity> iterator = dataList.iterator(); iterator.hasNext();) {
				Gratuity item = (Gratuity) iterator.next();
				Gratuity existingItem = get(item.getId());
				if (existingItem != null) {
					if (!BaseDataServiceDao.get().shouldSave(item.getLastUpdateTime(), existingItem.getLastUpdateTime())) {
						PosLog.info(getClass(), "Gratuity " + item.getId() + " already updated"); //$NON-NLS-1$
						continue;
					}
					long version = existingItem.getVersion();
					PropertyUtils.copyProperties(existingItem, item);
					existingItem.setVersion(version);
					existingItem.setUpdateLastUpdateTime(updateLastUpdateTime);
					existingItem.setUpdateSyncTime(updateSyncTime);
					update(existingItem, session);
				}
				else {
					item.setUpdateLastUpdateTime(updateLastUpdateTime);
					item.setUpdateSyncTime(updateSyncTime);
					save(item, session);
				}
			}
			tx.commit();
		} catch (Exception e) {
			tx.rollback();
			throw e;
		}
	}

}