/**
 * ************************************************************************
 * * The contents of this file are subject to the MRPL 1.2
 * * (the  "License"),  being   the  Mozilla   Public  License
 * * Version 1.1  with a permitted attribution clause; you may not  use this
 * * file except in compliance with the License. You  may  obtain  a copy of
 * * the License at http://www.floreantpos.org/license.html
 * * Software distributed under the License  is  distributed  on  an "AS IS"
 * * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * * License for the specific  language  governing  rights  and  limitations
 * * under the License.
 * * The Original Code is FLOREANT POS.
 * * The Initial Developer of the Original Code is OROCUBE LLC
 * * All portions are Copyright (C) 2015 OROCUBE LLC
 * * All Rights Reserved.
 * ************************************************************************
 */
package com.floreantpos.model.dao;

import java.io.Serializable;
import java.util.Calendar;
import java.util.Date;
import java.util.Iterator;
import java.util.List;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.WordUtils;
import org.hibernate.Criteria;
import org.hibernate.Query;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.hibernate.criterion.Criterion;
import org.hibernate.criterion.Disjunction;
import org.hibernate.criterion.Projections;
import org.hibernate.criterion.Restrictions;

import com.floreantpos.DuplicateDataException;
import com.floreantpos.Messages;
import com.floreantpos.POSConstants;
import com.floreantpos.PosException;
import com.floreantpos.PosLog;
import com.floreantpos.model.InventoryStock;
import com.floreantpos.model.Store;
import com.floreantpos.model.Terminal;
import com.floreantpos.model.TimedModel;
import com.floreantpos.model.util.DateUtil;

public class GenericDAO extends _RootDAO {

	private final static GenericDAO instance = new GenericDAO();

	public GenericDAO() {
		super();
	}

	public static GenericDAO getInstance() {
		return instance;
	}

	@Override
	protected Class getReferenceClass() {
		return null;
	}

	@Override
	public Serializable save(Object obj) {
		return super.save(obj);
	}

	@Override
	public void saveOrUpdate(Object obj) {
		super.saveOrUpdate(obj);
	}

	@Override
	public Serializable save(Object obj, Session s) {
		return super.save(obj, s);
	}

	@Override
	public void saveOrUpdate(Object obj, Session s) {
		super.saveOrUpdate(obj, s);
	}

	@Override
	public void delete(Object obj, Session s) {
		super.delete(obj, s);
	}

	@Override
	public Session getSession(String configFile, boolean createNew) {
		return super.getSession(configFile, createNew);
	}

	public List findAll(Class clazz, Session session) {
		Criteria crit = session.createCriteria(clazz);
		return crit.list();
	}

	public void saveAll(List list, Session session) {
		Transaction tx = session.beginTransaction();
		for (Object object : list) {
			session.saveOrUpdate(object);
		}
		tx.commit();
	}

	@Override
	public void closeSession(Session session) {
		try {
			super.closeSession(session);
		} catch (Exception x) {
		}
	}

	@SuppressWarnings("rawtypes")
	public List findAllUnSyncItem(Class refClass) {
		return findAllUnSyncItem(refClass, false, null);
	}

	@SuppressWarnings("rawtypes")
	public List findAllUnSyncItem(Class<?> refClass, boolean findAll, Date syncTime) {
		List unSyncItems = getAllUnSyncItems(refClass, findAll, syncTime);
		if (unSyncItems == null || unSyncItems.size() == 0) {
			return unSyncItems;
		}
		if (!(unSyncItems.get(0) instanceof TimedModel)) {
			return unSyncItems;
		}
		for (Iterator<?> iterator = unSyncItems.iterator(); iterator.hasNext();) {
			TimedModel data = (TimedModel) iterator.next();
			data.setLastUpdateTime(syncTime);
		}
		return unSyncItems;
	}

	private List getAllUnSyncItems(Class<?> refClass, boolean findAll, Date syncTime) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(refClass);
			if (!findAll) {
				Criterion nullUpdateTime = Restrictions.isNull(InventoryStock.PROP_LAST_UPDATE_TIME);
				Criterion nullSyncTime = Restrictions.isNull(InventoryStock.PROP_LAST_SYNC_TIME);
				Criterion gtQuery = Restrictions.gtProperty(InventoryStock.PROP_LAST_UPDATE_TIME, InventoryStock.PROP_LAST_SYNC_TIME);
				criteria.add(Restrictions.or(nullUpdateTime, nullSyncTime, gtQuery));
			}
			if (findAll && syncTime != null) {
				Criterion nullUpdateTime = Restrictions.isNull(InventoryStock.PROP_LAST_SYNC_TIME);
				Criterion ltQuery = Restrictions.lt(InventoryStock.PROP_LAST_SYNC_TIME, syncTime);
				criteria.add(Restrictions.or(nullUpdateTime, ltQuery));
			}
			criteria.setResultTransformer(Criteria.DISTINCT_ROOT_ENTITY);
			return criteria.list();
		}
	}

	public void updateItemsLastSyncTime(List<String> ids, String tableName) {
		updateItemsLastSyncTime(null, ids, tableName);
	}

	public void updateItemsLastSyncTime(Date syncTime, List<String> ids, String tableName) {
		if (ids == null || ids.isEmpty())
			return;

		if (syncTime == null) {
			syncTime = new Date();
		}
		String whereCondition = "("; //$NON-NLS-1$
		for (Iterator iterator = ids.iterator(); iterator.hasNext();) {
			String id = (String) iterator.next();
			whereCondition += "'" + id + "'"; //$NON-NLS-1$ //$NON-NLS-2$
			if (iterator.hasNext())
				whereCondition += ","; //$NON-NLS-1$
		}
		whereCondition += ")"; //$NON-NLS-1$
		Transaction tx = null;
		try (Session session = createNewSession()) {
			tx = session.beginTransaction();
			String hqlString = "update " + tableName + " set %s=:lastSyncTime where %s in %s"; //$NON-NLS-1$ //$NON-NLS-2$
			hqlString = String.format(hqlString, InventoryStock.PROP_LAST_SYNC_TIME, InventoryStock.PROP_ID, whereCondition);
			Query query = session.createQuery(hqlString);
			query.setParameter("lastSyncTime", syncTime); //$NON-NLS-1$
			query.executeUpdate();
			tx.commit();

		} catch (Exception e) {
			tx.rollback();
			throw e;
		}

		//update last update time only if last update time is null
		Calendar calendar = Calendar.getInstance();
		calendar.setTime(syncTime);
		calendar.add(Calendar.SECOND, -1);

		updateLastUpdateTimeIfNull(tableName, calendar.getTime());
	}

	public void updateItemsLastSyncTimeByInt(Date syncTime, List<Integer> ids, String tableName) {
		if (ids == null || ids.isEmpty())
			return;
		if (syncTime == null) {
			syncTime = new Date();
		}
		String whereCondition = "("; //$NON-NLS-1$
		for (Iterator iterator = ids.iterator(); iterator.hasNext();) {
			int id = (Integer) iterator.next();
			whereCondition += id;
			if (iterator.hasNext())
				whereCondition += ","; //$NON-NLS-1$
		}
		whereCondition += ")"; //$NON-NLS-1$
		Transaction tx = null;
		try (Session session = createNewSession()) {
			tx = session.beginTransaction();
			String hqlString = "update " + tableName + " set %s=:lastSyncTime where %s in" + whereCondition; //$NON-NLS-1$ //$NON-NLS-2$
			hqlString = String.format(hqlString, InventoryStock.PROP_LAST_SYNC_TIME, InventoryStock.PROP_ID);
			Query query = session.createQuery(hqlString);
			query.setParameter("lastSyncTime", syncTime); //$NON-NLS-1$
			query.executeUpdate();
			tx.commit();

		} catch (Exception e) {
			tx.rollback();
			throw e;
		}

		//update last update time only if last update time is null
		Calendar calendar = Calendar.getInstance();
		calendar.setTime(syncTime);
		calendar.add(Calendar.SECOND, -1);

		updateLastUpdateTimeIfNull(tableName, calendar.getTime());
	}

	public void updateLastUpdateTimeIfNull(String tableName) {
		Calendar realTime = Calendar.getInstance();
		realTime.set(Calendar.MILLISECOND, realTime.get(Calendar.MILLISECOND));
		updateLastUpdateTimeIfNull(tableName, realTime.getTime());
	}

	public void updateLastUpdateTimeIfNull(String tableName, Date realTime) {
		Transaction tx = null;
		try (Session session = createNewSession()) {
			tx = session.beginTransaction();
			String hqlString = "update " + tableName + " set %s=:lastUpdateTime where %s is null"; //$NON-NLS-1$ //$NON-NLS-2$
			hqlString = String.format(hqlString, Terminal.PROP_LAST_UPDATE_TIME, Terminal.PROP_LAST_UPDATE_TIME);
			Query query = session.createQuery(hqlString);
			query.setParameter("lastUpdateTime", realTime); //$NON-NLS-1$
			query.executeUpdate();
			tx.commit();
		} catch (Exception e) {
			tx.rollback();
			PosLog.error(getClass(), Messages.getString("GenericDAO.19")); //$NON-NLS-1$
		}
	}

	public void checkIdOrNameExists(String existingId, String name, Class modelClass) {
		try (Session session = createNewSession()) {
			checkIdOrNameExists(existingId, name, modelClass, session);
		}
	}

	public void checkIdOrNameExists(String existingId, String name, Class modelClass, String outletId) {
		try (Session session = createNewSession()) {
			checkIdOrNameExists(existingId, name, modelClass, outletId, session);
		}
	}

	public void checkIdOrNameExists(String existingId, String name, Class modelClass, Session session) {
		checkIdOrNameExists(existingId, name, modelClass, null, session); //$NON-NLS-1$
	}

	public void checkIdOrNameExists(String existingId, String name, Class modelClass, String outletId, Session session) {
		if (StringUtils.isBlank(name)) {
			throw new PosException(POSConstants.NAME_IS_EMPTY);
		}
		checkDifferentObjectExists(existingId, name, modelClass, "name", outletId, session); //$NON-NLS-1$
	}

	public void checkDifferentObjectExists(String existingId, String fieldValue, Class modelClass, String colName) {
		try (Session session = createNewSession()) {
			checkDifferentObjectExists(existingId, fieldValue, modelClass, colName, session);
		}
	}

	public void checkDifferentObjectExists(String existingId, String fieldValue, Class modelClass, String colName, String outletId) {
		try (Session session = createNewSession()) {
			checkDifferentObjectExists(existingId, fieldValue, modelClass, colName, outletId, session);
		}
	}

	public void checkDifferentObjectExists(String existingId, String fieldValue, Class modelClass, String colName, Session session) {
		checkDifferentObjectExists(existingId, fieldValue, modelClass, colName, null, session);
	}

	public void checkDifferentObjectExists(String existingId, String fieldValue, Class modelClass, String colName, String outletId, Session session) {
		Criteria criteria = session.createCriteria(modelClass);
		criteria.setProjection(Projections.rowCount());
		criteria.add(Restrictions.eq(colName, fieldValue).ignoreCase());
		if (StringUtils.isNotEmpty(existingId)) {
			criteria.add(Restrictions.ne("id", existingId)); //$NON-NLS-1$
		}
		if (StringUtils.isNotBlank(outletId)) {
			criteria.add(Restrictions.eq("outletId", outletId)); //$NON-NLS-1$
		}
		addDeletedFilter(criteria);
		Number rowCount = (Number) criteria.uniqueResult();
		if (rowCount != null && rowCount.intValue() > 0) {
			throw new DuplicateDataException(String.format("%s" + " (%s) " + Messages.getString("GenericDAO.22"), WordUtils.capitalizeFully(colName), fieldValue), //$NON-NLS-1$//$NON-NLS-2$//$NON-NLS-3$
					colName);
		}
	}

	public Object findObjectByFieldValue(String value, Class modelClass, String id, String... colNames) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(modelClass);
			Disjunction disjunction = Restrictions.disjunction();
			for (String field : colNames) {
				disjunction.add(Restrictions.eq(field, value).ignoreCase());
			}
			criteria.add(disjunction);
			if (StringUtils.isNotEmpty(id)) {
				criteria.add(Restrictions.ne("id", id)); //$NON-NLS-1$
			}
			criteria.setMaxResults(1);
			addDeletedFilter(criteria);
			return criteria.uniqueResult();
		}
	}

	public void updateLastUpdateTimeProperty(Class class1, Date time) {
		Store store = StoreDAO.getRestaurant();
		updateLastUpdateTimeProperty(store, class1, time);
		StoreDAO.getInstance().saveOrUpdate(store);
	}

	public void updateLastUpdateTimeProperty(Store store, Class class1, Date time) {
		updateLastUpdateTimeProperty(store, class1.getSimpleName(), time);
	}

	public void updateLastUpdateTimeProperty(Store store, String simpleName, Date time) {
		store.addProperty(simpleName + "." + Store.PROP_LAST_UPDATE_TIME, DateUtil.formatSyncTime(time)); //$NON-NLS-1$
	}

	@SuppressWarnings("unchecked")
	public List<String> getForeignDataListNames(Session session, Class<?> clazz, String column, String id) {
		return getForeignDataList(session, clazz, column, id, "name");//$NON-NLS-1$
	}

	public List getForeignDataList(Session session, Class<?> clazz, String column, String id, String fetchColumn) {
		Criteria criteria = session.createCriteria(clazz);
		criteria.setProjection(Projections.property(fetchColumn));
		criteria.add(Restrictions.eq(column, id));
		addDeletedFilter(criteria, clazz);
		return criteria.list();
	}

	@SuppressWarnings("unchecked")
	public List<String> getForeignDataListNames(Session session, Class<?> clazz, String collectionName, String column, String id) {
		return getForeignDataList(session, clazz, collectionName, column, id, "name"); //$NON-NLS-1$
	}

	@SuppressWarnings("rawtypes")
	public List getForeignDataList(Session session, Class<?> clazz, String collectionName, String column, String id, String fetchColumn) {
		Criteria criteria = session.createCriteria(clazz);
		criteria.createAlias(collectionName, "c"); //$NON-NLS-1$
		criteria.setProjection(Projections.property(fetchColumn));
		criteria.add(Restrictions.eq("c." + column, id)); //$NON-NLS-1$
		addDeletedFilter(criteria, clazz);
		return criteria.list();
	}
}
