package com.floreantpos.model.dao;

import java.io.Serializable;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.apache.commons.beanutils.PropertyUtils;
import org.apache.commons.lang.StringUtils;
import org.hibernate.Criteria;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.hibernate.criterion.Criterion;
import org.hibernate.criterion.DetachedCriteria;
import org.hibernate.criterion.Disjunction;
import org.hibernate.criterion.MatchMode;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.ProjectionList;
import org.hibernate.criterion.Projections;
import org.hibernate.criterion.Property;
import org.hibernate.criterion.Restrictions;
import org.hibernate.transform.Transformers;

import com.floreantpos.PosLog;
import com.floreantpos.model.InventoryLocation;
import com.floreantpos.model.InventoryStock;
import com.floreantpos.model.InventoryTransaction;
import com.floreantpos.model.InventoryVendor;
import com.floreantpos.model.InventoryVendorItems;
import com.floreantpos.model.MenuGroup;
import com.floreantpos.model.MenuItem;
import com.floreantpos.model.Outlet;
import com.floreantpos.model.util.InventoryUnitConvertionUtil;
import com.floreantpos.util.NumberUtil;
import com.orocube.rest.service.server.BaseDataServiceDao;

public class InventoryStockDAO extends BaseInventoryStockDAO {

	/**
	 * Default constructor.  Can be used in place of getInstance()
	 */
	public InventoryStockDAO() {
	}

	@Override
	protected Serializable save(Object obj, Session s) {
		updateTime(obj);
		return super.save(obj, s);
	}

	@Override
	protected void update(Object obj, Session s) {
		updateTime(obj);
		super.update(obj, s);
	}

	@Override
	protected void saveOrUpdate(Object obj, Session s) {
		updateTime(obj);
		super.saveOrUpdate(obj, s);
	}

	public InventoryStock initialize(InventoryStock inventoryStock) {
		if (inventoryStock == null || inventoryStock.getId() == null)
			return inventoryStock;

		Session session = null;

		try {
			session = createNewSession();
			session.refresh(inventoryStock);

			/*	Hibernate.initialize(InventoryStock.PROP_INVENTORY_ITEM);*/

			return inventoryStock;
		} finally {
			closeSession(session);
		}
	}

	@SuppressWarnings({ "unchecked" })
	public List<InventoryStock> getInventoryStock(String itemName, Object selectedType) {
		try (Session session = createNewSession()) {
			DetachedCriteria detachedCriteria = DetachedCriteria.forClass(MenuItem.class);
			detachedCriteria.setProjection(Projections.property(MenuItem.PROP_ID));
			detachedCriteria.add(Restrictions.or(Restrictions.isNull(MenuItem.PROP_DELETED), Restrictions.eq(MenuItem.PROP_DELETED, Boolean.FALSE)));
			detachedCriteria.add(Restrictions.eq(MenuItem.PROP_INVENTORY_ITEM, Boolean.TRUE));

			Criteria criteria = session.createCriteria(InventoryStock.class);

			if (StringUtils.isNotEmpty(itemName)) {
				Disjunction or = Restrictions.disjunction();
				or.add(Restrictions.ilike(InventoryStock.PROP_ITEM_NAME, itemName.trim(), MatchMode.ANYWHERE));
				or.add(Restrictions.ilike(InventoryStock.PROP_SKU, itemName.trim(), MatchMode.ANYWHERE));
				or.add(Restrictions.ilike(InventoryStock.PROP_BARCODE, itemName.trim(), MatchMode.ANYWHERE));
				criteria.add(or);
			}
			if (selectedType instanceof InventoryLocation) {
				InventoryLocation location = (InventoryLocation) selectedType;
				criteria.add(Restrictions.eq(InventoryStock.PROP_LOCATION_ID, location.getId()));
			}

			criteria.add(Property.forName(InventoryStock.PROP_MENU_ITEM_ID).in(detachedCriteria));

			return criteria.list();
		}
	}

	public void addNewStock(InventoryStock stock, InventoryTransaction transaction, MenuItem menuItem) {
		Session session = null;
		Transaction tx = null;

		try {

			session = createNewSession();
			tx = session.beginTransaction();

			session.saveOrUpdate(stock);
			session.saveOrUpdate(transaction);
			session.saveOrUpdate(menuItem);

			tx.commit();

		} catch (Exception e) {
			if (tx != null) {
				tx.rollback();
			}

			throw e;

		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

	public InventoryStock getInventoryStock(MenuItem menuItem, InventoryLocation location, String unitCode) {
		if (location == null) {
			return getInventoryStock(menuItem, (String) null, null, unitCode);
		}
		return getInventoryStock(menuItem, (String) location.getId(), location.getOutletId(), unitCode);
	}

	public InventoryStock getInventoryStock(MenuItem menuItem, String locationId, String outletId, String unitCode) {
		Session session = null;
		try {
			session = createNewSession();
			return getInventoryStock(menuItem, locationId, outletId, unitCode, session);
		} catch (Exception e) {
			PosLog.error(InventoryStockDAO.class, e.getMessage(), e);
		} finally {
			closeSession(session);
		}
		return null;
	}

	public InventoryStock getInventoryStock(MenuItem menuItem, String locationId, String outletId, String unitCode, Session session) {
		Criteria criteria = session.createCriteria(InventoryStock.class);
		if (menuItem != null) {
			criteria.add(Restrictions.eq(InventoryStock.PROP_MENU_ITEM_ID, menuItem.getId()));
		}
		if (locationId != null) {
			criteria.add(Restrictions.eq(InventoryStock.PROP_LOCATION_ID, locationId));
		}
		if (outletId != null) {
			criteria.add(Restrictions.eq(InventoryStock.PROP_OUTLET_ID, outletId));
		}
		if (unitCode != null)
			criteria.add(Restrictions.eq(InventoryStock.PROP_UNIT, unitCode));
		List list = criteria.list();
		if (list != null && list.size() > 0) {
			return (InventoryStock) list.get(0);
		}
		return null;
	}

	public List<InventoryStock> getInventoryStocks(MenuItem menuItem) {
		return getInventoryStocks(menuItem, null);
	}

	public List<InventoryStock> getInventoryStocks(MenuItem menuItem, InventoryLocation location) {
		Session session = null;
		Criteria criteria = null;
		try {
			session = getSession();
			criteria = session.createCriteria(InventoryStock.class);

			criteria.add(Restrictions.eq(InventoryStock.PROP_MENU_ITEM_ID, menuItem.getId()));
			if (location != null) {
				criteria.add(Restrictions.eq(InventoryStock.PROP_LOCATION_ID, location.getId()));
			}
			return criteria.list();

		} catch (Exception e) {
		} finally {
			closeSession(session);
		}
		return null;
	}

	/**
	 * Returns menu item's on hand quantity for each outlet.
	 * 
	 * Map key: outlet id
	 * Map value: on hand quantity in that outlet.
	 * 
	 * @param menuItem
	 * @return
	 */
	public Map<String, String> getOutletwiseOnHandQuantity(MenuItem menuItem) {
		Map<String, String> outletWiseOnhandQuantity = new HashMap<String, String>();
		Criteria criteria = null;
		try (Session session = createNewSession()) {

			criteria = session.createCriteria(InventoryStock.class);
			if (menuItem != null) {
				criteria.add(Restrictions.eq(InventoryStock.PROP_MENU_ITEM_ID, menuItem.getId()));
			}

			ProjectionList projectionList = Projections.projectionList();
			projectionList.add(Projections.groupProperty(InventoryStock.PROP_OUTLET_ID));
			projectionList.add(Projections.sum(InventoryStock.PROP_QUANTITY_IN_HAND));
			projectionList.add(Projections.groupProperty(InventoryStock.PROP_UNIT));
			criteria.setProjection(projectionList);

			List<Object[]> list = criteria.list();
			if (list == null || list.isEmpty()) {
				return outletWiseOnhandQuantity;
			}
			String oldOutletId = "";
			double totalBaseUnit = 0d;
			for (Iterator iterator = list.iterator(); iterator.hasNext();) {
				Object[] objects = (Object[]) iterator.next();
				if (objects[0] == null) {
					return outletWiseOnhandQuantity;
				}

				String outletId = (objects[0] != null) ? (objects[0]).toString() : "";
				double onHandQuantity = (objects[1] != null) ? ((Number) objects[1]).doubleValue() : 0;
				String currrnetUnitName = (objects[2] != null) ? (objects[2]).toString() : "";

				if (!oldOutletId.equals(outletId)) {
					oldOutletId = outletId;
					totalBaseUnit = 0d;
				}

				totalBaseUnit += InventoryUnitConvertionUtil.getBaseUnitQuantity(currrnetUnitName, menuItem) * onHandQuantity;

				String unitName = menuItem.getUnit() == null ? "" : menuItem.getUnit().getName();
				outletWiseOnhandQuantity.put(outletId, NumberUtil.format6DigitNumber(totalBaseUnit) + " " + unitName);

			}

		}
		return outletWiseOnhandQuantity;
	}

	public void addTransfer(InventoryTransaction inTransaction, InventoryTransaction outTransaction, InventoryStock inStock, InventoryStock outStock,
			MenuItem menuItem) {
		Session session = null;
		Transaction tx = null;

		try {

			session = createNewSession();
			tx = session.beginTransaction();

			session.saveOrUpdate(inStock);
			session.saveOrUpdate(outStock);
			session.saveOrUpdate(inTransaction);
			session.saveOrUpdate(outTransaction);
			session.saveOrUpdate(menuItem);

			tx.commit();

		} catch (Exception e) {
			if (tx != null) {
				tx.rollback();
			}

			throw e;

		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

	public List<InventoryStock> getInventoryOnhandReprotData(String nameOrSku, InventoryLocation location) {
		Session session = null;
		Criteria criteria = null;
		Map<String, String> inventoryLocMap = new HashMap<>();
		Map<String, Double> menuItemMap = new HashMap<>();
		session = getSession();
		criteria = session.createCriteria(InventoryStock.class);
		criteria.addOrder(Order.asc(InventoryStock.PROP_ITEM_NAME));
		if (StringUtils.isNotEmpty(nameOrSku)) {
			criteria.add(Restrictions.or(Restrictions.ilike(InventoryStock.PROP_ITEM_NAME, nameOrSku, MatchMode.ANYWHERE),
					Restrictions.eq(InventoryStock.PROP_SKU, nameOrSku)));
		}
		if (location != null) {
			criteria.add(Restrictions.eq(InventoryStock.PROP_LOCATION_ID, location.getId()));
		}
		List<InventoryStock> stockList = criteria.list();
		for (Iterator<InventoryStock> iterator = stockList.iterator(); iterator.hasNext();) {
			InventoryStock inventoryStock = iterator.next();
			String menuItemId = inventoryStock.getMenuItemId();
			String locationId = inventoryStock.getLocationId();
			Double cost = menuItemMap.get(menuItemId);
			String locationName = inventoryLocMap.get(locationId);
			if (cost == null) {
				cost = getMenuItemCost(menuItemId, session);
			}
			if (locationName == null) {
				locationName = getLocationName(locationId, session);
			}
			inventoryStock.setMenuItemCost(cost);
			inventoryStock.setLocationName(locationName);
		}
		return stockList;
	}

	private String getLocationName(String locationId, Session session) {
		Criteria criteria = session.createCriteria(InventoryLocation.class);
		criteria.add(Restrictions.eq(InventoryLocation.PROP_ID, locationId));
		criteria.setProjection(Projections.property(InventoryLocation.PROP_NAME));
		Object result = criteria.uniqueResult();
		if (result instanceof String) {
			return ((String) result).toString();
		}
		return null;
	}

	private Double getMenuItemCost(String menuItemId, Session session) {
		Criteria criteria = session.createCriteria(MenuItem.class);
		criteria.add(Restrictions.eq(MenuItem.PROP_ID, menuItemId));
		criteria.setProjection(Projections.property(MenuItem.PROP_COST));
		Object result = criteria.uniqueResult();
		if (result instanceof Number) {
			return ((Number) result).doubleValue();
		}
		return 0.0;
	}

	@SuppressWarnings("unchecked")
	public List<InventoryStock> findStocksForStockCountMultiItemSelection(MenuGroup group, InventoryVendor vendor, InventoryLocation location) {
		try (Session session = this.createNewSession()) {
			if (location == null) {
				return null;
			}
			List<String> commonMenuItemsId = null;
			Criteria criteria = null;
			if (vendor != null) {
				criteria = session.createCriteria(InventoryVendorItems.class);
				criteria.add(Restrictions.eq(InventoryVendorItems.PROP_VENDOR, vendor));
				criteria.createAlias(InventoryVendorItems.PROP_ITEM, "item");
				criteria.setProjection(Projections.property("item.id"));
				commonMenuItemsId = criteria.list();
			}

			// menu item
			criteria = session.createCriteria(MenuItem.class);
			this.addDeletedFilter(criteria);
			criteria.add(Restrictions.eq(MenuItem.PROP_INVENTORY_ITEM, Boolean.TRUE));
			criteria.add(Restrictions.isNotNull(MenuItem.PROP_UNIT_ID));

			if (group != null) {
				criteria.add(Restrictions.eq(MenuItem.PROP_MENU_GROUP_ID, group.getId()));
			}
			if (commonMenuItemsId != null && !commonMenuItemsId.isEmpty()) {
				criteria.add(Restrictions.in(MenuItem.PROP_ID, commonMenuItemsId));
			}

			criteria.setProjection(Projections.property(MenuItem.PROP_ID));

			if (commonMenuItemsId == null) {
				commonMenuItemsId = criteria.list();
			}
			else {
				commonMenuItemsId.retainAll(criteria.list());
			}

			criteria = session.createCriteria(InventoryStock.class);
			criteria.addOrder(Order.asc(InventoryStock.PROP_ITEM_NAME));
			criteria.addOrder(Order.asc(InventoryStock.PROP_UNIT));
			criteria.add(Restrictions.eq(InventoryStock.PROP_LOCATION_ID, location.getId()));

			if (commonMenuItemsId != null && !commonMenuItemsId.isEmpty()) {
				criteria.add(Restrictions.in(InventoryStock.PROP_MENU_ITEM_ID, commonMenuItemsId));
			}

			return criteria.list();
		}
	}

	public void saveOrUpdateInventoryStock(List<InventoryStock> dataList, boolean updateLastUpdateTime, boolean updateSyncTime) throws Exception {

		if (dataList == null)
			return;

		Transaction tx = null;
		Session session = null;
		try {
			session = createNewSession();
			tx = session.beginTransaction();

			for (Iterator<InventoryStock> iterator = dataList.iterator(); iterator.hasNext();) {
				InventoryStock item = (InventoryStock) iterator.next();
				InventoryStock existingItem = get(item.getId());
				if (existingItem != null) {
					if (!BaseDataServiceDao.get().shouldSave(item.getLastUpdateTime(), existingItem.getLastUpdateTime())) {
						PosLog.info(getClass(), item.getId() + " already updated"); //$NON-NLS-1$
						continue;
					}
					long version = existingItem.getVersion();
					PropertyUtils.copyProperties(existingItem, item);
					existingItem.setVersion(version);
					existingItem.setUpdateLastUpdateTime(updateLastUpdateTime);
					existingItem.setUpdateSyncTime(updateSyncTime);
					update(existingItem, session);
				}
				else {
					item.setUpdateLastUpdateTime(updateLastUpdateTime);
					item.setUpdateSyncTime(updateSyncTime);
					save(item, session);
				}
			}
			tx.commit();
		} catch (Exception e) {
			tx.rollback();
			throw e;
		} finally {
			closeSession(session);
		}
	}

	@SuppressWarnings("unchecked")
	public List<InventoryStock> findBy(String itemNameOrBarcodeOrSku, Outlet outlet) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(InventoryStock.class);
			ProjectionList projectionList = Projections.projectionList();
			projectionList.add(Projections.groupProperty(InventoryStock.PROP_BARCODE), InventoryStock.PROP_BARCODE);
			projectionList.add(Projections.groupProperty(InventoryStock.PROP_SKU), InventoryStock.PROP_SKU);
			projectionList.add(Projections.groupProperty(InventoryStock.PROP_MENU_ITEM_ID), InventoryStock.PROP_MENU_ITEM_ID);
			projectionList.add(Projections.groupProperty(InventoryStock.PROP_ITEM_NAME), InventoryStock.PROP_ITEM_NAME);
			projectionList.add(Projections.sum(InventoryStock.PROP_QUANTITY_IN_HAND), InventoryStock.PROP_QUANTITY_IN_HAND);
			projectionList.add(Projections.groupProperty(InventoryStock.PROP_UNIT), InventoryStock.PROP_UNIT);
			projectionList.add(Projections.groupProperty(InventoryStock.PROP_OUTLET_ID), InventoryStock.PROP_OUTLET_ID);
			projectionList.add(Projections.groupProperty(InventoryStock.PROP_LOCATION_ID), InventoryStock.PROP_LOCATION_ID);
			criteria.setProjection(projectionList);
			if (outlet != null) {
				criteria.add(Restrictions.eq(InventoryStock.PROP_OUTLET_ID, outlet.getId()));
			}
			if (StringUtils.isNotEmpty(itemNameOrBarcodeOrSku)) {
				Criterion name = Restrictions.ilike(InventoryStock.PROP_ITEM_NAME, itemNameOrBarcodeOrSku.trim(), MatchMode.ANYWHERE);
				Criterion barcode = Restrictions.ilike(InventoryStock.PROP_SKU, itemNameOrBarcodeOrSku.trim(), MatchMode.ANYWHERE);
				Criterion sku = Restrictions.ilike(InventoryStock.PROP_BARCODE, itemNameOrBarcodeOrSku.trim(), MatchMode.ANYWHERE);
				Disjunction or = Restrictions.or(name, barcode, sku);
				criteria.add(or);
			}
			addDeletedFilter(criteria);
			criteria.setResultTransformer(Transformers.aliasToBean(InventoryStock.class));
			return criteria.list();
		}
	}

}