package com.floreantpos.model.dao;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.Iterator;
import java.util.List;

import org.apache.commons.beanutils.PropertyUtils;
import org.apache.commons.lang.StringUtils;
import org.hibernate.Criteria;
import org.hibernate.SQLQuery;
import org.hibernate.Session;
import org.hibernate.criterion.Conjunction;
import org.hibernate.criterion.Criterion;
import org.hibernate.criterion.Projections;
import org.hibernate.criterion.Restrictions;
import org.hibernate.transform.Transformers;

import com.floreantpos.DuplicateDataException;
import com.floreantpos.Messages;
import com.floreantpos.PosException;
import com.floreantpos.model.Customer;
import com.floreantpos.model.Department;
import com.floreantpos.model.MenuItem;
import com.floreantpos.model.OrderType;
import com.floreantpos.model.PriceRule;
import com.floreantpos.model.PriceShift;
import com.floreantpos.model.PriceTable;
import com.floreantpos.model.PriceTableItem;
import com.floreantpos.model.SalesArea;
import com.floreantpos.model.Shift;
import com.floreantpos.util.POSUtil;
import com.floreantpos.util.ShiftUtil;

public class PriceRuleDAO extends BasePriceRuleDAO {

	/**
	 * Default constructor.  Can be used in place of getInstance()
	 */
	public PriceRuleDAO() {
	}

	@Override
	protected Serializable save(Object obj, Session session) {
		validateData((PriceRule) obj, session);
		updateTime(obj);
		return super.save(obj, session);
	}

	@Override
	protected void update(Object obj, Session session) {
		validateData((PriceRule) obj, session);
		updateTime(obj);
		super.update(obj, session);
	}

	@Override
	protected void saveOrUpdate(Object obj, Session session) {
		validateData((PriceRule) obj, session);
		updateTime(obj);
		super.saveOrUpdate(obj, session);
	}

	private void validateData(PriceRule priceRule, Session session) {
		if (priceRuleExists(priceRule, session)) {
			throw new DuplicateDataException(Messages.getString("PriceRuleDAO.1")); //$NON-NLS-1$
		}
	}

	private boolean priceRuleExists(PriceRule priceRule, Session session) {
		Criteria criteria = session.createCriteria(getReferenceClass());
		String existingId = priceRule.getId();
		if (StringUtils.isNotEmpty(existingId)) {
			Criterion existingIdExp = Restrictions.ne("id", existingId); //$NON-NLS-1$
			criteria.add(existingIdExp);
		}
		addDeletedFilter(criteria);

		String outletId = priceRule.getOutletId();
		Criterion outletExp = outletId != null ? Restrictions.eq(PriceRule.PROP_OUTLET_ID, outletId) : Restrictions.isNull(PriceRule.PROP_OUTLET_ID);

		String departmentId = priceRule.getDepartmentId();
		Criterion departmentExp = departmentId != null ? Restrictions.eq(PriceRule.PROP_DEPARTMENT_ID, departmentId)
				: Restrictions.isNull(PriceRule.PROP_DEPARTMENT_ID);

		String salesAreaId = priceRule.getSalesAreaId();
		Criterion salesAreaExp = salesAreaId != null ? Restrictions.eq(PriceRule.PROP_SALES_AREA_ID, salesAreaId)
				: Restrictions.isNull(PriceRule.PROP_SALES_AREA_ID);

		String orderTypeId = priceRule.getOrderTypeId();
		Criterion orderTypeExp = orderTypeId != null ? Restrictions.eq(PriceRule.PROP_ORDER_TYPE_ID, orderTypeId)
				: Restrictions.isNull(PriceRule.PROP_ORDER_TYPE_ID);

		String customerGroupId = priceRule.getCustomerGroupId();
		Criterion customerGroupExp = customerGroupId != null ? Restrictions.eq(PriceRule.PROP_CUSTOMER_GROUP_ID, customerGroupId)
				: Restrictions.isNull(PriceRule.PROP_CUSTOMER_GROUP_ID);

		String priceShiftId = priceRule.getPriceShiftId();
		Criterion priceShiftExp = priceShiftId != null ? Restrictions.eq(PriceRule.PROP_PRICE_SHIFT_ID, priceShiftId)
				: Restrictions.isNull(PriceRule.PROP_PRICE_SHIFT_ID);

		String priceTableId = priceRule.getPriceTableId();
		Criterion priceTableExp = priceTableId != null ? Restrictions.eq(PriceRule.PROP_PRICE_TABLE_ID, priceTableId)
				: Restrictions.isNull(PriceRule.PROP_PRICE_TABLE_ID);

		Conjunction priceRuleExp = Restrictions.and(outletExp, departmentExp, salesAreaExp, orderTypeExp, customerGroupExp, priceShiftExp, priceTableExp);
		criteria.add(priceRuleExp);

		List list = criteria.list();
		if (list != null && list.size() > 0) {
			return true;
		}
		return false;
	}

	@Override
	public List<PriceRule> findAll() {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			addDeletedFilter(criteria);
			return criteria.list();
		}
	}

	@Override
	protected void delete(Object obj, Session s) {
		if (obj instanceof PriceRule) {
			PriceRule priceRule = (PriceRule) obj;
			priceRule.setDeleted(Boolean.TRUE);
			super.update(priceRule, s);
		}
		else {
			throw new PosException(Messages.getString("PriceRuleDAO.0")); //$NON-NLS-1$
		}
	}

	public Double getPrice(MenuItem menuItem, OrderType orderType, Department department, SalesArea salesArea, Customer customer) {
		return getPrice(menuItem, orderType, department, salesArea, customer, null);
	}

	public Double getPrice(MenuItem menuItem, OrderType orderType, Department department, SalesArea salesArea, Customer customer, String unit) {
		Session session = null;
		try {
			session = createNewSession();
			return getPrice(menuItem, orderType, department, salesArea, customer, unit, session);
		} finally {
			if (session != null) {
				try {
					session.close();
				} catch (Exception e) {
				}
			}
		}
	}

	public Double getPrice(MenuItem menuItem, OrderType orderType, Department department, SalesArea salesArea, Customer customer, String unit,
			Session session) {

		List<PriceShift> currentPriceShifts = ShiftUtil.getCurrentPriceShifts();

		List<PriceRule> priceRules = getPriceRule(menuItem, orderType, department, salesArea, customer, currentPriceShifts, session);

		List<String> priceShiftIds = null;
		if (currentPriceShifts != null && !currentPriceShifts.isEmpty()) {
			priceShiftIds = POSUtil.getStringIds(currentPriceShifts, PriceShift.class);
		}
		if (priceRules == null || priceRules.isEmpty())
			return null;

		if (priceShiftIds != null && !priceShiftIds.isEmpty()) {
			List<String> nameIdList = new ArrayList<String>(priceShiftIds);
			priceRules.sort(Comparator.comparing(v -> nameIdList.indexOf(v.getPriceShiftId())));
		}
		List<String> priceTableIds = POSUtil.getStringIds(priceRules, PriceRule.class, "getPriceTableId"); //$NON-NLS-1$
		if (priceTableIds == null || priceTableIds.isEmpty()) {
			return null;
		}
		//search in PriceTableItem
		Criteria criteria = session.createCriteria(PriceTableItem.class);
		criteria.add(Restrictions.eq(PriceTableItem.PROP_MENU_ITEM_ID, menuItem.getId()));
		if (StringUtils.isNotBlank(unit)) {
			if (menuItem.getUnit() != null && menuItem.getUnit().getCode().equals(unit)) {
				criteria.add(Restrictions.or(Restrictions.isNull(PriceTableItem.PROP_UNIT), Restrictions.eq(PriceTableItem.PROP_UNIT, unit)));
			}
			else {
				criteria.add(Restrictions.eq(PriceTableItem.PROP_UNIT, unit));
			}
		}
		addDeletedFilter(criteria);
		criteria.add(Restrictions.in(PriceTableItem.PROP_PRICE_TABLE_ID, priceTableIds));

		List<PriceTableItem> prices = criteria.list();
		if (prices != null && prices.size() > 0) {
			if (prices.size() == 1) {
				PriceTableItem item = prices.get(0);
				return item.getPrice();
			}
			else {
				prices.sort(Comparator.comparing(v -> priceTableIds.indexOf(v.getPriceTableId())));
				PriceTableItem item = prices.get(0);
				return item.getPrice();
			}
		}
		return null;
	}

	private List<String> getFilteredPriceShift() {
		int firstPriorityValue = 1;
		List<String> filteredPriceShift = new ArrayList<>();
		List<PriceShift> priceShifts = ShiftUtil.getCurrentPriceShifts();
		if (priceShifts != null) {
			for (PriceShift priceShift : priceShifts) {
				Integer priorityValue = priceShift.getPriority();
				if (firstPriorityValue >= priorityValue) {
					firstPriorityValue = priorityValue;
					filteredPriceShift.add(priceShift.getId());
				}
			}
		}
		return filteredPriceShift;
	}

	public List<PriceRule> findByPriceShiftId(String shiftId, Session session) {
		return this.findByPriceShiftId(shiftId, session, Boolean.FALSE);
	}

	public List<PriceRule> findByPriceShiftId(String shiftId, Session session, boolean addDeletedFilter) {
		if (StringUtils.isBlank(shiftId)) {
			return null;
		}

		Criteria criteria = session.createCriteria(getReferenceClass());
		if (addDeletedFilter) {
			this.addDeletedFilter(criteria);
		}
		criteria.add(Restrictions.eq(PriceRule.PROP_PRICE_SHIFT_ID, shiftId));
		criteria.setProjection(Projections.alias(Projections.property(PriceRule.PROP_NAME), PriceRule.PROP_NAME));
		criteria.setResultTransformer(Transformers.aliasToBean(PriceRule.class));
		return criteria.list();
	}

	@SuppressWarnings("unchecked")
	public List<PriceRule> getPriceRulesByPriceTable(PriceTable priceTable, Session session) {
		if (priceTable == null || priceTable.getId() == null) {
			return null;
		}
		Criteria criteria = session.createCriteria(this.getReferenceClass());
		this.addDeletedFilter(criteria);
		criteria.add(Restrictions.eq(PriceRule.PROP_PRICE_TABLE_ID, priceTable.getId()));
		criteria.setProjection(Projections.alias(Projections.property(PriceRule.PROP_NAME), PriceRule.PROP_NAME));
		return criteria.setResultTransformer(Transformers.aliasToBean(this.getReferenceClass())).list();
	}

	public void saveOrUpdatePriceRuleList(List<PriceRule> dataList, String clientOutletId, boolean updateLastUpdateTime, boolean updateSyncTime)
			throws Exception {
		if (dataList != null && dataList.size() > 0) {
			for (Iterator<PriceRule> iterator = dataList.iterator(); iterator.hasNext();) {
				PriceRule item = (PriceRule) iterator.next();
				PriceRuleDAO dao = PriceRuleDAO.getInstance();
				PriceRule existingItem = dao.get(item.getId());
				if (existingItem != null) {
					final String id = existingItem.getId();
					long version = existingItem.getVersion();
					PropertyUtils.copyProperties(existingItem, item);
					existingItem.setId(id);
					existingItem.setVersion(version);
					existingItem.setUpdateLastUpdateTime(updateLastUpdateTime);
					existingItem.setUpdateSyncTime(updateSyncTime);
					dao.update(existingItem);
				}
				else {
					item.setVersion(0);
					item.setUpdateLastUpdateTime(updateLastUpdateTime);
					item.setUpdateSyncTime(updateSyncTime);
					dao.save(item);
				}
			}
		}
	}

	/**
	 * Contains sql_query
	 * 
	 * @param menuItem
	 * @param orderType
	 * @param department
	 * @param salesArea
	 * @param customer
	 * @param currentPriceShifts
	 * @param session
	 * @return
	 */
	public List<PriceRule> getPriceRule(MenuItem menuItem, OrderType orderType, Department department, SalesArea salesArea, Customer customer,
			List<PriceShift> currentPriceShifts, Session session) {

		String priceShiftIds = null;
		if (currentPriceShifts != null && !currentPriceShifts.isEmpty()) {
			priceShiftIds = POSUtil.createStringByField(currentPriceShifts, Shift.class, "getId", true); //$NON-NLS-1$
		}

		String hql = "SELECT * FROM PRICE_RULE as pr"; //$NON-NLS-1$
		hql += " LEFT JOIN CUSTOMER_CUST_GROUP ccg ON"; //$NON-NLS-1$
		hql += " pr.CUSTOMER_GROUP_ID = ccg.CUST_GROUP_ID"; //$NON-NLS-1$
		hql += " LEFT JOIN PRICE_TABLE pt ON"; //$NON-NLS-1$
		hql += " pt.ID = pr.PRICE_TABLE_ID"; //$NON-NLS-1$

		hql += " WHERE"; //$NON-NLS-1$
		hql += " (pr.DELETED IS NULL OR pr.DELETED = FALSE)"; //$NON-NLS-1$
		hql += " AND (pt.DELETED IS NULL OR pt.DELETED = FALSE)"; //$NON-NLS-1$

		hql += " AND"; //$NON-NLS-1$
		hql += " pr.ACTIVE = true"; //$NON-NLS-1$

		//orderType filter
		boolean includeOrderTypeFilter = orderType != null;
		hql += " AND"; //$NON-NLS-1$
		if (includeOrderTypeFilter) {
			hql += " ("; //$NON-NLS-1$
		}
		hql += " pr.ORDER_TYPE_ID IS NULL"; //$NON-NLS-1$
		if (includeOrderTypeFilter) {
			hql += " OR"; //$NON-NLS-1$
			hql += " pr.ORDER_TYPE_ID = '" + orderType.getId() + "')"; //$NON-NLS-1$ //$NON-NLS-2$
		}

		//department filter
		boolean includeDepartmentFilter = department != null;
		hql += " AND"; //$NON-NLS-1$ 
		if (includeDepartmentFilter) {
			hql += " ("; //$NON-NLS-1$ 
		}
		hql += " pr.DEPARTMENT_ID IS NULL"; //$NON-NLS-1$ 
		if (includeDepartmentFilter) {
			hql += " OR"; //$NON-NLS-1$ 
			hql += " pr.DEPARTMENT_ID = '" + department.getId() + "')"; //$NON-NLS-1$ //$NON-NLS-2$
		}

		//salesArea filter
		boolean includeSalesAreaFilter = salesArea != null;
		hql += " AND"; //$NON-NLS-1$ 
		if (includeSalesAreaFilter) {
			hql += " ("; //$NON-NLS-1$ 
		}
		hql += " pr.SALES_AREA_ID IS NULL"; //$NON-NLS-1$ 

		if (includeSalesAreaFilter) {
			hql += " OR"; //$NON-NLS-1$
			hql += " pr.SALES_AREA_ID = '" + salesArea.getId() + "')"; //$NON-NLS-1$ //$NON-NLS-2$
		}

		if (StringUtils.isNotBlank(priceShiftIds)) {
			//priceShiftIds filter
			hql += " AND ("; //$NON-NLS-1$ 
			hql += " pr.PRICE_SHIFT_ID in (" + priceShiftIds + ")"; //$NON-NLS-1$ //$NON-NLS-2$

			hql += " OR"; //$NON-NLS-1$ 
			hql += " pr.PRICE_SHIFT_ID IS NULL )"; //$NON-NLS-1$ 
		}
		else {
			hql += " AND ("; //$NON-NLS-1$ 
			hql += " pr.PRICE_SHIFT_ID IS NULL )"; //$NON-NLS-1$ 
		}

		//customerGroup filter
		boolean includeCustomer = customer != null;
		hql += " AND "; //$NON-NLS-1$ 
		if (includeCustomer) {
			hql += " ("; //$NON-NLS-1$ 
		}
		hql += " pr.CUSTOMER_GROUP_ID IS NULL"; //$NON-NLS-1$ 
		if (includeCustomer) {
			hql += " OR"; //$NON-NLS-1$
			hql += " ccg.CUSTOMER_ID = '" + customer.getId() + "')"; //$NON-NLS-1$ //$NON-NLS-2$
		}

		//customerIsMember filter
		hql += " AND "; //$NON-NLS-1$ 
		if (includeCustomer) {
			hql += " ("; //$NON-NLS-1$ 
		}
		hql += " pr.CUSTOMER_IS_MEMBER IS NULL"; //$NON-NLS-1$ 
		if (includeCustomer) {
			hql += " OR"; //$NON-NLS-1$ 
			hql += " pr.CUSTOMER_IS_MEMBER = " + customer.isMembershipActive() + ")"; //$NON-NLS-1$ //$NON-NLS-2$
		}

		SQLQuery query = session.createSQLQuery(hql);
		query.addEntity(PriceRule.class);
		return query.list();
	}

}