package com.floreantpos.model.dao;

import java.io.Serializable;
import java.util.Iterator;
import java.util.List;

import org.hibernate.Criteria;
import org.hibernate.Query;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.hibernate.criterion.Restrictions;

import com.floreantpos.PosLog;
import com.floreantpos.model.ShopFloor;
import com.floreantpos.model.ShopFloorTemplate;
import com.orocube.rest.service.server.BaseDataServiceDao;

public class ShopFloorTemplateDAO extends BaseShopFloorTemplateDAO {

	public ShopFloorTemplateDAO() {
	}

	@Override
	protected Serializable save(Object obj, Session s) {
		updateTime(obj);
		return super.save(obj, s);
	}

	@Override
	protected void update(Object obj, Session s) {
		updateTime(obj);
		super.update(obj, s);
	}

	@Override
	protected void saveOrUpdate(Object obj, Session s) {
		updateTime(obj);
		super.saveOrUpdate(obj, s);
	}

	@SuppressWarnings("unchecked")
	public List<ShopFloorTemplate> findByParent(ShopFloor selectedValue) {
		try (Session session = this.createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.eq(ShopFloorTemplate.PROP_FLOOR, selectedValue));
			return criteria.list();

		} catch (Exception e) {
		}
		return null;
	}

	public void saveOrUpdateTemplates(List<ShopFloorTemplate> templates) {
		if (templates == null) {
			return;
		}
		Session session = null;
		Transaction tx = null;
		try {
			session = createNewSession();
			tx = session.beginTransaction();
			for (ShopFloorTemplate template : templates) {
				session.saveOrUpdate(template);
			}
			tx.commit();
		} catch (Exception e) {
			tx.rollback();
		} finally {
			closeSession(session);
		}

	}

	public void deleteTemplates(List<ShopFloorTemplate> templates) {
		if (templates == null) {
			return;
		}
		Session session = null;
		Transaction tx = null;
		try {
			session = createNewSession();
			tx = session.beginTransaction();
			for (ShopFloorTemplate template : templates) {
				session.delete(template);
			}
			tx.commit();
		} catch (Exception e) {
			tx.rollback();
		} finally {
			closeSession(session);
		}
	}

	public void setDefaultTemplate(ShopFloorTemplate template, ShopFloor shopFloor) {
		Transaction transaction = null;
		try (Session session = this.createNewSession()) {
			transaction = session.beginTransaction();

			String hql = "UPDATE %s template SET template.%s = :default WHERE template.%s = :floor"; //$NON-NLS-1$
			hql = String.format(hql, ShopFloorTemplate.REF, ShopFloorTemplate.PROP_DEFAULT_FLOOR, ShopFloorTemplate.PROP_FLOOR);
			Query query = session.createQuery(hql);
			query.setParameter("default", Boolean.FALSE); //$NON-NLS-1$
			query.setParameter("floor", shopFloor); //$NON-NLS-1$
			query.executeUpdate();

			template.setDefaultFloor(true);
			session.update(template);
			session.update(shopFloor);

			transaction.commit();
		} catch (Exception e) {
			if (transaction != null) {
				transaction.commit();
			}
		}
	}

	//download and upload
	public void saveOrUpdateShopFloorTemplates(List<ShopFloorTemplate> dataList, boolean updateLastUpdateTime, boolean updateSyncTime) throws Exception {

		if (dataList == null)
			return;

		Transaction tx = null;
		try (Session session = createNewSession()) {
			tx = session.beginTransaction();
			for (Iterator<ShopFloorTemplate> iterator = dataList.iterator(); iterator.hasNext();) {
				ShopFloorTemplate shopFloor = (ShopFloorTemplate) iterator.next();
				ShopFloorTemplate existingShopFloorTemplate = get(shopFloor.getId());

				shopFloor.setUpdateLastUpdateTime(updateLastUpdateTime);
				shopFloor.setUpdateSyncTime(updateSyncTime);

				if (existingShopFloorTemplate == null) {
					save(shopFloor, session);
				}
				else {
					if (!BaseDataServiceDao.get().shouldSave(shopFloor.getLastUpdateTime(), existingShopFloorTemplate.getLastUpdateTime())) {
						PosLog.info(getClass(), shopFloor.getName() + " already updated"); //$NON-NLS-1$
						continue;
					}
					shopFloor.setVersion(existingShopFloorTemplate.getVersion());
				}
				update(shopFloor, session);
			}
			tx.commit();
		} catch (Exception e) {
			tx.rollback();
			throw e;
		}
	}
}