/**
 * ************************************************************************
 * * The contents of this file are subject to the MRPL 1.2
 * * (the  "License"),  being   the  Mozilla   Public  License
 * * Version 1.1  with a permitted attribution clause; you may not  use this
 * * file except in compliance with the License. You  may  obtain  a copy of
 * * the License at http://www.floreantpos.org/license.html
 * * Software distributed under the License  is  distributed  on  an "AS IS"
 * * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * * License for the specific  language  governing  rights  and  limitations
 * * under the License.
 * * The Original Code is FLOREANT POS.
 * * The Initial Developer of the Original Code is OROCUBE LLC
 * * All portions are Copyright (C) 2015 OROCUBE LLC
 * * All Rights Reserved.
 * ************************************************************************
 */
package com.floreantpos.model;

import java.util.Set;

import javax.xml.bind.annotation.XmlRootElement;

import org.apache.commons.lang.StringUtils;

import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.floreantpos.PosLog;
import com.floreantpos.model.base.BaseGratuity;
import com.floreantpos.model.dao.TicketDAO;
import com.google.gson.Gson;
import com.google.gson.JsonObject;

@JsonIgnoreProperties(ignoreUnknown = true, value = { "ticket" })
@XmlRootElement
public class Gratuity extends BaseGratuity implements TimedModel, PropertyContainer {
	private static final String AUTO_CALCULATED = "auto_calculated";
	private static final long serialVersionUID = 1L;
	private transient boolean syncEdited;
	private transient Ticket ticket;

	private boolean updateLastUpdateTime = true;
	private boolean updateSyncTime = false;
	private Double transactionTipsAmount;

	private transient com.google.gson.JsonObject propertiesContainer;

	/*[CONSTRUCTOR MARKER BEGIN]*/
	public Gratuity() {
		setAutoCalculated(false);
	}

	public void calculateTransactionTips() {
		try {
			transactionTipsAmount = 0.0;
			String ticketId2 = getTicketId();
			if (StringUtils.isEmpty(ticketId2))
				return;
			Ticket ticket = TicketDAO.getInstance().loadCouponsAndTransactions(ticketId2, getOutletId());
			if (ticket == null)
				return;
			Set<PosTransaction> transactions = ticket.getTransactions();
			if (transactions != null && transactions.size() > 0) {
				for (PosTransaction posTransaction : transactions) {
					if (posTransaction.isVoided())
						continue;
					else if (posTransaction instanceof RefundTransaction) {
						transactionTipsAmount -= posTransaction.getTipsAmount();
					}
					else {
						transactionTipsAmount += posTransaction.getTipsAmount();
					}
				}
			}
		} catch (Exception e) {
			PosLog.error(getClass(), e);
		}
	}

	public Double getTransactionTipsAmount() {
		if (transactionTipsAmount == null) {
			calculateTransactionTips();
		}
		return transactionTipsAmount;
	}

	public Ticket getTicket() {
		if (ticket == null && StringUtils.isNotEmpty(getTicketId())) {
			ticket = TicketDAO.getInstance().get(getTicketId(), getOutletId());
		}
		return ticket;
	}

	public boolean isSyncEdited() {
		return syncEdited;
	}

	public void setSyncEdited(boolean syncEdited) {
		this.syncEdited = syncEdited;
	}

	public boolean isUpdateSyncTime() {
		return updateSyncTime;
	}

	public void setUpdateSyncTime(boolean shouldUpdateSyncTime) {
		this.updateSyncTime = shouldUpdateSyncTime;
	}

	public boolean isUpdateLastUpdateTime() {
		return updateLastUpdateTime;
	}

	public void setUpdateLastUpdateTime(boolean shouldUpdateUpdateTime) {
		this.updateLastUpdateTime = shouldUpdateUpdateTime;
	}

	@Override
	public String getProperties() {
		if (propertiesContainer != null) {
			return propertiesContainer.toString();
		}

		String properties = super.getProperties();
		if (StringUtils.isEmpty(properties)) {
			return null;
		}

		propertiesContainer = new Gson().fromJson(properties, com.google.gson.JsonObject.class);
		return properties;
	}

	@Override
	public void setProperties(String properties) {
		super.setProperties(properties);
		propertiesContainer = new Gson().fromJson(properties, com.google.gson.JsonObject.class);
	}

	@Override
	public JsonObject getPropertyStore() {
		if (propertiesContainer == null) {
			propertiesContainer = new JsonObject();
		}
		return propertiesContainer;
	}

	public void addProperty(String key, String value) {
		this.getPropertyStore().addProperty(key, value);
		setProperties(this.getPropertyStore().toString());
	}

	public void removeProperty(String propertyName) {
		if (this.getPropertyStore() != null) {
			this.getPropertyStore().remove(propertyName);
		}
		setProperties(this.getPropertyStore().toString());
	}

	public boolean isAutoCalculated() {
		return Boolean.valueOf(getProperty(AUTO_CALCULATED));
	}

	public void setAutoCalculated(boolean autoCalculated) {
		addProperty(AUTO_CALCULATED, String.valueOf(autoCalculated));
	}
}