package com.floreantpos.util;

import java.io.InputStream;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.LinkedList;
import java.util.List;

import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.StringUtils;

import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.floreantpos.PosLog;

public class AddressUtil {
	private List<Country> countries;

	public AddressUtil() {
		this.initCountries();
	}

	public List<String> getCityNames(String countryName, String stateName) {
		List<String> names = new LinkedList<>();
		for (Country country : this.countries) {
			if (StringUtils.isBlank(country.getName())) {
				continue;
			}
			if (country.getName().equalsIgnoreCase(countryName)) {
				for (State state : country.getStates()) {
					if (state.getName().equalsIgnoreCase(stateName)) {
						for (City city : state.getCities()) {
							names.add(city.getName());
						}
						return names;
					}
				}
			}
		}
		return names;
	}

	public List<String> getStateNames(String countryName) {
		List<String> names = new LinkedList<>();
		for (Country country : this.countries) {
			if (StringUtils.isBlank(country.getName())) {
				continue;
			}
			if (country.getName().equalsIgnoreCase(countryName)) {
				for (State state : country.getStates()) {
					names.add(state.getName());
				}
				return names;
			}
		}
		return names;
	}

	public List<String> getCountryNames() {
		List<String> names = new LinkedList<>();
		for (Country country : this.countries) {
			names.add(country.getName());
		}
		return names;
	}

	private void initCountries() {
		if (this.countries != null && this.countries.size() > 0) {
			return;
		}
		String fileName = "/country-list.json"; //$NON-NLS-1$

		try (InputStream countryResourceStream = getClass().getResourceAsStream(fileName)) {
			String countryString = IOUtils.toString(countryResourceStream);
			ObjectMapper mapper = new ObjectMapper();
			this.countries = mapper.readValue(countryString, new TypeReference<List<Country>>() {
			});
			Collections.sort(countries, Comparator.comparing(Country::getName, Comparator.nullsFirst(String.CASE_INSENSITIVE_ORDER)));
		} catch (Exception e0) {
			this.countries = new ArrayList<>(0);
			PosLog.error(getClass(), e0);
		}
	}

	public static class Country implements Serializable {
		private String name;
		private List<State> states;

		public Country() {
		}

		public String getName() {
			return name;
		}

		public void setName(String name) {
			this.name = name;
		}

		public List<State> getStates() {
			return states;
		}

		public void setStates(List<State> states) {
			this.states = states;
		}

		@Override
		public String toString() {
			return this.name;
		}

		@Override
		public int hashCode() {
			final int prime = 31;
			int result = 1;
			result = prime * result + ((name == null) ? 0 : name.hashCode());
			return result;
		}

		@Override
		public boolean equals(Object obj) {
			if (this == obj)
				return true;
			if (obj == null)
				return false;
			if (getClass() != obj.getClass())
				return false;
			Country other = (Country) obj;
			if (name == null) {
				if (other.name != null)
					return false;
			}
			else if (!name.equals(other.name))
				return false;
			return true;
		}
	}

	public static class State implements Serializable {
		private String name;
		private List<City> cities;

		public State() {
		}

		public String getName() {
			return name;
		}

		public void setName(String name) {
			this.name = name;
		}

		public List<City> getCities() {
			return cities;
		}

		public void setCities(List<City> cities) {
			this.cities = cities;
		}

		@Override
		public String toString() {
			return this.name;
		}

		@Override
		public int hashCode() {
			final int prime = 31;
			int result = 1;
			result = prime * result + ((name == null) ? 0 : name.hashCode());
			return result;
		}

		@Override
		public boolean equals(Object obj) {
			if (this == obj)
				return true;
			if (obj == null)
				return false;
			if (getClass() != obj.getClass())
				return false;
			State other = (State) obj;
			if (name == null) {
				if (other.name != null)
					return false;
			}
			else if (!name.equals(other.name))
				return false;
			return true;
		}
	}

	public static class City implements Serializable {
		private String name;

		public City() {
		}

		public String getName() {
			return name;
		}

		public void setName(String name) {
			this.name = name;
		}

		@Override
		public String toString() {
			return this.name;
		}

		@Override
		public int hashCode() {
			final int prime = 31;
			int result = 1;
			result = prime * result + ((name == null) ? 0 : name.hashCode());
			return result;
		}

		@Override
		public boolean equals(Object obj) {
			if (this == obj)
				return true;
			if (obj == null)
				return false;
			if (getClass() != obj.getClass())
				return false;
			City other = (City) obj;
			if (name == null) {
				if (other.name != null)
					return false;
			}
			else if (!name.equals(other.name))
				return false;
			return true;
		}
	}
}
