package com.orocube.rest.service.server;

import java.beans.PropertyDescriptor;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Calendar;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.beanutils.PropertyUtils;
import org.apache.commons.lang.StringUtils;
import org.hibernate.Criteria;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.hibernate.criterion.Restrictions;
import org.json.JSONArray;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.floreantpos.PosLog;
import com.floreantpos.model.AttendenceHistory;
import com.floreantpos.model.Attribute;
import com.floreantpos.model.CashDrawer;
import com.floreantpos.model.ComboGroup;
import com.floreantpos.model.ComboItem;
import com.floreantpos.model.ComboTicketItem;
import com.floreantpos.model.Discount;
import com.floreantpos.model.InventoryStockUnit;
import com.floreantpos.model.InventoryTransaction;
import com.floreantpos.model.KitchenTicket;
import com.floreantpos.model.KitchenTicketItem;
import com.floreantpos.model.MenuCategory;
import com.floreantpos.model.MenuGroup;
import com.floreantpos.model.MenuItem;
import com.floreantpos.model.MenuItemModifierPage;
import com.floreantpos.model.MenuItemModifierPageItem;
import com.floreantpos.model.MenuItemModifierSpec;
import com.floreantpos.model.MenuModifier;
import com.floreantpos.model.MenuShift;
import com.floreantpos.model.ModifiableTicketItem;
import com.floreantpos.model.ModifierGroup;
import com.floreantpos.model.ModifierMultiplierPrice;
import com.floreantpos.model.OrderType;
import com.floreantpos.model.PizzaModifierPrice;
import com.floreantpos.model.PizzaPrice;
import com.floreantpos.model.PosTransaction;
import com.floreantpos.model.StoreSession;
import com.floreantpos.model.Terminal;
import com.floreantpos.model.Ticket;
import com.floreantpos.model.TicketItem;
import com.floreantpos.model.TicketItemModifier;
import com.floreantpos.model.User;
import com.floreantpos.model.VoidItem;
import com.floreantpos.model.dao.AttendenceHistoryDAO;
import com.floreantpos.model.dao.AttributeDAO;
import com.floreantpos.model.dao.CashDrawerDAO;
import com.floreantpos.model.dao.ComboGroupDAO;
import com.floreantpos.model.dao.ComboItemDAO;
import com.floreantpos.model.dao.DiscountDAO;
import com.floreantpos.model.dao.InventoryStockUnitDAO;
import com.floreantpos.model.dao.InventoryTransactionDAO;
import com.floreantpos.model.dao.KitchenTicketDAO;
import com.floreantpos.model.dao.KitchenTicketItemDAO;
import com.floreantpos.model.dao.MenuItemDAO;
import com.floreantpos.model.dao.MenuItemModifierPageDAO;
import com.floreantpos.model.dao.MenuItemModifierPageItemDAO;
import com.floreantpos.model.dao.MenuItemModifierSpecDAO;
import com.floreantpos.model.dao.MenuModifierDAO;
import com.floreantpos.model.dao.ModifierGroupDAO;
import com.floreantpos.model.dao.ModifierMultiplierPriceDAO;
import com.floreantpos.model.dao.PizzaModifierPriceDAO;
import com.floreantpos.model.dao.PizzaPriceDAO;
import com.floreantpos.model.dao.StoreDAO;
import com.floreantpos.model.dao.TerminalDAO;
import com.floreantpos.model.dao.UserDAO;
import com.floreantpos.model.dao.VoidItemDAO;
import com.floreantpos.model.ext.KitchenStatus;
import com.floreantpos.model.util.DataProvider;
import com.floreantpos.util.POSUtil;
import com.floreantpos.util.StoreUtil;
import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.orocube.rest.service.JsonBeanFieldExcluder;
import com.orocube.rest.service.ServiceUtils;

public class BaseDataServiceDao {
	public static final String GSON_PARSING_DATE_FORMAT = "MMM dd, yyyy HH:mm:ss.SSSZ"; //$NON-NLS-1$
	public static final String DATA = "data"; //$NON-NLS-1$

	private static Logger LOGGER = LoggerFactory.getLogger(BaseDataServiceDao.class);

	private static BaseDataServiceDao instance;

	private BaseDataServiceDao() {

	}

	public void saveOrUpdateTickets(String request) throws Exception {
		saveOrUpdateTickets(request, true);
	}

	public List<Ticket> saveOrUpdateTickets(String request, boolean shouldPublishMqtt) throws Exception {
		JSONObject rootElement = new JSONObject(request);
		JSONArray ticketsArray = rootElement.getJSONArray(DATA);

		debug("Request received: " + request); //$NON-NLS-1$

		List<Ticket> tickets = populateTickets(ticketsArray.toString());
		return saveOrUpdateTickets(rootElement, tickets, shouldPublishMqtt);
	}

	public List<Ticket> saveOrUpdateTickets(JSONObject rootElement, List<Ticket> tickets, boolean shouldPublishMqtt) throws Exception {
		Map<String, CashDrawer> cashDrawerMap = new HashMap<String, CashDrawer>();

		Terminal currentTerminal = DataProvider.get().getCurrentTerminal();

		Calendar calendar = Calendar.getInstance();
		calendar.setTime(StoreDAO.getServerTimestamp());
		calendar.set(Calendar.MILLISECOND, calendar.getMinimum(Calendar.MILLISECOND));

		Date lastUpdateTime = calendar.getTime();
		for (Ticket ticket : tickets) {
			if (ticket.getTerminalId() == 0 && currentTerminal != null) {
				ticket.setTerminalId(currentTerminal.getId());
			}
			ticket.setLastUpdateTime(lastUpdateTime);
			ticket.setShouldPublishMqtt(shouldPublishMqtt);
			Set<PosTransaction> convertedTransactions = new HashSet<PosTransaction>();
			Set<PosTransaction> sourceTransactions = ticket.getTransactions();
			if (sourceTransactions != null && !sourceTransactions.isEmpty()) {
				for (PosTransaction transaction : sourceTransactions) {
					try {
						PosTransaction convertedTransaction = ServiceUtils.convertPosTransactionUsingClassType(transaction);
						convertedTransactions.add(convertedTransaction);
						if (StringUtils.isNotEmpty(convertedTransaction.getCashDrawerId())) {
							CashDrawer requestedCashDrawer = cashDrawerMap.get(convertedTransaction.getCashDrawerId());
							if (requestedCashDrawer == null) {
								requestedCashDrawer = convertedTransaction.getCashDrawer();
								if (requestedCashDrawer == null) {
									continue;
								}
								cashDrawerMap.put(requestedCashDrawer.getId(), requestedCashDrawer);
							}
						}
					} catch (Exception e) {
						PosLog.error(getClass(), e);
					}
				}
				ticket.setTransactions(convertedTransactions);
			}
		}
		if (cashDrawerMap.size() > 0) {
			for (Iterator<CashDrawer> iterator = cashDrawerMap.values().iterator(); iterator.hasNext();) {
				CashDrawer cashDrawer = (CashDrawer) iterator.next();
				BaseDataServiceDao.get().saveOrUpdateCashDrawer(cashDrawer);
			}
		}
		saveOrUpdateTickets(tickets, false, true);
		debug("Successfully saved ticket"); //$NON-NLS-1$

		if (rootElement.has("linkedData")) //$NON-NLS-1$
			saveOrUpdateInventoryTransactions(rootElement.getJSONArray("linkedData")); //$NON-NLS-1$
		return tickets;
	}

	public void saveOrUpdateInventoryTransactions(String request) throws Exception {
		JSONObject jsonObject = new JSONObject(request);
		JSONArray jsonArray = new JSONArray(jsonObject.getJSONArray(DATA).toString());
		saveOrUpdateInventoryTransactions(jsonArray);
	}

	private void saveOrUpdateInventoryTransactions(JSONArray jsonArray) throws Exception {
		List<InventoryTransaction> inventoryTransactions = new ArrayList<>();

		if (jsonArray.length() > 0) {
			for (int i = 0; i < jsonArray.length(); i++) {
				JSONObject json = jsonArray.getJSONObject(i);
				InventoryTransaction invTransaction = getGsonBuilder().setExclusionStrategies(new JsonBeanFieldExcluder(MenuShift.class, null)).create()
						.fromJson(json.toString(), InventoryTransaction.class);
				if (invTransaction != null) {
					inventoryTransactions.add(invTransaction);
				}
			}
		}
		saveOrUpdateInventoryTransactions(inventoryTransactions, false, true);
	}

	public List<Ticket> populateTickets(String request) {
		try {
			if (request == null)
				return null;

			Map<String, OrderType> orderTypeMap = new HashMap<String, OrderType>();
			Map<String, User> userMap = new HashMap<String, User>();
			Map<Integer, Terminal> terminalMap = new HashMap<Integer, Terminal>();

			List<Ticket> tickets = new ArrayList<>();
			JSONArray jsonArray = new JSONArray(request);

			for (int i = 0; i < jsonArray.length(); i++) {
				JSONObject json = jsonArray.getJSONObject(i);

				populateMap(json, "orderType", orderTypeMap, OrderType.class); //$NON-NLS-1$
				populateMap(json, "terminal", terminalMap, Terminal.class); //$NON-NLS-1$
				populateMap(json, "owner", userMap, User.class); //$NON-NLS-1$

				Map<String, List<TicketItemModifier>> ticketItemModifierMap = new HashMap<String, List<TicketItemModifier>>();
				populateTicketItemModifiers(json, ticketItemModifierMap);

				Map<String, ComboTicketItem> comboTicketItemMap = new HashMap<String, ComboTicketItem>();
				populateComboTicketItemFromJson(json, comboTicketItemMap);

				Ticket ticket = getGsonBuilder().setExclusionStrategies(new JsonBeanFieldExcluder(MenuShift.class, null)).create().fromJson(json.toString(),
						Ticket.class);

				ticket.convertDiscountPropertyToList();

				StoreSession currentStoreSession = StoreUtil.getCurrentStoreSession();
				if (currentStoreSession != null)
					ticket.setStoreSessionId(currentStoreSession.getId());

				populateComboTicketItems(comboTicketItemMap, ticket);
				populateTicketItems(ticketItemModifierMap, ticket);
				tickets.add(ticket);
			}

			//saveOrderTypesIfNotExists(new ArrayList<OrderType>(orderTypeMap.values()));
			//saveTerminalsIfNotExists(new ArrayList<Terminal>(terminalMap.values()));
			//saveUserIfNotExists(new ArrayList<User>(userMap.values()));
			return tickets;
		} catch (Exception e) {
			PosLog.error(ServiceUtils.class, e);
		}
		return null;
	}

	//	private void saveOrderTypesIfNotExists(List<OrderType> orderTypes) {
	//		if (orderTypes == null || orderTypes.isEmpty())
	//			return;
	//		for (OrderType orderType : orderTypes) {
	//			OrderType existingOrderType = DataProvider.get().getOrderType(orderType.getId(), orderType.getOutletId());
	//			if (existingOrderType == null) {
	//				OrderTypeDAO.getInstance().save(orderType);
	//				debug("New order type " + orderType.getId() + " " + orderType.getName() + " has been created."); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
	//			}
	//		}
	//	}
	//
	//	private void saveTerminalsIfNotExists(List<Terminal> terminals) {
	//		if (terminals == null || terminals.isEmpty())
	//			return;
	//		for (Terminal terminal : terminals) {
	//			Terminal existingTerminal = TerminalDAO.getInstance().getByTerminalKey(terminal.getTerminalKey());
	//			if (existingTerminal == null) {
	//				terminal.setCurrentCashDrawer(null);
	//				TerminalDAO.getInstance().save(terminal);
	//				debug("New terminal " + terminal.getId() + " " + terminal.getName() + " has been created."); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
	//			}
	//		}
	//	}
	//
	//	private void saveUserIfNotExists(List<User> users) {
	//		if (users == null || users.isEmpty())
	//			return;
	//		for (User user : users) {
	//			User existingUser = DataProvider.get().getUserById(user.getId());
	//			if (existingUser == null) {
	//				UserDAO.getInstance().save(user);
	//				debug("New user " + user.getId() + " " + user.getFirstName() + " has been created."); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
	//			}
	//		}
	//	}

	private void populateMap(final JSONObject json, String objectName, Map dataMap, Class refClass) throws Exception {
		if (!json.has(objectName)) {
			return;
		}
		JSONObject jsonObject = json.getJSONObject(objectName);
		Object data = convertJson(jsonObject.toString(), refClass);
		if (data != null) {
			PropertyDescriptor descriptor = new PropertyDescriptor("id", data.getClass()); //$NON-NLS-1$
			dataMap.put(descriptor.getReadMethod().invoke(data), data);
		}
	}

	private Object convertJson(String json, Class class1) throws Exception {
		return getGsonBuilder().setExclusionStrategies(new JsonBeanFieldExcluder(MenuShift.class)).create().fromJson(json, class1);
	}

	private void populateTicketItems(Map<String, List<TicketItemModifier>> ticketItemModifierMap, Ticket ticket) {
		List<TicketItem> ticketItems = new ArrayList<>();
		for (TicketItem ticketItem : ticket.getTicketItems()) {
			updateTicketItemModifiers(ticketItemModifierMap, ticketItems, ticketItem);

			if (ticketItem instanceof ComboTicketItem) {
				List<TicketItem> newComboChildItems = new ArrayList<>();
				List<TicketItem> comboChildItems = ticketItem.getComboItems();
				if (comboChildItems != null) {
					for (TicketItem comboChildItem : comboChildItems) {
						updateTicketItemModifiers(ticketItemModifierMap, newComboChildItems, comboChildItem);
					}
				}
				ticketItem.setComboItems(newComboChildItems);
			}
		}
		ticket.setTicketItems(ticketItems);
	}

	private void updateTicketItemModifiers(Map<String, List<TicketItemModifier>> ticketItemModifierMap, List<TicketItem> ticketItems, TicketItem ticketItem) {
		List<TicketItemModifier> ticketItemModifiers = ticketItemModifierMap.get(ticketItem.getId());
		if (ticketItemModifiers != null && ticketItemModifiers.size() > 0) {
			ModifiableTicketItem modifiableTicketItem = new ModifiableTicketItem(ticketItem.getId());
			try {
				PropertyUtils.copyProperties(modifiableTicketItem, ticketItem);
			} catch (Exception e) {
			}
			modifiableTicketItem.setId(ticketItem.getId());
			modifiableTicketItem.setTicketItemModifiers(ticketItemModifiers);
			ticketItems.add(modifiableTicketItem);
		}
		else {
			ticketItems.add(ticketItem);
		}
	}

	private void populateComboTicketItems(Map<String, ComboTicketItem> comboTicketItemMap, Ticket ticket) {
		List<TicketItem> ticketItems = new ArrayList<>();
		for (TicketItem ticketItem : ticket.getTicketItems()) {
			ComboTicketItem comboTicketItem = comboTicketItemMap.get(ticketItem.getId());
			if (comboTicketItem != null) {
				ticketItems.add(comboTicketItem);
			}
			else {
				ticketItems.add(ticketItem);
			}
		}
		ticket.setTicketItems(ticketItems);
	}

	private static void populateTicketItemModifiers(JSONObject json, Map<String, List<TicketItemModifier>> ticketItemModifierMap)
			throws NoSuchFieldException, ClassNotFoundException {
		if (json.has("ticketItemModifiers")) { //$NON-NLS-1$
			populateTicketItemModifiersToMap(null, ticketItemModifierMap, json.getJSONArray("ticketItemModifiers")); //$NON-NLS-1$
			return;
		}
		if (!json.has("ticketItems")) { //$NON-NLS-1$
			return;
		}
		JSONArray ticketItemArray = json.getJSONArray("ticketItems"); //$NON-NLS-1$
		for (int tm = 0; tm < ticketItemArray.length(); tm++) {
			JSONObject ticketItemJson = ticketItemArray.getJSONObject(tm);
			if (ticketItemJson.has("ticketItemModifiers")) { //$NON-NLS-1$
				populateTicketItemModifiersToMap(ticketItemJson.getString(TicketItem.PROP_ID), ticketItemModifierMap,
						ticketItemJson.getJSONArray("ticketItemModifiers")); //$NON-NLS-1$
			}

			if (!ticketItemJson.has("comboItems")) { //$NON-NLS-1$
				continue;
			}
			JSONArray comboTicketItemArray = ticketItemJson.getJSONArray("comboItems"); //$NON-NLS-1$
			for (int cti = 0; cti < comboTicketItemArray.length(); cti++) {
				JSONObject comboTicketObject = comboTicketItemArray.getJSONObject(cti);

				if (!comboTicketObject.has("ticketItemModifiers")) { //$NON-NLS-1$
					continue;
				}

				populateTicketItemModifiersToMap(comboTicketObject.getString(TicketItem.PROP_ID), ticketItemModifierMap,
						comboTicketObject.getJSONArray("ticketItemModifiers")); //$NON-NLS-1$
			}
		}
	}

	private static void populateComboTicketItemFromJson(JSONObject json, Map<String, ComboTicketItem> comboTicketItemMap)
			throws NoSuchFieldException, ClassNotFoundException {
		if (json.has("ticketItems")) { //$NON-NLS-1$
			JSONArray ticketItemArray = json.getJSONArray("ticketItems"); //$NON-NLS-1$
			for (int tm = 0; tm < ticketItemArray.length(); tm++) {
				JSONObject ticketItemJson = ticketItemArray.getJSONObject(tm);
				if (!ticketItemJson.has("comboItems")) { //$NON-NLS-1$
					continue;
				}

				ComboTicketItem convertedComboTicketItem = getGsonBuilder().setExclusionStrategies(new JsonBeanFieldExcluder(MenuShift.class, null)).create()
						.fromJson(ticketItemJson.toString(), ComboTicketItem.class);
				String comboTicketItemId = convertedComboTicketItem.getId();
				if (StringUtils.isNotBlank(comboTicketItemId)) {
					ComboTicketItem comboTicketItem = comboTicketItemMap.get(comboTicketItemId);
					if (comboTicketItem == null) {
						comboTicketItemMap.put(comboTicketItemId, convertedComboTicketItem);
					}
				}
			}
		}
	}

	private static void populateTicketItemModifiersToMap(String ticketItemId, Map<String, List<TicketItemModifier>> ticketItemModifierMap,
			JSONArray ticketItemModifierJson) throws NoSuchFieldException, ClassNotFoundException {
		if (ticketItemModifierJson == null || ticketItemModifierJson.length() == 0) {
			return;
		}
		for (int tm = 0; tm < ticketItemModifierJson.length(); tm++) {
			JSONObject modifierJson = ticketItemModifierJson.getJSONObject(tm);
			TicketItemModifier ticketItemModifier = getGsonBuilder().setExclusionStrategies(new JsonBeanFieldExcluder(MenuShift.class, null)).create()
					.fromJson(modifierJson.toString(), TicketItemModifier.class);
			String modifierTicketItemId = ticketItemId;
			if (modifierTicketItemId == null && ticketItemModifier.getTicketItem() != null) {
				modifierTicketItemId = ticketItemModifier.getTicketItem().getId();
			}
			if (modifierTicketItemId != null) {
				List<TicketItemModifier> ticketItemModifiers = ticketItemModifierMap.get(modifierTicketItemId);
				if (ticketItemModifiers == null) {
					ticketItemModifiers = new ArrayList<>();
					ticketItemModifierMap.put(modifierTicketItemId, ticketItemModifiers);
				}
				ticketItemModifiers.add(ticketItemModifier);
			}
		}
	}

	public void saveOrUpdateTickets(List<Ticket> dataList, boolean updateLastUpdateTime, boolean updateSyncTime) throws Exception {
		if (dataList == null || dataList.isEmpty()) {
			return;
		}

		for (Ticket ticket : dataList) {
			debug("Start saving ticket id: " + ticket.getId()); //$NON-NLS-1$
			TicketDataServiceDao.get().saveOrUpdateTicket(ticket, updateLastUpdateTime, updateSyncTime);
		}
	}

	public void saveOrUpdateCashDrawers(String request) {
		saveOrUpdateCashDrawers(request, getOutletId(request));
	}

	private String getOutletId(String request) {
		JSONObject rootElement = new JSONObject(request);
		String terminalKey = rootElement.has("deviceId") ? rootElement.getString("deviceId") : null; //$NON-NLS-1$ //$NON-NLS-2$
		if (StringUtils.isNotEmpty(terminalKey)) {
			Terminal terminal = TerminalDAO.getInstance().getByTerminalKey(terminalKey);
			return terminal.getOutletId();
		}
		return StoreDAO.getRestaurant().getDefaultOutletId();
	}

	public void saveOrUpdateCashDrawers(String request, String outletId) {
		try {
			JSONObject rootElement = new JSONObject(request);
			JSONArray data = rootElement.getJSONArray(DATA);
			if (data != null && data.length() > 0) {
				List<CashDrawer> cashDrawers = new ArrayList<>();
				for (int i = 0; i < data.length(); i++) {
					JSONObject jsonObject = data.getJSONObject(i);
					Gson gson = getGsonBuilder().setExclusionStrategies(new JsonBeanFieldExcluder(MenuShift.class)).create();
					CashDrawer sourceCashDrawer = gson.fromJson(jsonObject.toString(), CashDrawer.class);
					if (sourceCashDrawer != null) {
						cashDrawers.add(sourceCashDrawer);
					}
				}
				for (CashDrawer cashDrawer : cashDrawers) {
					cashDrawer.setOutletId(outletId);
					saveOrUpdateCashDrawer(cashDrawer, true);
				}
			}
			JSONObject terminalJson = rootElement.has("linkedData") ? rootElement.getJSONObject("linkedData") : null; //$NON-NLS-1$ //$NON-NLS-2$
			if (terminalJson != null) {
				Gson gson = getGsonBuilder().setExclusionStrategies(new JsonBeanFieldExcluder(MenuShift.class)).create();
				Terminal sourceTerminal = gson.fromJson(terminalJson.toString(), Terminal.class);
				if (sourceTerminal != null) {
					Terminal existingTerminal = TerminalDAO.getInstance().get(sourceTerminal);
					if (existingTerminal != null) {
						existingTerminal.setAssignedUser(sourceTerminal.getAssignedUser());
						existingTerminal.setCurrentCashDrawer(sourceTerminal.getCurrentCashDrawer());
						TerminalDAO.getInstance().update(existingTerminal);
					}
				}
			}
		} catch (Exception e) {
			PosLog.error(ServiceUtils.class, e);
		}
	}

	private void saveOrUpdateCashDrawer(CashDrawer sourceCashDrawer) {
		saveOrUpdateCashDrawer(sourceCashDrawer, false);
	}

	private void saveOrUpdateCashDrawer(CashDrawer sourceCashDrawer, boolean updateCashDrawer) {
		Session session = null;
		Transaction tx = null;
		CashDrawerDAO dao = CashDrawerDAO.getInstance();
		try {
			session = dao.createNewSession();
			tx = session.beginTransaction();
			sourceCashDrawer.setStoreSession(StoreUtil.getCurrentStoreSession());
			CashDrawer existingCashDrawer = CashDrawerDAO.getInstance().get(sourceCashDrawer.getId());
			if (existingCashDrawer == null) {
				dao.save(sourceCashDrawer);
				debug("New cash drawer " + sourceCashDrawer.getId() + ", asssignd user " + sourceCashDrawer.getAssignedUserId() + " has been created."); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
			}
			else {
				if (updateCashDrawer) {
					sourceCashDrawer.setVersion(existingCashDrawer.getVersion());
					dao.update(sourceCashDrawer);
				}
				updateTerminal(session, sourceCashDrawer.getTerminalId(), sourceCashDrawer.getOutletId(), sourceCashDrawer.getAssignedUser(), sourceCashDrawer);
			}
			tx.commit();
		} catch (Exception e) {
			if (tx != null)
				tx.rollback();
			throw e;
		} finally {
			dao.closeSession(session);
		}
	}

	private static void updateTerminal(Session session, Integer terminalId, String outletId, User assignedUser, CashDrawer sourceCashDrawer) {
		TerminalDAO terminalDAO = new TerminalDAO();
		Terminal terminal = terminalDAO.get(terminalId, outletId, session);
		if (terminal != null) {
			CashDrawer existingTerminalCashDrawer = terminal.getActiveCurrentCashDrawer();
			if (sourceCashDrawer.isClosed() && existingTerminalCashDrawer != null && existingTerminalCashDrawer.getId().equals(sourceCashDrawer.getId())) {
				assignedUser = null;
				sourceCashDrawer = null;
			}
			//terminal.setAssignedUser(assignedUser);
			//terminal.setCurrentCashDrawer(sourceCashDrawer);
			terminal.setActive(true);
			terminalDAO.saveOrUpdate(terminal, session);
		}
	}

	public static BaseDataServiceDao get() {
		return getInstance();
	}

	public void saveOrUpdateMenuItems(String requestData) throws Exception {
		//		saveOrUpdateDataList(requestData, MenuItem.class);
		saveOrUpdateMenuItemDataList(requestData, MenuItem.class);
	}

	public void saveOrUpdateMenuGroups(String requestData) throws Exception {
		saveOrUpdateDataList(requestData, MenuGroup.class);
	}

	public void saveOrUpdateMenuCategories(String requestData) throws Exception {
		saveOrUpdateDataList(requestData, MenuCategory.class);
	}

	public void saveOrUpdateTerminals(String requestData) throws Exception {
		List dataList = getDataListFromJson(requestData, Terminal.class);
		if (dataList != null && dataList.size() > 0) {
			for (Iterator iterator = dataList.iterator(); iterator.hasNext();) {
				Terminal terminal = (Terminal) iterator.next();
				Terminal existingTerminal = TerminalDAO.getInstance().get(terminal);
				if (existingTerminal == null) {
					TerminalDAO.getInstance().save(terminal);
				}
				else {
					existingTerminal.setName(terminal.getName());
					existingTerminal.setActive(terminal.isActive());
					existingTerminal.setDepartmentId(terminal.getDepartmentId());
					existingTerminal.setHasCashDrawer(terminal.isHasCashDrawer());
					TerminalDAO.getInstance().update(existingTerminal);
				}
			}
		}
	}

	public void saveOrUpdateMenuItemDataList(String request, Class referenceClass) throws Exception {
		List dataList = getDataListFromJson(request, referenceClass);
		for (Iterator iterator = dataList.iterator(); iterator.hasNext();) {
			MenuItem item = (MenuItem) iterator.next();
			saveOrUpdateMenuItems(Arrays.asList(item), false, false);
		}
	}

	public List saveOrUpdateDataList(String request, Class referenceClass) throws Exception {
		TerminalDAO dao = new TerminalDAO();
		Session session = null;
		Transaction tx = null;
		try {
			session = dao.createNewSession();
			tx = session.beginTransaction();
			List dataList = saveOrUpdateDataList(request, referenceClass, session);
			tx.commit();
			return dataList;
		} catch (Exception e) {
			if (tx != null)
				tx.rollback();
			throw e;
		} finally {
			dao.closeSession(session);
		}
	}

	public List saveOrUpdateDataList(String request, Class referenceClass, Session session) throws Exception {
		List dataList = getDataListFromJson(request, referenceClass);
		for (Iterator iterator = dataList.iterator(); iterator.hasNext();) {
			Object item = (Object) iterator.next();
			PropertyDescriptor propertyDescriptor = new PropertyDescriptor("id", item.getClass()); //$NON-NLS-1$
			Object id = propertyDescriptor.getReadMethod().invoke(item);
			Object existingItem = getExistingData(item.getClass(), id);
			if (existingItem == null) {
				session.save(item);
			}
			else {
				PropertyDescriptor propertyDescriptorVersion = new PropertyDescriptor("version", item.getClass()); //$NON-NLS-1$
				long version = (long) propertyDescriptorVersion.getReadMethod().invoke(existingItem);
				propertyDescriptorVersion.getWriteMethod().invoke(item, version);
				session.update(item);
			}
		}
		return dataList;
	}

	private Object getExistingData(Class<? extends Object> class1, Object id) {
		TerminalDAO dao = new TerminalDAO();
		Session session = null;
		try {
			session = dao.createNewSession();
			Criteria criteria = session.createCriteria(class1);
			criteria.add(Restrictions.eq("id", id)); //$NON-NLS-1$
			return criteria.uniqueResult();
		} finally {
			dao.closeSession(session);
		}
	}

	public List getDataListFromJson(String request, Class referenceClass) throws Exception {
		JSONObject rootElement = new JSONObject(request);
		JSONArray data = rootElement.getJSONArray(DATA);
		if (data == null || data.length() == 0) {
			return null;
		}
		List dataList = new ArrayList();
		for (int i = 0; i < data.length(); i++) {
			JSONObject jsonObject = data.getJSONObject(i);
			dataList.add(toObject(jsonObject, referenceClass));
		}
		return dataList;
	}

	public Object toObject(JSONObject request, Class<?> type) throws Exception {
		Gson gson = getGsonBuilder().setExclusionStrategies(new JsonBeanFieldExcluder(MenuShift.class)).create();
		return gson.fromJson(request.toString(), type);
	}

	private void debug(String msg) {
		PosLog.debug(getClass(), msg);
	}

	public void saveOrUpdateInventoryTransactions(List<InventoryTransaction> dataList, boolean updateLastUpdateTime, boolean updateSyncTime) throws Exception {
		if (dataList == null || dataList.isEmpty())
			return;

		InventoryTransactionDAO dao = new InventoryTransactionDAO();
		Transaction tx = null;
		Session session = null;
		Map<String, MenuItem> itemMap = new HashMap<>();

		try {
			session = dao.createNewSession();
			tx = session.beginTransaction();

			for (Iterator<InventoryTransaction> iterator = dataList.iterator(); iterator.hasNext();) {
				InventoryTransaction item = (InventoryTransaction) iterator.next();
				MenuItem existingMenuItem = item.getMenuItem();
				if (existingMenuItem != null) {
					MenuItem mapItem = itemMap.get(existingMenuItem.getId());
					if (mapItem != null) {
						item.setMenuItem(mapItem);
					}
					else {
						MenuItemDAO.getInstance().initialize(existingMenuItem);
						itemMap.put(existingMenuItem.getId(), existingMenuItem);
					}
				}
				InventoryTransaction existingItem = null;
				if (StringUtils.isNotEmpty(item.getId())) {
					existingItem = dao.get(item.getId(), session);
				}
				if (existingItem == null) {
					//debug(item.getMenuItem().getDisplayName() + "'s inventory adjusted"); //$NON-NLS-1$
					item.setUpdateLastUpdateTime(updateLastUpdateTime);
					item.setUpdateSyncTime(updateSyncTime);
					dao.save(item, session);
					dao.adjustInventoryStock(item, session);
				}
			}
			tx.commit();
			debug("Successfully inventory adjusted"); //$NON-NLS-1$
		} catch (Exception e) {
			tx.rollback();
			throw e;
		} finally {
			dao.closeSession(session);
		}
	}

	//TODO : Menu modifier Upload and download
	public void saveOrUpdateMenuModifiers(List<MenuModifier> dataList, boolean updateLastUpdateTime, boolean updateSyncTime) throws Exception {
		if (dataList == null || dataList.isEmpty())
			return;
		MenuModifierDAO modifierDAO = MenuModifierDAO.getInstance();

		for (Iterator<MenuModifier> iterator = dataList.iterator(); iterator.hasNext();) {
			MenuModifier menuModifier = (MenuModifier) iterator.next();
			MenuModifier existingMenuModifier = modifierDAO.loadFullModifier(menuModifier.getId());

			try (Session session = modifierDAO.createNewSession()) {
				Transaction tx = session.beginTransaction();

				List<PizzaModifierPrice> pizzaModifierPriceList = menuModifier.getPizzaModifierPriceList();
				List<ModifierMultiplierPrice> multiplierPriceList = menuModifier.getMultiplierPriceList();
				List<ModifierGroup> modifierGroups = menuModifier.getModifierGroups();

				menuModifier.setUpdateLastUpdateTime(updateLastUpdateTime);
				menuModifier.setUpdateSyncTime(updateSyncTime);
				menuModifier.setPizzaModifierPriceList(null);
				menuModifier.setMultiplierPriceList(null);
				menuModifier.setModifierGroups(null);

				saveOrUpdateVerMenuModifier(session, menuModifier, existingMenuModifier);
				saveOrUpdatePizzaModifierPrice(session, existingMenuModifier, pizzaModifierPriceList);
				saveOrUpdateModifierMultiplierPrice(session, menuModifier, existingMenuModifier, multiplierPriceList);
				saveOrUpdateModifierGroup(session, existingMenuModifier, modifierGroups, updateLastUpdateTime, updateSyncTime);

				menuModifier.setPizzaModifierPriceList(pizzaModifierPriceList);
				menuModifier.setMultiplierPriceList(multiplierPriceList);
				menuModifier.setModifierGroups(modifierGroups);

				modifierDAO.saveOrUpdate(menuModifier, session);

				tx.commit();
			}
		}
	}

	private void saveOrUpdateModifierGroup(Session session, MenuModifier existingMenuModifier, List<ModifierGroup> modifierGroups, boolean updateLastUpdateTime,
			boolean updateSyncTime) {

		if (modifierGroups == null) {
			return;
		}
		List<ModifierGroup> existingModifierGroupList = null;
		if (existingMenuModifier != null) {
			existingModifierGroupList = existingMenuModifier.getModifierGroups();
		}

		for (ModifierGroup modifierGroup : modifierGroups) {
			saveOrUpdateModifierGroup(session, modifierGroup, existingModifierGroupList, updateLastUpdateTime, updateSyncTime);
		}

	}

	private void saveOrUpdateModifierGroup(Session session, ModifierGroup modifierGroup, List<ModifierGroup> existingModifierGroup,
			boolean updateLastUpdateTime, boolean updateSyncTime) {

		ModifierGroupDAO modifierGroupDAO = new ModifierGroupDAO();

		ModifierGroup existModifierGroup = null;

		modifierGroup.setUpdateLastUpdateTime(updateLastUpdateTime);
		modifierGroup.setUpdateSyncTime(updateSyncTime);

		if (existingModifierGroup == null || existingModifierGroup.isEmpty()) {
			saveOrSetVersionModifierGroup(session, modifierGroup, modifierGroupDAO);
		}
		else {
			int idx = existingModifierGroup.indexOf(modifierGroup);
			if (idx != -1) {
				existModifierGroup = existingModifierGroup.get(idx);
				if (existModifierGroup == null) {
					modifierGroupDAO.save(modifierGroup, session);
				}
				else {
					modifierGroup.setVersion(existModifierGroup.getVersion());
				}
			}
			else {
				saveOrSetVersionModifierGroup(session, modifierGroup, modifierGroupDAO);
			}
		}

	}

	private void saveOrSetVersionModifierGroup(Session session, ModifierGroup modifierGroup, ModifierGroupDAO modifierGroupDAO) {
		ModifierGroup modifierGroupInSession = modifierGroupDAO.get(modifierGroup.getId(), session);
		if (modifierGroupInSession == null) {
			modifierGroupDAO.save(modifierGroup, session);
		}
		else {
			modifierGroup.setVersion(modifierGroupInSession.getVersion());
		}
	}

	private void saveOrUpdatePizzaModifierPrice(Session session, MenuModifier existingMenuModifier, List<PizzaModifierPrice> pizzaModifierPriceList) {

		if (pizzaModifierPriceList == null) {
			return;
		}
		//List<PizzaModifierPrice> existingPizzaModifierPrice = null;
		//if (existingMenuModifier != null) {
		//	existingPizzaModifierPrice = existingMenuModifier.getPizzaModifierPriceList();
		//}

		for (PizzaModifierPrice pizzaModifierPrice : pizzaModifierPriceList) {

			PizzaModifierPrice existingPizzaModifierPric = PizzaModifierPriceDAO.getInstance().get(pizzaModifierPrice.getId());

			List<ModifierMultiplierPrice> modiMultiPrices = pizzaModifierPrice.getMultiplierPriceList();
			pizzaModifierPrice.setMultiplierPriceList(null);

			saveOrUpdateVerPizzaModifierPrice(session, pizzaModifierPrice, existingPizzaModifierPric);
			if (modiMultiPrices != null) {
				saveOrUpdateModifierMultiplierPrice(session, pizzaModifierPrice, existingPizzaModifierPric, modiMultiPrices);
				pizzaModifierPrice.setMultiplierPriceList(modiMultiPrices);
			}
			//saveOrUpdatePizzaModifierPrice(session, pizzaModifierPrice, existingPizzaModifierPrice);
		}

	}

	private void saveOrUpdateVerPizzaModifierPrice(Session session, PizzaModifierPrice pizzaModifierPrice, PizzaModifierPrice existingPizzaModifierPrice) {
		if (existingPizzaModifierPrice == null) {
			PizzaModifierPriceDAO.getInstance().save(pizzaModifierPrice, session);
		}
		else {
			pizzaModifierPrice.setVersion(existingPizzaModifierPrice.getVersion());
		}
	}

	private void saveOrUpdateModifierMultiplierPrice(Session session, PizzaModifierPrice pizzaModifierPrice, PizzaModifierPrice existingPizzaModifierPrice,
			List<ModifierMultiplierPrice> modiMultiPrices) {

		List<ModifierMultiplierPrice> existingModiMultiPrices = null;
		if (existingPizzaModifierPrice != null) {
			existingModiMultiPrices = existingPizzaModifierPrice.getMultiplierPriceList();
		}
		for (ModifierMultiplierPrice multiplierPrice : modiMultiPrices) {
			multiplierPrice.setPizzaModifierPriceId(pizzaModifierPrice.getId());

			saveOrUpdateModifierMultiplierPrice(session, multiplierPrice, existingModiMultiPrices);
		}
	}

	private void saveOrUpdateModifierMultiplierPrice(Session session, MenuModifier menuModifier, MenuModifier existingMenuModifier,
			List<ModifierMultiplierPrice> multiplierPriceList) {

		if (multiplierPriceList == null) {
			return;
		}

		List<ModifierMultiplierPrice> existingModiMultiPrices = null;
		if (existingMenuModifier != null) {
			existingModiMultiPrices = existingMenuModifier.getMultiplierPriceList();
		}

		for (ModifierMultiplierPrice multiplierPrice : multiplierPriceList) {
			multiplierPrice.setModifierId(menuModifier.getId());
			saveOrUpdateModifierMultiplierPrice(session, multiplierPrice, existingModiMultiPrices);
		}

	}

	private void saveOrUpdatePizzaModifierPrice(Session session, PizzaModifierPrice pizzaModifierPrice, List<PizzaModifierPrice> existingPizzaModifierPrice) {
		PizzaModifierPriceDAO pizzaModifierPriceDAO = new PizzaModifierPriceDAO();

		PizzaModifierPrice existPizzaModifierPrice = null;

		if (existingPizzaModifierPrice == null || existingPizzaModifierPrice.isEmpty()) {
			pizzaModifierPriceDAO.save(pizzaModifierPrice, session);
		}
		else {
			int idx = existingPizzaModifierPrice.indexOf(pizzaModifierPrice);
			if (idx != -1) {
				existPizzaModifierPrice = existingPizzaModifierPrice.get(idx);
				if (existPizzaModifierPrice == null) {
					pizzaModifierPriceDAO.save(pizzaModifierPrice, session);
				}
				else {
					pizzaModifierPrice.setVersion(existPizzaModifierPrice.getVersion());
				}
			}
			else {
				pizzaModifierPriceDAO.save(pizzaModifierPrice, session);
			}
		}
	}

	private void saveOrUpdateModifierMultiplierPrice(Session session, ModifierMultiplierPrice multiplierPrice,
			List<ModifierMultiplierPrice> existingModiMultiPrices) {
		ModifierMultiplierPriceDAO multiplierPriceDAO = new ModifierMultiplierPriceDAO();

		ModifierMultiplierPrice existingModifierMultiplierPrice = null;
		if (existingModiMultiPrices == null || existingModiMultiPrices.isEmpty()) {
			multiplierPrice.setVersion(0);
			multiplierPriceDAO.save(multiplierPrice, session);
		}
		else {
			int idx = existingModiMultiPrices.indexOf(multiplierPrice);
			if (idx != -1) {
				existingModifierMultiplierPrice = existingModiMultiPrices.get(idx);
				if (existingModifierMultiplierPrice == null) {
					multiplierPriceDAO.save(multiplierPrice, session);
				}
				else {
					multiplierPrice.setVersion(existingModifierMultiplierPrice.getVersion());
				}
			}
			else {
				multiplierPriceDAO.save(multiplierPrice, session);
			}
		}
	}

	private void saveOrUpdateVerMenuModifier(Session session, MenuModifier menuModifier, MenuModifier existingMenuModifier) {
		if (existingMenuModifier == null) {
			MenuModifierDAO.getInstance().save(menuModifier, session);
		}
		else {
			menuModifier.setVersion(existingMenuModifier.getVersion());
		}
	}

	//TODO : User Upload and download
	public void saveOrUpdateUsers(List<User> dataList, boolean updateLastUpdateTime, boolean updateSyncTime) throws Exception {
		if (dataList == null || dataList.isEmpty())
			return;

		List<User> parentUser = new ArrayList<User>();
		List<User> childUser = new ArrayList<User>();

		UserDAO instance = UserDAO.getInstance();

		for (Iterator<User> iterator = dataList.iterator(); iterator.hasNext();) {
			User user = (User) iterator.next();
			if (user.isRoot()) {
				parentUser.add(user);
			}
			else {
				childUser.add(user);
			}
		}

		saveUser(parentUser, updateLastUpdateTime, updateSyncTime, instance);
		saveUser(childUser, updateLastUpdateTime, updateSyncTime, instance);
	}

	private void saveUser(List<User> dataList, boolean updateLastUpdateTime, boolean updateSyncTime, UserDAO instance) {
		if (dataList == null || dataList.isEmpty())
			return;

		Transaction tx = null;
		Session session = null;
		try {

			session = instance.createNewSession();
			tx = session.beginTransaction();
			for (Iterator<User> iterator = dataList.iterator(); iterator.hasNext();) {
				User user = (User) iterator.next();
				User existingUser = instance.get(user.getId(), user.getOutletId());

				if (existingUser != null) {
					if (!shouldSave(user.getLastUpdateTime(), existingUser.getLastUpdateTime())) {
						printAreadyUpdated(user.getFullName());
						continue;
					}
				}
				if (!updateLastUpdateTime && existingUser != null) {
					user.setLastClockInTime(existingUser.getLastClockInTime());
					user.setClockedIn(existingUser.isClockedIn());
				}
				saveOrSetVersionUsers(updateLastUpdateTime, updateSyncTime, session, user, existingUser, instance);
				instance.saveOrUpdate(user, session);

			}
			tx.commit();
		} catch (Exception e) {
			tx.rollback();
			throw e;
		} finally {
			instance.closeSession(session);
		}
	}

	private void saveOrSetVersionUsers(boolean updateLastUpdateTime, boolean updateSyncTime, Session session, User user, User existingUser, UserDAO instance) {
		User parentUser = user.getParentUser();
		user.setUpdateLastUpdateTime(updateLastUpdateTime);
		user.setUpdateSyncTime(updateSyncTime);
		user.setParentUser(null);
		if (existingUser != null) {
			user.setEmail(existingUser.getEmail());
			user.setCanLoginInCloud(existingUser.isCanLoginInCloud());
			user.setPosliveEnable(existingUser.isPosliveEnable());
		}

		saveOrSetVersionUsers(session, user, existingUser, instance);

		user.setParentUser(parentUser);
	}

	private void saveOrSetVersionUsers(Session session, User user, User existingUser, UserDAO instance) {
		if (existingUser == null) {
			User existingUserInSession = instance.get(user.getId(), user.getOutletId(), session);
			if (existingUserInSession == null) {
				instance.save(user, session);
			}
			else {
				user.setVersion(existingUserInSession.getVersion());
			}
		}
		else {
			user.setVersion(existingUser.getVersion());
		}
	}

	//TODO : Menu Item Upload and download
	public List<MenuItem> saveOrUpdateMenuItems(List<MenuItem> dataList, boolean updateLastUpdateTime, boolean updateSyncTime) throws Exception {
		if (dataList == null || dataList.isEmpty())
			return null;

		List<MenuItem> nonVariantMenuItems = new ArrayList<MenuItem>();
		List<MenuItem> variantMenuItems = new ArrayList<MenuItem>();
		List<MenuItem> comboMenuItems = new ArrayList<MenuItem>();
		List<MenuItem> savedErrorItems = new ArrayList<>();

		for (Iterator<MenuItem> iterator = dataList.iterator(); iterator.hasNext();) {
			MenuItem menuItem = (MenuItem) iterator.next();
			if (menuItem.isComboItem()) {
				comboMenuItems.add(menuItem);
			}
			else if (!menuItem.isVariant()) {
				nonVariantMenuItems.add(menuItem);
			}
			else {
				variantMenuItems.add(menuItem);
			}
		}
		for (MenuItem menuItem : nonVariantMenuItems) {
			try {
				saveMenuItem(menuItem, updateLastUpdateTime, updateSyncTime);
			} catch (Exception e) {
				//FIXME: should throw error.
				savedErrorItems.add(menuItem);
			}
		}
		for (MenuItem menuItem : variantMenuItems) {
			try {
				saveMenuItem(menuItem, updateLastUpdateTime, updateSyncTime);
			} catch (Exception e) {
				//FIXME: should throw error.
				savedErrorItems.add(menuItem);
			}
		}
		for (MenuItem menuItem : comboMenuItems) {
			try {
				saveMenuItem(menuItem, updateLastUpdateTime, updateSyncTime);
			} catch (Exception e) {
				//FIXME: should throw error.
				savedErrorItems.add(menuItem);
			}
		}

		if (savedErrorItems.size() == 0) {
			return null;
		}
		for (Iterator iterator = savedErrorItems.iterator(); iterator.hasNext();) {
			MenuItem menuItem = (MenuItem) iterator.next();
			try {
				saveMenuItem(menuItem, updateLastUpdateTime, updateSyncTime);
				iterator.remove();
			} catch (Exception e) {
				//FIXME: should throw error.
				PosLog.error(getClass(), "Failed to update menu item. Id: " + menuItem.getId() + ", name: " + menuItem.toString());
			}
		}
		return savedErrorItems;
	}

	private void saveMenuItem(MenuItem menuItem, boolean updateLastUpdateTime, boolean updateSyncTime) throws Exception {
		if (menuItem == null)
			return;

		MenuItemDAO menuItemDAO = MenuItemDAO.getInstance();
		MenuItem existingMenuItem = menuItemDAO.get(menuItem.getId());

		if (existingMenuItem != null) {
			if (!shouldSave(menuItem.getLastUpdateTime(), existingMenuItem.getLastUpdateTime())) {
				printAreadyUpdated(menuItem.getDisplayName());
				return;
			}
		}

		try (Session session = menuItemDAO.createNewSession()) {
			Transaction tx = session.beginTransaction();

			saveOrSetVersionMenuItem(updateLastUpdateTime, updateSyncTime, session, menuItem, existingMenuItem);
			menuItemDAO.saveOrUpdate(menuItem, session, updateLastUpdateTime);

			tx.commit();
		}
	}

	private void saveOrSetVersionMenuItem(boolean updateLastUpdateTime, boolean updateSyncTime, Session session, MenuItem menuItem, MenuItem existingMenuItem)
			throws Exception {
		MenuItemDAO.getInstance().initialize(existingMenuItem);

		List<Discount> discounts = copyList(menuItem.getDiscounts());
		List<MenuItemModifierSpec> menuItemModiferSpecs = copyList(menuItem.getMenuItemModiferSpecs());
		List<ComboGroup> comboGroups = copyList(menuItem.getComboGroups());
		List<ComboItem> comboItems = copyList(menuItem.getComboItems());
		List<InventoryStockUnit> stockUnits = copyList(menuItem.getStockUnits());
		List<PizzaPrice> pizzaPriceList = copyList(menuItem.getPizzaPriceList());
		List<MenuItem> variants = copyList(menuItem.getVariants());

		menuItem.setUpdateLastUpdateTime(updateLastUpdateTime);
		menuItem.setUpdateSyncTime(updateSyncTime);

		clearList(menuItem.getDiscounts());
		clearList(menuItem.getMenuItemModiferSpecs());
		clearList(menuItem.getComboGroups());
		clearList(menuItem.getComboItems());
		clearList(menuItem.getStockUnits());
		clearList(menuItem.getPizzaPriceList());
		clearList(menuItem.getVariants());

		saveOrSetVersionAttributes(session, existingMenuItem, menuItem.getAttributes());
		saveOrSetVersionMenuItem(session, menuItem, existingMenuItem);
		saveOrSetVersionComboItems(session, menuItem, existingMenuItem, comboItems);
		saveOrSetVersionDiscounts(session, existingMenuItem, discounts);
		saveOrSetVersionMenuItemModifierSpecs(session, existingMenuItem, menuItemModiferSpecs, updateLastUpdateTime, updateSyncTime);
		saveOrSetVersionComboGroups(session, existingMenuItem, comboGroups);

		saveOrSetVersionInvStockUnits(session, existingMenuItem, stockUnits);
		saveOrSetVersionPizzaPrices(session, existingMenuItem, pizzaPriceList);

		//if (menuItem.isHasVariant()) {
		saveOrSetVersionVariants(session, existingMenuItem, variants);
		//}
		copyList(menuItem.getDiscounts(), discounts);
		copyList(menuItem.getMenuItemModiferSpecs(), menuItemModiferSpecs);
		copyList(menuItem.getComboGroups(), comboGroups);
		copyList(menuItem.getComboItems(), comboItems);
		copyList(menuItem.getStockUnits(), stockUnits);
		copyList(menuItem.getPizzaPriceList(), pizzaPriceList);
		copyList(menuItem.getVariants(), variants);

		if (existingMenuItem != null) {
			//ORCL-312 Menu items automatically removed from online items tab when edited from Oropos 
			//do not disable online ordering from oropos
			//now enable online ordering property populate from online item table
			//menuItem.setEnableOnlineOrdering(existingMenuItem.isEnableOnlineOrdering());
		}
	}

	private ArrayList copyList(List sourceList) {
		if (sourceList == null) {
			return new ArrayList<>();
		}

		return new ArrayList<>(sourceList);
	}

	private void clearList(List sourceList) {
		if (sourceList != null) {
			sourceList.clear();
		}
	}

	private void copyList(List destList, List sourceList) {
		if (destList != null) {
			destList.addAll(sourceList);
		}
	}

	private void saveOrSetVersionVariants(Session session, MenuItem existingVarient, List<MenuItem> variants) throws Exception {
		if (variants == null) {
			return;
		}
		info(variants.size(), " variants found"); //$NON-NLS-1$
		for (MenuItem varientMenuItem : variants) {
			MenuItem existingVarientMenuItem = MenuItemDAO.getInstance().get(varientMenuItem.getId());
			//MenuItem existingVarientMenuItem = MenuItemDAO.getInstance().getInitialized(varientMenuItem.getId());
			if (existingVarientMenuItem != null) {
				if (!shouldSave(varientMenuItem.getLastUpdateTime(), existingVarientMenuItem.getLastUpdateTime())) {
					printAreadyUpdated(varientMenuItem.getDisplayName());
					continue;
				}
			}
			saveOrSetVersionMenuItem(false, false, session, varientMenuItem, existingVarientMenuItem);
		}
	}

	private void saveOrSetVersionPizzaPrices(Session session, MenuItem existingMenuItem, List<PizzaPrice> pizzaPriceList) {
		if (pizzaPriceList == null) {
			return;
		}
		info(pizzaPriceList.size(), " PizzaPrices found"); //$NON-NLS-1$
		List<PizzaPrice> existingPizzaPrices = null;
		if (existingMenuItem != null) {
			existingPizzaPrices = existingMenuItem.getPizzaPriceList();
		}
		for (PizzaPrice pizzaPrice : pizzaPriceList) {

			PizzaPriceDAO pizzaPriceDAO = new PizzaPriceDAO();
			PizzaPrice existingPizzaPrice = null;

			if (existingPizzaPrices == null || existingPizzaPrices.isEmpty()) {
				existingPizzaPrice = pizzaPriceDAO.get(pizzaPrice.getId(), session);
				if (existingPizzaPrice == null) {
					pizzaPriceDAO.save(pizzaPrice, session);
				}
				else {
					session.evict(existingPizzaPrice);
					pizzaPrice.setVersion(existingPizzaPrice.getVersion());
				}
			}
			else {
				existingPizzaPrice = pizzaPriceDAO.get(pizzaPrice.getId());
				if (existingPizzaPrice != null) {
					pizzaPrice.setVersion(existingPizzaPrice.getVersion());
				}
				else {
					pizzaPriceDAO.save(pizzaPrice, session);
				}
			}
		}
	}

	private void saveOrSetVersionInvStockUnits(Session session, MenuItem existingMenuItem, List<InventoryStockUnit> stockUnits) {
		if (stockUnits == null) {
			return;
		}
		info(stockUnits.size(), " stockUnits found"); //$NON-NLS-1$
		List<InventoryStockUnit> existingInventoryStockUnits = null;
		if (existingMenuItem != null) {
			existingInventoryStockUnits = existingMenuItem.getStockUnits();
		}
		for (InventoryStockUnit stockUnit : stockUnits) {
			InventoryStockUnitDAO stockUnitDAO = new InventoryStockUnitDAO();
			InventoryStockUnit existingInvStockUnit = null;

			if (existingInventoryStockUnits == null || existingInventoryStockUnits.isEmpty()) {
				existingInvStockUnit = stockUnitDAO.get(stockUnit.getId(), session);
				if (existingInvStockUnit == null) {
					stockUnitDAO.save(stockUnit, session);
				}
				else {
					session.evict(existingInvStockUnit);
					stockUnit.setVersion(existingInvStockUnit.getVersion());
				}
			}
			else {
				int idx = existingInventoryStockUnits.indexOf(stockUnit);
				if (idx != -1) {
					existingInvStockUnit = existingInventoryStockUnits.get(idx);
					if (existingInvStockUnit == null) {
						stockUnitDAO.save(stockUnit, session);
					}
					else {
						stockUnit.setVersion(existingInvStockUnit.getVersion());
					}
				}
				else {
					stockUnitDAO.save(stockUnit, session);
				}
			}
		}
	}

	private void saveOrSetVersionComboItems(Session session, MenuItem menuItem, MenuItem existingMenuItem, List<ComboItem> comboItems) {
		if (comboItems == null || comboItems.isEmpty()) {
			return;
		}
		info(comboItems.size(), " comboItems found"); //$NON-NLS-1$
		List<ComboItem> existingComboItems = null;
		if (existingMenuItem != null) {
			existingComboItems = existingMenuItem.getComboItems();
		}
		for (ComboItem comboItem : comboItems) {
			//info(menuItem.getId() + " menuItem " + menuItem.getDisplayName());
			comboItem.setMenuItemId(menuItem.getId());
			ComboItemDAO comboItemDAO = new ComboItemDAO();
			ComboItem existingComboItem = null;
			if (existingComboItems == null || existingComboItems.isEmpty()) {
				existingComboItem = comboItemDAO.get(comboItem.getId(), session);
				if (existingComboItem == null) {
					comboItemDAO.save(comboItem, session);
				}
				else {
					session.evict(existingComboItem);
					comboItem.setVersion(existingComboItem.getVersion());
				}
			}
			else {
				int idx = existingComboItems.indexOf(comboItem);
				if (idx != -1) {
					existingComboItem = existingComboItems.get(idx);
					if (existingComboItem == null) {
						comboItemDAO.save(comboItem, session);
					}
					else {
						comboItem.setVersion(existingComboItem.getVersion());
					}
				}
				else {
					comboItemDAO.save(comboItem, session);
				}
			}
		}
	}

	private void saveOrSetVersionComboGroups(Session session, MenuItem existingMenuItem, List<ComboGroup> comboGroups) throws Exception {
		if (comboGroups == null || comboGroups.isEmpty()) {
			return;
		}
		info(comboGroups.size(), " comboGroup found"); //$NON-NLS-1$

		for (ComboGroup comboGroup : comboGroups) {
			if (existingMenuItem == null) {
				ComboGroupDAO.getInstance().save(comboGroup, session);
				continue;
			}
			List<ComboGroup> existingComboGroups = existingMenuItem.getComboGroups();
			ComboGroup existingComboGroup = (ComboGroup) POSUtil.getFromList(comboGroup, existingComboGroups);
			if (existingComboGroup != null) {
				comboGroup.setVersion(existingComboGroup.getVersion());
			}
			else {
				ComboGroupDAO.getInstance().save(comboGroup, session);
			}
		}
	}

	private void saveOrSetVersionMenuItemModifierSpecs(Session session, MenuItem existingMenuItem, List<MenuItemModifierSpec> menuItemModiferSpecs,
			boolean updateLastUpdateTime, boolean updateSyncTime) {
		if (menuItemModiferSpecs == null || menuItemModiferSpecs.isEmpty()) {
			return;
		}
		info(menuItemModiferSpecs.size(), " menuItemModiferSpec found"); //$NON-NLS-1$
		List<MenuItemModifierSpec> existingMenuItemModifierSpecs = null;
		if (existingMenuItem != null) {
			existingMenuItemModifierSpecs = existingMenuItem.getMenuItemModiferSpecs();
		}

		for (MenuItemModifierSpec modifierSpec : menuItemModiferSpecs) {
			saveOrSetVersionMenuItemModifierSpecs(session, updateLastUpdateTime, updateSyncTime, existingMenuItemModifierSpecs, modifierSpec);
		}
	}

	public void saveOrSetVersionMenuItemModifierSpecs(Session session, boolean updateLastUpdateTime, boolean updateSyncTime,
			List<MenuItemModifierSpec> existingMenuItemModifierSpecs, MenuItemModifierSpec modifierSpec) {
		Set<MenuItemModifierPage> modifierPages = modifierSpec.getModifierPages();

		modifierSpec.setUpdateLastUpdateTime(updateLastUpdateTime);
		modifierSpec.setUpdateSyncTime(updateSyncTime);
		modifierSpec.setModifierPages(null);

		saveOrSetVersionMenuItemModifierSpec(session, modifierSpec);

		modifierSpec.setModifierPages(modifierPages);

		if (modifierPages != null && !modifierPages.isEmpty()) {
			for (MenuItemModifierPage menuItemModifierPage : modifierPages) {
				saveOrSetVersionMenuItemModiPage(session, menuItemModifierPage);
			}
		}

		MenuItemModifierSpecDAO modifierSpecDAO = new MenuItemModifierSpecDAO();
		MenuItemModifierSpec existingModifierSpec = null;
		if (existingMenuItemModifierSpecs == null || existingMenuItemModifierSpecs.isEmpty()) {

			existingModifierSpec = modifierSpecDAO.get(modifierSpec.getId(), session);
			if (existingModifierSpec == null) {
				modifierSpecDAO.save(modifierSpec, session);
			}
			else {
				session.evict(existingModifierSpec);
				modifierSpec.setVersion(existingModifierSpec.getVersion());
			}
		}
		else {
			int idx = existingMenuItemModifierSpecs.indexOf(modifierSpec);
			if (idx != -1) {
				existingModifierSpec = existingMenuItemModifierSpecs.get(idx);
				if (existingModifierSpec == null) {
					modifierSpecDAO.save(modifierSpec, session);
				}
				else {
					modifierSpec.setVersion(existingModifierSpec.getVersion());
				}
			}
			else {
				modifierSpecDAO.save(modifierSpec, session);
			}
		}
	}

	public void saveOrSetVersionMenuItemModiPage(Session session, MenuItemModifierPage menuItemModifierPage) {
		List<MenuItemModifierPageItem> pageItems = menuItemModifierPage.getPageItems();
		menuItemModifierPage.setPageItems(null);

		saveOrSetVersionMenuItemModifierPage(session, menuItemModifierPage);
		saveOrSetVersionMenuItemModifierPageItem(session, menuItemModifierPage, pageItems);

		menuItemModifierPage.setPageItems(pageItems);
	}

	private void saveOrSetVersionMenuItemModifierPageItem(Session session, MenuItemModifierPage menuItemModifierPage,
			List<MenuItemModifierPageItem> pageItems) {
		if (pageItems == null || pageItems.isEmpty()) {
			return;
		}
		info(pageItems.size(), " page items found"); //$NON-NLS-1$
		MenuItemModifierPageItemDAO instance = new MenuItemModifierPageItemDAO();
		MenuItemModifierPageItem existingModifierpageItem = null;

		for (MenuItemModifierPageItem menuItemModifierPageItem : pageItems) {

			existingModifierpageItem = instance.get(menuItemModifierPageItem.getId(), session);
			menuItemModifierPageItem.setParentPageId(menuItemModifierPage.getId());
			if (existingModifierpageItem == null) {
				instance.save(menuItemModifierPageItem, session);
			}
			else {
				session.evict(existingModifierpageItem);
				menuItemModifierPageItem.setVersion(existingModifierpageItem.getVersion());
			}
		}
	}

	private void saveOrSetVersionMenuItemModifierSpec(Session session, MenuItemModifierSpec modifierSpec) {
		MenuItemModifierSpecDAO instance = MenuItemModifierSpecDAO.getInstance();
		MenuItemModifierSpec existingItemModifierSpec = instance.get(modifierSpec.getId(), session);
		if (existingItemModifierSpec == null) {
			instance.save(modifierSpec, session);
		}
		else {
			session.evict(existingItemModifierSpec);
			modifierSpec.setVersion(existingItemModifierSpec.getVersion());
		}
	}

	private void saveOrSetVersionMenuItemModifierPage(Session session, MenuItemModifierPage menuItemModifierPage) {

		MenuItemModifierPageDAO instance = MenuItemModifierPageDAO.getInstance();
		MenuItemModifierPage existingMenuItemModiPageIn = instance.get(menuItemModifierPage.getId(), session);
		if (existingMenuItemModiPageIn == null) {
			instance.save(menuItemModifierPage, session);
		}
		else {
			session.evict(existingMenuItemModiPageIn);
			menuItemModifierPage.setVersion(existingMenuItemModiPageIn.getVersion());
		}

	}

	private void saveOrSetVersionDiscounts(Session session, MenuItem existingMenuItem, List<Discount> discounts) {
		if (discounts == null || discounts.isEmpty()) {
			return;
		}
		info(discounts.size(), " discount found"); //$NON-NLS-1$
		List<Discount> existingDiscounts = null;
		if (existingMenuItem != null) {
			existingDiscounts = existingMenuItem.getDiscounts();
		}
		for (Discount discount : discounts) {
			DiscountDAO discountDAO = new DiscountDAO();
			Discount existingDiscount = null;

			if (existingDiscounts == null || existingDiscounts.isEmpty()) {
				existingDiscount = discountDAO.get(discount.getId(), session);
				if (existingDiscount == null) {
					discountDAO.save(discount, session);
				}
				else {
					session.evict(existingDiscount);
					discount.setVersion(existingDiscount.getVersion());
				}
			}
			else {
				int idx = existingDiscounts.indexOf(discount);
				if (idx != -1) {
					existingDiscount = existingDiscounts.get(idx);
					if (existingDiscount == null) {
						discountDAO.save(discount, session);
					}
					else {
						discount.setVersion(existingDiscount.getVersion());
					}
				}
				else {
					discountDAO.save(discount, session);
				}
			}

		}
	}

	private void saveOrSetVersionAttributes(Session session, MenuItem existingMenuItem, List<Attribute> attributes) {
		if (attributes == null || attributes.isEmpty()) {
			return;
		}
		info(attributes.size(), " attribute found"); //$NON-NLS-1$
		List<Attribute> existingAttributes = null;
		if (existingMenuItem != null) {
			existingAttributes = existingMenuItem.getAttributes();
		}
		for (Attribute attribute : attributes) {
			AttributeDAO attributeDAO = new AttributeDAO();
			Attribute existingAttribute = null;

			if (existingAttributes == null || existingAttributes.isEmpty()) {
				existingAttribute = attributeDAO.get(attribute.getId(), session);
				if (existingAttribute == null) {
					attributeDAO.save(attribute, session);
				}
				else {
					session.evict(existingAttribute);
					attribute.setVersion(existingAttribute.getVersion());
				}
			}
			else {
				int idx = existingAttributes.indexOf(attribute);
				if (idx != -1) {
					existingAttribute = existingAttributes.get(idx);
					if (existingAttribute == null) {
						attributeDAO.save(attribute, session);
					}
					else {
						attribute.setVersion(existingAttribute.getVersion());
					}
				}
				else {
					attributeDAO.save(attribute, session);
				}
			}
		}
	}

	private void info(int size, String msg) {
		if (size == 0) {
			return;
		}
		LOGGER.info(size + " " + msg);
	}

	private void printAreadyUpdated(String name) {
		if (name == null) {
			return;
		}
		LOGGER.info("  " + name.toLowerCase() + " already updated.");
	}

	private void saveOrSetVersionMenuItem(Session session, MenuItem menuItem, MenuItem existingMenuItem) {
		if (existingMenuItem == null) {
			session.save(menuItem);
		}
		else {
			menuItem.setVersion(existingMenuItem.getVersion());
		}
	}

	public boolean shouldSave(Date clientDate, Date serverDate) {
		if (clientDate == null || serverDate == null) {
			return true;
		}
		if (clientDate.after(serverDate)) {
			return true;
		}
		return false;
	}

	public static BaseDataServiceDao getInstance() {
		if (instance == null) {
			instance = new BaseDataServiceDao();

		}
		return instance;
	}

	public void saveOrUpdateKitchenTickets(String request) {
		try {
			JSONObject rootElement = new JSONObject(request);
			JSONArray kitchentTicketsArray = rootElement.getJSONArray(DATA);

			debug("Request received: " + request); //$NON-NLS-1$

			String terminalKey = rootElement.has("deviceId") ? rootElement.getString("deviceId") : null; //$NON-NLS-1$ //$NON-NLS-2$
			if (StringUtils.isEmpty(terminalKey)) {
				return;
			}
			Terminal terminal = TerminalDAO.getInstance().getByTerminalKey(terminalKey);
			Terminal currentTerminal = DataProvider.get().getCurrentTerminal();

			if (terminal == null || currentTerminal == null) {
				return;
			}
			if (currentTerminal.getId().equals(terminal.getId())) {
				return;
			}
			else if (currentTerminal.isMasterTerminal()) {
				List<KitchenTicket> kitchenTickets = new ArrayList<>();
				if (kitchentTicketsArray != null && kitchentTicketsArray.length() > 0) {
					for (int i = 0; i < kitchentTicketsArray.length(); i++) {
						JSONObject json = kitchentTicketsArray.getJSONObject(i);
						try {
							KitchenTicket kitchenTicket = getGsonBuilder().setExclusionStrategies(new JsonBeanFieldExcluder(MenuShift.class)).create()
									.fromJson(json.toString(), KitchenTicket.class);
							kitchenTickets.add(kitchenTicket);
						} catch (Exception e) {
							PosLog.error(getClass(), e);
						}
					}
				}
				saveOrUpdateKitTickets(kitchenTickets);
			}

			debug("Successfully saved kitchen tickets"); //$NON-NLS-1$
		} catch (Exception e) {
			PosLog.error(getClass(), e);
		}
	}

	public void saveOrUpdateKitTickets(List<KitchenTicket> kitchenTickets) {
		Session session = null;
		Transaction tx = null;
		try {
			KitchenTicketDAO kitchenTicketDAO = KitchenTicketDAO.getInstance();
			session = kitchenTicketDAO.createNewSession();
			tx = session.beginTransaction();
			for (KitchenTicket kitchenTicket : kitchenTickets) {
				if (kitchenTicket == null || kitchenTicket.getId() == null) {
					continue;
				}
				KitchenStatus kitchenStatusValue = kitchenTicket.getKitchenStatusValue();
				if (kitchenStatusValue != null && kitchenStatusValue.equals(KitchenStatus.DISPATCHED)) {
					doPerformDispatchKitchenTicket(session, kitchenTicketDAO, kitchenTicket);
				}
				else {
					doSaveOrUpdateKitchenTicket(session, kitchenTicketDAO, kitchenTicket);
				}
			}
			tx.commit();
		} catch (Exception e) {
			if (tx != null) {
				tx.rollback();
			}
			throw e;
		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

	private void doPerformDispatchKitchenTicket(Session session, KitchenTicketDAO kitchenTicketDAO, KitchenTicket kitchenTicket) {
		KitchenTicket existingKitchenTicket = kitchenTicketDAO.get(kitchenTicket.getId());
		if (existingKitchenTicket == null) {
			return;
		}
		kitchenTicketDAO.delete(existingKitchenTicket, session);
	}

	private void doSaveOrUpdateKitchenTicket(Session session, KitchenTicketDAO kitchenTicketDAO, KitchenTicket kitchenTicket) {
		List<KitchenTicketItem> kitchenTicketItems = kitchenTicket.getTicketItems();
		KitchenTicket existedKitchenTicket = kitchenTicketDAO.get(kitchenTicket.getId());
		kitchenTicket.setTicketItems(null);

		if (existedKitchenTicket == null) {
			kitchenTicketDAO.save(kitchenTicket, session);
		}
		else {
			kitchenTicket.setVersion(existedKitchenTicket.getVersion());
		}
		List<KitchenTicketItem> existingKitchenTicketItems = null;
		if (existedKitchenTicket != null) {
			existingKitchenTicketItems = existedKitchenTicket.getTicketItems();
		}
		for (KitchenTicketItem kitchenTicketItem : kitchenTicketItems) {
			if (existingKitchenTicketItems == null || existingKitchenTicketItems.isEmpty()) {
				KitchenTicketItemDAO.getInstance().save(kitchenTicketItem, session);
				continue;
			}
			int idx = existingKitchenTicketItems.indexOf(kitchenTicketItem);
			if (idx != -1) {
				KitchenTicketItem existingKitchenTicketItem = existingKitchenTicketItems.get(idx);
				kitchenTicketItem.setVersion(existingKitchenTicketItem.getVersion());
			}
			else {
				KitchenTicketItemDAO.getInstance().save(kitchenTicketItem, session);
			}
		}
		kitchenTicket.setTicketItems(kitchenTicketItems);
		kitchenTicketDAO.saveOrUpdate(kitchenTicket, session);
	}

	public void saveOrUpdateVoidItems(String request, String clientOutletId) throws Exception {
		JSONObject rootElement = new JSONObject(request);
		JSONArray dataArray = rootElement.getJSONArray(DATA);

		List<VoidItem> voidItems = new ArrayList<>();
		for (int i = 0; i < dataArray.length(); i++) {
			JSONObject json = dataArray.getJSONObject(i);
			VoidItem voidItem = getGsonBuilder().create().fromJson(json.toString(), VoidItem.class);
			if (voidItem == null) {
				continue;
			}
			voidItem.setOutletId(clientOutletId);
			voidItems.add(voidItem);
		}
		saveOrUpdateVoidItems(voidItems);
	}

	public void saveOrUpdateVoidItems(List<VoidItem> voidItems) {
		if (voidItems == null || voidItems.isEmpty())
			return;
		Session session = null;
		Transaction tx = null;
		try {
			VoidItemDAO voidItemDao = VoidItemDAO.getInstance();
			session = voidItemDao.createNewSession();
			tx = session.beginTransaction();
			for (VoidItem voidItem : voidItems) {
				VoidItem existingVoidItem = voidItemDao.get(voidItem.getId(), voidItem.getOutletId());
				if (existingVoidItem == null) {
					voidItemDao.save(voidItem, session);
				}
				else {
					voidItem.setVersion(existingVoidItem.getVersion());
					voidItemDao.update(voidItem, session);
				}
				VoidItemDAO.getInstance().saveAndSentToKitchenIfNeeded(voidItem, session);
			}
			tx.commit();
		} catch (Exception e) {
			if (tx != null) {
				tx.rollback();
			}
			throw e;
		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

	public void saveOrUpdateAttendanceHistory(List<AttendenceHistory> attendanceHistories) {
		if (attendanceHistories == null || attendanceHistories.isEmpty())
			return;
		Session session = null;
		Transaction tx = null;
		try {
			AttendenceHistoryDAO attHistoryDao = AttendenceHistoryDAO.getInstance();
			session = attHistoryDao.createNewSession();
			tx = session.beginTransaction();
			for (AttendenceHistory att : attendanceHistories) {
				AttendenceHistory existingVoidItem = attHistoryDao.get(att.getId());
				if (existingVoidItem == null) {
					attHistoryDao.save(att, session);
				}
				else {
					att.setVersion(existingVoidItem.getVersion());
					attHistoryDao.update(att, session);
				}
			}
			tx.commit();
		} catch (Exception e) {
			if (tx != null) {
				tx.rollback();
			}
			throw e;
		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

	public void saveOrUpdateUsersClockInOutTime(String request) {
		JSONObject rootElement = new JSONObject(request);
		JSONArray dataArray = rootElement.getJSONArray(DATA);

		List<User> users = new ArrayList<>();
		for (int i = 0; i < dataArray.length(); i++) {
			JSONObject json = dataArray.getJSONObject(i);
			User user = getGsonBuilder().create().fromJson(json.toString(), User.class);
			if (user == null) {
				continue;
			}
			users.add(user);
		}
		Session session = null;
		Transaction tx = null;
		try {
			UserDAO userDao = UserDAO.getInstance();
			session = userDao.createNewSession();
			tx = session.beginTransaction();
			for (User user : users) {
				User existingUser = userDao.get(user.getId(), user.getOutletId());
				if (existingUser == null) {
					continue;
				}
				existingUser.setClockedIn(user.isClockedIn());
				existingUser.setLastClockInTime(user.getLastClockInTime());
				existingUser.setLastClockOutTime(user.getLastClockOutTime());
				userDao.update(existingUser);
			}
			tx.commit();
		} catch (Exception e) {
			if (tx != null) {
				tx.rollback();
			}
			throw e;
		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

	public static GsonBuilder getGsonBuilder() {
		return new GsonBuilder().setDateFormat(GSON_PARSING_DATE_FORMAT);
	}

	public void saveOrSetVersionModifierGroup(Session session, boolean updateLastUpdateTime, boolean updateSyncTime, ModifierGroup modifierGroup,
			ModifierGroup existingModifierGroup) throws Exception {

		Set<MenuItemModifierPage> modifierPages = modifierGroup.getModifierPages();

		modifierGroup.setUpdateLastUpdateTime(updateLastUpdateTime);
		modifierGroup.setUpdateSyncTime(updateSyncTime);
		modifierGroup.setModifierPages(null);
		saveOrSetVersionModifierGroup(session, modifierGroup, existingModifierGroup);

		modifierGroup.setModifierPages(modifierPages);

		if (modifierPages != null && !modifierPages.isEmpty()) {
			for (MenuItemModifierPage menuItemModifierPage : modifierPages) {
				saveOrSetVersionMenuItemModiPage(session, menuItemModifierPage);
			}
		}
	}

	private void saveOrSetVersionModifierGroup(Session session, ModifierGroup modifierGroup, ModifierGroup existingModifierGroup) {
		ModifierGroupDAO instance = ModifierGroupDAO.getInstance();
		if (existingModifierGroup == null) {
			instance.save(modifierGroup, session);
		}
		else {
			session.evict(existingModifierGroup);
			modifierGroup.setVersion(existingModifierGroup.getVersion());
		}
	}

}
