package com.floreantpos.model;

import javax.xml.bind.annotation.XmlTransient;

import org.apache.commons.lang.StringUtils;

import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.floreantpos.model.base.BaseComboItem;
import com.floreantpos.model.dao.MenuItemDAO;
import com.google.gson.Gson;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;

@JsonIgnoreProperties(ignoreUnknown = true, value = { "menuItem" })
public class ComboItem extends BaseComboItem implements PropertyContainer {
	private static final long serialVersionUID = 1L;
	private transient JsonObject propertiesContainer;

	/*[CONSTRUCTOR MARKER BEGIN]*/
	public ComboItem() {
	}

	/**
	 * Constructor for primary key
	 */
	public ComboItem(java.lang.String id) {
		super(id);
	}

	/*[CONSTRUCTOR MARKER END]*/

	private MenuItem menuItem;

	@XmlTransient
	@JsonIgnore
	public MenuItem getMenuItem() {
		if (menuItem == null) {
			String itemId = getMenuItemId();
			if (StringUtils.isEmpty(itemId)) {
				return null;
			}
			menuItem = MenuItemDAO.getInstance().loadInitialized(itemId);
		}
		return menuItem;
	}

	public void setMenuItem(MenuItem menuItem) {
		this.menuItem = menuItem;
		if (menuItem != null) {
			if (StringUtils.isBlank(super.getMenuItemId())) {
				setItemId(menuItem.getId());
			}

			setName(menuItem.getName());
			setDisplayName(menuItem.getDisplayName());
			double itemPrice = menuItem.getPrice();
			if (menuItem.isVariant()) {
				MenuItem parentMenuItem = menuItem.getParentMenuItem();
				if (parentMenuItem != null) {
					itemPrice = parentMenuItem.getPrice() + menuItem.getPrice();
				}
			}
			setPrice(itemPrice);
		}
	}

	public void setDisplayName(String name) {
		addProperty("translatedName", name); //$NON-NLS-1$
	}

	@JsonIgnore
	public String getDisplayName() {
		return getProperty("translatedName", getName()); //$NON-NLS-1$
	}

	@Override
	public void addProperty(String key, String value) {
		initPropertyContainer();
		propertiesContainer.addProperty(key, value);
		super.setProperties(propertiesContainer.toString());
	}

	@Override
	public String getProperty(String key) {
		initPropertyContainer();

		if (propertiesContainer.has(key)) {
			JsonElement jsonElement = propertiesContainer.get(key);
			if (!jsonElement.isJsonNull()) {
				return jsonElement.getAsString();
			}
		}
		return null;
	}

	@Override
	public boolean hasProperty(String key) {
		return getProperty(key) != null;
	}

	private void initPropertyContainer() {
		if (propertiesContainer == null) {
			if (StringUtils.isBlank(super.getProperties())) {
				propertiesContainer = new com.google.gson.JsonObject();
			}
			else {
				propertiesContainer = new Gson().fromJson(super.getProperties(), com.google.gson.JsonObject.class);
			}
		}
	}

	@Override
	public String getProperties() {
		if (propertiesContainer != null) {
			return propertiesContainer.toString();
		}

		String properties = super.getProperties();
		if (StringUtils.isEmpty(properties)) {
			return null;
		}

		propertiesContainer = new Gson().fromJson(properties, JsonObject.class);
		return properties;
	}

	@Override
	public void setProperties(String properties) {
		super.setProperties(properties);
		propertiesContainer = new Gson().fromJson(properties, JsonObject.class);
	}

	@Override
	public JsonObject getPropertyStore() {
		if (propertiesContainer == null) {
			propertiesContainer = new JsonObject();
		}
		return propertiesContainer;
	}

	public double getSubtotalAmount() {
		return super.getQuantity() * super.getPrice();
	}

	public void setSubtotalAmount(double subtotalAmount) {
	}

}