package com.floreantpos.model.dao;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import org.apache.commons.lang.StringUtils;
import org.hibernate.Criteria;
import org.hibernate.Query;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.hibernate.criterion.Disjunction;
import org.hibernate.criterion.MatchMode;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.ProjectionList;
import org.hibernate.criterion.Projections;
import org.hibernate.criterion.Restrictions;
import org.hibernate.transform.Transformers;

import com.floreantpos.PosLog;
import com.floreantpos.model.AccessLog;
import com.floreantpos.model.Customer;
import com.floreantpos.model.OnlineStore;
import com.floreantpos.model.Outlet;
import com.floreantpos.model.Pagination;
import com.floreantpos.model.Store;
import com.floreantpos.model.User;

public class AccessLogDAO extends BaseAccessLogDAO {

	/**
	 * Default constructor.  Can be used in place of getInstance()
	 */
	public AccessLogDAO() {
	}

	public void create(String ipAddress) {
		create(null, null, null, null, null, null, null, ipAddress);
	}

	public void create(Customer currentCustomer, String clientIpAddress) {
		create(currentCustomer, clientIpAddress, null, null);
	}

	public void create(Customer currentCustomer, String ipAddress, String orderId, String action) {
		if (currentCustomer == null) {
			create(null, null, null, null, null, orderId, null, ipAddress, null);
			return;
		}
		create(currentCustomer.getId(), currentCustomer.getName(), currentCustomer.getEmail(), currentCustomer.getAddress(), currentCustomer.getCountry(),
				orderId, null, ipAddress, action);
	}

	public void create(String customerId, String customerName, String customerEmail, String address, String country, String orderId, String cvcCheck, String ipAddress) {
		create(customerId, customerName, customerEmail, address, country, orderId, cvcCheck, ipAddress, null);
	}

	public void create(String customerId, String name, String email, String address, String country, String orderId, String cvcCheck, String ipAddress,
			String action) {
		try {
			AccessLog accessLog = new AccessLog();
			accessLog.setAccessLogDate(new Date());
			accessLog.setCountry(country);
			accessLog.setCustomerEmail(email);
			accessLog.setIpAddress(ipAddress);
			accessLog.setOrderId(orderId);
			if (customerId != null) {
				accessLog.addProperty("customerId", customerId); //$NON-NLS-1$
			}
			if (name != null) {
				accessLog.addProperty("customerName", name); //$NON-NLS-1$
			}
			if (address != null) {
				accessLog.addProperty("customerAddress", address); //$NON-NLS-1$
			}
			if (cvcCheck != null) {
				accessLog.addProperty("cvcCheck", cvcCheck); //$NON-NLS-1$
			}
			if (action != null) {
				accessLog.setAction(action);
			}
			save(accessLog);
		} catch (Exception e) {
			PosLog.error(getReferenceClass(), "Failed to save access log: Customer ID: " + customerId + " IP Address:" + ipAddress + " Name: " + name //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
					+ " Address: " + address + " Country: " + country + " CVC Check: " + cvcCheck + " Order Id: " + orderId); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$
		}
	}

	@SuppressWarnings("unchecked")
	public void loadData(String searchKeyword, Pagination<AccessLog> pagination) {
		loadData(searchKeyword, pagination, null, null, false);
	}

	@SuppressWarnings("unchecked")
	public void loadData(String searchKeyword, Pagination<AccessLog> pagination, String productName, String storeId, boolean isShowUniqueRecord) {
		try (Session session = this.createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.setProjection(Projections.rowCount());
			if (StringUtils.isNotBlank(searchKeyword)) {
				Disjunction disjunction = Restrictions.disjunction();
				disjunction.add(Restrictions.like(AccessLog.PROP_CUSTOMER_EMAIL, searchKeyword, MatchMode.ANYWHERE));
				disjunction.add(Restrictions.eq(AccessLog.PROP_ORDER_ID, searchKeyword));
				disjunction.add(Restrictions.like(AccessLog.PROP_USER_EMAIL, searchKeyword, MatchMode.ANYWHERE));

				criteria.add(disjunction);
			}
			if (StringUtils.isNotBlank(productName)) {
				criteria.add(Restrictions.eq(AccessLog.PROP_PRODUCT_NAME, productName));
			}
			if (StringUtils.isNotBlank(storeId)) {
				criteria.add(Restrictions.like(AccessLog.PROP_STORE_ID, storeId, MatchMode.ANYWHERE));
			}
			if (isShowUniqueRecord) {
				ProjectionList projections = Projections.projectionList();
				projections.add((Projections.property(AccessLog.PROP_STORE_ID)), AccessLog.PROP_STORE_ID);
				projections.add(Projections.max(AccessLog.PROP_ACCESS_LOG_DATE), AccessLog.PROP_ACCESS_LOG_DATE);
				projections.add(Projections.groupProperty(AccessLog.PROP_PRODUCT_NAME), AccessLog.PROP_PRODUCT_NAME);
				projections.add(Projections.groupProperty(AccessLog.PROP_STORE_ID), AccessLog.PROP_STORE_ID);
				criteria.setProjection(projections);
				criteria.setResultTransformer(Transformers.aliasToBean(AccessLog.class));
				criteria.addOrder(Order.desc(AccessLog.PROP_ACCESS_LOG_DATE));
				List list = criteria.list();
				pagination.setData(list);
				pagination.setNumRows(list == null ? 0 : list.size());
			}
			else {
				criteria.setProjection(Projections.rowCount());
				Number uniqueResult = (Number) criteria.uniqueResult();
				int rowCount = uniqueResult == null ? 0 : uniqueResult.intValue();
				pagination.setNumRows(rowCount);
				if (rowCount == 0) {
					pagination.setData(new ArrayList<>());
					return;
				}
				criteria.setProjection(null);
				criteria.setFirstResult(pagination.getCurrentRowIndex());
				criteria.setMaxResults(pagination.getPageSize());
				criteria.addOrder(Order.desc(AccessLog.PROP_ACCESS_LOG_DATE));
				pagination.setData(criteria.list());
			}
		}
	}

	public void createOrohubLoginAccessLog(Store store, String siiopaCustomerEmail, User loggedInUser, String clientIpAddress) {
		createOrohubAccessLog(store, siiopaCustomerEmail, loggedInUser, clientIpAddress, null, null, null);
	}

	public void createOrohubAccessLog(Store store, String siiopaCustomerEmail, User loggedInUser, String clientIpAddress, String action,
			String terminalKey, String source) {
		try {
			AccessLog accessLog = createNewAccessLog(AccessLog.PRODUCT.OROHUB); //$NON-NLS-1$ //$NON-NLS-2$
			loadStoreInfo(accessLog, store);
			loadLoginInfo(accessLog, siiopaCustomerEmail, loggedInUser, clientIpAddress);
			if (action != null) {
				accessLog.setAction(action);
			}
			if (terminalKey != null) {
				accessLog.setTerminalKey(terminalKey);
			}
			if (source != null) {
				accessLog.addProperty("source", source); //$NON-NLS-1$
			}
			save(accessLog);
		} catch (Exception e) {
			error(store.getUuid(), siiopaCustomerEmail, clientIpAddress, e);
		}
	}
	
	public void createOrderManagerAccessLog(Store store, String siiopaCustomerEmail, User loggedInUser, String clientIpAddress) {
		try {
			AccessLog accessLog = createNewAccessLog(AccessLog.PRODUCT.ORDER_MANAGER); //$NON-NLS-1$ //$NON-NLS-2$
			loadStoreInfo(accessLog, store);
			loadLoginInfo(accessLog, siiopaCustomerEmail, loggedInUser, clientIpAddress);
			save(accessLog);
		} catch (Exception e) {
			error(store.getUuid(), siiopaCustomerEmail, clientIpAddress, e);
		}
	}

	public void createOroposLiveLoginAccessLog(Store store, String customerEmail, User loggedInUser, String clientIpAddress, String terminalKey) {
		try {
			AccessLog accessLog = createNewAccessLog(AccessLog.PRODUCT.OROPOS_LIVE); //$NON-NLS-1$ //$NON-NLS-2$
			loadStoreInfo(accessLog, store);
			loadLoginInfo(accessLog, customerEmail, loggedInUser, clientIpAddress);
			accessLog.setOutletName(getUserOutletName(loggedInUser, store));
			accessLog.setTerminalKey(terminalKey);
			save(accessLog);
		} catch (Exception e) {
			error(store.getUuid(), customerEmail, clientIpAddress, e);
		}
	}

	public void createMenugreatAccessLog(OnlineStore store, Customer loggedInCustomer, String clientIpAddress) {
		try {
			AccessLog accessLog = createNewAccessLog(AccessLog.PRODUCT.MENUGREAT);
			if (store != null) {
				accessLog.setStoreName(store.getStoreName());
				accessLog.setStoreId(store.getStoreId());
				accessLog.setStoreUrl(store.getFriendlyUrl());
				accessLog.setOutletName(store.getOutletName());
			}
			if (loggedInCustomer != null) {
				accessLog.setUserEmail(loggedInCustomer.getEmail());
				accessLog.addProperty("customerId", loggedInCustomer.getId()); //$NON-NLS-1$
			}
			accessLog.setIpAddress(clientIpAddress);
			save(accessLog);
		} catch (Exception e) {
			error(store == null ? "" : store.getStoreId(), loggedInCustomer == null ? "" : loggedInCustomer.getEmail(), clientIpAddress, e); //$NON-NLS-1$ //$NON-NLS-2$
		}
	}

	public void loadStoreInfo(AccessLog accessLog, Store store) {
		accessLog.setStoreName(store.getName());
		accessLog.setStoreId(store.getUuid());
		accessLog.setStoreUrl(store.getFriendlyUid());
	}

	private void loadLoginInfo(AccessLog accessLog, String siiopaCustomerEmail, User loggedInUser, String clientIpAddress) {
		accessLog.setCustomerEmail(siiopaCustomerEmail);
		accessLog.setIpAddress(clientIpAddress);
		if (loggedInUser != null) {
			accessLog.setUserEmail(loggedInUser.getEmail());
		}
	}

	private AccessLog createNewAccessLog(AccessLog.PRODUCT product) {
		AccessLog accessLog = new AccessLog();
		accessLog.setAccessLogDate(new Date());
		accessLog.setProductId(product.getId());
		accessLog.setProductName(product.getDisplayString());
		return accessLog;
	}

	//	private AccessLog createNewAccessLog() {
	//		return createNewAccessLog(null, null);
	//	}

	private void error(String storeId, String customerEmail, String clientIpAddress, Exception e) {
		PosLog.error(getReferenceClass(),
				"Failed to save access log: Store Id: " + storeId + " Customer Email: " + customerEmail + " IP Address:" + clientIpAddress, e); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ 
	}

	private String getUserOutletName(User user, Store store) {
		if (user.getOutletId() != null && store.getDefaultOutletId() != null && store.getDefaultOutletId().equals(user.getOutletId())) {
			return store.getOutletName();
		}
		Outlet outlet = OutletDAO.getInstance().get(user.getOutletId());
		if (outlet != null) {
			return outlet.getName();
		}
		return null;
	}
	
	public void deleteLogs(String storeId) {
		Session session = null;
		Transaction transaction = null;
		
		try {
			session = createNewSession();
			transaction = session.beginTransaction();
			
			Query query = session.createQuery("delete from " + AccessLog.REF + " where " + AccessLog.PROP_STORE_ID + "=:storeId");
			query.setParameter("storeId", storeId);
			int rowCount = query.executeUpdate();
			
			transaction.commit();
			
			PosLog.debug(getClass(), "Deleted " + rowCount + " access log.");
		} catch (Exception e) {
			if (transaction != null) {
				transaction.rollback();
			}
			
			throw e;
		} finally {
			closeSession(session);
		}
	}
}