package com.orocube.workspace.subscription;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;

import org.apache.commons.lang.StringUtils;
import org.joda.time.Days;
import org.joda.time.LocalDate;
import org.joda.time.Period;

import com.floreantpos.Messages;
import com.floreantpos.PosException;
import com.floreantpos.PosLog;
import com.floreantpos.model.dao.GlobalConfigDAO;
import com.orocube.siiopa.common.model.SubscriptionStatus;
import com.stripe.Stripe;
import com.stripe.exception.CardException;
import com.stripe.exception.InvalidRequestException;
import com.stripe.exception.StripeException;
import com.stripe.model.Card;
import com.stripe.model.Coupon;
import com.stripe.model.Customer;
import com.stripe.model.PaymentSource;
import com.stripe.model.Price;
import com.stripe.model.PriceCollection;
import com.stripe.model.Product;
import com.stripe.model.ProductCollection;
import com.stripe.model.PromotionCode;
import com.stripe.model.PromotionCodeCollection;
import com.stripe.model.Source;
import com.stripe.model.Subscription;
import com.stripe.model.SubscriptionCollection;
import com.stripe.model.SubscriptionItem;
import com.stripe.model.SubscriptionSearchResult;
import com.stripe.model.Token;
import com.stripe.net.RequestOptions;
import com.stripe.param.CustomerRetrieveParams;
import com.stripe.param.CustomerUpdateParams;
import com.stripe.param.PriceListParams;
import com.stripe.param.ProductListParams;
import com.stripe.param.SubscriptionCancelParams;
import com.stripe.param.SubscriptionCreateParams;
import com.stripe.param.SubscriptionCreateParams.Builder;
import com.stripe.param.SubscriptionCreateParams.CollectionMethod;
import com.stripe.param.SubscriptionCreateParams.PaymentBehavior;
import com.stripe.param.SubscriptionCreateParams.ProrationBehavior;
import com.stripe.param.SubscriptionListParams;
import com.stripe.param.SubscriptionSearchParams;
import com.stripe.param.SubscriptionUpdateParams;

public class StripeUtil {
	private static final String STRIPE_CARD_HOLDER_NAME = "name"; //$NON-NLS-1$
	private static final String STRIPE_METADATA = "metadata"; //$NON-NLS-1$
	private static final String STRIPE_CARD_EXP_YEAR = "exp_year"; //$NON-NLS-1$
	private static final String STRIPE_CARD_EXP_MONTH = "exp_month"; //$NON-NLS-1$
	private static final String STRIPE_CARD_NUMBER = "number"; //$NON-NLS-1$
	private static final String STRIPE_CARD_CVC = "cvc"; //$NON-NLS-1$
	private static final String STRIPE_MODEL_CARD = "card"; //$NON-NLS-1$
	private static final String STRIPE_MODEL_SOURCE = "source"; //$NON-NLS-1$
	private static final String STRIPE_MODEL_ID = "id"; //$NON-NLS-1$
	private static final String STRIPE_CUSTOMER_EMAIL = "email"; //$NON-NLS-1$

	public static List<Product> getProducts(final String apiKey) {
		Stripe.apiKey = apiKey;
		try {
			ProductCollection collection = Product.list(ProductListParams.builder().setLimit(100L).setActive(true).build());
			List<Product> products = collection.getData();
			while (collection.getHasMore()) {
				products.addAll((collection = Product.list(ProductListParams.builder().setLimit(100L)
						.setStartingAfter(collection.getData().get(collection.getData().size() - 1).getId()).setActive(true).build())).getData());
			}
			return products;
		} catch (Exception e0) {
			PosLog.error(StripeUtil.class, e0);
			throw new RuntimeException(e0);
		}
	}

	public static List<Price> getPrices(String apiKey, String productId) {
		Stripe.apiKey = apiKey;
		try {
			PriceCollection collection = Price.list(PriceListParams.builder().setActive(Boolean.TRUE).setLimit(100L).setProduct(productId).build());
			List<Price> plans = collection.getData();
			while (collection.getHasMore()) {
				plans.addAll((collection = Price.list(PriceListParams.builder().setActive(Boolean.TRUE).setLimit(100L).setProduct(productId)
						.setStartingAfter(collection.getData().get(collection.getData().size() - 1).getId()).build())).getData());
			}
			return plans;
		} catch (Exception e0) {
			PosLog.error(StripeUtil.class, e0);
			throw new RuntimeException(e0);
		}
	}

	public static List<Card> getCards(String apiKey, String customerId) {
		Stripe.apiKey = apiKey;
		try {
			Customer stripeCustomer = getCustomerById(apiKey, customerId);
			if (stripeCustomer == null || stripeCustomer.getSources() == null) {
				return null;
			}
			List<Card> cards = new ArrayList<>();
			List<PaymentSource> accounts = stripeCustomer.getSources().getData();
			if (accounts != null && !accounts.isEmpty()) {
				for (Iterator<PaymentSource> iterator = accounts.iterator(); iterator.hasNext();) {
					PaymentSource account = (PaymentSource) iterator.next();
					if (account instanceof Card) {
						Card creditCard = (Card) account;
						cards.add(creditCard);
					}
				}
			}
			return cards;
		} catch (Exception e0) {
			PosLog.error(StripeUtil.class, e0);
			throw new RuntimeException(e0);
		}
	}

	public static Subscription getSubscriptionById(String apiKey, String subscriptionId) {
		return getSubscriptionById(apiKey, subscriptionId, null);
	}

	public static Subscription getSubscriptionById(String apiKey, String subscriptionId, String productId) {
		Stripe.apiKey = apiKey;
		try {
			Subscription subscription = Subscription.retrieve(subscriptionId);
			if (subscription == null || StringUtils.isBlank(productId) || subscription.getItems() == null) {
				return subscription;
			}
			List<SubscriptionItem> items = subscription.getItems().getData();
			if (items != null && items.size() > 0) {
				for (SubscriptionItem subscriptionItem : items) {
					if (subscriptionItem.getPrice() != null && subscriptionItem.getPrice().getProduct().equals(productId)
							&& subscriptionItem.getQuantity() > 0) {
						return subscription;
					}
				}
			}
			return null;
		} catch (Exception e) {
			PosLog.error(StripeUtil.class, e.getMessage());
		}
		return null;
	}

	public static String getSubscriptionStatus(String apiKey, String customerId, String productId) {
		Stripe.apiKey = apiKey;
		try {
			Subscription subscription = getSubscription(apiKey, customerId, productId);
			return getSubscriptionStatus(subscription);
		} catch (Exception e0) {
			PosLog.error(StripeUtil.class, e0);
			throw new RuntimeException(e0);
		}
	}

	public static String getSubscriptionStatus(Subscription subscription) {
		if (subscription == null) {
			return null;
		}
		if (subscription.getTrialEnd() != null && !subscription.getStatus().equals("active")) {
			java.util.Date currentDate = new java.util.Date();
			java.util.Date expiredDate = new java.util.Date(subscription.getTrialEnd() * 1000L);
			Period p = new Period(currentDate.getTime(), expiredDate.getTime());
			long days = calculateDays(currentDate, expiredDate);
			long hours = p.getHours();
			long minutes = p.getMinutes();
			long seconds = p.getSeconds();
			if (days < 0 || hours < 0 || minutes < 0 || seconds < 0) {
				return "trail_end"; //$NON-NLS-1$
			}
		}
		return subscription.getStatus();
	}

	public static boolean hasActiveTrialPeriod(Subscription subscription) {
		if (subscription == null) {
			return false;
		}
		if (subscription.getTrialEnd() == null || subscription.getStatus().equals("active")) { //$NON-NLS-1$
			return false;
		}
		java.util.Date currentDate = new java.util.Date();
		java.util.Date expiredDate = new java.util.Date(subscription.getTrialEnd() * 1000L);

		Period p = new Period(currentDate.getTime(), expiredDate.getTime());
		long days = calculateDays(currentDate, expiredDate);
		long hours = p.getHours();
		long minutes = p.getMinutes();
		long seconds = p.getSeconds();
		if (days > 0 || hours > 0 || minutes > 0 || seconds > 0) {
			return true;
		}
		return false;
	}

	public static String getSubscriptionTrialStatus(String stripeApiKey, Subscription subscription, SubscriptionProduct product) {
		if (subscription == null) {
			subscription = getLastSubscription(stripeApiKey, product);
			if (subscription == null) {
				return null;
			}
		}
		if (subscription.getTrialEnd() == null || subscription.getStatus().equals("active")) { //$NON-NLS-1$
			return null;
		}
		java.util.Date currentDate = new java.util.Date();
		java.util.Date expiredDate = new java.util.Date(subscription.getTrialEnd() * 1000L);
		Period p = new Period(currentDate.getTime(), expiredDate.getTime());

		long days = calculateDays(currentDate, expiredDate);
		long hours = p.getHours();
		long minutes = p.getMinutes();
		long seconds = p.getSeconds();
		if (days < 0 || hours < 0 || minutes < 0 || seconds < 0) {
			return Messages.getString("StripeUtil.1") + " " + product.getLabel() + " " + Messages.getString("StripeUtil.2"); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$
		}

		String trialMsg = Messages.getString("StripeUtil.3"); //$NON-NLS-1$
		if (days > 0) {
			days += 1;
			return trialMsg + " " + days + " " + (days > 1 ? Messages.getString("StripeUtil.4") : Messages.getString("StripeUtil.5")); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$
		}
		if (hours > 0) {
			return trialMsg + " " + hours + " " + (hours > 1 ? Messages.getString("StripeUtil.6") : Messages.getString("StripeUtil.7")); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$
		}
		if (minutes > 0) {
			return trialMsg + " " + minutes + " " + (minutes > 1 ? Messages.getString("StripeUtil.8") : Messages.getString("StripeUtil.9")); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$
		}
		if (seconds > 0) {
			return trialMsg + " " + seconds + " " + (seconds > 1 ? Messages.getString("StripeUtil.10") : Messages.getString("StripeUtil.11")); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$
		}
		return null;
	}

	private static long calculateDays(java.util.Date currentDate, java.util.Date expiredDate) {
		return TimeUnit.MILLISECONDS.toDays(expiredDate.getTime() - currentDate.getTime());
	}

	public static Subscription getCurrentSubscription(String stripeApiKey, String storeId, SubscriptionProduct product) {
		Subscription subscription = getSubscription(stripeApiKey, storeId, product.getId());
		if (subscription != null) {
			return subscription;
		}
		return getLastSubscription(stripeApiKey, product);
	}

	public static Subscription getLastSubscription(String stripeApiKey, SubscriptionProduct product) {
		String lastSubscriptionId = GlobalConfigDAO.getInstance().getProperty(product.getSubscriptionKey() + ".id"); //$NON-NLS-1$
		if (StringUtils.isBlank(lastSubscriptionId)) {
			return null;
		}
		return getSubscriptionById(stripeApiKey, lastSubscriptionId, product.getId());
	}

	public static Integer getSubscriptionTrailEndDayCount(String apiKey, String customerId, String productId) {
		Stripe.apiKey = apiKey;
		try {
			Subscription subscription = getSubscription(apiKey, customerId, productId);
			if (subscription == null) {
				return null;
			}
			if (subscription.getTrialEnd() != null && !subscription.getStatus().equals("active")) { //$NON-NLS-1$
				java.util.Date currentDate = new java.util.Date();
				java.util.Date expiredDate = new java.util.Date(subscription.getTrialEnd() * 1000L);
				return Days.daysBetween(new LocalDate(currentDate.getTime()), new LocalDate(expiredDate.getTime())).getDays();
			}
			return null;
		} catch (Exception e0) {
			PosLog.error(StripeUtil.class, e0);
			throw new RuntimeException(e0);
		}
	}

	public static Subscription getActiveSubscription(String apiKey, String customerId, String productId) {
		List<Subscription> subscriptions = getSubscriptions(apiKey, customerId, productId);
		if (subscriptions == null || subscriptions.isEmpty()) {
			return null;
		}
		for (Subscription subscription : subscriptions) {
			if (subscription.getItems() == null || subscription.getItems().getData() == null) {
				continue;
			}
			for (SubscriptionItem item : subscription.getItems().getData()) {
				if (item.getPrice().getProduct().equals(productId)) {
					if (subscription.getTrialEnd() != null || !isCanceled(subscription)) {
						return subscription;
					}
				}
			}
		}
		return null;
	}

	public static Subscription getSubscription(String apiKey, String storeId, String productId) {
		List<Subscription> subscriptions = getSubscriptions(apiKey, storeId, productId);
		if (subscriptions != null && !subscriptions.isEmpty()) {
			for (Subscription subscription : subscriptions) {
				if (subscription.getStatus() != null && subscription.getStatus().equals("active")) { //$NON-NLS-1$
					return subscription;
				}
			}
			return subscriptions.get(0);
		}
		return null;
	}

	public static boolean isValidSubscriptionProduct(Subscription subscription, String productId) {
		return getActivePrice(subscription, productId) != null;
	}

	public static Price getActivePrice(Subscription subscription, String productId) {
		if (subscription == null || subscription.getItems() == null || subscription.getItems().getData() == null) {
			return null;
		}
		if (isCanceled(subscription) && !StripeUtil.hasActiveTrialPeriod(subscription)) {
			return null;
		}
		for (SubscriptionItem subscriptionItem : subscription.getItems().getData()) {
			if (subscriptionItem.getPrice() == null) {
				continue;
			}
			if (subscriptionItem.getPrice().getProduct().equals(productId) && subscriptionItem.getQuantity() > 0) {
				return subscriptionItem.getPrice();
			}
		}
		return null;
	}

	public static boolean isCanceled(Subscription subscription) {
		return subscription.getStatus() == null || !subscription.getStatus().equals(SubscriptionStatus.ACTIVE.getStatusName());
	}

	public static List<Subscription> getSubscriptions(String apiKey, String storeId, String productId) {
		Stripe.apiKey = apiKey;
		try {
			if (StringUtils.isBlank(storeId) || StringUtils.isBlank(productId)) {
				return null;
			}
			List<Subscription> validSubscriptions = new ArrayList<>();
			List<Subscription> subscriptions = getAllSubscriptionsByStoreId(apiKey, storeId);
			for (Subscription s : subscriptions) {
				List<SubscriptionItem> subscriptionItems = s.getItems().getData();
				if (subscriptionItems != null && subscriptionItems.size() > 0) {
					for (SubscriptionItem subscriptionItem : subscriptionItems) {
						if (subscriptionItem.getPrice() != null && subscriptionItem.getPrice().getProduct().equals(productId)
								&& subscriptionItem.getQuantity() > 0) {
							validSubscriptions.add(s);
						}
					}
				}
			}
			Collections.sort(validSubscriptions, new Comparator<Subscription>() {

				@Override
				public int compare(Subscription o1, Subscription o2) {
					if (o1.getStatus() == null || o2.getStatus() == null) {
						return 0;
					}
					return o1.getStatus().compareTo(o2.getStatus());
				}
			});
			return validSubscriptions;
		} catch (Exception e0) {
			PosLog.error(StripeUtil.class, e0);
			throw new RuntimeException(e0);
		}
	}

	public static List<Subscription> getActiveSubscriptionsForStore(String apiKey, String storeId) {
		Stripe.apiKey = apiKey;
		try {
			if (StringUtils.isBlank(storeId)) {
				return null;
			}
			SubscriptionCollection subscriptionCollection = Subscription.list(SubscriptionListParams.builder().setCustomer(storeId).build());
			return subscriptionCollection.getData();
		} catch (Exception e0) {
			//PosLog.error(StripeUtil.class, e0);
			throw new RuntimeException(e0);
		}
	}

	public static void doCancelNonActiveSubscriptionIfNeeded(String stripeApiKey, String storeId, SubscriptionProduct product) {
		try {
			List<Subscription> subscriptions = getSubscriptions(stripeApiKey, storeId, product.getId());
			doCancelNonActiveSubscriptionIfNeeded(stripeApiKey, product, subscriptions);
		} catch (Exception e) {
			PosLog.error(StripeUtil.class, e);
		}
	}

	public static void doCancelNonActiveSubscriptionIfNeeded(String stripeApiKey, SubscriptionProduct product, List<Subscription> subscriptions)
			throws Exception {
		if (subscriptions == null || subscriptions.isEmpty()) {
			return;
		}
		for (Subscription oldSubscription : subscriptions) {
			if (oldSubscription.getStatus() == null || !oldSubscription.getStatus().equals("active")) { //$NON-NLS-1$
				List<SubscriptionItem> items = oldSubscription.getItems().getData();
				if (items == null || items.size() == 0) {
					continue;
				}
				for (SubscriptionItem subscriptionItem : items) {
					Price price = subscriptionItem.getPrice();
					if (price != null && price.getProduct().equals(product.getId())) {
						doCancelSubscription(stripeApiKey, oldSubscription);
						break;
					}
				}
			}
		}
	}

	public static boolean hasMultipleProducts(Subscription subscription) {
		return subscription.getItems() != null && subscription.getItems().getData() != null && subscription.getItems().getData().size() > 1;
	}

	public static Subscription doCancelSubscription(String apiKey, Subscription subscription) {
		Stripe.apiKey = apiKey;
		try {
			subscription = Subscription.retrieve(subscription.getId());
			if (subscription.getStatus().equals("canceled")) { //$NON-NLS-1$
				return subscription;
			}
			subscription = subscription.cancel(SubscriptionCancelParams.builder().setInvoiceNow(false).setProrate(false).build());
			return subscription;
		} catch (Exception e0) {
			PosLog.error(StripeUtil.class, e0);
			throw new RuntimeException(e0);
		}
	}

	public static Customer getCustomerById(String apiKey, String customerId) {
		Stripe.apiKey = apiKey;
		try {
			Customer customer = Customer.retrieve(customerId, CustomerRetrieveParams.builder().addExpand("sources").build(), RequestOptions.getDefault()); //$NON-NLS-1$
			return customer;
		} catch (Exception e0) {
			//PosLog.error(StoreUtil.class, e0.getMessage());
		}
		return null;
	}

	public static Customer createCustomer(String apiKey, String customerEmail, String customerId) {
		Stripe.apiKey = apiKey;
		try {
			Customer customer = getCustomerById(apiKey, customerId);
			if (customer == null) {
				customer = createCustomer(customerEmail, customerId);
			}
			return customer;
		} catch (Exception e0) {
			PosLog.error(StripeUtil.class, e0);
			throw new RuntimeException(e0);
		}
	}

	private static Customer createCustomer(String customerEmail, String customerId) throws StripeException {
		Map<String, Object> customerParams = new HashMap<>();
		customerParams.put(STRIPE_CUSTOMER_EMAIL, customerEmail);
		if (StringUtils.isNotEmpty(customerId)) {
			customerParams.put(STRIPE_MODEL_ID, customerId);
		}
		return Customer.create(customerParams);
	}

	public static Token createToken(String apiKey, String cardNumber, Integer cardExpMonth, Integer cardExpYear, String cvc, String outletId) {
		return createToken(apiKey, null, cardNumber, cardExpMonth, cardExpYear, cvc, outletId);
	}

	public static Token createToken(String apiKey, String cardHolderName, String cardNumber, Integer cardExpMonth, Integer cardExpYear, String cvc,
			String outletId) {
		Stripe.apiKey = apiKey;
		try {
			Map<String, Object> tokenParams = new HashMap<String, Object>();

			Map<String, Object> cardMetaData = new HashMap<String, Object>();
			cardMetaData.put(outletId, true);

			Map<String, Object> cardParams = new HashMap<String, Object>();
			if (cardHolderName != null) {
				cardParams.put("name", cardHolderName); //$NON-NLS-1$
			}
			cardParams.put(STRIPE_CARD_NUMBER, cardNumber);
			cardParams.put(STRIPE_CARD_EXP_MONTH, cardExpMonth);
			cardParams.put(STRIPE_CARD_EXP_YEAR, cardExpYear);
			cardParams.put(STRIPE_CARD_CVC, cvc);
			cardParams.put(STRIPE_METADATA, cardMetaData);

			tokenParams.put(STRIPE_MODEL_CARD, cardParams);
			Token token = Token.create(tokenParams);
			return token;
		} catch (CardException ex) {
			PosLog.error(StripeUtil.class, ex.getMessage());
			throw new PosException(ex.getStripeError().getMessage());
		} catch (Exception e0) {
			throw new RuntimeException(e0);
		}
	}

	public static Card createOrUpdateCard(String apiKey, String customerId, String stripeCardId, String cardHolderName, String cardNumber, Integer cardExpMonth,
			Integer cardExpYear, String cvc, String outletId) {
		Customer customer = getCustomerById(apiKey, customerId);
		Card card = null;
		if (StringUtils.isNotBlank(stripeCardId)) {
			card = getCardById(customer, stripeCardId);
		}
		return createOrUpdateCard(apiKey, customer, card, cardHolderName, cardNumber, cardExpMonth, cardExpYear, cvc, outletId);
	}

	public static Card createOrUpdateCard(String apiKey, Customer customer, Card stripeCard, String cardHolderName, String cardNumber, Integer cardExpMonth,
			Integer cardExpYear, String cvc, String outletId) {
		Stripe.apiKey = apiKey;
		try {
			Map<String, Object> cardMetaData = new HashMap<String, Object>();
			if (outletId != null) {
				cardMetaData.put(outletId, true);
			}
			if (stripeCard == null) {
				Map<String, Object> tokenParams = new HashMap<String, Object>();
				Map<String, Object> cardParams = new HashMap<String, Object>();
				if (StringUtils.isNotBlank(cardHolderName)) {
					cardParams.put(STRIPE_CARD_HOLDER_NAME, cardHolderName);
				}
				cardParams.put(STRIPE_CARD_NUMBER, cardNumber);
				cardParams.put(STRIPE_CARD_EXP_MONTH, cardExpMonth);
				cardParams.put(STRIPE_CARD_EXP_YEAR, cardExpYear);
				cardParams.put(STRIPE_CARD_CVC, cvc);
				cardParams.put(STRIPE_METADATA, cardMetaData);

				tokenParams.put(STRIPE_MODEL_CARD, cardParams);
				Token token = Token.create(tokenParams);

				Map<String, Object> sourceParams = new HashMap<>();
				sourceParams.put(STRIPE_MODEL_SOURCE, token.getId());
				return (Card) customer.getSources().create(sourceParams);
			}
			else {
				Map<String, Object> cardParams = new HashMap<String, Object>();
				if (StringUtils.isNotBlank(cardHolderName)) {
					cardParams.put(STRIPE_CARD_HOLDER_NAME, cardHolderName);
				}
				cardParams.put(STRIPE_CARD_EXP_MONTH, cardExpMonth);
				cardParams.put(STRIPE_CARD_EXP_YEAR, cardExpYear);
				cardParams.put(STRIPE_METADATA, cardMetaData);
				return (Card) stripeCard.update(cardParams);
			}
		} catch (CardException ex) {
			PosLog.error(StripeUtil.class, ex.getMessage());
			throw new PosException(ex.getStripeError().getMessage());
		} catch (Exception e0) {
			throw new RuntimeException(e0);
		}
	}

	public static Subscription createSubscription(String apiKey, String customerId, String stripePriceId, String lookupKey, String cardId, String promoCode,
			String discountId, String productId, Long planQuantity) {
		return createSubscription(apiKey, customerId, stripePriceId, lookupKey, cardId, promoCode, discountId, productId, planQuantity, null);
	}

	public static Subscription createSubscription(String apiKey, String customerId, String stripePriceId, String lookupKey, String cardId, String promoCode,
			String discountId, String productId, Long planQuantity, String storeId) {
		Stripe.apiKey = apiKey;
		try {
			PromotionCode promotion = getPromotionByCode(apiKey, promoCode, productId);
			Coupon defaultCoupon = promotion != null ? null : getCoupon(apiKey, discountId, productId);

			//@formatter:off
			Builder builder =
					SubscriptionCreateParams.builder()
			    .setCancelAtPeriodEnd(false)
			    .setCustomer(customerId)
			    .setDefaultPaymentMethod(cardId)
			    .setDefaultSource(cardId)
			    .setPaymentBehavior(PaymentBehavior.ERROR_IF_INCOMPLETE)
			    .setCollectionMethod(CollectionMethod.CHARGE_AUTOMATICALLY)
			    .setProrationBehavior(ProrationBehavior.ALWAYS_INVOICE)
			    .setPromotionCode(promotion==null?null:promotion.getId())
			    .setCoupon(defaultCoupon==null?null:defaultCoupon.getId())
			    .addItem(
			    		SubscriptionCreateParams.Item.builder()
			        .setQuantity(planQuantity)
			        .setPrice(stripePriceId)
			        .build())
			    ;
			if(StringUtils.isNotBlank(storeId)) {
				builder=builder.putMetadata("storeId", storeId);
			}
			//@formatter:on
			return Subscription.create(builder.build());
		} catch (CardException ex) {
			PosLog.error(StripeUtil.class, ex.getMessage());
			throw new PosException(ex.getStripeError().getMessage());
		} catch (Exception e0) {
			PosLog.error(StripeUtil.class, e0);
			throw new RuntimeException(e0);
		}
	}

	public static Subscription updateSubscriptionProductQuantity(String apiKey, Subscription subscription, SubscriptionProduct subscriptionProduct,
			Integer quantity) throws Exception {
		for (SubscriptionItem subscriptionItem : subscription.getItems().getData()) {
			if (subscriptionItem.getPrice().getProduct().equals(subscriptionProduct.getId())) {
				try {
					//@formatter:off
					Stripe.apiKey = apiKey;
					com.stripe.param.SubscriptionUpdateParams.Builder builder = SubscriptionUpdateParams.builder()
					.setPaymentBehavior(com.stripe.param.SubscriptionUpdateParams.PaymentBehavior.ERROR_IF_INCOMPLETE)
					.setProrationBehavior(SubscriptionUpdateParams.ProrationBehavior.NONE);
					SubscriptionUpdateParams params =
							  builder
							    .addItem(
							      SubscriptionUpdateParams.Item.builder()
							        .setId(subscriptionItem.getId())
							        .setQuantity(quantity.longValue())
							        .setPrice(subscriptionItem.getPrice().getId())
							        .build())
							    .build();
					return subscription.update(params);
					//@formatter:on
				} catch (CardException ex) {
					PosLog.error(StripeUtil.class, ex.getMessage());
					throw ex;
				} catch (InvalidRequestException ex) {
					PosLog.error(StripeUtil.class, ex.getMessage());
					throw ex;
				} catch (Exception e0) {
					throw new RuntimeException(e0);
				}
			}
		}
		return subscription;
	}

	public static Subscription createTrialSubscription(String apiKey, String storeId, String customerId, String customerEmail, String productId,
			String planId) {
		Stripe.apiKey = apiKey;
		boolean testMode = apiKey.startsWith("sk_test");

		try {
			Customer customer = getCustomerById(apiKey, customerId);
			if (customer == null) {
				createCustomer(customerEmail, customerId);
			}
			else {
				Subscription subscription = getSubscription(apiKey, customerId, productId);
				if (subscription != null) {
					return subscription;
				}
			}
			String priceId = null;
			List<Price> stripePricePlans = getPrices(apiKey, productId);
			if (stripePricePlans != null && stripePricePlans.size() > 0) {
				for (Price price : stripePricePlans) {
					if (price.getLookupKey() != null && price.getLookupKey().equals(planId)) {
						priceId = price.getId();
					}
				}
			}
			if (priceId == null) {
				return null;
			}
			Calendar calendar = Calendar.getInstance();
			if (testMode) { //$NON-NLS-1$
				PosLog.info(StripeUtil.class, "Creating one day trial for test mode");
				calendar.add(Calendar.DAY_OF_MONTH, 1);
			}
			else {
				PosLog.info(StripeUtil.class, "Creating one month trial for live mode");
				calendar.add(Calendar.MONTH, 1);
			}
			long timeInMillis = calendar.getTimeInMillis();
			//@formatter:off
			Builder builder = SubscriptionCreateParams.builder()
			    .setCancelAtPeriodEnd(true)
			    .setCustomer(customerId)
			    .setPaymentBehavior(PaymentBehavior.ERROR_IF_INCOMPLETE)
			    .setTrialEnd(timeInMillis / 1000L)
			    .addItem(
			    		SubscriptionCreateParams.Item.builder()
			        .setQuantity(1L)
			        .setPrice(priceId)
			        .build())
			    ;
			//@formatter:on
			if (StringUtils.isNotBlank(storeId)) {
				builder = builder.putMetadata("storeId", storeId);
			}
			return Subscription.create(builder.build());
		} catch (CardException ex) {
			PosLog.error(StripeUtil.class, ex.getMessage());
			throw new PosException(ex.getStripeError().getMessage());
		} catch (Exception e0) {
			PosLog.error(StripeUtil.class, e0);
			throw new RuntimeException(e0);
		}
	}

	public static Card getCardById(String apiKey, String customerId, String cardId) {
		Stripe.apiKey = apiKey;
		try {
			Customer customer = getCustomerById(apiKey, customerId);
			PaymentSource account = null;
			Card card = null;
			try {
				account = customer.getSources().retrieve(cardId);
				card = (Card) account;
			} catch (Exception e1) {
				if (account != null) {
					Source source = Source.retrieve(cardId);
					source.detach();
				}
			}
			return card;
		} catch (Exception e0) {
			PosLog.error(StripeUtil.class, e0);
			throw new RuntimeException(e0);
		}
	}

	public static Card getCardById(Customer stripeCustomer, String cardId) {
		if (stripeCustomer == null || stripeCustomer.getSources() == null) {
			return null;
		}
		List<PaymentSource> accounts = stripeCustomer.getSources().getData();
		if (accounts != null && !accounts.isEmpty()) {
			for (Iterator<PaymentSource> iterator = accounts.iterator(); iterator.hasNext();) {
				PaymentSource account = (PaymentSource) iterator.next();
				if (account instanceof Card) {
					Card creditCard = (Card) account;
					if (creditCard.getId().equals(cardId)) {
						return creditCard;
					}
				}
			}
		}
		return null;
	}

	public static Card removeStripeCard(String apiKey, Card stripeCard) {
		Stripe.apiKey = apiKey;
		try {
			return stripeCard.delete();
		} catch (Exception e0) {
			throw new RuntimeException(e0);
		}
	}

	public static PromotionCode getPromotionByCode(String apiKey, String promoCode, String productId) {
		if (StringUtils.isBlank(promoCode)) {
			return null;
		}
		Stripe.apiKey = apiKey;
		try {
			Map<String, Object> params = new HashMap<>();
			params.put("code", promoCode); //$NON-NLS-1$  
			params.put("limit", 1); //$NON-NLS-1$  
			PromotionCodeCollection promotionCodes = PromotionCode.list(params);
			if (promotionCodes == null || promotionCodes.getData() == null || promotionCodes.getData().isEmpty()) {
				return null;
			}
			PromotionCode promotion = promotionCodes.getData().get(0);
			if (promotion == null) {
				return null;
			}
			if (!promotion.getActive()) {
				if (promotion.getExpiresAt() != null) {
					java.util.Date currentDate = new java.util.Date();
					java.util.Date expiredDate = new java.util.Date(promotion.getExpiresAt() * 1000L);
					if (currentDate.after(expiredDate)) {
						throw new PosException(Messages.getString("StripeUtil.12")); //$NON-NLS-1$
					}
				}
				return null;
			}
			if (getCoupon(apiKey, promotion.getCoupon().getId(), productId) == null) {
				return null;
			}
			return promotion;
		} catch (PosException e0) {
			throw e0;
		} catch (Exception e0) {
			PosLog.error(StripeUtil.class, e0.getMessage());
		}
		return null;
	}

	public static Coupon getCoupon(String apiKey, String couponId, String productId) {
		if (StringUtils.isBlank(couponId)) {
			return null;
		}
		Stripe.apiKey = apiKey;
		try {
			List<String> expandList = new ArrayList<>();
			expandList.add("applies_to"); //$NON-NLS-1$  

			Map<String, Object> params = new HashMap<String, Object>();
			params.put("expand", expandList); //$NON-NLS-1$  
			Coupon coupon = Coupon.retrieve(couponId, params, null);
			if (!isValidCoupon(coupon, productId)) {
				return null;
			}
			return coupon;
		} catch (Exception e0) {
			PosLog.error(StripeUtil.class, e0.getMessage());
		}
		return null;
	}

	public static boolean isValidCoupon(Coupon coupon, String productId) {
		if (coupon == null || !coupon.getValid()) {
			return false;
		}
		if (StringUtils.isNotBlank(productId) && coupon.getAppliesTo() != null) {
			List<String> products = coupon.getAppliesTo().getProducts();
			if (products != null && products.size() > 0) {
				if (!products.contains(productId)) {
					return false;
				}
			}
		}
		return true;
	}

	public static void setSubscriptionPropertiesToGlobalConfig(Subscription subscription, String productId) {
		SubscriptionProduct product = SubscriptionProduct.getById(productId);
		if (product != null) {
			boolean isSubscriptionActive = subscription != null && subscription.getStatus().equals("active"); //$NON-NLS-1$ 

			String plan = ""; //$NON-NLS-1$
			if (isSubscriptionActive) {
				List<SubscriptionItem> items = subscription.getItems().getData();
				if (items != null && items.size() > 0) {
					for (SubscriptionItem subscriptionItem : items) {
						if (subscriptionItem.getPrice() != null && subscriptionItem.getPrice().getProduct().equals(productId)) {
							if (subscriptionItem.getQuantity() > 0) {
								plan = subscriptionItem.getPrice().getLookupKey();
							}
							else {
								isSubscriptionActive = false;
								if (product == SubscriptionProduct.Menugreat) {
									plan = SubscriptionPlan.Starter.getId();
								}
								else if (product == SubscriptionProduct.Siiopa) {
									plan = SubscriptionPlan.SiiopaStarter.getId();
								}
							}
							break;
						}
					}
				}
			}
			else if (product == SubscriptionProduct.Menugreat) {
				plan = SubscriptionPlan.Starter.getId();
			}
			else if (product == SubscriptionProduct.Siiopa) {
				plan = SubscriptionPlan.SiiopaStarter.getId();
			}

			String active = isSubscriptionActive ? "true" : "false";//$NON-NLS-1$ //$NON-NLS-2$
			String subscriptionId = subscription == null ? "" : subscription.getId(); //$NON-NLS-1$
			GlobalConfigDAO instance = GlobalConfigDAO.getInstance();
			instance.addProperty(product.getSubscriptionKey(), active);
			instance.addProperty(product.getSubscriptionKey() + ".id", subscriptionId); //$NON-NLS-1$
			if (StringUtils.isNotBlank(plan)) {
				instance.addProperty(product.getSubscriptionKey() + ".plan", plan); //$NON-NLS-1$
			}
			PosLog.info(StripeUtil.class, String.format("Global config updated>>product: %s current plan: %s subscription id: %s status: %s", //$NON-NLS-1$
					product.getLabel(), plan, subscriptionId, subscription == null ? "" : subscription.getStatus()));
		}
	}

	public static boolean isTrialingOrActive(Subscription subscription) {
		return subscription.getStatus().equals("active") || subscription.getStatus().equals("trialing"); //$NON-NLS-1$ //$NON-NLS-2$
	}

	public static boolean isMenugreateStarterPlan(String apiKey, String storeId) {
		SubscriptionProduct productId = SubscriptionProduct.Menugreat;
		String planId = GlobalConfigDAO.getInstance().getProperty(productId.getSubscriptionKey() + ".plan"); //$NON-NLS-1$
		if (StringUtils.isBlank(planId)/* || isMenugreatStarterPlan(planId)*/) {
			planId = getCurrentPlanId(apiKey, storeId, productId);
		}
		return isMenugreatStarterPlan(planId);
	}

	private static boolean isMenugreatStarterPlan(String planId) {
		SubscriptionPlan subscriptionPlan = SubscriptionPlan.getById(planId);
		if (subscriptionPlan == null) {
			return true;
		}
		if (subscriptionPlan == SubscriptionPlan.StandardMonthly || subscriptionPlan == SubscriptionPlan.StandardYearly
				|| subscriptionPlan == SubscriptionPlan.ProMonthly || subscriptionPlan == SubscriptionPlan.ProYearly) {
			return false;
		}
		return true;
	}

	private static String getCurrentPlanId(String apiKey, String storeId, SubscriptionProduct productId) {
		try {
			Subscription subscription = getSubscription(apiKey, storeId, productId.getId());
			setSubscriptionPropertiesToGlobalConfig(subscription, productId.getId());
			if (subscription != null) {
				List<SubscriptionItem> dataList = subscription.getItems().getData();
				if (dataList != null && dataList.size() > 0) {
					for (SubscriptionItem subscriptionItem : dataList) {
						if (subscriptionItem.getPrice() != null && subscriptionItem.getPrice().getProduct().equals(productId)
								&& subscriptionItem.getQuantity() > 0) {
							return subscriptionItem.getPrice().getLookupKey();
						}
					}
				}
			}
		} catch (Exception e) {
		}
		return null;
	}

	public static boolean hasPosLiveSubscription(String stripeApiKey, String storeId, String storeOwnerEmail) {
		String oldSubscriptionId = GlobalConfigDAO.getInstance().getProperty(SubscriptionProduct.OroposLive.getSubscriptionKey() + ".id"); //$NON-NLS-1$
		if (StringUtils.isBlank(oldSubscriptionId)) {
			return false;
		}
		String productId = SubscriptionProduct.OroposLive.getId();
		String subscriptionStatus = getSubscriptionStatus(stripeApiKey, storeId, productId);
		if (subscriptionStatus == null) {
			if (StringUtils.isNotBlank(oldSubscriptionId)) {
				Subscription oldSubscription = getSubscriptionById(stripeApiKey, oldSubscriptionId, productId);
				subscriptionStatus = StripeUtil.getSubscriptionStatus(oldSubscription);
				if (subscriptionStatus != null) {
					if (subscriptionStatus.equals("trail_end")) { //$NON-NLS-1$
						throw new PosException(Messages.getString("SubsTrailEndMsg")); //$NON-NLS-1$
					}
					if (subscriptionStatus.equals("canceled")) { //$NON-NLS-1$
						throw new PosException(Messages.getString("SubsCanceledMsg")); //$NON-NLS-1$
					}
				}
			}
			throw new PosException(Messages.getString("SubsActiveMsg")); //$NON-NLS-1$
		}
		if (subscriptionStatus.equals("active")) { //$NON-NLS-1$
			return true;
		}
		if (subscriptionStatus.equals("trialing")) { //$NON-NLS-1$
			return true;
		}
		if (subscriptionStatus.equals("trail_end")) { //$NON-NLS-1$
			throw new PosException(Messages.getString("SubsTrailEndMsg")); //$NON-NLS-1$
		}
		if (subscriptionStatus.equals("canceled") || subscriptionStatus.equals("invalid")) { //$NON-NLS-1$ //$NON-NLS-2$
			throw new PosException(Messages.getString("SubsCanceledMsg")); //$NON-NLS-1$
		}
		throw new PosException(Messages.getString("SubsPaymentFailedMsg")); //$NON-NLS-1$
	}

	public static void createPosLiveTrialSubscription(String stripeApiKey, String storeId, String storeOwnerEmail) {
		if (storeOwnerEmail == null) {
			return;
		}
		Subscription trialSubscription = createTrialSubscription(stripeApiKey, storeId, storeId, storeOwnerEmail, SubscriptionProduct.OroposLive.getId(),
				SubscriptionPlan.OroposLiveMonthly.getId());
		setSubscriptionPropertiesToGlobalConfig(trialSubscription, SubscriptionProduct.OroposLive.getId());
	}

	public static void updateCustomerIfNeeded(String apiKey, String customerId, String customerEmail, String customerName) {
		Customer stripeCustomer = getCustomerById(apiKey, customerId);
		if (stripeCustomer == null) {
			return;
		}
		boolean equalsName = stripeCustomer.getName() != null && customerName != null && customerName.equalsIgnoreCase(stripeCustomer.getName());
		boolean equalsEmail = stripeCustomer.getEmail() != null && customerEmail != null && customerEmail.equalsIgnoreCase(stripeCustomer.getEmail());
		if (equalsName && equalsEmail) {
			return;
		}
		try {
			stripeCustomer.update(CustomerUpdateParams.builder().setName(customerName).setEmail(customerEmail).build());
			stripeCustomer.setName(customerName);
			stripeCustomer.setEmail(customerEmail);
		} catch (StripeException e) {
			PosLog.error(StripeUtil.class, e);
		}
	}

	public static List<Subscription> getAllSubscriptionsByStoreId(String apiKey, String storeUuId) throws Exception {
		Stripe.apiKey = apiKey;
		SubscriptionSearchParams params = SubscriptionSearchParams.builder().setQuery("-status:'canceled' metadata['storeId']:'" + storeUuId + "'").build();
		SubscriptionSearchResult result = Subscription.search(params);
		return result.getData();

	}

	public static List<Subscription> getSubscriptionsByStoreId(String apiKey, String storeUuId) throws Exception {
		Stripe.apiKey = apiKey;
		SubscriptionSearchParams params = SubscriptionSearchParams.builder().setQuery("status:'active' AND metadata['storeId']:'" + storeUuId + "'").build();
		SubscriptionSearchResult result = Subscription.search(params);
		List<Subscription> activeSubscriptions = result.getData();
		if (activeSubscriptions == null) {
			activeSubscriptions = new ArrayList<Subscription>();
		}
		params = SubscriptionSearchParams.builder().setQuery("status:'trialing' AND metadata['storeId']:'" + storeUuId + "'").build();
		result = Subscription.search(params);
		List<Subscription> trialingSubs = result.getData();
		if (trialingSubs != null && trialingSubs.size() > 0) {
			activeSubscriptions.addAll(trialingSubs);
		}
		return activeSubscriptions;
	}
}
