/**
 * ************************************************************************
 * * The contents of this file are subject to the MRPL 1.2
 * * (the  "License"),  being   the  Mozilla   Public  License
 * * Version 1.1  with a permitted attribution clause; you may not  use this
 * * file except in compliance with the License. You  may  obtain  a copy of
 * * the License at http://www.floreantpos.org/license.html
 * * Software distributed under the License  is  distributed  on  an "AS IS"
 * * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * * License for the specific  language  governing  rights  and  limitations
 * * under the License.
 * * The Original Code is FLOREANT POS.
 * * The Initial Developer of the Original Code is OROCUBE LLC
 * * All portions are Copyright (C) 2015 OROCUBE LLC
 * * All Rights Reserved.
 * ************************************************************************
 */
package com.floreantpos.model.dao;

import java.io.Serializable;
import java.util.Iterator;
import java.util.List;

import org.apache.commons.beanutils.PropertyUtils;
import org.hibernate.Criteria;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.hibernate.criterion.Projections;
import org.hibernate.criterion.Restrictions;

import com.floreantpos.Messages;
import com.floreantpos.PosException;
import com.floreantpos.PosLog;
import com.floreantpos.model.IUnit;
import com.floreantpos.model.InventoryStock;
import com.floreantpos.model.InventoryTransaction;
import com.floreantpos.model.InventoryUnit;
import com.floreantpos.model.MenuItem;
import com.floreantpos.model.PackagingUnit;
import com.floreantpos.util.POSUtil;
import com.orocube.rest.service.server.BaseDataServiceDao;

public class InventoryUnitDAO extends BaseInventoryUnitDAO {

	/**
	 * Default constructor.  Can be used in place of getInstance()
	 */
	public InventoryUnitDAO() {
	}

	@Override
	protected Serializable save(Object obj, Session s) {
		updateTime(obj);
		return super.save(obj, s);
	}

	@Override
	protected void update(Object obj, Session s) {
		updateTime(obj);
		super.update(obj, s);
	}

	@Override
	protected void saveOrUpdate(Object obj, Session s) {
		updateTime(obj);
		super.saveOrUpdate(obj, s);
	}

	@Override
	public List<InventoryUnit> findAll() {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			addDeletedFilter(criteria);
			return criteria.list();
		}
	}

	@Override
	protected void delete(Object obj, Session s) {
		InventoryUnit inventoryUnit = (InventoryUnit) obj;
		if (inventoryUnit == null) {
			throw new PosException(Messages.getString("InventoryUnitDAO.0")); //$NON-NLS-1$
		}
		if (isInventoryUnitUsed(inventoryUnit)) {
			throw new PosException(Messages.getString("InventoryUnitDAO.1")); //$NON-NLS-1$
		}

		inventoryUnit.setDeleted(true);
		update(inventoryUnit, s);
	}

	public boolean nameExists(InventoryUnit unit, String code) {
		Session session = null;
		try {
			session = getSession();
			Criteria criteria = session.createCriteria(InventoryUnit.class);
			if (unit.getId() != null)
				criteria.add(Restrictions.ne(InventoryUnit.PROP_ID, unit.getId()));
			criteria.add(Restrictions.eq(InventoryUnit.PROP_CODE, code).ignoreCase());
			criteria.setProjection(Projections.rowCount());
			Number rowCount = (Number) criteria.uniqueResult();
			if (rowCount != null && ((Number) rowCount).intValue() > 0) {
				return true;
			}
			criteria = session.createCriteria(PackagingUnit.class);
			criteria.add(Restrictions.eq(InventoryUnit.PROP_CODE, code).ignoreCase());
			criteria.setProjection(Projections.rowCount());
			rowCount = (Number) criteria.uniqueResult();
			if (rowCount != null && ((Number) rowCount).intValue() > 0) {
				return true;
			}

			return false;

		} finally {
			if (session != null) {
				closeSession(session);
			}
		}
	}

	public List<InventoryUnit> findByGroupId(String unitGroupId) {
		Session session = null;
		try {
			session = getSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.eq(InventoryUnit.PROP_UNIT_GROUP_ID, unitGroupId));
			addDeletedFilter(criteria);
			return criteria.list();
		} finally {
			if (session != null) {
				closeSession(session);
			}
		}
	}

	public List<InventoryUnit> findUnGrouped() {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.isNull(InventoryUnit.PROP_UNIT_GROUP_ID));
			criteria.add(Restrictions.or(Restrictions.isNull(InventoryUnit.PROP_DELETED), Restrictions.ne(InventoryUnit.PROP_DELETED, Boolean.TRUE)));
			return criteria.list();
		}
	}

	public IUnit findByCode(String code) {
		Session session = null;
		try {
			session = getSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.eq(InventoryUnit.PROP_CODE, code));
			addDeletedFilter(criteria);
			List list = criteria.list();
			if (list != null && !list.isEmpty()) {
				return (InventoryUnit) list.get(0);
			}
			criteria = session.createCriteria(PackagingUnit.class);
			criteria.add(Restrictions.eq(PackagingUnit.PROP_CODE, code));
			addDeletedFilter(criteria);
			list = criteria.list();
			if (list != null && !list.isEmpty()) {
				return (PackagingUnit) list.get(0);
			}
			return null;
		} finally {
			if (session != null) {
				closeSession(session);
			}
		}
	}

	public InventoryUnit findByName(String name) {
		Session session = null;
		try {
			session = getSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.eq(InventoryUnit.PROP_NAME, name));
			List list = criteria.list();
			if (list != null && !list.isEmpty()) {
				return (InventoryUnit) list.get(0);
			}

			return null;
		} finally {
			if (session != null) {
				closeSession(session);
			}
		}
	}

	public boolean isInventoryUnitUsed(InventoryUnit unit) {
		Session session = null;
		Criteria criteria = null;
		try {
			session = createNewSession();
			criteria = session.createCriteria(MenuItem.class);
			criteria.add(Restrictions.eq(MenuItem.PROP_UNIT_NAME, unit.getCode()));
			criteria.setProjection(Projections.rowCount());
			Object uniqueResult1 = criteria.uniqueResult();
			if (POSUtil.getInteger(uniqueResult1) > 0) {
				return true;
			}

			criteria = session.createCriteria(InventoryStock.class);
			criteria.add(Restrictions.eq(InventoryStock.PROP_UNIT, unit.getCode()));
			criteria.setProjection(Projections.rowCount());
			Object uniqueResult2 = criteria.uniqueResult();
			if (POSUtil.getInteger(uniqueResult2) > 0) {
				return true;
			}

			criteria = session.createCriteria(InventoryTransaction.class);
			criteria.add(Restrictions.eq(InventoryTransaction.PROP_UNIT, unit.getCode()));
			criteria.setProjection(Projections.rowCount());
			Object uniqueResult3 = criteria.uniqueResult();
			if (POSUtil.getInteger(uniqueResult3) > 0) {
				return true;
			}

		} finally {
			closeSession(session);
		}
		return false;
	}
	/*public void saveOrUpdate(InventoryUnit unit) {
		Session session = null;
		Transaction tx = null;
		try {
			session = createNewSession();
			tx = session.beginTransaction();
			if (unit.isBaseUnit()) {
				unit.setConversionRate(1.0);
				Criteria criteria = session.createCriteria(getReferenceClass());
				criteria.add(Restrictions.eq(InventoryUnit.PROP_UNIT_GROUP, unit.getUnitGroup()));
				criteria.add(Restrictions.eq(InventoryUnit.PROP_BASE_UNIT, Boolean.TRUE));
	
				InventoryUnit oldBaseUnit = (InventoryUnit) criteria.uniqueResult();
				if (oldBaseUnit != null) {
					oldBaseUnit.setBaseUnit(false);
					session.saveOrUpdate(oldBaseUnit);
				}
			}
			else
				session.saveOrUpdate(unit);
			tx.commit();
		} catch (Exception ex) {
			tx.rollback();
		} finally {
			if (session != null) {
				closeSession(session);
			}
		}
	}*/

	public void saveOrUpdateInventoryUnits(List<InventoryUnit> dataList, boolean updateLastUpdateTime, boolean updateSyncTime) throws Exception {

		if (dataList == null)
			return;

		Transaction tx = null;
		Session session = null;
		try {
			session = createNewSession();
			tx = session.beginTransaction();

			for (Iterator<InventoryUnit> iterator = dataList.iterator(); iterator.hasNext();) {
				InventoryUnit item = (InventoryUnit) iterator.next();
				InventoryUnit existingItem = get(item.getId());
				if (existingItem != null) {
					if (!BaseDataServiceDao.get().shouldSave(item.getLastUpdateTime(), existingItem.getLastUpdateTime())) {
						PosLog.info(getClass(), item.getName() + " already updated"); //$NON-NLS-1$
						continue;
					}
					long version = existingItem.getVersion();
					PropertyUtils.copyProperties(existingItem, item);
					existingItem.setVersion(version);
					existingItem.setUpdateLastUpdateTime(updateLastUpdateTime);
					existingItem.setUpdateSyncTime(updateSyncTime);
					update(existingItem, session);
				}
				else {
					item.setUpdateLastUpdateTime(updateLastUpdateTime);
					item.setUpdateSyncTime(updateSyncTime);
					save(item, session);
				}
			}
			tx.commit();
		} catch (Exception e) {
			tx.rollback();
			throw e;
		} finally {
			closeSession(session);
		}
	}

	public IUnit findById(String unitId) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.eq(InventoryUnit.PROP_ID, unitId));
			criteria.setMaxResults(1);
			addDeletedFilter(criteria);
			List list = criteria.list();
			if (list != null && !list.isEmpty()) {
				return (InventoryUnit) list.get(0);
			}
			criteria = session.createCriteria(PackagingUnit.class);
			criteria.add(Restrictions.eq(PackagingUnit.PROP_ID, unitId));
			criteria.setMaxResults(1);
			addDeletedFilter(criteria);
			list = criteria.list();
			if (list != null && !list.isEmpty()) {
				return (PackagingUnit) list.get(0);
			}
			return null;
		}
	}

	//	@Override
	//	public List<InventoryUnit> findAll() {
	//		try (Session session = createNewSession()) {
	//			Criteria criteria = session.createCriteria(getReferenceClass());
	//			criteria.addOrder(Order.asc(InventoryUnit.PROP_NAME));
	//			addDeletedFilter(criteria);
	//			return criteria.list();
	//		}
	//	}

}