/**
 * ************************************************************************
 * * The contents of this file are subject to the MRPL 1.2
 * * (the  "License"),  being   the  Mozilla   Public  License
 * * Version 1.1  with a permitted attribution clause; you may not  use this
 * * file except in compliance with the License. You  may  obtain  a copy of
 * * the License at http://www.floreantpos.org/license.html
 * * Software distributed under the License  is  distributed  on  an "AS IS"
 * * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * * License for the specific  language  governing  rights  and  limitations
 * * under the License.
 * * The Original Code is FLOREANT POS.
 * * The Initial Developer of the Original Code is OROCUBE LLC
 * * All portions are Copyright (C) 2015 OROCUBE LLC
 * * All Rights Reserved.
 * ************************************************************************
 */
package com.floreantpos.model.dao;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.apache.commons.beanutils.PropertyUtils;
import org.hibernate.Criteria;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.hibernate.criterion.Projections;
import org.hibernate.criterion.Restrictions;

import com.floreantpos.Messages;
import com.floreantpos.PosException;
import com.floreantpos.PosLog;
import com.floreantpos.model.InventoryUnit;
import com.floreantpos.model.MenuItem;
import com.floreantpos.model.PackagingUnit;
import com.orocube.rest.service.server.BaseDataServiceDao;

public class PackagingUnitDAO extends BasePackagingUnitDAO {

	private static final String EMPTY_STRING = ""; //$NON-NLS-1$
	private static final String EMPTY_DOT_STRING = "."; //$NON-NLS-1$
	private static final String EMPTY_NEWLINE_STRING = "\n"; //$NON-NLS-1$
	private static final String EMPTY_SPACE_STRING = " "; //$NON-NLS-1$

	/**
	 * Default constructor.  Can be used in place of getInstance()
	 */
	public PackagingUnitDAO() {
	}

	@Override
	protected Serializable save(Object obj, Session s) {
		updateTime(obj);
		return super.save(obj, s);
	}

	@Override
	protected void update(Object obj, Session s) {
		updateTime(obj);
		super.update(obj, s);
	}

	@Override
	protected void saveOrUpdate(Object obj, Session s) {
		updateTime(obj);
		super.saveOrUpdate(obj, s);
	}

	@Override
	protected void delete(Object obj, Session s) {
		if (obj instanceof PackagingUnit) {
			PackagingUnit packagingUnit = (PackagingUnit) obj;
			List<MenuItem> menuItems = MenuItemDAO.getInstance().getMenuItemsByPackagingUnit(packagingUnit, s);
			if (menuItems != null && !menuItems.isEmpty()) {
				String message = packagingUnit.getName() + EMPTY_SPACE_STRING + Messages.getString("PackagingUnitDAO.4"); //$NON-NLS-1$
				String details = this.constructExceptionDetailsByMenuItems(packagingUnit, menuItems);
				throw new PosException(message, details);
			}
			packagingUnit.setDeleted(Boolean.TRUE);
			super.update(packagingUnit, s);
		}
		else {
			super.delete(obj, s);
		}
	}

	private String constructExceptionDetailsByMenuItems(PackagingUnit packagingUnit, List<MenuItem> menuItems) {
		if (menuItems != null && !menuItems.isEmpty()) {
			StringBuilder builder = new StringBuilder(packagingUnit.getName() + EMPTY_SPACE_STRING + Messages.getString("PackagingUnitDAO.5")); //$NON-NLS-1$
			for (int i = 0; i < menuItems.size(); i++) {
				String message = (i + 1) + EMPTY_DOT_STRING + EMPTY_SPACE_STRING + menuItems.get(i).getName();
				builder.append(EMPTY_NEWLINE_STRING).append(message);
			}
			return builder.toString();
		}
		return EMPTY_STRING;
	}

	public boolean nameExists(PackagingUnit packagingUnit, String code) {
		try (Session session = this.createNewSession()) {
			Criteria criteria = session.createCriteria(InventoryUnit.class);
			this.addDeletedFilter(criteria);
			criteria.add(Restrictions.eq(InventoryUnit.PROP_CODE, code).ignoreCase());
			criteria.setProjection(Projections.rowCount());
			Number rowCount = (Number) criteria.uniqueResult();
			if (rowCount != null && ((Number) rowCount).intValue() > 0) {
				return true;
			}
			criteria = session.createCriteria(PackagingUnit.class);
			this.addDeletedFilter(criteria);
			if (packagingUnit != null && packagingUnit.getId() != null) {
				criteria.add(Restrictions.ne(InventoryUnit.PROP_ID, packagingUnit.getId()));
			}
			criteria.add(Restrictions.eq(InventoryUnit.PROP_CODE, code).ignoreCase());
			criteria.setProjection(Projections.rowCount());
			rowCount = (Number) criteria.uniqueResult();
			if (rowCount != null && ((Number) rowCount).intValue() > 0) {
				return true;
			}
			return false;
		}
	}

	@Override
	public List<PackagingUnit> findAll() {
		try (Session session = this.createNewSession()) {
			Criteria criteria = session.createCriteria(this.getReferenceClass());
			this.addDeletedFilter(criteria);
			return criteria.list();
		}
	}

	@SuppressWarnings("unchecked")
	public List<PackagingUnit> findAll(boolean recipeUnit) {
		try (Session session = this.createNewSession()) {
			Criteria criteria = session.createCriteria(this.getReferenceClass());
			this.addDeletedFilter(criteria);
			if (recipeUnit) {
				criteria.add(Restrictions.eq(PackagingUnit.PROP_RECIPE_UNIT, true));
			}
			else {
				criteria.add(Restrictions.or(Restrictions.isNull(PackagingUnit.PROP_RECIPE_UNIT), Restrictions.eq(PackagingUnit.PROP_RECIPE_UNIT, recipeUnit)));
			}
			return criteria.list();
		}
	}

	public void saveOrUpdatePackagingUnits(List<PackagingUnit> dataList, boolean updateLastUpdateTime, boolean updateSyncTime) throws Exception {

		if (dataList == null)
			return;

		Transaction tx = null;
		Session session = null;
		try {
			session = createNewSession();
			tx = session.beginTransaction();
			List<String> uniqueNames = new ArrayList<>();
			for (Iterator<PackagingUnit> iterator = dataList.iterator(); iterator.hasNext();) {
				PackagingUnit item = (PackagingUnit) iterator.next();
				if (uniqueNames.contains(item.getName())) {
					item.setName(item.getName() + "_"); //$NON-NLS-1$
				}
				uniqueNames.add(item.getName());
				PackagingUnit existingItem = get(item.getId());
				if (existingItem != null) {
					if (!BaseDataServiceDao.get().shouldSave(item.getLastUpdateTime(), existingItem.getLastUpdateTime())) {
						PosLog.info(getClass(), item.getName() + " already updated"); //$NON-NLS-1$
						continue;
					}
					long version = existingItem.getVersion();
					PropertyUtils.copyProperties(existingItem, item);
					existingItem.setVersion(version);
					existingItem.setUpdateLastUpdateTime(updateLastUpdateTime);
					existingItem.setUpdateSyncTime(updateSyncTime);
					update(existingItem, session);
				}
				else {
					item.setUpdateLastUpdateTime(updateLastUpdateTime);
					item.setUpdateSyncTime(updateSyncTime);
					save(item, session);
				}
			}
			tx.commit();
		} catch (Exception e) {
			tx.rollback();
			throw e;
		} finally {
			closeSession(session);
		}
	}

}