/**
 * ************************************************************************
 * * The contents of this file are subject to the MRPL 1.2
 * * (the  "License"),  being   the  Mozilla   Public  License
 * * Version 1.1  with a permitted attribution clause; you may not  use this
 * * file except in compliance with the License. You  may  obtain  a copy of
 * * the License at http://www.floreantpos.org/license.html
 * * Software distributed under the License  is  distributed  on  an "AS IS"
 * * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * * License for the specific  language  governing  rights  and  limitations
 * * under the License.
 * * The Original Code is FLOREANT POS.
 * * The Initial Developer of the Original Code is OROCUBE LLC
 * * All portions are Copyright (C) 2015 OROCUBE LLC
 * * All Rights Reserved.
 * ************************************************************************
 */
package com.floreantpos.services;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.lang.StringUtils;
import org.hibernate.Session;
import org.hibernate.Transaction;

import com.floreantpos.Messages;
import com.floreantpos.PosException;
import com.floreantpos.PosLog;
import com.floreantpos.config.CardConfig;
import com.floreantpos.constants.AppConstants;
import com.floreantpos.extension.PaymentGatewayPlugin;
import com.floreantpos.main.Application;
import com.floreantpos.model.BalanceType;
import com.floreantpos.model.CardReader;
import com.floreantpos.model.CreditCardTransaction;
import com.floreantpos.model.Currency;
import com.floreantpos.model.Customer;
import com.floreantpos.model.DebitCardTransaction;
import com.floreantpos.model.GiftCard;
import com.floreantpos.model.OrderType;
import com.floreantpos.model.Outlet;
import com.floreantpos.model.PaymentType;
import com.floreantpos.model.PosTransaction;
import com.floreantpos.model.RefundTransaction;
import com.floreantpos.model.Store;
import com.floreantpos.model.StoreSession;
import com.floreantpos.model.Terminal;
import com.floreantpos.model.Ticket;
import com.floreantpos.model.TicketDiscount;
import com.floreantpos.model.TicketItem;
import com.floreantpos.model.TransactionSubType;
import com.floreantpos.model.TransactionType;
import com.floreantpos.model.User;
import com.floreantpos.model.dao.BalanceUpdateTransactionDAO;
import com.floreantpos.model.dao.CustomerDAO;
import com.floreantpos.model.dao.GiftCardDAO;
import com.floreantpos.model.dao.StoreDAO;
import com.floreantpos.model.dao.TerminalDAO;
import com.floreantpos.model.dao.TicketDAO;
import com.floreantpos.model.util.DataProvider;
import com.floreantpos.report.ReceiptPrintService;
import com.floreantpos.ui.views.payment.CardProcessor;
import com.floreantpos.util.CurrencyUtil;
import com.floreantpos.util.NumberUtil;

public class PosTransactionService {
	private static PosTransactionService paymentService = new PosTransactionService();

	public void settleTicket(Ticket ticket, PosTransaction transaction, User currentUser) throws Exception {
		settleTicket(ticket, transaction, currentUser, null);
	}

	public void settleTicket(Ticket ticket, PosTransaction transaction, User currentUser, PostPaymentProcessor postPaymentService) throws Exception {
		settleTicket(ticket, transaction, currentUser, postPaymentService, false);
	}

	public void settleTicket(Ticket ticket, PosTransaction transaction, User currentUser, PostPaymentProcessor postPaymentService,
			boolean closeTicketIfApplicable) throws Exception {
		Terminal terminal = DataProvider.get().getCurrentTerminal();

		Session session = null;
		Transaction tx = null;

		try {
			session = TerminalDAO.getInstance().createNewSession();
			Date currentDate = StoreDAO.getServerTimestamp();

			tx = session.beginTransaction();

			//			adjustGiftCardBalances(ticket, transaction, session);

			//			CashDrawer cashDrawer = currentUser.getActiveDrawerPullReport();

			transaction.setTransactionType(TransactionType.CREDIT.name());
			transaction.setPaymentType(transaction.getPaymentType());
			transaction.setTerminal(terminal);
			User owner = ticket.getOwner();
			if (currentUser != null) {
				transaction.setUser(currentUser);
			}
			else {
				transaction.setUser(owner);
			}
			transaction.setServer(owner);
			//			transaction.setCashDrawer(cashDrawer);
			transaction.setTransactionTime(currentDate);
			transaction.setLastUpdateTime(currentDate);
			transaction.setCustomerId(ticket.getCustomerId());
			transaction.setCustomerName(ticket.getCustomerName());
			transaction.setOutletId(ticket.getOutletId());
			if (StringUtils.isBlank(transaction.getStoreSessionId())) {
				StoreSession storeSession = DataProvider.get().getStoreSession();
				if (storeSession != null) {
					transaction.setStoreSessionId(storeSession.getId());
				}
			}
			transaction.setTicket(ticket);
			ticket.setCashier(currentUser);

			if (transaction.getAmount() > 0) {
				ticket.addTotransactions(transaction);
			}

			ticket.setVoided(false);
			ticket.setTerminal(terminal);
			ticket.setPaidAmount(NumberUtil.round(ticket.getPaidAmount() + transaction.getAmount()));
			ticket.setShouldUpdateStock(true);
			ticket.calculatePrice();

			calculateToleranceAmount(ticket, transaction);

			if (NumberUtil.isZero(ticket.getDueAmount())) {
				ticket.setPaid(true);
				if (closeTicketIfApplicable) {
					closeTicketIfApplicable(ticket, currentDate);
				}
			}
			else {
				ticket.setPaid(false);
				//				ticket.setClosed(false);
			}
			if (ticket.getOrderType() != null && ticket.getOrderType().isBarTab()) {
				ticket.removeProperty(Ticket.PROPERTY_PAYMENT_METHOD);
				ticket.removeProperty(Ticket.PROPERTY_CARD_NAME);
				ticket.removeProperty(Ticket.PROPERTY_CARD_TRANSACTION_ID);
				ticket.removeProperty(Ticket.PROPERTY_CARD_TRACKS);
				ticket.removeProperty(Ticket.PROPERTY_CARD_READER);
				ticket.removeProperty(Ticket.PROPERTY_ADVANCE_PAYMENT);
				ticket.removeProperty(Ticket.PROPERTY_CARD_NUMBER);
				ticket.removeProperty(Ticket.PROPERTY_CARD_EXP_YEAR);
				ticket.removeProperty(Ticket.PROPERTY_CARD_EXP_MONTH);
				ticket.removeProperty(Ticket.PROPERTY_CARD_AUTH_CODE);
			}

			this.updateCustomerLoyaltyPoint(ticket, transaction, session);
			adjustGratuityIfNeeded(ticket, transaction);
			markItemsAsPaid(ticket);

			TicketDAO.getInstance().saveOrUpdate(ticket, session);

			if (postPaymentService != null) {
				postPaymentService.paymentDone(transaction, session);
			}
			tx.commit();

		} catch (Exception e) {
			try {
				tx.rollback();
			} catch (Exception x) {
				PosLog.error(PosTransactionService.class, x);
			}
			throw e;
		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

	public void voidTicket(Ticket ticket, User currentUser) throws Exception {
		Terminal terminal = Application.getInstance().getTerminal();
		ticket.setVoidedBy(currentUser);
		ticket.setTerminal(terminal);
		ticket.calculatePrice();
		TicketDAO.getInstance().voidTicket(ticket);
		try {
			ReceiptPrintService.printVoidTicket(ticket);
		} catch (Exception e) {
		}
	}

	public PosTransaction refundTicket(Ticket ticket, PosTransaction selectedTransaction, final double refundTenderedAmount, User currentUser,
			PaymentType refundPaymentType, Map<String, String> paymentProperties) throws Exception {
		if (paymentProperties == null) {
			paymentProperties = new HashMap<>();
		}
		RefundTransaction refundTransaction = createRefundTransaction(ticket, selectedTransaction, refundTenderedAmount, refundPaymentType, paymentProperties,
				currentUser);
		refundTransaction.setUser(currentUser);
		refundTransaction.setServer(ticket.getOwner());

		if (NumberUtil.isZero(ticket.getDueAmount())) {
			ticket.setClosed(true);
			ticket.setClosingDate(StoreDAO.getServerTimestamp());
		}
		Transaction hbnTransaction = null;
		try (Session session = TicketDAO.getInstance().createNewSession()) {
			hbnTransaction = session.beginTransaction();
			ticket.setShouldUpdateStock(true);
			markItemsAsPaid(ticket);
			adjustGiftCardBalances(ticket, refundTransaction, session);

			BalanceType balanceType = null;
			String accountNumber = null;
			Double balanceBefore = null;
			TransactionSubType description = null;
			if (refundPaymentType == PaymentType.GIFT_CERTIFICATE) {
				//				String giftCardNo = paymentProperties.get(PosTransaction.PROP_GIFT_CERT_NUMBER);
				//				if (StringUtils.isBlank(giftCardNo)) {
				//					throw new PosException(Messages.getString("GiftCardCannotBeEmpty")); //$NON-NLS-1$
				//				}
				//				GiftCardPaymentPlugin paymentGateway = GiftCardConfig.getPaymentGateway();
				//				GiftCardProcessor giftCardProcessor = paymentGateway.getProcessor();
				//				GiftCard giftCard = giftCardProcessor.getCard(giftCardNo);
				//				if (giftCard == null) {
				//					throw new PosException(Messages.getString("PosTransactionService.7")); //$NON-NLS-1$
				//				}
				//				balanceBefore = giftCard.getBalance();
				//				description = TransactionSubType.REFUNDED;
				//				giftCardProcessor.refund(giftCard.getCardNumber(), refundTenderedAmount, session);
				//				balanceType = BalanceType.GIFT_CARD;
				//				accountNumber = giftCardNo;
			}
			else if (refundPaymentType == PaymentType.MEMBER_ACCOUNT) {
				Customer customer = ticket.getCustomer();
				if (customer == null) {
					throw new PosException(Messages.getString("PosTransactionService.1")); //$NON-NLS-1$
				}

				balanceBefore = customer.getBalance();
				refundTransaction.setCustomerBalanceBefore(balanceBefore);
				customer.setBalance(balanceBefore + refundTenderedAmount);
				refundTransaction.setCustomerBalanceAfter(customer.getBalance());
				refundTransaction.setRefunded(true);

				CustomerDAO.getInstance().saveOrUpdate(ticket.getCustomer(), session);
				balanceType = BalanceType.CUSTOMER;
				accountNumber = customer.getId();
			}
			else if (refundPaymentType == PaymentType.CREDIT_CARD || refundPaymentType == PaymentType.DEBIT_CARD) {
				if (StringUtils.isNotBlank(selectedTransaction.getCardMerchantGateway())
						&& CardReader.fromString(selectedTransaction.getCardReader()) != CardReader.EXTERNAL_TERMINAL) {
					final double transactionAmount = selectedTransaction.getAmount();
					PaymentGatewayPlugin paymentGateway = CardConfig.getPaymentGatewayByName(selectedTransaction.getCardMerchantGateway());
					//TODO: only colud pos
					paymentGateway.setForOnlineOrder(false);
					CardProcessor cardProcessor = paymentGateway.getProcessor();
					try {
						selectedTransaction.setAmount(refundTenderedAmount);
						cardProcessor.refundTransaction(selectedTransaction, refundTenderedAmount);
					} catch (PosException e) {
						removeRefundedAmount(selectedTransaction);
						throw e;
					} catch (Exception e) {
						removeRefundedAmount(selectedTransaction);
						throw new PosException(e.getMessage(), e);
					} finally {
						selectedTransaction.setAmount(transactionAmount);
					}
				}
			}
			if (selectedTransaction != null) {
				updateRefundTransactionProperties(selectedTransaction, refundTenderedAmount, refundTransaction);
			}

			ticket.addTotransactions(refundTransaction);
			ticket.setRefunded(true);
			ticket.setCashier(currentUser);
			ticket.calculateRefundAmount();
			ticket.setPaidAmount(NumberUtil.round(ticket.getPaidAmount() - refundTransaction.getAmount()));
			ticket.calculatePrice();
			ticket.closeIfApplicable();

			TicketDAO.getInstance().saveOrUpdate(ticket, session);
			if (refundTenderedAmount > 0) {
				this.deductCustomerLoyaltyPoint(ticket.getCustomer(), refundTenderedAmount, refundTransaction, session);
			}
			Terminal terminal = DataProvider.get().getCurrentTerminal();
			if (terminal != null && terminal.isEnableMultiCurrency()) {
				//adjustMulticurrencyBalance(session, terminal, currentUser.getActiveDrawerPullReport(), null, refundTransaction);
			}
			if (balanceType != null) {
				BalanceUpdateTransactionDAO.getInstance().saveBalanceUpdateTrans(balanceType, ticket, refundTransaction, TransactionType.CREDIT, accountNumber,
						refundTenderedAmount, balanceBefore, description, session);
			}
			hbnTransaction.commit();
		}
		return refundTransaction;
	}

	private void removeRefundedAmount(PosTransaction selectedTransaction) {
		selectedTransaction.removeProperty(PosTransaction.JSON_PROP_REFUNDED_AMOUNT);
		selectedTransaction.removeProperty("REFUNDED_TIPS_AMOUNT");
	}

	//	public static void adjustMulticurrencyBalance(Session session, Terminal terminal, CashDrawer cashDrawer, List<PaymentByCurrency> paymentByCurrencyList,
	//			PosTransaction transaction) {
	//		if (cashDrawer == null) {
	//			return;
	//		}
	//		if (transaction.getPaymentType() != PaymentType.CASH) {
	//			return;
	//		}
	//		if (terminal.isEnableMultiCurrency()) {
	//			CashDrawerDAO.getInstance().refresh(cashDrawer, session);
	//
	//			//if multi currency is enabled, update main currency balance of cash breakdown
	//			Ticket ticket = transaction.getTicket();
	//			if (paymentByCurrencyList == null) {
	//				if (transaction.isVoided()) {
	//					paymentByCurrencyList = new ArrayList<>();
	//
	//					String paymentCurrencies = transaction.getProperty(PosTransaction.PAYMENT_CURRENCIES);
	//					if (StringUtils.isNotBlank(paymentCurrencies)) {
	//						List<PaymentByCurrency> createPaymentByCurrencies = CurrencyUtil.createPaymentByCurrencies(paymentCurrencies);
	//						for (PaymentByCurrency paymentByCurrency : createPaymentByCurrencies) {
	//							if (paymentByCurrency.tenderedAmount > 0) {
	//								double creditAmount = paymentByCurrency.tenderedAmount - paymentByCurrency.cashBackAmount;
	//								paymentByCurrency.tenderedAmount = -1 * creditAmount;
	//								paymentByCurrency.cashBackAmount = 0;
	//								paymentByCurrencyList.add(paymentByCurrency);
	//							}
	//						}
	//					}
	//					else {
	//						List<Currency> currencies = CurrencyUtil.getAllCurrency();
	//						if (currencies != null && currencies.size() > 0) {
	//							for (Currency currency : currencies) {
	//								PaymentByCurrency paymentByCurrency = new PaymentByCurrency();
	//								paymentByCurrency.currency = currency;
	//								String cashBackKey = currency.getId() + PosTransaction.PROP_CASH_BACK_POSTFIX;
	//								String tenderedKey = currency.getId() + PosTransaction.PROP_TENDERED_POSTFIX;
	//								double tenderedAmount = POSUtil.parseDouble(transaction.getProperty(tenderedKey));
	//								if (tenderedAmount > 0) {
	//									double cashBackAmount = POSUtil.parseDouble(transaction.getProperty(cashBackKey));
	//									double creditAmount = tenderedAmount - cashBackAmount;
	//									paymentByCurrency.tenderedAmount = -1 * creditAmount;
	//									paymentByCurrencyList.add(paymentByCurrency);
	//								}
	//							}
	//						}
	//					}
	//				}
	//				else {
	//					PaymentByCurrency paymentByCurrency = new PaymentByCurrency();
	//					paymentByCurrency.currency = CurrencyUtil.getMainCurrency();
	//					paymentByCurrency.tenderedAmount = transaction.getTenderAmount();
	//					if (transaction instanceof RefundTransaction || transaction instanceof PayOutTransaction || transaction instanceof CashDropTransaction) {
	//						paymentByCurrency.tenderedAmount = -1 * transaction.getAmount();
	//					}
	//					else {
	//						Double dueAmount = ticket.getDueAmount();
	//						paymentByCurrency.cashBackAmount = dueAmount > paymentByCurrency.tenderedAmount ? 0 : paymentByCurrency.tenderedAmount - dueAmount;
	//						if (Math.abs(paymentByCurrency.cashBackAmount) < ticket.getToleranceFactor()) {
	//							paymentByCurrency.cashBackAmount = 0d;
	//						}
	//					}
	//					paymentByCurrencyList = Arrays.asList(paymentByCurrency);
	//				}
	//			}
	//
	//			JSONArray currencyArray = new JSONArray();
	//			paymentByCurrencyList.forEach(payByCurrency -> {
	//				Currency currency = payByCurrency.currency;
	//				CashBreakdown breakdown = cashDrawer.getCurrencyBalance(currency);
	//				double balance = payByCurrency.tenderedAmount - payByCurrency.cashBackAmount;
	//				if (breakdown != null) {
	//					breakdown.setBalance(breakdown.getBalance() + balance);
	//				}
	//				if (!transaction.isVoided()) {
	//					double paidAmount = payByCurrency.tenderedAmount - payByCurrency.cashBackAmount;
	//					JSONObject currencyObject = new JSONObject();
	//					currencyObject.put(PosTransaction.CURRENCY_ID, currency.getId()); //$NON-NLS-1$
	//					currencyObject.put(PosTransaction.CURRENCY_NAME, currency.getName()); //$NON-NLS-1$
	//					currencyObject.put(PosTransaction.CURRENCY_CODE, currency.getCode()); //$NON-NLS-1$
	//					currencyObject.put(PosTransaction.CURRENCY_SYMBOL, currency.getSymbol()); //$NON-NLS-1$
	//					currencyObject.put(PosTransaction.CURRENCY_RATE, currency.getExchangeRate()); //$NON-NLS-1$
	//					currencyObject.put(PosTransaction.CURRENCY_TENDERED, NumberUtil.format(payByCurrency.tenderedAmount)); //$NON-NLS-1$
	//					currencyObject.put(PosTransaction.CURRENCY_CHANGE, NumberUtil.format(payByCurrency.cashBackAmount)); //$NON-NLS-1$
	//					currencyObject.put(PosTransaction.CURRENCY_PAID_AMOUNT, NumberUtil.format(paidAmount)); //$NON-NLS-1$
	//					currencyArray.put(currencyObject);
	//
	//					//					transaction.addProperty(paidAmountKey, String.valueOf(paidAmount));
	//				}
	//			});
	//			transaction.addProperty(PosTransaction.PAYMENT_CURRENCIES, currencyArray.toString()); //$NON-NLS-1$
	//			session.saveOrUpdate(cashDrawer);
	//			if (ticket != null) {
	//				ticket.addProperty("enableMultiCurrency", String.valueOf(Boolean.TRUE)); //$NON-NLS-1$
	//			}
	//		}
	//	}

	private void updateRefundTransactionProperties(PosTransaction selectedTransaction, final double refundTenderedAmount, RefundTransaction refundTransaction) {
		String refundedAmountText = selectedTransaction.getProperty(PosTransaction.JSON_PROP_REFUNDED_AMOUNT);
		double previousRefundedAmount = 0;
		if (StringUtils.isNotEmpty(refundedAmountText)) {
			try {
				previousRefundedAmount = NumberUtil.parseDouble(refundedAmountText);
			} catch (Exception e2) {
			}
		}

		refundTransaction.addProperty(PosTransaction.JSON_PROP_REFUNDED_AMOUNT, String.valueOf(refundTenderedAmount + previousRefundedAmount));

		if (selectedTransaction.getTipsAmount() > 0) {
			String refundedTipsAmountText = selectedTransaction.getProperty("REFUNDED_TIPS_AMOUNT"); //$NON-NLS-1$
			double previousRefundedTipsAmount = 0;
			if (StringUtils.isNotEmpty(refundedAmountText)) {
				try {
					previousRefundedTipsAmount = NumberUtil.parseDouble(refundedTipsAmountText);
				} catch (Exception e2) {
				}
			}
			double transactionAmountWithoutTips = selectedTransaction.getAmount() - selectedTransaction.getTipsAmount();
			double totalRefundedAmount = previousRefundedAmount + refundTenderedAmount;
			double tipsRefundAmount = totalRefundedAmount - transactionAmountWithoutTips - previousRefundedTipsAmount;
			if (tipsRefundAmount > 0) {
				selectedTransaction.setTipsAmount(tipsRefundAmount);
				refundTransaction.addProperty("REFUNDED_TIPS_AMOUNT", String.valueOf(tipsRefundAmount + previousRefundedTipsAmount)); //$NON-NLS-1$
			}
		}
	}

	public void deductCustomerLoyaltyPoint(Customer customer, double refundedAmount, PosTransaction refundTransaction, Session session) {
		if (customer == null || refundedAmount <= 0) {
			return;
		}
		Outlet outlet = DataProvider.get().getOutlet();
		Boolean loyaltyEnabled = Boolean.valueOf(outlet.getProperty(AppConstants.LOYALTY_ENABLED));
		Boolean loyaltyDeduct = Boolean.valueOf(outlet.getProperty(AppConstants.LOYALTY_DEDUCT_POINT));
		if (!(loyaltyEnabled && loyaltyDeduct)) {
			return;
		}
		Integer beforeLoyaltyPoint = customer.getLoyaltyPoint();
		Integer loyaltyPoint = customer.getLoyaltyPoint();
		Integer loyaltyPointToDeduct = 0;
		try {
			int loyaltyAmount = Integer.parseInt(outlet.getProperty(AppConstants.LOYALTY_POINT_FOR_PURCHASES));
			if (loyaltyAmount > 0) {
				loyaltyPointToDeduct = (int) Math.ceil((refundedAmount / loyaltyAmount));
			}
		} catch (Exception e) {
		}
		loyaltyPoint = loyaltyPoint - loyaltyPointToDeduct;
		refundTransaction.addExtraProperty(AppConstants.LOYALTY_DEDUCTED, String.valueOf(loyaltyPointToDeduct));
		customer.setLoyaltyPoint(loyaltyPoint > 0 ? loyaltyPoint : 0);
		CustomerDAO.getInstance().update(customer, session);
		BalanceUpdateTransactionDAO.getInstance().saveBalanceUpdateTrans(BalanceType.LOYALTY, refundTransaction.getTicket(), refundTransaction,
				TransactionType.DEBIT, customer.getId(), (double) loyaltyPointToDeduct, (double) beforeLoyaltyPoint, null, session);
	}

	//adjust gift card add balance
	private void adjustGiftCardBalances(Ticket ticket, PosTransaction transaction, Session session) {
		boolean hasGiftCard = Boolean.FALSE;

		List<TicketItem> ticketItems = ticket.getTicketItems();
		for (TicketItem ticketItem : ticketItems) {
			if (ticketItem.isService() && !ticketItem.getBooleanProperty("processed", false)) { //$NON-NLS-1$
				//				saveTicketIfNotSaved(ticket, session);
				//
				//				ServiceProcessor service = new ServiceProcessor();
				//				service.prcocess(ticketItem, ticket, session);
				//				ticketItem.addProperty("processed", String.valueOf(Boolean.TRUE)); //$NON-NLS-1$
			}
			else if (ticketItem.isGiftCard() && !(transaction instanceof RefundTransaction)) {
				String giftCardNo = ticketItem.getGiftCardNo();
				GiftCard giftCard = GiftCardDAO.getInstance().findByCardNumber(session, giftCardNo);
				if (giftCard == null) {
					if (transaction instanceof CreditCardTransaction || transaction instanceof DebitCardTransaction) {
						PosLog.error(PosTransactionService.class, String.format(Messages.getString("PosTransactionService.8"), giftCardNo, //$NON-NLS-1$
								transaction.getCardNumber()));
						return;
					}
					else {
						throw new PosException(String.format(Messages.getString("GiftCardNoNotFound"), giftCardNo)); //$NON-NLS-1$
					}
				}
				double balanceToAdd = ticketItem.getUnitPrice();

				double giftCardPaidAmount = ticketItem.getGiftCardPaidAmount();
				if (balanceToAdd == giftCardPaidAmount) {
					return;
				}
				saveTicketIfNotSaved(ticket, session);

				hasGiftCard = Boolean.TRUE;
				ticketItem.setGiftCardPaidAmount(balanceToAdd);
				transaction.setGiftCertNumber(giftCardNo);
				transaction.setGiftCertFaceValue(balanceToAdd);

				transaction.addGiftCardBalanceAddInfo(ticketItem.getId(), giftCardNo, balanceToAdd);

				Double balanceBefore = giftCard.getBalance();
				giftCard.setBalance(balanceBefore + balanceToAdd);
				GiftCardDAO.getInstance().saveOrUpdate(giftCard, session);
				BalanceUpdateTransactionDAO.getInstance().saveBalanceUpdateTrans(BalanceType.GIFT_CARD, ticket, transaction, TransactionType.CREDIT, giftCardNo,
						balanceToAdd, balanceBefore, TransactionSubType.BALANCE_ADDED, session);
			}
		}
		if (hasGiftCard) {
			TicketDAO.getInstance().saveOrUpdate(ticket, session);
		}
	}

	/**
	 * Save ticket to create ticket id.
	 * 
	 * @param ticket
	 * @param session
	 */
	private void saveTicketIfNotSaved(Ticket ticket, Session session) {
		if (StringUtils.isBlank(ticket.getId())) {
			TicketDAO.getInstance().saveOrUpdate(ticket, session);
		}
	}

	private void updateCustomerLoyaltyPoint(Ticket ticket, PosTransaction transaction, Session session) {
		Outlet outlet = ticket.getOutlet();
		Boolean loyaltyEnabled = Boolean.valueOf(outlet.getProperty(AppConstants.LOYALTY_ENABLED));
		if (!loyaltyEnabled) {
			return;
		}
		Customer customer = ticket.getCustomer();
		if (customer == null) {
			return;
		}

		customer = CustomerDAO.getInstance().get(customer.getId(), session);

		this.chargeCustomerLoyaltyPoint(ticket, customer, transaction, session);
		this.adjustCustomerLoyaltyPoint(ticket, transaction, outlet, customer, session);

		ticket.updateCustomer(customer);
	}

	private void chargeCustomerLoyaltyPoint(Ticket ticket, Customer customer, PosTransaction transaction, Session session) {
		if (ticket == null) {
			throw new PosException(Messages.getString("PosTransactionService.0")); //$NON-NLS-1$
		}
		if (ticket.getDiscounts() == null) {
			return;
		}
		List<TicketDiscount> applicableDiscounts = new ArrayList<>();
		ticket.getDiscounts().forEach(discount -> {
			if (!discount.isLoyaltyCharged()) {
				applicableDiscounts.add(discount);
			}
		});
		if (applicableDiscounts.isEmpty()) {
			return;
		}
		if (customer == null) {
			throw new PosException(Messages.getString("PosTransactionService.3")); //$NON-NLS-1$
		}

		int loyaltyChargedAmt = 0;
		int customerLogalty = customer.getLoyaltyPoint();
		for (TicketDiscount discount : applicableDiscounts) {
			int discountLoyalty = discount.getLoyaltyPoint();
			customer.setLoyaltyPoint(customerLogalty - discountLoyalty);
			discount.setLoyaltyCharged(Boolean.TRUE);
			loyaltyChargedAmt += discountLoyalty;
		}
		transaction.addExtraProperty(AppConstants.LOYALTY_CHARGED_AMOUNT, String.valueOf(loyaltyChargedAmt));
		ticket.buildDiscounts();
		if (loyaltyChargedAmt > 0) {
			BalanceUpdateTransactionDAO.getInstance().saveBalanceUpdateTrans(BalanceType.LOYALTY, ticket, transaction, TransactionType.DEBIT, customer.getId(),
					(double) loyaltyChargedAmt, (double) customerLogalty, null, session);
		}
	}

	private void adjustCustomerLoyaltyPoint(Ticket ticket, PosTransaction transaction, Outlet outlet, Customer customer, Session session) {
		if (ticket == null) {
			throw new PosException(Messages.getString("PosTransactionService.0")); //$NON-NLS-1$
		}
		if (transaction == null) {
			throw new PosException(Messages.getString("PosTransactionService.5")); //$NON-NLS-1$
		}
		if (outlet == null) {
			throw new PosException(Messages.getString("PosTransactionService.66")); //$NON-NLS-1$
		}
		if (customer == null) {
			return;
		}
		Integer beforeLoyaltyPoint = customer.getLoyaltyPoint();
		Integer loyaltyPoint = customer.getLoyaltyPoint();
		int ticketLoyaltyPoint = 0;
		try {
			String ticketLoyaltyProperty = ticket.getProperty(Ticket.PROPERTY_LOYALTY_ADDED);
			Boolean addedTicketLoyalty = ticketLoyaltyProperty != null && Boolean.valueOf(ticketLoyaltyProperty);
			if (!addedTicketLoyalty) {
				int pointForVisit = Integer.parseInt(outlet.getProperty(AppConstants.LOYALTY_POINT_FOR_VISIT));
				if (pointForVisit > 0) {
					ticketLoyaltyPoint = pointForVisit;
				}
				ticket.addProperty(Ticket.PROPERTY_LOYALTY_ADDED, String.valueOf(Boolean.TRUE));
			}
		} catch (Exception e) {
		}
		loyaltyPoint += ticketLoyaltyPoint;
		int purchaseLoyaltyPoint = 0;
		try {
			int loyaltyAmount = Integer.parseInt(outlet.getProperty(AppConstants.LOYALTY_POINT_FOR_PURCHASES));
			if (loyaltyAmount > 0) {
				purchaseLoyaltyPoint = (int) (transaction.getAmount() / loyaltyAmount);
			}
		} catch (Exception e) {
		}
		loyaltyPoint += purchaseLoyaltyPoint;
		transaction.addExtraProperty(AppConstants.LOYALTY_POINT_EARNED, String.valueOf(purchaseLoyaltyPoint));
		customer.setLoyaltyPoint(loyaltyPoint);
		CustomerDAO.getInstance().update(customer, session);
		BalanceUpdateTransactionDAO.getInstance().saveBalanceUpdateTrans(BalanceType.LOYALTY, ticket, transaction, TransactionType.CREDIT, customer.getId(),
				(double) (ticketLoyaltyPoint + purchaseLoyaltyPoint), (double) beforeLoyaltyPoint, null, session);
	}

	/*	
	public void deductCustomerLoyaltyPoint(Customer customer, double refundedAmount, PosTransaction refundTransaction, Session session) {
		if (customer == null || refundedAmount <= 0) {
			return;
		}
		Store store = DataProvider.get().getStore();
		Boolean loyaltyEnabled = Boolean.valueOf(store.getProperty(AppConstants.LOYALTY_ENABLED));
		Boolean loyaltyDeduct = Boolean.valueOf(store.getProperty(AppConstants.LOYALTY_DEDUCT_POINT));
		if (!(loyaltyEnabled && loyaltyDeduct)) {
			return;
		}
		Integer beforeLoyaltyPoint = customer.getLoyaltyPoint();
		Integer loyaltyPoint = customer.getLoyaltyPoint();
		Integer loyaltyPointToDeduct = 0;
		try {
			int loyaltyAmount = Integer.parseInt(store.getProperty(AppConstants.LOYALTY_POINT_FOR_PURCHASES));
			if (loyaltyAmount > 0) {
				loyaltyPointToDeduct = (int) Math.ceil((refundedAmount / loyaltyAmount));
			}
		} catch (Exception e) {
		}
		loyaltyPoint = loyaltyPoint - loyaltyPointToDeduct;
		refundTransaction.addExtraProperty(AppConstants.LOYALTY_DEDUCTED, String.valueOf(loyaltyPointToDeduct));
		customer.setLoyaltyPoint(loyaltyPoint > 0 ? loyaltyPoint : 0);
		CustomerDAO.getInstance().update(customer, session);
		BalanceUpdateTransactionDAO.getInstance().saveBalanceUpdateTrans(BalanceType.LOYALTY, refundTransaction.getTicket(), refundTransaction,
				TransactionType.DEBIT, customer.getId(), (double) loyaltyPointToDeduct, (double) beforeLoyaltyPoint, null, session);
	}*/

	private void calculateToleranceAmount(Ticket ticket, PosTransaction transaction) {
		double toleranceAmountFactor = 0.0;
		Currency mainCurrency = CurrencyUtil.getMainCurrency();
		if (mainCurrency != null) {
			toleranceAmountFactor = mainCurrency.getTolerance();
		}

		//if (ticket.getRoundedDueAmount() == 0.0 || ticket.getRoundedDueAmount() <= toleranceAmountFactor) {
		transaction.setToleranceAmount(ticket.getToleranceAmount());
		//}
		double changeAmount = NumberUtil.round(transaction.getTenderAmount() - transaction.getAmount());
		if (changeAmount == 0) {
			return;
		}

		double roundedChangeAmount;
		Store store = DataProvider.get().getStore();
		if (store.isAllowPenyRounding()) {
			roundedChangeAmount = Math.round(changeAmount * 100.0 / 5.0) * 5.0 / 100.0;
		}
		else {
			roundedChangeAmount = changeAmount;
		}

		double toleranceAmount = changeAmount - roundedChangeAmount;

		if (Math.abs(roundedChangeAmount) <= toleranceAmountFactor) {
			transaction.setToleranceAmount(changeAmount);
			transaction.setChangeAmount(0.0);
		}
		else {
			transaction.setToleranceAmount(toleranceAmount);
			transaction.setChangeAmount(roundedChangeAmount);
		}
		transaction.setAmount(transaction.getAmount() + transaction.getToleranceAmount());
	}

	public static void closeTicketIfApplicable(Ticket ticket, Date currentDate) {
		OrderType ticketType = ticket.getOrderType();

		if (ticketType != null && (ticketType.isCloseOnPaid() || ticketType.isBarTab())) {//fix
			ticket.setClosed(true);
			ticket.setClosingDate(currentDate);
		}
	}

	private void adjustGratuityIfNeeded(Ticket ticket, PosTransaction transaction) {
		double gratuityAmount = ticket.getGratuityAmount();
		if (gratuityAmount <= 0)
			return;

		double gratuityPaidAmount = 0;

		Set<PosTransaction> transactions = ticket.getTransactions();
		if (transactions != null && transactions.size() > 0) {
			for (PosTransaction posTransaction : transactions) {
				if (posTransaction instanceof RefundTransaction || posTransaction.isVoided())
					continue;
				gratuityPaidAmount += posTransaction.getTipsAmount();
			}
		}
		double gratuityDueAmount = gratuityAmount - gratuityPaidAmount;
		double payableTipsAmount = (gratuityDueAmount + ticket.getPaidAmount()) - ticket.getTotalAmountWithTips();

		if (gratuityDueAmount > 0) {
			if (ticket.getDueAmount() == 0) {
				transaction.setTipsAmount(gratuityDueAmount);
			}
			else if (payableTipsAmount > 0) {
				Double transactionAmount = transaction.getAmount();
				if (payableTipsAmount > transactionAmount) {
					transaction.setTipsAmount(transactionAmount);
				}
				else {
					transaction.setTipsAmount((payableTipsAmount));
				}
			}
		}
	}

	public RefundTransaction createRefundTransaction(Ticket ticket, PosTransaction transaction, double refundAmount, PaymentType refundPaymentType,
			Map<String, String> paymentProperties, User currentUser) {

		if (transaction != null) {
			double refundableAmount = transaction.getRefundableAmount();
			if (refundAmount > refundableAmount) {
				throw new PosException(Messages.getString("RefundDialog.36") + " " + refundableAmount); //$NON-NLS-1$ //$NON-NLS-2$
			}
		}
		RefundTransaction posTransaction = new RefundTransaction();
		if (paymentProperties != null) {
			posTransaction.setCustomPaymentFieldName(paymentProperties.get(PosTransaction.PROP_CUSTOM_PAYMENT_FIELD_NAME));
			posTransaction.setCustomPaymentName(paymentProperties.get(PosTransaction.PROP_CUSTOM_PAYMENT_NAME));
			posTransaction.setCustomPaymentRef(paymentProperties.get(PosTransaction.PROP_CUSTOM_PAYMENT_REF));
			posTransaction.setGiftCertNumber(paymentProperties.get(PosTransaction.PROP_GIFT_CERT_NUMBER));
		}

		posTransaction.setPaymentType(refundPaymentType);
		posTransaction.setTenderAmount(refundAmount);
		posTransaction.setAmount(refundAmount);
		posTransaction.setTicket(ticket);
		posTransaction.setCustomerId(ticket.getCustomerId());
		posTransaction.setTerminal(DataProvider.get().getCurrentTerminal());
		posTransaction.setTransactionType(TransactionType.DEBIT.name());
		posTransaction.setCashDrawer(currentUser.getActiveDrawerPullReport());
		posTransaction.setOutletId(DataProvider.get().getStore().getDefaultOutletId());
		posTransaction.setTransactionTime(StoreDAO.getServerTimestamp());
		if (transaction == null) {
			return posTransaction;
		}
		posTransaction.setCardExpMonth(transaction.getCardExpMonth());
		posTransaction.setCardHolderName(transaction.getCardHolderName());
		posTransaction.setCardAuthCode(transaction.getCardAuthCode());
		posTransaction.setCardMerchantGateway(transaction.getCardMerchantGateway());
		posTransaction.setCardNumber(transaction.getCardNumber());
		posTransaction.setCardTrack(transaction.getCardTrack());
		posTransaction.setCardTransactionId(transaction.getCardTransactionId());
		posTransaction.setCardReader(transaction.getCardReader());
		posTransaction.setCardType(transaction.getCardType());
		posTransaction.addProperty("REFUNDED_TRANSACTION_ID", transaction.getId()); //$NON-NLS-1$

		String refundedAmountText = transaction.getProperty(PosTransaction.JSON_PROP_REFUNDED_AMOUNT);
		double previousRefundedAmount = 0;
		if (StringUtils.isNotEmpty(refundedAmountText)) {
			try {
				previousRefundedAmount = NumberUtil.parseDouble(refundedAmountText);
			} catch (Exception e2) {
			}
		}

		transaction.addProperty(PosTransaction.JSON_PROP_REFUNDED_AMOUNT, String.valueOf(refundAmount + previousRefundedAmount));

		if (transaction.getTipsAmount() > 0) {
			String refundedTipsAmountText = transaction.getProperty("REFUNDED_TIPS_AMOUNT"); //$NON-NLS-1$
			double previousRefundedTipsAmount = 0;
			if (StringUtils.isNotEmpty(refundedAmountText)) {
				try {
					previousRefundedTipsAmount = NumberUtil.parseDouble(refundedTipsAmountText);
				} catch (Exception e2) {
				}
			}
			double transactionAmountWithoutTips = transaction.getAmount() - transaction.getTipsAmount();
			double totalRefundedAmount = previousRefundedAmount + refundAmount;
			double tipsRefundAmount = totalRefundedAmount - transactionAmountWithoutTips - previousRefundedTipsAmount;
			if (tipsRefundAmount > 0) {
				posTransaction.setTipsAmount(tipsRefundAmount);
				transaction.addProperty("REFUNDED_TIPS_AMOUNT", String.valueOf(tipsRefundAmount + previousRefundedTipsAmount)); //$NON-NLS-1$
			}
		}
		return posTransaction;
	}

	private void markItemsAsPaid(Ticket ticket) {
		List<TicketItem> ticketItems = ticket.getTicketItems();
		for (TicketItem ticketItem : ticketItems) {
			ticketItem.setPaid(true);
		}
	}

	//adjust gift card add balance
	/*private void adjustGiftCardBalances(Ticket ticket, PosTransaction transaction, Session session) {
		boolean hasGiftCard = Boolean.FALSE;
	
		List<TicketItem> ticketItems = ticket.getTicketItems();
		for (TicketItem ticketItem : ticketItems) {
			if (ticketItem.isGiftCard() && !(transaction instanceof RefundTransaction)) {
				String giftCardNo = ticketItem.getGiftCardNo();
				GiftCard giftCard = GiftCardDAO.getInstance().findByCardNumber(session, giftCardNo);
				if (giftCard == null) {
					if (transaction instanceof CreditCardTransaction || transaction instanceof DebitCardTransaction) {
						PosLog.error(PosTransactionService.class,
								String.format("Gift card add balance: Gift card with number %s not found while payment with card %s.", giftCardNo, //$NON-NLS-1$
										transaction.getCardNumber()));
						return;
					}
					else {
						throw new PosException(String.format(Messages.getString("GiftCardNoNotFound"), giftCardNo)); //$NON-NLS-1$
					}
				}
				double balanceToAdd = ticketItem.getUnitPrice();
	
				double giftCardPaidAmount = ticketItem.getGiftCardPaidAmount();
				if (balanceToAdd == giftCardPaidAmount) {
					return;
				}
				hasGiftCard = Boolean.TRUE;
				ticketItem.setGiftCardPaidAmount(balanceToAdd);
				transaction.setGiftCertNumber(giftCardNo);
				transaction.setGiftCertFaceValue(balanceToAdd);
	
				transaction.addGiftCardBalanceAddInfo(ticketItem.getId(), giftCardNo, balanceToAdd);
	
				Double balanceBefore = giftCard.getBalance();
				giftCard.setBalance(balanceBefore + balanceToAdd);
				GiftCardDAO.getInstance().saveOrUpdate(giftCard, session);
				BalanceUpdateTransactionDAO.getInstance().saveBalanceUpdateTrans(BalanceType.GIFT_CARD, ticket, transaction, TransactionType.CREDIT, giftCardNo,
						balanceToAdd, balanceBefore, TransactionSubType.BALANCE_ADDED, session);
			}
		}
		if (hasGiftCard) {
			TicketDAO.getInstance().saveOrUpdate(ticket, session);
		}
	}*/

	public static PosTransactionService getInstance() {
		return paymentService;
	}

	public static void sortTransactionsByDateDesc(List<PosTransaction> transactions) {
		Collections.sort(transactions, new Comparator<PosTransaction>() {
			@Override
			public int compare(PosTransaction o1, PosTransaction o2) {
				if (o1.getTransactionTime() == null) {
					return -1;
				}
				else if (o2.getTransactionTime() == null) {
					return 1;
				}
				return o2.getTransactionTime().compareTo(o1.getTransactionTime());
			}
		});
	}
}
