/**
 * ************************************************************************
 * * The contents of this file are subject to the MRPL 1.2
 * * (the  "License"),  being   the  Mozilla   Public  License
 * * Version 1.1  with a permitted attribution clause; you may not  use this
 * * file except in compliance with the License. You  may  obtain  a copy of
 * * the License at http://www.floreantpos.org/license.html
 * * Software distributed under the License  is  distributed  on  an "AS IS"
 * * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * * License for the specific  language  governing  rights  and  limitations
 * * under the License.
 * * The Original Code is FLOREANT POS.
 * * The Initial Developer of the Original Code is OROCUBE LLC
 * * All portions are Copyright (C) 2015 OROCUBE LLC
 * * All Rights Reserved.
 * ************************************************************************
 */
package com.floreantpos.model.dao;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.List;

import org.apache.commons.lang.StringUtils;
import org.hibernate.Criteria;
import org.hibernate.Session;
import org.hibernate.criterion.DetachedCriteria;
import org.hibernate.criterion.MatchMode;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.Projection;
import org.hibernate.criterion.ProjectionList;
import org.hibernate.criterion.Projections;
import org.hibernate.criterion.Property;
import org.hibernate.criterion.Restrictions;
import org.hibernate.transform.Transformers;

import com.floreantpos.PosLog;
import com.floreantpos.model.Customer;
import com.floreantpos.model.MenuGroup;
import com.floreantpos.model.Outlet;
import com.floreantpos.model.Patient;
import com.floreantpos.model.Terminal;
import com.floreantpos.model.Ticket;
import com.floreantpos.model.TicketItem;
import com.floreantpos.model.TicketItemModifier;
import com.floreantpos.model.TicketSource;
import com.floreantpos.model.User;
import com.floreantpos.model.ext.LabWorkStatus;
import com.floreantpos.model.util.DateUtil;
import com.floreantpos.swing.PaginationSupport;
import com.floreantpos.util.POSUtil;

public class TicketItemDAO extends BaseTicketItemDAO {

	/**
	 * Default constructor.  Can be used in place of getInstance()
	 */
	public TicketItemDAO() {
	}

	public boolean deleteTicketItemWithTicket(Integer itemID) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(TicketItem.class);
			criteria.add(Restrictions.eq(TicketItem.PROP_MENU_ITEM_ID, itemID));
			List<TicketItem> result = criteria.list();
			if (result == null || result.isEmpty()) {
				return false;
			}
			for (TicketItem ticketItem : result) {
				ticketItem.setTicket(null);
				super.delete(ticketItem, session);
			}
			return true;
		}
	}

	@SuppressWarnings("unchecked")
	public List<TicketItem> getSalesItems(Date startDate, Date endDate, Terminal terminal) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(TicketItem.class);
			criteria.createAlias(TicketItem.PROP_TICKET, "t");

			criteria.add(Restrictions.ge(TicketItem.PROP_CREATE_DATE, DateUtil.getUTCStartOfDay(startDate)));
			criteria.add(Restrictions.lt(TicketItem.PROP_CREATE_DATE, DateUtil.getUTCEndOfDay(endDate)));
			criteria.add(Restrictions.eq(TicketItem.PROP_VOIDED, Boolean.FALSE));
			criteria.add(Restrictions.eq("t." + Ticket.PROP_VOIDED, Boolean.FALSE));
			if (terminal != null) {
				criteria.add(Restrictions.eq("t." + Ticket.PROP_TERMINAL_ID, terminal.getId()));
			}

			return criteria.list();
		}
	}

	public Double getItemsCountWithinHour(Session session, String outletId, Integer hourDuration) {
		try {
			Calendar calendar = Calendar.getInstance();
			Date toDate = calendar.getTime();
			calendar.add(Calendar.HOUR, -(hourDuration == null ? 24 : hourDuration));
			Date fromDate = calendar.getTime();
			Criteria criteria = session.createCriteria(TicketItem.class).createAlias(TicketItem.PROP_TICKET, "ticket"); //$NON-NLS-1$
			criteria.add(Restrictions.between("ticket." + Ticket.PROP_CREATE_DATE, DateUtil.getUTCStartOfDay(fromDate), DateUtil.getUTCEndOfDay(toDate))); //$NON-NLS-1$
			if (StringUtils.isNotBlank(outletId)) {
				criteria.add(Restrictions.eq("ticket." + Ticket.PROP_OUTLET_ID, outletId)); //$NON-NLS-1$
			}
			criteria.add(Restrictions.eqOrIsNull(TicketItem.PROP_VOIDED, false));
			ProjectionList projectionList = Projections.projectionList();
			Projection projection = Projections.sum(TicketItem.PROP_QUANTITY);
			projectionList.add(projection);
			criteria.setProjection(projectionList);
			Number number = (Number) criteria.uniqueResult();
			if (number != null) {
				return number.doubleValue();
			}
		} catch (Exception e0) {
			PosLog.error(this.getReferenceClass(), e0);
		}
		return 0D;
	}

	public List<TicketItem> findTicketItemWithinDate(Date startDate, Date endDate, Terminal terminal, List<MenuGroup> groups, Boolean isInventoryItem) {
		return findTicketItemWithinDate(startDate, endDate, terminal, groups, isInventoryItem, null, Boolean.FALSE);
	}

	public List<TicketItem> findTicketItemWithinDate(Date startDate, Date endDate, Terminal terminal, List<MenuGroup> groups, Boolean isInventoryItem,
			Outlet outlet, Boolean isShowOnlineTicket) {
		return findTicketItemWithinDate(startDate, endDate, terminal, groups, isInventoryItem, outlet, isShowOnlineTicket, null);
	}

	public List<TicketItem> findTicketItemWithinDate(Date startDate, Date endDate, Terminal terminal, List<MenuGroup> groups, Boolean isInventoryItem,
			Outlet outlet, Boolean isShowOnlineTicket, List<User> users) {
		try (Session session = this.createNewSession()) {
			List<String> groupIdList = POSUtil.getStringIds(groups, MenuGroup.class);

			Criteria criteria = session.createCriteria(TicketItem.class).createAlias(TicketItem.PROP_TICKET, "ticket"); //$NON-NLS-1$
			if (startDate != null) {
				criteria.add(Restrictions.ge(TicketItem.PROP_CREATE_DATE, startDate));
			}
			if (endDate != null) {
				criteria.add(Restrictions.lt(TicketItem.PROP_CREATE_DATE, endDate));
			}

			criteria.add(Restrictions.eq(TicketItem.PROP_VOIDED, Boolean.FALSE));
			criteria.add(Restrictions.eq(TicketItem.PROP_TREAT_AS_SEAT, Boolean.FALSE));
			criteria.add(Restrictions.eq("ticket." + Ticket.PROP_VOIDED, Boolean.FALSE)); //$NON-NLS-1$
			if (isShowOnlineTicket) {
				criteria.add(Restrictions.eq("ticket." + Ticket.PROP_SOURCE, TicketSource.Online.name())); //$NON-NLS-1$
			}

			if (users != null) {
				List<String> allUserIds = new ArrayList<String>();
				for (User user : users) {
					allUserIds.addAll(user.getRoleIds());
				}
				criteria.add(Restrictions.in("ticket." + Ticket.PROP_OWNER_ID, allUserIds)); //$NON-NLS-1$
			}

			if (outlet != null) {
				criteria.add(Restrictions.eq("ticket." + Ticket.PROP_OUTLET_ID, outlet.getId())); //$NON-NLS-1$
			}

			if (terminal != null) {
				criteria.add(Restrictions.eq("ticket." + Ticket.PROP_TERMINAL_ID, terminal.getId())); //$NON-NLS-1$
			}
			if (!groupIdList.isEmpty()) {
				criteria.add(Restrictions.in(TicketItem.PROP_GROUP_ID, groupIdList));
			}
			if (isInventoryItem != null) {
				criteria.add(Restrictions.eq(TicketItem.PROP_INVENTORY_ITEM, isInventoryItem));
			}

			criteria.addOrder(Order.asc(TicketItem.PROP_NAME));
			List<TicketItem> list = criteria.list();
			return list;
		}
	}

	public List<TicketItemModifier> findTicketItemModifierWithinDate(String outletId, Date startDate, Date endDate, Terminal terminal, List<MenuGroup> groups,
			Boolean isInventoryItem) {
		try (Session session = this.createNewSession()) {
			List groupIdList = new ArrayList<>();
			if (groups != null) {
				for (MenuGroup menuGroup : groups) {
					groupIdList.add(menuGroup.getId());
				}
			}
			Criteria criteria = session.createCriteria(TicketItemModifier.class).createAlias(TicketItemModifier.PROP_TICKET_ITEM, "ticketItem") //$NON-NLS-1$
					.createAlias("ticketItem." + TicketItem.PROP_TICKET, "ticket"); //$NON-NLS-1$ //$NON-NLS-2$
			criteria.add(Restrictions.between("ticket." + Ticket.PROP_CREATE_DATE, startDate, endDate)); //$NON-NLS-1$
			if (isInventoryItem != null) {
				criteria.add(Restrictions.eq("ticketItem." + TicketItem.PROP_INVENTORY_ITEM, isInventoryItem)); //$NON-NLS-1$
			}
			if (!groupIdList.isEmpty()) {
				criteria.add(Restrictions.in("ticketItem." + TicketItem.PROP_GROUP_ID, groupIdList)); //$NON-NLS-1$
			}
			ProjectionList pList = Projections.projectionList();
			pList.add(Projections.property(TicketItemModifier.PROP_ITEM_ID), TicketItemModifier.PROP_ITEM_ID);
			pList.add(Projections.property(TicketItemModifier.PROP_NAME), TicketItemModifier.PROP_NAME);
			pList.add(Projections.property(TicketItemModifier.PROP_MODIFIER_TYPE), TicketItemModifier.PROP_MODIFIER_TYPE);
			pList.add(Projections.property(TicketItemModifier.PROP_UNIT_PRICE), TicketItemModifier.PROP_UNIT_PRICE);
			pList.add(Projections.property(TicketItemModifier.PROP_ITEM_QUANTITY), TicketItemModifier.PROP_ITEM_QUANTITY);
			pList.add(Projections.property(TicketItemModifier.PROP_SERVICE_CHARGE), TicketItemModifier.PROP_SERVICE_CHARGE);
			pList.add(Projections.property(TicketItemModifier.PROP_ADJUSTED_UNIT_PRICE), TicketItemModifier.PROP_ADJUSTED_UNIT_PRICE);
			pList.add(Projections.property(TicketItemModifier.PROP_ADJUSTED_DISCOUNT), TicketItemModifier.PROP_ADJUSTED_DISCOUNT);
			pList.add(Projections.property(TicketItemModifier.PROP_ADJUSTED_SUBTOTAL), TicketItemModifier.PROP_ADJUSTED_SUBTOTAL);
			pList.add(Projections.property(TicketItemModifier.PROP_ADJUSTED_TOTAL), TicketItemModifier.PROP_ADJUSTED_TOTAL);
			pList.add(Projections.property(TicketItemModifier.PROP_ADJUSTED_TAX), TicketItemModifier.PROP_ADJUSTED_TAX);
			pList.add(Projections.property(TicketItemModifier.PROP_TAX_INCLUDED), TicketItemModifier.PROP_TAX_INCLUDED);
			pList.add(Projections.property("ticketItem." + TicketItem.PROP_QUANTITY), TicketItemModifier.TRANSIENT_PROP_TICKET_ITEM_QUANTITY); //$NON-NLS-1$
			criteria.setProjection(pList);

			criteria.setResultTransformer(Transformers.aliasToBean(TicketItemModifier.class));
			criteria.addOrder(Order.asc(TicketItemModifier.PROP_NAME));
			List<TicketItemModifier> list = criteria.list();
			return list;
		}
	}

	public List<TicketItem> findTicketItemWithinDateForDashboard(String outletId, Date startDate, Date endDate, Integer numberOfItems, Boolean sortAsc,
			Session session) {
		Criteria criteria = session.createCriteria(TicketItem.class).createAlias(TicketItem.PROP_TICKET, "ticket"); //$NON-NLS-1$
		criteria.add(Restrictions.between(TicketItem.PROP_CREATE_DATE, DateUtil.getUTCStartOfDay(startDate), DateUtil.getUTCEndOfDay(endDate)));
		if (StringUtils.isNotBlank(outletId)) {
			criteria.add(Restrictions.eq("ticket." + Ticket.PROP_OUTLET_ID, outletId)); //$NON-NLS-1$
		}
		criteria.add(Restrictions.or(Restrictions.isNull(TicketItem.PROP_VOIDED), Restrictions.eq(TicketItem.PROP_VOIDED, false)));

		ProjectionList projectionList = Projections.projectionList();
		projectionList.add(Projections.property(TicketItem.PROP_NAME), TicketItem.PROP_NAME);
		projectionList.add(Projections.property(TicketItem.PROP_QUANTITY), TicketItem.PROP_QUANTITY);
		criteria.setProjection(projectionList);
		criteria.setMaxResults(numberOfItems == null ? 20 : numberOfItems);
		criteria.setResultTransformer(Transformers.aliasToBean(TicketItem.class));
		if (sortAsc != null) {
			if (sortAsc) {
				criteria.addOrder(Order.asc(TicketItem.PROP_QUANTITY));
			}
			else {
				criteria.addOrder(Order.desc(TicketItem.PROP_QUANTITY));
			}
		}
		return criteria.list();
	}

	public void findLabTestItems(String orderId, String patientName, String patientMobile, Date dob, LabWorkStatus labWorkStatus,
			PaginationSupport tableModel) {
		try (Session session = createNewSession()) {

			DetachedCriteria patientCriteria = DetachedCriteria.forClass(Patient.class);
			patientCriteria.setProjection(Property.forName(Patient.PROP_ID));

			if (StringUtils.isNotBlank(patientName)) {
				patientCriteria.add(Restrictions.ilike(Customer.PROP_NAME, patientName, MatchMode.ANYWHERE));
			}
			if (StringUtils.isNotBlank(patientMobile)) {
				patientCriteria.add(Restrictions.ilike(Customer.PROP_MOBILE_NO, patientMobile, MatchMode.ANYWHERE));
			}
			if (dob != null) {
				//patientCriteria.add(Restrictions.between(Customer.PROP_DATE_OF_BIRTH, DateUtil.startOfDay(dob), DateUtil.endOfDay(dob)));
				patientCriteria.add(Restrictions.eq(Customer.PROP_DATE_OF_BIRTH, DateUtil.startOfDay(dob)));
			}

			Criteria criteria = session.createCriteria(TicketItem.class);
			criteria.createAlias(TicketItem.PROP_TICKET, "ticket"); //$NON-NLS-1$

			criteria.add(Restrictions.eq(TicketItem.PROP_VOIDED, Boolean.FALSE));
			criteria.add(Restrictions.eq("ticket." + Ticket.PROP_VOIDED, Boolean.FALSE)); //$NON-NLS-1$

			if (StringUtils.isNotBlank(orderId)) {
				criteria.add(Restrictions.eq("ticket." + Ticket.PROP_ID, orderId)); //$NON-NLS-1$
			}

			if (labWorkStatus != null) {
				if (labWorkStatus == LabWorkStatus.PENDING_SAMPLE) {
					criteria.add(Restrictions.isNull(TicketItem.PROP_KITCHEN_STATUS));
				}
				else {
					criteria.add(Restrictions.eq(TicketItem.PROP_KITCHEN_STATUS, labWorkStatus.name()));
				}
			}

			criteria.add(Property.forName("ticket." + Ticket.PROP_CUSTOMER_ID).in(patientCriteria)); //$NON-NLS-1$

			tableModel.setNumRows(rowCount(criteria));
			criteria.setFirstResult(tableModel.getCurrentRowIndex());
			criteria.setMaxResults(tableModel.getPageSize());
			criteria.setResultTransformer(Transformers.aliasToBean(TicketItem.class));
			criteria.setResultTransformer(Criteria.DISTINCT_ROOT_ENTITY);
			criteria.addOrder(Order.desc(TicketItem.PROP_CREATE_DATE));

			tableModel.setRows(criteria.list());
		}
	}

	public List<TicketItem> findTicketItems(Date startDate, Date endDate, Boolean voided, Boolean returned, Outlet outlet, User user) {
		try (Session session = this.createNewSession()) {

			Criteria criteria = session.createCriteria(TicketItem.class).createAlias(TicketItem.PROP_TICKET, "ticket"); //$NON-NLS-1$
			criteria.add(Restrictions.ge(TicketItem.PROP_CREATE_DATE, startDate));
			criteria.add(Restrictions.lt(TicketItem.PROP_CREATE_DATE, endDate));
			if (voided != null) {
				criteria.add(Restrictions.eq(TicketItem.PROP_VOIDED, voided));
			}
			if (returned != null) {
				criteria.add(Restrictions.eq(TicketItem.PROP_ITEM_RETURNED, returned));
			}
			criteria.add(Restrictions.eq(TicketItem.PROP_TREAT_AS_SEAT, Boolean.FALSE));
			criteria.add(Restrictions.eq("ticket." + Ticket.PROP_VOIDED, Boolean.FALSE)); //$NON-NLS-1$
			if (outlet != null) {
				criteria.add(Restrictions.eq("ticket." + Ticket.PROP_OUTLET_ID, outlet.getId())); //$NON-NLS-1$
			}

			if (user != null) {
				criteria.add(Restrictions.in("ticket." + Ticket.PROP_OWNER_ID, user.getRoleIds())); //$NON-NLS-1$
			}

			criteria.addOrder(Order.asc(TicketItem.PROP_NAME));
			List<TicketItem> list = criteria.list();
			return list;
		}
	}

	public List<TicketItem> findTopItems(Session session, String outletId, Date startDate, Date endDate, int itemCount) {
		Criteria criteria = session.createCriteria(TicketItem.class).createAlias(TicketItem.PROP_TICKET, "ticket"); //$NON-NLS-1$
		criteria.add(Restrictions.between(TicketItem.PROP_CREATE_DATE, DateUtil.getUTCStartOfDay(startDate), DateUtil.getUTCEndOfDay(endDate)));
		if (StringUtils.isNotBlank(outletId)) {
			criteria.add(Restrictions.eq("ticket." + Ticket.PROP_OUTLET_ID, outletId)); //$NON-NLS-1$
		}
		criteria.add(Restrictions.or(Restrictions.isNull(TicketItem.PROP_VOIDED), Restrictions.eq(TicketItem.PROP_VOIDED, false)));
		criteria.add(Restrictions.eq("ticket." + Ticket.PROP_VOIDED, Boolean.FALSE));//$NON-NLS-1$

		ProjectionList projectionList = Projections.projectionList();
		projectionList.add(Projections.property(TicketItem.PROP_NAME), TicketItem.PROP_NAME);
		projectionList.add(Projections.sum(TicketItem.PROP_QUANTITY), TicketItem.PROP_QUANTITY);
		projectionList.add(Projections.groupProperty(TicketItem.PROP_NAME));
		criteria.setProjection(projectionList);

		criteria.setMaxResults(itemCount);
		criteria.setResultTransformer(Transformers.aliasToBean(TicketItem.class));
		criteria.addOrder(Order.desc(TicketItem.PROP_QUANTITY));
		criteria.addOrder(Order.asc(TicketItem.PROP_NAME));
		return criteria.list();
	}

	public TicketItem findByLabTest(String labTestId, Session session) {
		Criteria criteria = session.createCriteria(TicketItem.class);
		criteria.add(Restrictions.ilike(TicketItem.PROP_PROPERTIES, "\"lab.test.id\":\"" + labTestId.trim() + "\"", MatchMode.ANYWHERE)); //$NON-NLS-1$ //$NON-NLS-2$
		return (TicketItem) criteria.uniqueResult();
	}

}