package com.floreantpos.report;

import java.io.Serializable;

import org.apache.commons.lang3.StringUtils;

import com.floreantpos.model.Agent;
import com.floreantpos.model.AgentTypeEnum;
import com.floreantpos.model.Doctor;
import com.floreantpos.model.Ticket;
import com.floreantpos.model.dao.AgentDAO;
import com.floreantpos.model.dao.DoctorDAO;
import com.floreantpos.util.NumberUtil;

public class ReferralCommissionReportData implements Serializable {

	private String referralType;

	private String referralBy;
	private double receivable;
	private double due;
	private double received;
	private String reportDoctor;
	private double agentCommissionOnNetSales;
	private double agentCommissionOnReport;
	private double agentFee;

	public ReferralCommissionReportData(Ticket ticket) {
		double subtotalAmount = ticket.getSubtotalAmount();
		setReceivable(subtotalAmount + ticket.getLabDoctorFee());
		setDue(ticket.getDueAmount());
		setReceived(ticket.getPaidAmount());

		String agentId = ticket.getAgentId();
		if (StringUtils.isNotBlank(agentId)) {
			Agent agent = AgentDAO.getInstance().get(agentId);
			if (agent != null) {
				String agentType = agent.getAgentType();
				if (StringUtils.isNotBlank(agentType)) {
					setReferralType(AgentTypeEnum.fromString(agentType).getAgentTypeName());
				}
				else {
					setReferralType("Self");
				}
				setReferralBy(agent.getName());

				double totalCommissionOnReport = 0d;
				String commissionOnReport = agent.getAgentCommissionOnReport();
				if (commissionOnReport.endsWith("%")) { //$NON-NLS-1$
					double commissionPercentage = NumberUtil.parseDouble(commissionOnReport.replaceAll("%", "")); //$NON-NLS-1$ //$NON-NLS-2$
					totalCommissionOnReport = calculateCommission(subtotalAmount, commissionPercentage);
				}
				else {
					totalCommissionOnReport = NumberUtil.parseDouble(commissionOnReport);
				}

				double totalCommissionOnNetSales = 0d;
				String commissionOnNetSales = agent.getAgentCommissionOnNetSales();
				if (commissionOnNetSales.endsWith("%")) { //$NON-NLS-1$
					double commissionPercentage = NumberUtil.parseDouble(commissionOnNetSales.replaceAll("%", "")); //$NON-NLS-1$ //$NON-NLS-2$
					totalCommissionOnNetSales = calculateCommission(subtotalAmount - totalCommissionOnReport, commissionPercentage);
				}
				else {
					totalCommissionOnNetSales = NumberUtil.parseDouble(commissionOnNetSales);
				}
				setAgentCommissionOnNetSales(totalCommissionOnNetSales);

				setAgentCommissionOnReport(totalCommissionOnReport);
				setAgentFee(totalCommissionOnNetSales + totalCommissionOnReport);
			}
			else {
				setReferralType("Self");
				setReferralBy("");
			}
		}
		else {
			setReferralType("Self");
			setReferralBy("");
		}

		String doctorId = ticket.getDoctorId();
		if (StringUtils.isNotBlank(doctorId)) {
			Doctor doctor = DoctorDAO.getInstance().get(doctorId);
			if (doctor != null) {
				setReportDoctor(doctor.getName());
			}
		}

	}

	public double calculateCommission(double subtotalAmount, double commissionPercentage) {
		if (commissionPercentage > 0) {
			return subtotalAmount * (commissionPercentage / 100.0);
		}
		return 0d;
	}

	public String getReferralType() {
		return referralType;
	}

	public void setReferralType(String referralType) {
		this.referralType = referralType;
	}

	public String getReferralBy() {
		return referralBy;
	}

	public void setReferralBy(String referralBy) {
		this.referralBy = referralBy;
	}

	public double getReceivable() {
		return receivable;
	}

	public void setReceivable(double receivable) {
		this.receivable = receivable;
	}

	public double getDue() {
		return due;
	}

	public void setDue(double due) {
		this.due = due;
	}

	public double getReceived() {
		return received;
	}

	public void setReceived(double received) {
		this.received = received;
	}

	public String getReportDoctor() {
		return reportDoctor;
	}

	public void setReportDoctor(String reportDoctor) {
		this.reportDoctor = reportDoctor;
	}

	public double getAgentCommissionOnNetSales() {
		return agentCommissionOnNetSales;
	}

	public void setAgentCommissionOnNetSales(double agentCommissionOnNetSales) {
		this.agentCommissionOnNetSales = agentCommissionOnNetSales;
	}

	public double getAgentCommissionOnReport() {
		return agentCommissionOnReport;
	}

	public void setAgentCommissionOnReport(double agentCommissionOnReport) {
		this.agentCommissionOnReport = agentCommissionOnReport;
	}

	public double getAgentFee() {
		return agentFee;
	}

	public void setAgentFee(double agentFee) {
		this.agentFee = agentFee;
	}

}
