/**
 * ************************************************************************
 * * The contents of this file are subject to the MRPL 1.2
 * * (the  "License"),  being   the  Mozilla   Public  License
 * * Version 1.1  with a permitted attribution clause; you may not  use this
 * * file except in compliance with the License. You  may  obtain  a copy of
 * * the License at http://www.floreantpos.org/license.html
 * * Software distributed under the License  is  distributed  on  an "AS IS"
 * * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * * License for the specific  language  governing  rights  and  limitations
 * * under the License.
 * * The Original Code is FLOREANT POS.
 * * The Initial Developer of the Original Code is OROCUBE LLC
 * * All portions are Copyright (C) 2015 OROCUBE LLC
 * * All Rights Reserved.
 * ************************************************************************
 */
package com.floreantpos.model.dao;

import java.io.Serializable;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import java.util.Random;

import org.apache.commons.beanutils.PropertyUtils;
import org.apache.commons.lang.StringUtils;
import org.hibernate.Criteria;
import org.hibernate.Query;
import org.hibernate.SQLQuery;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.hibernate.criterion.Criterion;
import org.hibernate.criterion.Projections;
import org.hibernate.criterion.Restrictions;
import org.hibernate.transform.Transformers;

import com.floreantpos.PosLog;
import com.floreantpos.model.CashDrawer;
import com.floreantpos.model.DataSyncInfo;
import com.floreantpos.model.Outlet;
import com.floreantpos.model.Terminal;
import com.floreantpos.model.TerminalType;
import com.floreantpos.model.Ticket;
import com.floreantpos.model.User;
import com.floreantpos.model.ext.DeviceType;
import com.floreantpos.model.util.SyncUtil;
import com.orocube.rest.service.server.BaseDataServiceDao;

public class TerminalDAO extends BaseTerminalDAO {

	/**
	 * Default constructor.  Can be used in place of getInstance()
	 */
	public TerminalDAO() {
	}

	@Override
	protected Serializable save(Object obj, Session s) {
		updateTime(obj);
		return super.save(obj, s);
	}

	@Override
	protected void update(Object obj, Session s) {
		updateTime(obj);
		super.update(obj, s);
	}

	@Override
	protected void delete(Object obj, Session s) {
		if (obj instanceof Terminal) {
			Terminal terminal = (Terminal) obj;
			terminal.setDeleted(Boolean.TRUE);
			super.update(terminal, s);
		}
		else {
			super.delete(obj, s);
		}
	}

	@SuppressWarnings("unchecked")
	@Override
	public List<Terminal> findAll() {
		try (Session session = this.createNewSession()) {
			Criteria criteria = session.createCriteria(this.getReferenceClass());
			this.addDeletedFilter(criteria);
			return criteria.list();
		}
	}

	@Override
	protected void saveOrUpdate(Object obj, Session s) {
		updateTime(obj);
		super.saveOrUpdate(obj, s);
	}

	/**
	 * @param terminal
	 * 
	 * see {@link Terminal#isCashDrawerAssigned()}
	 * @return
	 */
	@Deprecated
	public boolean isDrawerAssigned(Terminal terminal) {
		//		try (Session session = this.createNewSession()) {
		//			Criteria criteria = session.createCriteria(getReferenceClass());
		//			this.addDeletedFilter(criteria);
		//			criteria.add(Restrictions.eq(Terminal.PROP_ID, terminal.getId()));
		//			criteria.setProjection(Projections.property(Terminal.PROP_ASSIGNED_USER_ID));
		//			Object result = criteria.uniqueResult();
		//			return result != null;
		//		}

		return terminal.getActiveCurrentCashDrawer() != null;
	}

	public String createNewTerminalName() {
		try (Session session = this.createNewSession()) {
			return createNewTerminalName(session);
		}
	}

	public Integer createNewTerminalId(String terminalKey) {
		try {
			String hashCode = String.valueOf(terminalKey.hashCode()).replaceFirst("-", "0");
			return Integer.parseInt(hashCode);
		} catch (Exception e) {
			return terminalKey.hashCode();
		}
	}

	public String createNewTerminalName(Session session) {
		if (session == null) {
			return createNewTerminalName();
		}
		Random random = new Random();

		while (true) {
			String terminalId = String.valueOf(random.nextInt(10000) + 1);
			Criteria criteria = session.createCriteria(getClass());
			criteria.add(Restrictions.eq(Terminal.PROP_ID, terminalId));
			Object uniqueResult = criteria.uniqueResult();
			if (uniqueResult == null) {
				return terminalId;
			}
		}
	}

	public Terminal createNewTerminal(String terminalKey, Outlet outlet) {
		return createNewTerminal(terminalKey, outlet, null, DeviceType.DESKTOP);
	}

	public Terminal createNewTerminal(String terminalKey, Outlet outlet, DeviceType deviceType) {
		return createNewTerminal(terminalKey, outlet, null, deviceType);
	}

	public Terminal createNewTerminal(String terminalKey, Outlet outlet, Session session, DeviceType deviceType) {
		Terminal terminal = new Terminal();
		terminal.setId(createNewTerminalId(terminalKey));
		terminal.setName(createNewTerminalName(session));
		terminal.setTerminalKey(terminalKey);
		if (outlet != null) {
			terminal.setOutletId(outlet.getId());
		}
		if (session == null) {
			save(terminal);
		}
		else {
			save(terminal, session);
		}

		return terminal;
	}

	public void refresh(Terminal terminal) {
		try (Session session = this.createNewSession()) {
			session.refresh(terminal);
		}
	}

	@SuppressWarnings("unchecked")
	public List<Terminal> findOpenTerminals() {
		try (Session session = this.createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			this.addDeletedFilter(criteria);
			criteria.add(Restrictions.isNotNull(Terminal.PROP_ASSIGNED_USER_ID));
			return criteria.list();
		}
	}

	@SuppressWarnings("unchecked")
	public List<Terminal> findCashDrawerTerminals() {
		try (Session session = this.createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			this.addDeletedFilter(criteria);
			criteria.add(Restrictions.eq(Terminal.PROP_HAS_CASH_DRAWER, Boolean.TRUE));
			return criteria.list();
		}
	}

	/*
	 * Save a list of objects in a batch operation
	 */
	public void performBatchSave(Object... objects) {
		Transaction tx = null;
		try (Session session = this.createNewSession()) {
			tx = session.beginTransaction();
			for (Object object : objects) {
				if (object != null) {
					updateTime(object);
					session.saveOrUpdate(object);
				}
			}
			tx.commit();
		} catch (Exception e) {
			if (tx != null) {
				tx.rollback();
			}
			throw e;
		}
	}

	public void performBatchSave(List<String> customQueryList, Object... objects) {
		Transaction tx = null;
		try (Session session = this.createNewSession()) {
			tx = session.beginTransaction();
			for (Object object : objects) {
				if (object != null) {
					session.saveOrUpdate(object);
				}
			}
			if (customQueryList != null) {
				for (String sql : customQueryList) {
					Query query = session.createQuery(sql);
					query.executeUpdate();
				}
			}
			tx.commit();
		} catch (Exception e) {
			if (tx != null) {
				tx.rollback();
			}
			throw e;
		}
	}

	public void resetCashDrawer(CashDrawer cashDrawer, Terminal terminal, User user, double balance) throws Exception {
		Transaction tx = null;

		cashDrawer.setClosedBy(user);
		cashDrawer.setReportTime(new Date());

		try (Session session = this.createNewSession()) {
			tx = session.beginTransaction();
			CashDrawer terminalCashDrawer = terminal.getActiveCurrentCashDrawer();
			if (StringUtils.isNotBlank(cashDrawer.getId()) && terminalCashDrawer != null && StringUtils.isNotBlank(terminalCashDrawer.getId())
					&& !cashDrawer.getId().equals(terminalCashDrawer.getId())) {
				saveOrUpdate(cashDrawer, session);
			}
			else {
				//terminal.setAssignedUserId(null);
				//terminal.setCurrentCashDrawer(null);
				//update(terminal, session);
				saveOrUpdate(cashDrawer, session);
			}
			tx.commit();
		} catch (Exception e) {
			if (tx != null) {
				tx.rollback();
			}
			throw e;
		}
	}

	public void resetStaffBank(User closedByUser, CashDrawer cashDrawer) throws Exception {
		Transaction tx = null;

		try (Session session = this.createNewSession()) {
			tx = session.beginTransaction();

			resetStaffBank(closedByUser, cashDrawer, session);

			tx.commit();
		} catch (Exception e) {
			if (tx != null) {
				tx.rollback();
			}
			throw e;
		}
	}

	public void resetStaffBank(User closedByUser, CashDrawer cashDrawer, Session session) throws Exception {
		cashDrawer.setClosedBy(closedByUser);
		cashDrawer.setReportTime(StoreDAO.getServerTimestamp());
		CashDrawerDAO.getInstance().saveOrUpdate(cashDrawer, session);

		String assignedUserId = cashDrawer.getAssignedUserId();
		if (StringUtils.isNotEmpty(assignedUserId)) {
			User assignedUser = UserDAO.getInstance().get(assignedUserId, cashDrawer.getOutletId());
			if (assignedUser != null) {
				assignedUser.setStaffBankStarted(false);
				assignedUser.setCurrentCashDrawer(null);
				UserDAO.getInstance().update(assignedUser, session);
			}
		}
	}

	public Terminal initialize(Terminal terminal) {
		if (terminal == null || terminal.getId() == null) {
			return terminal;
		}
		try (Session session = this.createNewSession()) {
			session.refresh(terminal);
			//Hibernate.initialize(InventoryTransaction.);
			return terminal;
		}
	}

	/**
	 * Get terminal by terminal key
	 * 
	 * @param terminalKey
	 * @return
	 */
	public Terminal getByTerminalKey(String terminalKey) {
		try (Session session = this.createNewSession()) {
			return getByTerminalKey(terminalKey, null, session);
		}
	}

	public Terminal getByTerminalKey(String terminalKey, String outletId) {
		try (Session session = this.createNewSession()) {
			return getByTerminalKey(terminalKey, outletId, session);
		}
	}

	@SuppressWarnings("unchecked")
	public Terminal getByTerminalKey(String terminalKey, String outletId, Session session) {
		Criteria criteria = session.createCriteria(getReferenceClass());
		//this.addDeletedFilter(criteria);
		if (StringUtils.isNotBlank(outletId)) {
			criteria.add(Restrictions.eq(Terminal.PROP_OUTLET_ID, outletId));
		}
		criteria.add(Restrictions.eq(Terminal.PROP_TERMINAL_KEY, terminalKey));
		List<Terminal> list = criteria.list();
		if (list.size() > 0) {
			return list.get(0);
		}
		return null;
	}

	@SuppressWarnings("unchecked")
	public Terminal findByAssignedUser(User user) {
		try (Session session = this.createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			this.addDeletedFilter(criteria);
			criteria.add(Restrictions.eq(Terminal.PROP_ASSIGNED_USER_ID, user.getId()));
			List<Terminal> list = criteria.list();
			if (list.size() > 0) {
				return list.get(0);
			}
			return null;
		}
	}

	@SuppressWarnings("rawtypes")
	public boolean isVersionEqual(Class beanClass, Object id, long versionToCompare) {
		try (Session session = this.createNewSession()) {
			Criteria criteria = session.createCriteria(beanClass);
			criteria.add(Restrictions.eq("id", id)); //$NON-NLS-1$
			criteria.setProjection(Projections.property("version")); //$NON-NLS-1$
			Object result = criteria.uniqueResult();
			if (result instanceof Number) {
				return ((Number) result).longValue() == versionToCompare;
			}
			return true;
		}
	}

	@SuppressWarnings("rawtypes")
	public long getVersion(Class beanClass, String column, Object id) {
		try (Session session = this.createNewSession()) {
			Criteria criteria = session.createCriteria(beanClass);
			criteria.add(Restrictions.eq(column, id));
			criteria.setProjection(Projections.property("version")); //$NON-NLS-1$
			return (long) criteria.uniqueResult();
		}
	}

	public void executeSqlQuery(List<String> sqlList) {
		if (sqlList == null) {
			return;
		}
		Transaction tx = null;
		try (Session session = this.createNewSession()) {
			tx = session.beginTransaction();
			for (String sql : sqlList) {
				Query query = session.createSQLQuery(sql);
				query.executeUpdate();
			}
			tx.commit();
		} catch (Exception e) {
			if (tx != null) {
				tx.rollback();
			}
			PosLog.error(getClass(), e);
		}
	}

	@SuppressWarnings("rawtypes")
	public List executeSqlQuery(String sql) {
		return executeSqlQuery(sql, 0, 0);
	}

	@SuppressWarnings("rawtypes")
	public List executeSqlQuery(String sql, long startIndex, long maxResult) {
		try (Session session = this.createNewSession()) {
			SQLQuery sqlQuery = session.createSQLQuery(sql);
			if (maxResult > 0) {
				sqlQuery.setFirstResult((int) startIndex);
				sqlQuery.setMaxResults((int) maxResult);
			}
			return sqlQuery.list();
		}
	}

	@SuppressWarnings("unchecked")
	public List<Terminal> findTerminalsByDeviceType(int deviceType, int limit) {
		try (Session session = this.createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			this.addDeletedFilter(criteria);
			criteria.add(Restrictions.eq(Terminal.PROP_DEVICE_TYPE, deviceType));
			if (limit > 0) {
				criteria.setMaxResults(limit);
			}
			return criteria.list();
		}
	}

	@SuppressWarnings("rawtypes")
	public int rowCount(Class beanClass, Date lastUpdateTime) {
		return rowCount(beanClass, lastUpdateTime, null);
	}

	@SuppressWarnings("rawtypes")
	public int rowCount(Class beanClass, Date lastUpdateTime, String outletId) {
		return rowCount(beanClass, lastUpdateTime, outletId, null);
	}

	@SuppressWarnings("rawtypes")
	public int rowCount(Class beanClass, Date lastUpdateTime, String outletId, Criterion restriction) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(beanClass);
			criteria.setProjection(Projections.rowCount());
			if (restriction != null) {
				criteria.add(restriction);
			}
			if (lastUpdateTime != null) {
				criteria.add(Restrictions.or(Restrictions.isNull("lastUpdateTime"), Restrictions.gt("lastUpdateTime", lastUpdateTime))); //$NON-NLS-1$ //$NON-NLS-2$
			}
			if (StringUtils.isNotBlank(outletId)) {
				criteria.add(Restrictions.eq("outletId", outletId)); //$NON-NLS-1$ 
			}
			criteria.setResultTransformer(Criteria.DISTINCT_ROOT_ENTITY);
			Number uniqueResult = (Number) criteria.uniqueResult();
			return uniqueResult == null ? 0 : uniqueResult.intValue();
		}
	}

	@SuppressWarnings("rawtypes")
	public List findDataList(Class beanClass, Date lastUpdateTime) {
		return findDataList(beanClass, lastUpdateTime, null);
	}

	@SuppressWarnings("rawtypes")
	public List findDataList(Class beanClass, Date lastUpdateTime, String outletId) {
		return findDataList(beanClass, lastUpdateTime, outletId, null);
	}

	public List findDataList(Class beanClass, Date lastUpdateTime, String outletId, Criterion criterion) {
		return findDataList(beanClass, lastUpdateTime, outletId, criterion, 0, 0);
	}

	//Deleted filter is strictly prohibited for this method 
	@SuppressWarnings("rawtypes")
	public List findDataList(Class beanClass, Date lastUpdateTime, String outletId, Criterion criterion, int offset, int limit) {
		try (Session session = this.createNewSession()) {
			Criteria criteria = session.createCriteria(beanClass);
			if (criterion != null) {
				criteria.add(criterion);
			}
			if (lastUpdateTime != null) {
				criteria.add(Restrictions.or(Restrictions.isNull("lastUpdateTime"), Restrictions.gt("lastUpdateTime", lastUpdateTime))); //$NON-NLS-1$ //$NON-NLS-2$
			}
			if (StringUtils.isNotBlank(outletId)) {
				criteria.add(Restrictions.eq("outletId", outletId)); //$NON-NLS-1$ 
			}
			if (limit > 0 && offset >= 0) {
				criteria.setFirstResult(offset);
				criteria.setMaxResults(limit);
			}
			criteria.setResultTransformer(Criteria.DISTINCT_ROOT_ENTITY);
			List list = criteria.list();

			if (Ticket.class.equals(beanClass)) {
				for (Object object : list) {
					Ticket ticket = (Ticket) object;
					TicketDAO.getInstance().initialize(ticket, session);
				}
			}

			return list;
		}
	}

	public int getActiveTerminalCount() {
		return getActiveTerminalCountByDeviceType(null);
	}

	public int getActiveTerminalCountByDeviceType(DeviceType deviceType) {
		try (Session session = this.createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			this.addDeletedFilter(criteria);
			criteria.add(Restrictions.neOrIsNotNull(Terminal.PROP_TERMINAL_KEY, "")); //$NON-NLS-1$
			criteria.add(Restrictions.eq(Terminal.PROP_ACTIVE, Boolean.TRUE));
			if (deviceType != null) {
				criteria.add(Restrictions.eq(Terminal.PROP_DEVICE_TYPE, deviceType.getDeviceType()));
			}
			criteria.setProjection(Projections.rowCount());
			Number rowCount = (Number) criteria.uniqueResult();
			if (rowCount != null) {
				return rowCount.intValue();
			}
			return 0;
		}
	}

	public Boolean deactivateAllActiveTerminal() {
		Transaction transaction = null;
		try (Session session = this.createNewSession()) {
			transaction = session.beginTransaction();
			String hql = "UPDATE " + Terminal.REF + " T SET T" + "." + Terminal.PROP_ACTIVE //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
					+ "=:newValue WHERE T" + "." + Terminal.PROP_ACTIVE + "=:oldValue"; //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
			Query query = session.createQuery(hql);
			query.setParameter("newValue", Boolean.FALSE); //$NON-NLS-1$
			query.setParameter("oldValue", Boolean.TRUE); //$NON-NLS-1$
			query.executeUpdate();
			transaction.commit();
			return Boolean.TRUE;
		} catch (Exception e0) {
			if (transaction != null) {
				transaction.rollback();
			}
			PosLog.error(getReferenceClass(), e0);
		}
		return Boolean.FALSE;
	}

	@SuppressWarnings("unchecked")
	public List<Terminal> findMasterTerminal(Terminal currentTerminalToExclude) {
		try (Session session = this.createNewSession()) {
			Criteria criteria = session.createCriteria(Terminal.class);
			this.addDeletedFilter(criteria);
			criteria.add(Restrictions.eq(Terminal.PROP_MASTER_TERMINAL, true));
			if (currentTerminalToExclude != null) {
				criteria.add(Restrictions.ne(Terminal.PROP_ID, currentTerminalToExclude.getId()));
			}
			return criteria.list();
		}
	}

	public void resetMasterTerminal(List<Terminal> masterTerminals) {
		Transaction tx = null;
		try (Session session = this.createNewSession()) {
			tx = session.beginTransaction();
			if (masterTerminals != null && !masterTerminals.isEmpty()) {
				for (Terminal terminal : masterTerminals) {
					terminal.setMasterTerminal(false);
					update(terminal, session);
				}
			}
			tx.commit();
		}
	}

	public Integer getMasterTerminalId() {
		try (Session session = this.createNewSession()) {
			Criteria criteria = session.createCriteria(Terminal.class);
			this.addDeletedFilter(criteria);
			criteria.setProjection(Projections.property(Terminal.PROP_ID));
			criteria.add(Restrictions.eq(Terminal.PROP_MASTER_TERMINAL, true));
			criteria.setMaxResults(1);

			Number number = (Number) criteria.uniqueResult();
			return number == null ? null : number.intValue();
		}
	}

	public Object getFieldValueByTerminalKey(String terminaKey, String fieldName) {
		try (Session session = this.createNewSession()) {
			Criteria criteria = session.createCriteria(Terminal.class);
			this.addDeletedFilter(criteria);
			criteria.setProjection(Projections.property(fieldName));
			criteria.add(Restrictions.eq(Terminal.PROP_TERMINAL_KEY, terminaKey));
			criteria.setMaxResults(1);
			return criteria.uniqueResult();
		}
	}

	//download and upload
	public void saveOrUpdateTerminal(List<Terminal> terminals, boolean updateLastUpdateTime, boolean updateSyncTime) throws Exception {
		if (terminals == null) {
			return;
		}
		Transaction tx = null;
		try (Session session = this.createNewSession()) {
			tx = session.beginTransaction();
			for (Iterator<Terminal> iterator = terminals.iterator(); iterator.hasNext();) {
				Terminal item = iterator.next();
				TerminalDAO dao = TerminalDAO.getInstance();
				Terminal existingItem = dao.get(item.getId(), item.getOutletId());
				if (existingItem != null) {
					if (!BaseDataServiceDao.get().shouldSave(item.getLastUpdateTime(), existingItem.getLastUpdateTime())) {
						PosLog.info(getClass(), item.getName() + " already updated"); //$NON-NLS-1$
						continue;
					}
					CashDrawer sourceCashDrawer = item.getActiveCurrentCashDrawer();
					CashDrawer existingCashDrawer = existingItem.getActiveCurrentCashDrawer();

					long version = existingItem.getVersion();
					PropertyUtils.copyProperties(existingItem, item);
					existingItem.setVersion(version);
					existingItem.setUpdateLastUpdateTime(updateLastUpdateTime);
					existingItem.setUpdateSyncTime(updateSyncTime);

					if (sourceCashDrawer != null && !CashDrawerDAO.getInstance().isExist(sourceCashDrawer.getId())) {
						CashDrawerDAO.getInstance().save(sourceCashDrawer);
					}
					if (sourceCashDrawer == null && existingCashDrawer != null && existingCashDrawer.getReportTime() == null) {
						existingItem.setCurrentCashDrawer(existingCashDrawer);
					}
					if (sourceCashDrawer != null && existingCashDrawer != null && !sourceCashDrawer.getId().equals(existingCashDrawer.getId())) {
						if (existingCashDrawer.getStartTime().after(sourceCashDrawer.getStartTime())) {
							existingItem.setCurrentCashDrawer(existingCashDrawer);
						}
					}
					dao.update(existingItem);
				}
				else {
					item.setUpdateLastUpdateTime(updateLastUpdateTime);
					item.setUpdateSyncTime(updateSyncTime);
					dao.save(item);
				}

			}
			tx.commit();
		} catch (Exception e) {
			if (tx != null) {
				tx.rollback();
			}
			throw e;
		}
	}

	public List<String> getActiveTerminalKeys() {
		return getActiveTerminalKeys(null);
	}

	@SuppressWarnings("unchecked")
	public List<String> getActiveTerminalKeys(DeviceType deviceType) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			this.addDeletedFilter(criteria);
			criteria.add(Restrictions.eqOrIsNull(Terminal.PROP_ACTIVE, true));
			criteria.add(Restrictions.eqOrIsNull(Terminal.PROP_DELETED, false));
			if (deviceType != null) {
				criteria.add(Restrictions.eqOrIsNull(Terminal.PROP_DEVICE_TYPE, deviceType.getDeviceType()));
			}
			criteria.add(Restrictions.neOrIsNotNull(Terminal.PROP_TERMINAL_KEY, "")); //$NON-NLS-1$
			criteria.setProjection(Projections.distinct(Projections.property(Terminal.PROP_TERMINAL_KEY)));
			return criteria.list();
		}
	}

	@SuppressWarnings("unchecked")
	public List<Terminal> getTerminalsByTerminalType(TerminalType terminalType, Session session) {
		Criteria criteria = session.createCriteria(this.getReferenceClass());
		this.addDeletedFilter(criteria);
		criteria.add(Restrictions.eq(Terminal.PROP_TERMINAL_TYPE_ID, terminalType.getId()));
		criteria.setProjection(Projections.alias(Projections.property(Terminal.PROP_NAME), Terminal.PROP_NAME));
		return criteria.setResultTransformer(Transformers.aliasToBean(this.getReferenceClass())).list();
	}

	public Terminal get(Integer terminalId, String outletId) {
		return super.get(new Terminal(terminalId, outletId));
	}

	public Terminal get(Integer terminalId, String outletId, Session session) {
		return super.get(new Terminal(terminalId, outletId), session);
	}

	public boolean existsTerminalId(Integer terminalId) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.setProjection(Projections.rowCount());
			criteria.add(Restrictions.eq(Terminal.PROP_ID, terminalId)); //$NON-NLS-1$ 
			criteria.setResultTransformer(Criteria.DISTINCT_ROOT_ENTITY);
			Number uniqueResult = (Number) criteria.uniqueResult();
			return uniqueResult != null && uniqueResult.intValue() > 0;
		}
	}

	public Date getLastUpdateTime(String outletId) {
		List<DataSyncInfo> dataSyncInfoList = SyncUtil.getSyncModelClasses(outletId);
		try (Session session = createNewSession()) {
			Date maxLastUpdateTime = null;
			for (Iterator<DataSyncInfo> iterator = dataSyncInfoList.iterator(); iterator.hasNext();) {
				DataSyncInfo syncInfo = (DataSyncInfo) iterator.next();
				Criteria criteria = session.createCriteria(syncInfo.getBeanClass());
				criteria.setProjection(Projections.max("lastUpdateTime")); //$NON-NLS-1$ 
				if (StringUtils.isNotBlank(syncInfo.getOutletId())) {
					criteria.add(Restrictions.eq("outletId", syncInfo.getOutletId())); //$NON-NLS-1$ 
				}
				criteria.setMaxResults(1);
				Date lastUpdateTime = (Date) criteria.uniqueResult();
				if (lastUpdateTime == null) {
					continue;
				}
				if (maxLastUpdateTime == null) {
					maxLastUpdateTime = lastUpdateTime;
				}
				else if (lastUpdateTime.compareTo(maxLastUpdateTime) > 0) {
					maxLastUpdateTime = lastUpdateTime;
				}
			}
			return maxLastUpdateTime;
		}
	}

	public int rowCountByDeviceType(DeviceType deviceType) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.setProjection(Projections.rowCount());
			criteria.add(Restrictions.eq(Terminal.PROP_DEVICE_TYPE, deviceType.getDeviceType())); //$NON-NLS-1$ 
			Number uniqueResult = (Number) criteria.uniqueResult();
			return uniqueResult == null ? 0 : uniqueResult.intValue();
		}
	}
}