/**
 * ************************************************************************
 * * The contents of this file are subject to the MRPL 1.2
 * * (the  "License"),  being   the  Mozilla   Public  License
 * * Version 1.1  with a permitted attribution clause; you may not  use this
 * * file except in compliance with the License. You  may  obtain  a copy of
 * * the License at http://www.floreantpos.org/license.html
 * * Software distributed under the License  is  distributed  on  an "AS IS"
 * * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * * License for the specific  language  governing  rights  and  limitations
 * * under the License.
 * * The Original Code is FLOREANT POS.
 * * The Initial Developer of the Original Code is OROCUBE LLC
 * * All portions are Copyright (C) 2015 OROCUBE LLC
 * * All Rights Reserved.
 * ************************************************************************
 */
package com.floreantpos.model;

import java.util.ArrayList;
import java.util.List;

import javax.swing.ImageIcon;
import javax.xml.bind.annotation.XmlRootElement;
import javax.xml.bind.annotation.XmlTransient;

import org.apache.commons.lang.StringUtils;

import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.floreantpos.PosLog;
import com.floreantpos.model.base.BaseCustomer;
import com.floreantpos.model.dao.CustomerGroupDAO;
import com.floreantpos.report.PatientContactData;
import com.floreantpos.util.AESencrp;
import com.floreantpos.util.NumberUtil;
import com.floreantpos.util.POSUtil;
import com.google.gson.Gson;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.reflect.TypeToken;

@JsonIgnoreProperties(ignoreUnknown = true, value = { "deliveryAddresses", "deliveryInstructions" })
@XmlRootElement
public class Customer extends BaseCustomer implements TimedModel {
	private static final long serialVersionUID = 1L;
	public final static String APPETIZER_PREF = "appetizerPref";//$NON-NLS-1$
	public final static String BILLINGSTATEMENT_PREF = "billingStatementPref";//$NON-NLS-1$
	public final static String COCKTAIL_PREF = "cocktailPref";//$NON-NLS-1$
	public final static String DESSERT_PREF = "dessertPref";//$NON-NLS-1$
	public final static String ENTREE_PREF = "entreePref";//$NON-NLS-1$
	public final static String GOLFBALL_PREF = "golfBallPref";//$NON-NLS-1$
	public final static String MEAL_PREF = "mealPref";//$NON-NLS-1$
	public final static String PAYMENT_PREF = "paymentPref";//$NON-NLS-1$
	public final static String PILLOW_PREF = "pillowPref";//$NON-NLS-1$
	public final static String ROLLAWAYCRIB_PREF = "rollawayCribPref";//$NON-NLS-1$

	public final static String SEATING_PREF = "seatingPref";//$NON-NLS-1$
	public final static String SMOKING_PREF = "smokingPref";//$NON-NLS-1$
	public final static String WINE_PREF = "winePref";//$NON-NLS-1$
	public final static String ROOM_ACCESS_PREF = "roomAccessPref";//$NON-NLS-1$
	public final static String ROOM_LOC_PREF = "roomLocPref";//$NON-NLS-1$
	public final static String ROOM_NUMBER_PREF = "roomNumberPref";//$NON-NLS-1$
	public final static String REMAINING_CURRENT_QUARTER = "club62__Remaining_Current_Quarter_F_B_Spends__c";//$NON-NLS-1$
	public final static String STATUS = "club62__Status__c";//$NON-NLS-1$
	public final static String STREET_1 = "street_1";//$NON-NLS-1$
	public final static String STREET_2 = "street_2";//$NON-NLS-1$
	public final static String STREET_3 = "street_3";//$NON-NLS-1$

	private Boolean selected;
	private transient JsonObject propertiesContainer;

	private boolean updateLastUpdateTime = true;
	private boolean updateSyncTime = false;

	public boolean isUpdateSyncTime() {
		return updateSyncTime;
	}

	public void setUpdateSyncTime(boolean shouldUpdateSyncTime) {
		this.updateSyncTime = shouldUpdateSyncTime;
	}

	public boolean isUpdateLastUpdateTime() {
		return updateLastUpdateTime;
	}

	public void setUpdateLastUpdateTime(boolean shouldUpdateUpdateTime) {
		this.updateLastUpdateTime = shouldUpdateUpdateTime;
	}

	/*[CONSTRUCTOR MARKER BEGIN]*/
	public Customer() {
	}

	/**
	 * Constructor for primary key
	 */
	public Customer(java.lang.String id) {
		super(id);
	}

	/*[CONSTRUCTOR MARKER END]*/

	@Override
	public String toString() {
		String fName = getName();
		if (fName == null) {
			fName = getEmail();
		}
		return fName;
	}

	public enum MemberType {
		//@formatter:off
		MEMBER(0, "Member"),//$NON-NLS-1$
		GUEST(1, "Guest"),//$NON-NLS-1$
		EMPLOYEE(2, "Employee"),//$NON-NLS-1$
		STORE_OWNER(3, "StoreOwner")//$NON-NLS-1$
		;
		//@formatter:on

		private int typeInt;
		private String name;

		private MemberType(int type, String name) {
			this.typeInt = type;
			this.name = name;
		}

		public int getTypeInt() {
			return typeInt;
		}

		public String getName() {
			return name;
		}

		@Override
		public String toString() {
			return name;
		}

		public static MemberType fromInt(int type) {
			MemberType[] values = values();
			for (MemberType memberType : values) {
				if (memberType.getTypeInt() == type) {
					return memberType;
				}
			}

			return MemberType.MEMBER;
		}
	}

	public ImageIcon getImage() {
		return null;
	}

	public String getName() {
		String name = super.getFirstName();
		if (StringUtils.isNotEmpty(super.getLastName())) {
			name += " " + super.getLastName(); //$NON-NLS-1$
		}
		return name;
	}

	@Override
	public void setMobileNo(String mobileNo) {
		super.setMobileNo(mobileNo);
	}

	@Override
	public void setWorkPhoneNo(String workPhoneNo) {
		super.setWorkPhoneNo(workPhoneNo);
	}

	@Override
	public void setHomePhoneNo(String homePhoneNo) {
		super.setHomePhoneNo(homePhoneNo);
	}

	public Boolean isSelected() {
		return selected == null ? Boolean.FALSE : selected;
	}

	public void setSelected(Boolean enable) {
		this.selected = enable;
	}

	public CustomerGroup getCustomerGroup() {
		if (StringUtils.isNotEmpty(getCustomerGroupId())) {
			return CustomerGroupDAO.getInstance().get(getCustomerGroupId());
		}
		return null;
	}

	public void setCustomerGroup(CustomerGroup customerGroup) {
		String customerGroupId = null;
		if (customerGroup != null) {
			customerGroupId = customerGroup.getId();
		}
		super.setCustomerGroupId(customerGroupId);
	}

	public void setEncryptedSecretCode(String secretCode) {
		try {
			setSecretCode(AESencrp.encrypt(secretCode));
		} catch (Exception e) {
			setSecretCode(secretCode);
		}
	}

	public String getUnencryptedSecretCode() {
		String secretKey = getSecretCode();
		if (StringUtils.isNotEmpty(secretKey)) {
			try {
				secretKey = AESencrp.decrypt(secretKey);
			} catch (Exception e) {
			}
		}
		return secretKey;
	}

	public void setEncryptedPin(String pin) {
		try {
			setPin(AESencrp.encrypt(pin));
		} catch (Exception e) {
			setPin(pin);
		}
	}

	public String getUnencryptedPin() {
		String pin = getPin();
		if (StringUtils.isNotEmpty(pin)) {
			try {
				pin = AESencrp.decrypt(pin);
			} catch (Exception e) {
			}
		}
		return pin;
	}

	@Override
	public String getProperties() {
		if (propertiesContainer != null) {
			return propertiesContainer.toString();
		}

		String properties = super.getProperties();
		if (StringUtils.isEmpty(properties)) {
			return null;
		}

		propertiesContainer = new Gson().fromJson(properties, JsonObject.class);
		return properties;
	}

	@Override
	public void setProperties(String properties) {
		super.setProperties(properties);
		if (StringUtils.isNotEmpty(properties)) {
			propertiesContainer = new Gson().fromJson(properties, JsonObject.class);
		}
	}

	public void addProperty(String key, String value) {
		if (propertiesContainer == null) {
			propertiesContainer = new JsonObject();
		}
		//if (StringUtils.isNotEmpty(value)) {
		propertiesContainer.addProperty(key, value);
		//}
	}

	public String getProperty(String key) {
		return getProperty(key, null);
	}

	public String getProperty(String key, String defaultValue) {
		if (propertiesContainer == null) {
			return defaultValue;
		}
		if (propertiesContainer.has(key)) {
			JsonElement jsonElement = propertiesContainer.get(key);
			if (!jsonElement.isJsonNull()) {
				return jsonElement.getAsString();
			}
		}
		return defaultValue;
	}

	public boolean hasProperty(String key) {
		return getProperty(key) != null;
	}

	public boolean isPropertyValueTrue(String propertyName) {
		String property = getProperty(propertyName);

		return POSUtil.getBoolean(property);
	}

	public void removeProperty(String propertyName) {
		if (propertiesContainer != null) {
			propertiesContainer.remove(propertyName);
		}
	}

	public void putPatientAddressLine(String patientAddress) {
		addProperty("patient.address.line", patientAddress); //$NON-NLS-1$
	}

	public String getPatientAddressLine() {
		return getProperty("patient.address.line", ""); //$NON-NLS-1$ //$NON-NLS-2$
	}

	public void putPatientGender(String patientGender) {
		addProperty("patient.gender", patientGender); //$NON-NLS-1$
	}

	public String getPatientGender() {
		return getProperty("patient.gender", ""); //$NON-NLS-1$ //$NON-NLS-2$
	}

	public void putPatientRegistrationReason(String patientRegistrationReason) {
		addProperty("patient.registration.reason", patientRegistrationReason); //$NON-NLS-1$
	}

	public String getPatientRegistrationReason() {
		return getProperty("patient.registration.reason", ""); //$NON-NLS-1$ //$NON-NLS-2$
	}

	public void putPatientHealthHistory(String patientHealthHistory) {
		addProperty("patient.health.history", patientHealthHistory); //$NON-NLS-1$
	}

	public String getPatientHealthHistory() {
		return getProperty("patient.health.history", ""); //$NON-NLS-1$ //$NON-NLS-2$
	}

	public void addPatientEmergencyContacts(String patientEmergencyContacts) {
		addProperty("patient.emergency.contact", patientEmergencyContacts); //$NON-NLS-1$
	}

	public List<PatientContactData> getPatientEmergencyContacts() {
		List<PatientContactData> patientEmergencyContactList = new ArrayList<PatientContactData>();
		try {
			String emergencyContact = getProperty("patient.emergency.contact"); //$NON-NLS-1$
			if (StringUtils.isEmpty(emergencyContact)) {
				return patientEmergencyContactList;
			}
			patientEmergencyContactList = new Gson().fromJson(emergencyContact, new TypeToken<ArrayList<PatientContactData>>() {
			}.getType());
		} catch (Exception e) {
			PosLog.error(getClass(), e);
		}
		return patientEmergencyContactList;
	}

	public void addPatientDisclosureContacts(String patientDisclosureContacts) {
		addProperty("patient.disclosure.contact", patientDisclosureContacts); //$NON-NLS-1$
	}

	public List<PatientContactData> getPatientDisclosureContacts() {
		List<PatientContactData> patientDisclosureContactList = new ArrayList<PatientContactData>();
		try {
			String disclosureContact = getProperty("patient.disclosure.contact"); //$NON-NLS-1$
			if (StringUtils.isEmpty(disclosureContact)) {
				return patientDisclosureContactList;
			}

			patientDisclosureContactList = new Gson().fromJson(disclosureContact, new TypeToken<ArrayList<PatientContactData>>() {
			}.getType());
		} catch (Exception e) {
			PosLog.error(getClass(), e);
		}
		return patientDisclosureContactList;
	}

	public void putBloodGroup(String bloodGroup) {
		addProperty("customer.blood.group", bloodGroup); //$NON-NLS-1$
	}

	public String getBloodGroup() {
		return getProperty("customer.blood.group", ""); //$NON-NLS-1$ //$NON-NLS-2$
	}
	
	public void putAgentCommissionOnReportChargeType(String chargeType) {
		addProperty("agent.report.commission_charge.type", chargeType); //$NON-NLS-1$
	}

	public String getAgentCommissionOnReportChargeType() {
		return getProperty("agent.report.commission_charge.type", ""); //$NON-NLS-1$ //$NON-NLS-2$
	}

	public void putAgentCommissionOnReport(String commissionValue) {
		addProperty("agent.report.commission_value", commissionValue); //$NON-NLS-1$
	}

	public String getAgentCommissionOnReport() {
		return getProperty("agent.report.commission_value", ""); //$NON-NLS-1$ //$NON-NLS-2$
	}

	public void putAgentCommissionOnNetSalesChargeType(String chargeType) {
		addProperty("agent.netsales.commission_charge.type", chargeType); //$NON-NLS-1$
	}

	public String getAgentCommissionOnNetSalesChargeType() {
		return getProperty("agent.netsales.commission_charge.type", ""); //$NON-NLS-1$ //$NON-NLS-2$
	}
	
	public void putAgentCommissionOnNetSales(String commissionValue) {
		addProperty("agent.netsales.commission_value", commissionValue); //$NON-NLS-1$
	}

	public String getAgentCommissionOnNetSales() {
		return getProperty("agent.netsales.commission_value", ""); //$NON-NLS-1$ //$NON-NLS-2$
	}
	
	public void putAgentSalary(double salary) {
		addProperty("agent.salary", String.valueOf(salary)); //$NON-NLS-1$
	}

	public double getAgentSalary() {
		return PropertyContainer.parseDouble(getProperty("agent.salary", "")); //$NON-NLS-1$ //$NON-NLS-2$
	}
	
	public void putAgentType(String agentTypeName) {
		addProperty("agent.type", agentTypeName); //$NON-NLS-1$
	}

	public String getAgentType() {
		return getProperty("agent.type", ""); //$NON-NLS-1$ //$NON-NLS-2$
	}
	

	public void putDoctorAgentId(String agentId) {
		addProperty("doctor.agent.id", agentId); //$NON-NLS-1$
	}

	public String getDoctorAgentId() {
		return getProperty("doctor.agent.id", ""); //$NON-NLS-1$ //$NON-NLS-2$
	}

	public void putDoctorCommissionPercentage(String commissionPercentage) {
		addProperty("doctor.commission_percentage", commissionPercentage); //$NON-NLS-1$
	}

	public String getDoctorCommissionPercentage() {
		return getProperty("doctor.commission_percentage", ""); //$NON-NLS-1$ //$NON-NLS-2$
	}

	public void putDoctorDesignation(String doctorDesignation) {
		addProperty("doctor.designation", doctorDesignation); //$NON-NLS-1$
	}

	public String getDoctorDesignation() {
		return getProperty("doctor.designation", ""); //$NON-NLS-1$ //$NON-NLS-2$
	}

	public void putDoctorOrganization(String doctorOrganization) {
		addProperty("doctor.organization", doctorOrganization); //$NON-NLS-1$
	}

	public String getDoctorOrganization() {
		return getProperty("doctor.organization", ""); //$NON-NLS-1$ //$NON-NLS-2$
	}

	public void putDoctorDepartmentId(String doctorDepartmentId) {
		addProperty("doctor.department.id", doctorDepartmentId); //$NON-NLS-1$
	}

	public String getDoctorDepartmentId() {
		return getProperty("doctor.department.id", ""); //$NON-NLS-1$ //$NON-NLS-2$
	}

	public void putDoctorDegreesAndTraining(String degreesAndTraining) {
		addProperty("doctor.degrees.and.training", degreesAndTraining); //$NON-NLS-1$
	}

	public String getDoctorDegreesAndTraining() {
		return getProperty("doctor.degrees.and.training", ""); //$NON-NLS-1$ //$NON-NLS-2$
	}

	public void putDoctorSpecialistName(String specialistName) {
		addProperty("doctor.specialist.name", specialistName); //$NON-NLS-1$
	}

	public String getDoctorSpecialistName() {
		return getProperty("doctor.specialist.name", ""); //$NON-NLS-1$ //$NON-NLS-2$
	}

	public void putDoctorScheduleToVisit(String scheduleToVisit) {
		addProperty("doctor.schedule.to.visit", scheduleToVisit); //$NON-NLS-1$
	}

	public String getDoctorScheduleToVisit() {
		return getProperty("doctor.schedule.to.visit", ""); //$NON-NLS-1$ //$NON-NLS-2$
	}

	public void putDoctorAddressLine(String doctorAddress) {
		addProperty("doctor.address.line", doctorAddress); //$NON-NLS-1$
	}

	public String getDoctorAddressLine() {
		return getProperty("doctor.address.line", ""); //$NON-NLS-1$ //$NON-NLS-2$
	}

	public void putIndorDoctor(boolean doctor) {
		addProperty("doctor.indor", String.valueOf(doctor)); //$NON-NLS-1$
	}

	public Boolean getIndorDoctor() {
		return POSUtil.getBoolean(getProperty("doctor.indor"), false); //$NON-NLS-1$
	}

	public void putLabDoctor(boolean doctor) {
		addProperty("doctor.lab", String.valueOf(doctor)); //$NON-NLS-1$
	}

	public Boolean getLabDoctor() {
		return POSUtil.getBoolean(getProperty("doctor.lab"), false); //$NON-NLS-1$
	}
	

	public void setReceiveReferralFee(boolean isReceiveReferralFee) {
		addProperty("doctor.receive.referral.fee", String.valueOf(isReceiveReferralFee)); //$NON-NLS-1$
	}

	@XmlTransient
	@JsonIgnore
	public boolean isReceiveReferralFee() {
		return POSUtil.getBoolean(getProperty("doctor.receive.referral.fee"), false); //$NON-NLS-1$
	}

	public void putDoctorReferralFee(double doctorReferralFee) {
		addProperty("doctor.referral.fee", String.valueOf(doctorReferralFee)); //$NON-NLS-1$
	}

	@XmlTransient
	@JsonIgnore
	public double getDoctorReferralFee() {
		return NumberUtil.parseOrGetZero(getProperty("doctor.referral.fee")).doubleValue(); //$NON-NLS-1$
	}

}