/**
 * ************************************************************************
 * * The contents of this file are subject to the MRPL 1.2
 * * (the  "License"),  being   the  Mozilla   Public  License
 * * Version 1.1  with a permitted attribution clause; you may not  use this
 * * file except in compliance with the License. You  may  obtain  a copy of
 * * the License at http://www.floreantpos.org/license.html
 * * Software distributed under the License  is  distributed  on  an "AS IS"
 * * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * * License for the specific  language  governing  rights  and  limitations
 * * under the License.
 * * The Original Code is FLOREANT POS.
 * * The Initial Developer of the Original Code is OROCUBE LLC
 * * All portions are Copyright (C) 2015 OROCUBE LLC
 * * All Rights Reserved.
 * ************************************************************************
 */
package com.floreantpos.model;

import java.io.StringReader;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import javax.json.Json;
import javax.json.JsonArray;
import javax.json.JsonObject;
import javax.json.JsonReader;
import javax.xml.bind.annotation.XmlAccessType;
import javax.xml.bind.annotation.XmlAccessorType;
import javax.xml.bind.annotation.XmlRootElement;
import javax.xml.bind.annotation.XmlSeeAlso;
import javax.xml.bind.annotation.XmlTransient;

import org.apache.commons.lang.SerializationUtils;
import org.apache.commons.lang.StringUtils;
import org.json.JSONArray;
import org.json.JSONObject;

import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonSubTypes;
import com.fasterxml.jackson.annotation.JsonSubTypes.Type;
import com.fasterxml.jackson.annotation.JsonTypeInfo;
import com.fasterxml.jackson.annotation.JsonTypeInfo.Id;
import com.floreantpos.Messages;
import com.floreantpos.PosException;
import com.floreantpos.PosLog;
import com.floreantpos.constants.AppConstants;
import com.floreantpos.model.ProcessedStatus.ProcessedStatusResult;
import com.floreantpos.model.base.BaseTicketItem;
import com.floreantpos.model.dao.ActionHistoryDAO;
import com.floreantpos.model.dao.MenuItemDAO;
import com.floreantpos.model.dao.MenuModifierDAO;
import com.floreantpos.model.dao.StoreDAO;
import com.floreantpos.model.ext.KitchenStatus;
import com.floreantpos.model.ext.LabWorkStatus;
import com.floreantpos.model.util.DataProvider;
import com.floreantpos.model.util.DateUtil;
import com.floreantpos.model.util.pricecalc.TicketItemCalcFactory;
import com.floreantpos.util.CopyUtil;
import com.floreantpos.util.GsonUtil;
import com.floreantpos.util.JsonUtil;
import com.floreantpos.util.NumberUtil;
import com.floreantpos.util.OrgJsonUtil;
import com.floreantpos.util.POSUtil;
import com.google.gson.ExclusionStrategy;
import com.google.gson.FieldAttributes;
import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.JsonElement;
import com.google.gson.reflect.TypeToken;

@JsonIgnoreProperties(ignoreUnknown = true, value = { "inventoryAdjusted", "menuItem", "voidItem", "comboItemsModifierExtraPrice" })
@XmlSeeAlso({ ModifiableTicketItem.class, ComboTicketItem.class })
@XmlRootElement
@XmlAccessorType(XmlAccessType.FIELD)
@JsonTypeInfo(use = Id.CLASS, include = JsonTypeInfo.As.PROPERTY, property = "classType")
@JsonSubTypes({ @Type(value = ModifiableTicketItem.class), @Type(value = ComboTicketItem.class) })
public class TicketItem extends BaseTicketItem implements ITicketItem, PropertyContainer {

	public static final String JSON_PROP_DISCOUNT_ON_SC = "discountOnSc"; //$NON-NLS-1$
	public static final String JSON_PROP_VOID_ID = "voidId"; //$NON-NLS-1$
	public static final String JSON_PROP_VOIDED_BY_USER = "voidedByUser"; //$NON-NLS-1$
	public static final String JSON_PROP_MODIIFIER_VOID_IDS = "modifierVoidIds"; //$NON-NLS-1$
	public static final String JSON_PROP_VOID_REASON = "voidReason"; //$NON-NLS-1$
	public static final String JSON_PROP_VOID_QUANTITY = "voidQuantity"; //$NON-NLS-1$
	public static final String JSON_PROP_WASTED = "wasted"; //$NON-NLS-1$
	public static final String JSON_PROP_PRINTED_TO_KITCHEN = "originalItemPrintedToKitchen"; //$NON-NLS-1$
	public static final String JSON_PROP_RETURNED = "returned"; //$NON-NLS-1$
	public static final String JSON_PROP_RETURNED_ITEM_VOIDED = "returnedVoided"; //$NON-NLS-1$
	public static final String PROPERTY_COOKING_INSTRUCTION = "COOKING_INSTRUCTION"; //$NON-NLS-1$
	private static final String JSON_PROP_GIFT_CARD_PAID_AMOUNT = "gift_card_paid_amount"; //$NON-NLS-1$
	private static final String JSON_PROP_SC_WITHOUT_MODOFIERS = "scWithoutModifiers"; //$NON-NLS-1$

	public static final String PROPERTY_TAX = "TAX"; //$NON-NLS-1$
	public static final String PROPERTY_DISCOUNT = "DISCOUNT"; //$NON-NLS-1$
	private static final long serialVersionUID = 1L;
	private transient boolean syncEdited;
	private transient com.google.gson.JsonObject propertiesContainer;
	private String classType;
	private transient MenuItem menuItem;
	private int tableRowNum;
	private Double quantityToShip;

	public static final int HALF = 0;
	public static final int FULL = 1;
	public static final int SHARED_FULL = 2;
	public static final int SHARED_CUSTOM = 3;

	private Boolean includeVoidQuantity;
	private VoidItem voidItem;
	private java.util.List<TicketItemCookingInstruction> cookingInstructions;
	private java.util.List<TicketItemTax> taxes;
	private java.util.List<TicketItemDiscount> discounts;
	private TicketItem parentTicketItem;
	private Integer sortOrder;

	private transient List<TicketItem> treeChildren = new ArrayList<>();

	public TicketItem() {
		setDataVersion(2);
		setCreateDate(StoreDAO.getServerTimestamp());
	}

	/**
	 * Constructor for primary key
	 */
	public TicketItem(java.lang.String id) {
		super(id);
		setDataVersion(2);
		setCreateDate(StoreDAO.getServerTimestamp());
	}

	public TicketItem(Ticket ticket, String name, double unitPrice, double quantity, TicketItemTax tax) {
		setName(name);
		setUnitPrice(unitPrice);
		setQuantity(quantity);
		addTotaxes(tax);
		setTicket(ticket);
		setDataVersion(2);
		setCreateDate(StoreDAO.getServerTimestamp());
	}

	public TicketItem createNew() {
		if (this instanceof ModifiableTicketItem) {
			return new ModifiableTicketItem();
		}
		else {
			return new TicketItem();
		}
	}

	/**
	 * Return the value associated with the column: HAS_MODIIERS
	 */
	public java.lang.Boolean isHasModifiers() {
		List<TicketItemModifier> ticketItemModifiers = getTicketItemModifiers();
		return ticketItemModifiers != null && ticketItemModifiers.size() > 0;
	}

	/**
	 * Return the value associated with the column: SIZE_MODIFIER_ID
	 */
	public com.floreantpos.model.TicketItemModifier getSizeModifier() {
		return null;
	}

	/**
	 * Set the value related to the column: SIZE_MODIFIER_ID
	 * @param sizeModifier the SIZE_MODIFIER_ID value
	 */
	public void setSizeModifier(com.floreantpos.model.TicketItemModifier sizeModifier) {
	}

	/**
	 * Return the value associated with the column: ticketItemModifiers
	 */
	public java.util.List<com.floreantpos.model.TicketItemModifier> getTicketItemModifiers() {
		return null;
	}

	/**
	 * Set the value related to the column: ticketItemModifiers
	 * @param ticketItemModifiers the ticketItemModifiers value
	 */
	public void setTicketItemModifiers(java.util.List<com.floreantpos.model.TicketItemModifier> ticketItemModifiers) {
	}

	public void addToticketItemModifiers(com.floreantpos.model.TicketItemModifier ticketItemModifier) {
	}

	@Override
	public TicketItem clone() {
		return (TicketItem) SerializationUtils.clone(this);
	}

	public TicketItem cloneAsNew() {
		return cloneAsNew(false);
	}

	public TicketItem cloneAsNew(boolean needToCheeckPrintStatus) {
		try {
			TicketItem newTicketItem = (TicketItem) CopyUtil.deepCopy(this);
			boolean isPrintedToKitchen = needToCheeckPrintStatus && newTicketItem.isPrintedToKitchen();
			newTicketItem.setId(null);
			newTicketItem.setCreateDate(StoreDAO.getServerTimestamp());
			newTicketItem.setVersion(0);
			newTicketItem.setPaid(Boolean.FALSE);
			newTicketItem.setPrintedToKitchen(isPrintedToKitchen);
			newTicketItem.setInventoryAdjustQty(0.0);
			newTicketItem.setCloudSynced(false);
			newTicketItem.setHasSyncError(false);
			newTicketItem.setKitchenStatusValue(KitchenStatus.WAITING);
			newTicketItem.setSeat(null);
			if (newTicketItem.getSizeModifier() != null) {
				newTicketItem.getSizeModifier().setId(null);
				newTicketItem.getSizeModifier().setPrintedToKitchen(isPrintedToKitchen);
			}

			cloneModifiers(this, newTicketItem, isPrintedToKitchen);

			List<TicketItemCookingInstruction> cookingInstructions = newTicketItem.getCookingInstructions();
			if (cookingInstructions != null) {
				for (TicketItemCookingInstruction cookingInstruction : cookingInstructions) {
					if (cookingInstruction == null) {
						continue;
					}
					cookingInstruction.setPrintedToKitchen(isPrintedToKitchen);
					cookingInstruction.setKitchenStatusValue(KitchenStatus.WAITING);
				}
			}

			newTicketItem.setComboItems(null);
			if (isComboItem()) {
				List<TicketItem> comboItems = this.getComboItems();
				if (comboItems != null) {
					for (TicketItem ticketItem : comboItems) {
						TicketItem comboItem = (TicketItem) SerializationUtils.clone(ticketItem);
						comboItem.setId(null);
						comboItem.setCreateDate(StoreDAO.getServerTimestamp());
						comboItem.setParentTicketItem(newTicketItem);
						comboItem.setComboItems(null);
						if (comboItem.getSizeModifier() != null) {
							comboItem.getSizeModifier().setId(null);
							comboItem.getSizeModifier().setPrintedToKitchen(isPrintedToKitchen);
						}
						cloneModifiers(ticketItem, comboItem, isPrintedToKitchen);
						newTicketItem.addTocomboItems(comboItem);
					}
				}
			}
			return newTicketItem;
		} catch (PosException e) {
			throw e;
		} catch (Exception e) {
			throw new RuntimeException(e);
		}
	}

	private void cloneModifiers(TicketItem originalTicketItem, TicketItem newTicketItem, boolean isPrintedToKitchen) {
		newTicketItem.setTicketItemModifiers(null);
		List<TicketItemModifier> ticketItemModifiers = originalTicketItem.getTicketItemModifiers();
		if (ticketItemModifiers != null && ticketItemModifiers.size() > 0) {
			for (TicketItemModifier ticketItemModifier : ticketItemModifiers) {
				TicketItemModifier newTicketItemModifier = (TicketItemModifier) SerializationUtils.clone(ticketItemModifier);
				newTicketItemModifier.setId(null);
				newTicketItemModifier.setTicketItem(newTicketItem);
				newTicketItemModifier.setPrintedToKitchen(isPrintedToKitchen);
				newTicketItemModifier.setKitchenStatusValue(KitchenStatus.WAITING);
				newTicketItem.addToticketItemModifiers(newTicketItemModifier);
			}
		}
	}

	public void setTaxes(List<TicketItemTax> taxes) {
		this.taxes = taxes;
		buildTaxes();
	}

	public List<TicketItemTax> getTaxes() {
		if (taxes == null) {
			taxes = new ArrayList<>();

			String property = super.getTaxesProperty();
			if (StringUtils.isNotEmpty(property)) {
				JsonReader jsonParser = Json.createReader(new StringReader(property));
				JsonArray jsonArray = jsonParser.readArray();
				jsonParser.close();
				for (int i = 0; i < jsonArray.size(); i++) {
					JsonObject jsonObject = (JsonObject) jsonArray.get(i);
					TicketItemTax tit = new TicketItemTax();
					tit.setId(JsonUtil.getString(jsonObject, TicketItemTax.PROP_ID));
					tit.setName(JsonUtil.getString(jsonObject, TicketItemTax.PROP_NAME));
					tit.setRate(JsonUtil.getDouble(jsonObject, TicketItemTax.PROP_RATE));
					tit.setTaxAmount(JsonUtil.getDouble(jsonObject, TicketItemTax.PROP_TAX_AMOUNT));
					taxes.add(tit);
				}
			}
		}
		return taxes;
	}

	public void addTotaxes(TicketItemTax tax) {
		taxes = getTaxes();
		if (taxes == null) {
			taxes = new ArrayList<TicketItemTax>();
		}
		taxes.add(tax);
	}

	public void buildTaxes() {
		if (taxes == null || taxes.isEmpty()) {
			setTaxesProperty(null);
			return;
		}
		JSONArray jsonArray = new JSONArray();
		for (TicketItemTax ticketItemTax : taxes) {
			jsonArray.put(ticketItemTax.toJson());
		}
		setTaxesProperty(jsonArray.toString());
	}

	public void setDiscounts(List<TicketItemDiscount> discounts) {
		this.discounts = discounts;
	}

	public List<TicketItemDiscount> getDiscounts() {
		if (discounts == null) {
			discounts = new ArrayList<>();

			String property = super.getDiscountsProperty();
			if (StringUtils.isNotEmpty(property)) {
				JSONArray jsonArray = new JSONArray(property);
				for (int i = 0; i < jsonArray.length(); i++) {
					JSONObject jsonObject = jsonArray.getJSONObject(i);
					TicketItemDiscount tid = new TicketItemDiscount();
					tid.setTicketItem(this);
					tid.setDiscountId(OrgJsonUtil.getString(jsonObject, TicketItemDiscount.PROP_DISCOUNT_ID));
					tid.setName(OrgJsonUtil.getString(jsonObject, TicketItemDiscount.PROP_NAME));
					tid.setType(OrgJsonUtil.getInt(jsonObject, TicketItemDiscount.PROP_TYPE));
					tid.setAutoApply(OrgJsonUtil.getBoolean(jsonObject, TicketItemDiscount.PROP_AUTO_APPLY));
					tid.setApplyIfDivisible(OrgJsonUtil.getBoolean(jsonObject, TicketItemDiscount.PROP_APPLY_IF_DIVISIBLE));
					tid.setCouponQuantity(OrgJsonUtil.getDouble(jsonObject, TicketItemDiscount.PROP_COUPON_QUANTITY));
					tid.setMinimumAmount(OrgJsonUtil.getDouble(jsonObject, TicketItemDiscount.PROP_MINIMUM_AMOUNT));
					tid.setValue(OrgJsonUtil.getDouble(jsonObject, TicketItemDiscount.PROP_VALUE));
					tid.setAmount(OrgJsonUtil.getDouble(jsonObject, TicketItemDiscount.PROP_AMOUNT));
					tid.setAmountWithoutModifiers(OrgJsonUtil.getDouble(jsonObject, "amountWithoutModifier")); //$NON-NLS-1$
					tid.setDataVersion(OrgJsonUtil.getInt(jsonObject, "dataVersion")); //$NON-NLS-1$
					discounts.add(tid);
				}
			}
		}
		return discounts;
	}

	public void addTodiscounts(TicketItemDiscount discount) {
		discounts = getDiscounts();
		if (discounts == null) {
			discounts = new ArrayList<>();
		}
		discounts.add(discount);
		ActionHistoryDAO.addTicketItemDiscountAddedActionHistory(this.getTicket(), discount);
	}

	public void buildDiscounts() {
		if (discounts == null || discounts.isEmpty()) {
			setDiscountsProperty(null);
			return;
		}
		JSONArray jsonArray = new JSONArray();
		for (TicketItemDiscount ticketItemDiscount : discounts) {
			jsonArray.put(ticketItemDiscount.toJson());
		}
		setDiscountsProperty(jsonArray.toString());
	}

	public int getTableRowNum() {
		return tableRowNum;
	}

	public void setTableRowNum(int tableRowNum) {
		this.tableRowNum = tableRowNum;
	}

	@Override
	public boolean canAddCookingInstruction() {
		if (isPrintedToKitchen())
			return false;

		return true;
	}

	@Override
	public String toString() {
		return "name: " + getName() + " id: " + getId(); //$NON-NLS-1$ //$NON-NLS-2$
	}

	public void setCookingInstructions(java.util.List<TicketItemCookingInstruction> cookingInstructions) {
		this.cookingInstructions = cookingInstructions;
		buildCoookingInstructions();
	}

	public void addCookingInstructions(List<TicketItemCookingInstruction> instructions) {
		cookingInstructions = getCookingInstructions();
		if (cookingInstructions == null) {
			cookingInstructions = new ArrayList<TicketItemCookingInstruction>(2);
		}
		for (TicketItemCookingInstruction ticketItemCookingInstruction : instructions) {
			addCookingInstruction(ticketItemCookingInstruction);
		}
	}

	public void addCookingInstruction(TicketItemCookingInstruction cookingInstruction) {
		cookingInstructions = getCookingInstructions();
		if (cookingInstructions == null) {
			cookingInstructions = new ArrayList<>();
			return;
		}
		cookingInstructions.add(cookingInstruction);
		buildCoookingInstructions();
	}

	public void buildCoookingInstructions() {
		if (cookingInstructions == null || cookingInstructions.isEmpty()) {
			setCookingInstructionsProperty(null);
			return;
		}
		JSONArray jsonArray = new JSONArray();
		for (TicketItemCookingInstruction cookingIns : cookingInstructions) {
			jsonArray.put(cookingIns.toJson());
		}
		setCookingInstructionsProperty(jsonArray.toString());
	}

	public void removeCookingInstruction(TicketItemCookingInstruction itemCookingInstruction) {
		List<TicketItemCookingInstruction> cookingInstructions2 = getCookingInstructions();
		if (cookingInstructions2 == null || cookingInstructions2.size() == 0) {
			return;
		}
		for (Iterator<TicketItemCookingInstruction> iterator = cookingInstructions2.iterator(); iterator.hasNext();) {
			TicketItemCookingInstruction ticketItemCookingInstruction = iterator.next();
			if (ticketItemCookingInstruction.getTableRowNum() == itemCookingInstruction.getTableRowNum()) {
				iterator.remove();
				buildCoookingInstructions();
				return;
			}
		}
	}

	public TicketItemModifier findTicketItemModifierFor(MenuModifier menuModifier) {
		List<TicketItemModifier> modifiers = getTicketItemModifiers();
		if (modifiers == null) {
			return null;
		}
		for (TicketItemModifier ticketItemModifier : modifiers) {
			String itemId = ticketItemModifier.getItemId();
			if (itemId != null && itemId.equals(menuModifier.getId())) {
				return ticketItemModifier;
			}
		}
		return null;
	}

	public TicketItemModifier addTicketItemModifier(MenuModifier menuModifier, int modifierType, OrderType type, Multiplier multiplier) {
		return addTicketItemModifier(menuModifier, modifierType, type, multiplier, 1);
	}

	public TicketItemModifier addTicketItemModifier(MenuModifier menuModifier, int modifierType, OrderType type, Multiplier multiplier, double quantity) {
		return TicketItemCalcFactory.getCalc(this).addTicketItemModifier(this, menuModifier, modifierType, type, multiplier, quantity);
	}

	//	public TicketItemModifier addTicketItemModifier(MenuModifier menuModifier, boolean addOn) {
	//		TicketItemModifier ticketItemModifier = new TicketItemModifier();
	//		//TODO: SET TAX INCLUDE FIELD
	//		ticketItemModifier.setTaxIncluded(Application.getInstance().isPriceIncludesTax());
	//		ticketItemModifier.setItemId(menuModifier.getId());
	//		MenuItemModifierSpec menuItemModifierSpec = menuModifier.getMenuItemModifierSpec();
	//		if (menuItemModifierSpec != null) {
	//			ticketItemModifier.setGroupId(menuItemModifierSpec.getId());
	//		}
	//		ticketItemModifier.setItemCount(1);
	//		ticketItemModifier.setName(menuModifier.getDisplayName());
	//
	//		if (addOn) {
	//			ticketItemModifier.setUnitPrice(menuModifier.getExtraPrice());
	//			ticketItemModifier.setModifierType(TicketItemModifier.EXTRA_MODIFIER);
	//		}
	//		else {
	//			ticketItemModifier.setUnitPrice(menuModifier.getPrice());
	//			ticketItemModifier.setModifierType(TicketItemModifier.NORMAL_MODIFIER);
	//		}
	//		ticketItemModifier.setTaxes(menuModifier.getTaxByOrderType(ticketItemModifier.getTicketItem().getTicket().getOrderType(), this));
	//		ticketItemModifier.setShouldPrintToKitchen(menuModifier.isShouldPrintToKitchen());
	//		ticketItemModifier.setTicketItem(this);
	//
	//		addToticketItemModifiers(ticketItemModifier);
	//
	//		return ticketItemModifier;
	//	}

	//	public void updateModifiersUnitPrice(double defaultSellPortion) {
	//		List<TicketItemModifier> ticketItemModifiers = getTicketItemModifiers();
	//		if (ticketItemModifiers != null) {
	//			for (TicketItemModifier ticketItemModifier : ticketItemModifiers) {
	//				if (!ticketItemModifier.isInfoOnly()) {
	//					ticketItemModifier.setUnitPrice(ticketItemModifier.getUnitPrice() * defaultSellPortion / 100);
	//				}
	//			}
	//		}
	//	}

	public void updateModifiersUnitPriceByGroup(MenuItemModifierSpec modifierSpec) {
		TicketItemCalcFactory.getCalc(this).updateModifiersUnitPriceByGroup(this, modifierSpec);
	}

	@Deprecated
	public void updateModifiersUnitPriceByGroup(ModifierGroup group) {
		updateModifiersUnitPriceByGroup(group, group.getId());
	}

	@Deprecated
	public void updateModifiersUnitPriceByGroup(ModifierGroup group, String groupId) {
		updateModifiersUnitPriceByGroup(null, group, groupId);
	}

	@Deprecated
	public void updateModifiersUnitPriceByGroup(MenuItemModifierSpec spec, ModifierGroup group, String groupId) {
		List<TicketItemModifier> ticketItemModifiers = getTicketItemModifiers();
		List<TicketItemModifier> groupModifiers = new ArrayList<>();
		if (ticketItemModifiers != null) {
			double numOfModifiers = 0;
			for (TicketItemModifier ticketItemModifier : ticketItemModifiers) {
				if (!ticketItemModifier.isInfoOnly() && ticketItemModifier.getGroupId().equals(groupId)) {
					ticketItemModifier.setMultiplierName(null);
					groupModifiers.add(ticketItemModifier);
					numOfModifiers += ticketItemModifier.getItemQuantity();
				}
			}
			if (numOfModifiers == 0) {
				return;
			}
			Double groupPrice = spec != null ? spec.getPrice((int) numOfModifiers) : group.getPrice((int) numOfModifiers);
			if (groupPrice != null) {
				updateModifiersUnitPriceByGroup(groupModifiers, numOfModifiers, groupPrice);
			}
			else {
				updateTicketItemModifiersPrice(groupModifiers, spec, group);
			}
		}
	}

	private void updateModifiersUnitPriceByGroup(List<TicketItemModifier> groupModifiers, double numOfModifiers, Double groupPrice) {
		if (groupPrice == null || numOfModifiers == 0) {
			return;
		}
		double priceRemainder = groupPrice;
		for (TicketItemModifier ticketItemModifier : groupModifiers) {
			Double modifierPrice = ticketItemModifier.getMultiplierPrice(groupPrice / (int) numOfModifiers);
			modifierPrice = NumberUtil.round(modifierPrice);
			ticketItemModifier.setUnitPrice(modifierPrice);
			priceRemainder -= modifierPrice;
		}
		TicketItemModifier lastTicketItemModifier = groupModifiers.get(groupModifiers.size() - 1);
		lastTicketItemModifier.setUnitPrice(lastTicketItemModifier.getUnitPrice() + priceRemainder);
	}

	public void updateTicketItemModifiersPrice(List<TicketItemModifier> groupModifiers, MenuItemModifierSpec spec, ModifierGroup group) {
		Map<String, Multiplier> multiplierMap = new HashMap<String, Multiplier>();
		for (TicketItemModifier ticketItemModifier : groupModifiers) {
			String multiplierName = ticketItemModifier.getMultiplierName();
			Multiplier multiplier = null;
			if (StringUtils.isNotBlank(multiplierName)) {
				multiplier = multiplierMap.get(multiplierName);
				if (multiplier == null) {
					multiplier = DataProvider.get().getMultiplierById(multiplierName);
					multiplierMap.put(multiplierName, multiplier);
				}
			}
			MenuModifier menuModifier = ticketItemModifier.getMenuModifier();
			if (menuModifier == null) {
				menuModifier = MenuModifierDAO.getInstance().get(ticketItemModifier.getItemId());
			}
			if (menuModifier != null) {
				ticketItemModifier.setUnitPrice(menuModifier.getPriceForMultiplier(multiplier));
			}
		}
	}

	public boolean contains(TicketItemModifier ticketItemModifier) {
		List<TicketItemModifier> ticketItemModifiers = getTicketItemModifiers();
		int count = 0;
		if (ticketItemModifiers != null) {
			for (TicketItemModifier ticketItemModifier2 : ticketItemModifiers) {
				if (!ticketItemModifier2.isInfoOnly()) {
					if (ticketItemModifier.getName().trim().equals(ticketItemModifier2.getName().trim())) {
						count++;
					}
				}
			}
		}
		return (count > 1) ? true : false;
	}

	public TicketItemModifier removeTicketItemModifier(TicketItemModifier ticketItemModifier) {
		List<TicketItemModifier> ticketItemModifiers = getTicketItemModifiers();
		if (ticketItemModifiers == null)
			return ticketItemModifier;

		for (Iterator<TicketItemModifier> iter = ticketItemModifiers.iterator(); iter.hasNext();) {
			TicketItemModifier oldTicketItemModifier = iter.next();
			if (oldTicketItemModifier.getItemId().equals(ticketItemModifier.getItemId())
					&& oldTicketItemModifier.getModifierType() == ticketItemModifier.getModifierType()) {
				iter.remove();
				return oldTicketItemModifier;
			}
		}
		return ticketItemModifier;
	}

	public TicketItemModifier removeTicketItemModifier(int modifierIndex) {
		List<TicketItemModifier> ticketItemModifiers = getTicketItemModifiers();
		if (ticketItemModifiers == null) {
			return null;
		}
		TicketItemModifier ticketItemModifier = ticketItemModifiers.get(modifierIndex);
		ticketItemModifiers.remove(modifierIndex);
		return ticketItemModifier;
	}

	public void calculatePrice() {
		TicketItemCalcFactory.getCalc(this).calculatePrice(this);
	}

	public boolean isMergable(TicketItem otherItem, boolean merge) {
		if (this.isTreatAsSeat() || otherItem.isTreatAsSeat()) {
			return false;
		}
		if (this.isFractionalUnit() || otherItem.isFractionalUnit()) {
			return false;
		}
		if ((this.getQuantity() > 0 && otherItem.getQuantity() < 0) || (this.getQuantity() < 0 && otherItem.getQuantity() > 0)) {
			return false;
		}
		if (StringUtils.isEmpty(this.getMenuItemId()) || StringUtils.isEmpty(otherItem.getMenuItemId())) {
			return false;
		}
		if (this.hasCookingInstructions() || otherItem.hasCookingInstructions()) {
			return false;
		}
		if (this.isReturned() || otherItem.isReturned()) {
			return false;
		}
		if (!this.getMenuItemId().equals(otherItem.getMenuItemId())) {
			return false;
		}
		if (!this.getUnitPrice().equals(otherItem.getUnitPrice())) {
			return false;
		}
		if (this.isComboItem() || otherItem.isComboItem()) {
			return false;
		}
		if (!this.getSeatNumber().equals(otherItem.getSeatNumber())) {
			return false;
		}
		if (!this.isHasModifiers() && !otherItem.isHasModifiers()) {
			return true;
		}
		if (!isMergableModifiers(getTicketItemModifiers(), otherItem.getTicketItemModifiers(), merge)) {
			return false;
		}
		return true;
	}

	public boolean isMergableModifiers(List<TicketItemModifier> thisModifiers, List<TicketItemModifier> thatModifiers, boolean merge) {
		if (thatModifiers == null || thisModifiers == null) {
			return true;
		}
		if (thisModifiers.size() != thatModifiers.size()) {
			return false;
		}

		Comparator<TicketItemModifier> comparator = new Comparator<TicketItemModifier>() {
			@Override
			public int compare(TicketItemModifier o1, TicketItemModifier o2) {
				return o1.getItemId().compareTo(o2.getItemId());
			}
		};

		Collections.sort(thisModifiers, comparator);
		Collections.sort(thatModifiers, comparator);

		Iterator<TicketItemModifier> thisIterator = thisModifiers.iterator();
		Iterator<TicketItemModifier> thatIterator = thatModifiers.iterator();

		while (thisIterator.hasNext()) {
			TicketItemModifier next1 = thisIterator.next();
			TicketItemModifier next2 = thatIterator.next();

			if (comparator.compare(next1, next2) != 0) {
				return false;
			}

			if (merge) {
				next1.merge(next2);
			}
		}
		return true;
	}

	public void merge(TicketItem otherItem) {
		if (!this.isHasModifiers() && !otherItem.isHasModifiers()) {
			this.setQuantity(this.getQuantity() + otherItem.getQuantity());
			return;
		}
		if (isMergable(otherItem, false)) {
			this.setQuantity(this.getQuantity() + otherItem.getQuantity());
		}
	}

	public boolean hasCookingInstructions() {
		return this.getCookingInstructions() != null && this.getCookingInstructions().size() > 0;
	}

	//	public BigDecimal calculateDiscount(final BigDecimal subtotalAmount) {
	//		BigDecimal discount = convertToBigDecimal(0);
	//
	//		List<TicketItemDiscount> discounts = getDiscounts();
	//		if (discounts != null) {
	//			for (TicketItemDiscount ticketItemDiscount : discounts) {
	//				if (ticketItemDiscount.getType() == Discount.DISCOUNT_TYPE_REPRICE) {
	//					continue;
	//				}
	//				else if (ticketItemDiscount.getType() == Discount.DISCOUNT_TYPE_PERCENTAGE) {
	//					discount = discount.add(convertToBigDecimal(ticketItemDiscount.calculateDiscount(subtotalAmount.doubleValue() - discount.doubleValue())));
	//				}
	//				else if (ticketItemDiscount.getType() == Discount.DISCOUNT_TYPE_AMOUNT) {
	//					double discountAmount = subtotalAmount.doubleValue() == 0.0 ? 0.0
	//							: Math.abs((ticketItemDiscount.getValue() * 100.0) / subtotalAmount.doubleValue());
	//					TicketItemDiscount percentDiscount = new TicketItemDiscount();
	//					percentDiscount.setType(Discount.DISCOUNT_TYPE_PERCENTAGE);
	//					percentDiscount.setMinimumAmount(ticketItemDiscount.getMinimumAmount());
	//					percentDiscount.setValue(discountAmount);
	//					percentDiscount.setTicketItem(this);
	//					percentDiscount.setCouponQuantity(ticketItemDiscount.getCouponQuantity());
	//					BigDecimal convertToBigDecimal = convertToBigDecimal(percentDiscount.calculateDiscount(subtotalAmount.doubleValue()));
	//					discount = discount.add(convertToBigDecimal);
	//					ticketItemDiscount.setAmount(convertToBigDecimal.doubleValue());
	//				}
	//				//discount += ticketItemDiscount.calculateDiscount(subtotalAmount);
	//			}
	//		}
	//		buildDiscounts();
	//		if (discount.compareTo(subtotalAmount.abs()) > 0)
	//			return subtotalAmount;
	//		return round(discount);
	//	}
	//
	//	public double calculateDiscountV2(final double subtotalAmount) {
	//		double discount = 0.0;
	//
	//		List<TicketItemDiscount> discounts = getDiscounts();
	//		if (discounts != null) {
	//			for (TicketItemDiscount ticketItemDiscount : discounts) {
	//				if (ticketItemDiscount.getType() == Discount.DISCOUNT_TYPE_REPRICE) {
	//					continue;
	//				}
	//				else if (ticketItemDiscount.getType() == Discount.DISCOUNT_TYPE_PERCENTAGE) {
	//					discount += NumberUtil.round(ticketItemDiscount.calculateDiscount(Math.abs(subtotalAmount) - discount));
	//				}
	//				else if (ticketItemDiscount.getType() == Discount.DISCOUNT_TYPE_AMOUNT) {
	//					Double discountedAmount = Math.abs(subtotalAmount) - discount;
	//					double discountAmount = discountedAmount.doubleValue() == 0.0 ? 0.0
	//							: Math.abs((ticketItemDiscount.getValue() * 100.0) / discountedAmount.doubleValue());
	//					TicketItemDiscount percentDiscount = new TicketItemDiscount();
	//					percentDiscount.setType(Discount.DISCOUNT_TYPE_PERCENTAGE);
	//					percentDiscount.setMinimumAmount(ticketItemDiscount.getMinimumAmount());
	//					percentDiscount.setValue(discountAmount);
	//					percentDiscount.setTicketItem(this);
	//					percentDiscount.setCouponQuantity(ticketItemDiscount.getCouponQuantity());
	//					percentDiscount.setApplyIfDivisible(Boolean.FALSE);
	//					double ticketItemDiscountAmount = percentDiscount.calculateDiscount(discountedAmount.doubleValue());
	//
	//					if (ticketItemDiscountAmount >= discountedAmount) {
	//						ticketItemDiscount.setAmount(discountedAmount);
	//					}
	//					else {
	//						ticketItemDiscount.setAmount(ticketItemDiscountAmount);
	//					}
	//					discount += ticketItemDiscountAmount;
	//				}
	//			}
	//		}
	//
	//		if (subtotalAmount < 0) {
	//			return -1 * Math.abs(discount);
	//		}
	//
	//		if (discount > Math.abs(subtotalAmount))
	//			return subtotalAmount;
	//		return discount;
	//	}
	//
	//	public double getAmountByType(TicketItemDiscount discount) {
	//
	//		switch (discount.getType()) {
	//			case Discount.DISCOUNT_TYPE_AMOUNT:
	//				return discount.getValue();
	//
	//			case Discount.DISCOUNT_TYPE_PERCENTAGE:
	//				return (discount.getValue() * getUnitPrice()) / 100;
	//
	//			default:
	//				break;
	//		}
	//
	//		return 0;
	//	}

	public double getTotalTaxRate() {
		List<TicketItemTax> ticketItemTaxes = getTaxes();
		if (ticketItemTaxes == null || ticketItemTaxes.isEmpty())
			return 0;

		double totalTaxRate = 0;
		for (TicketItemTax tax : ticketItemTaxes) {
			totalTaxRate += tax.getRate();
		}
		return totalTaxRate;
	}

	@Override
	public String getNameDisplay() {
		return getNameDisplay(getName());
	}

	public String getNameDisplay(String name) {
		return getNameDisplay(name, false);
	}

	public String getNameDisplay(String name, boolean includeComboModifiers) {
		return getNameDisplay(name, includeComboModifiers, true);
	}

	public String getNameDisplay(String name, boolean includeComboModifiers, boolean markVoidedItem) {
		return getNameDisplay(name, includeComboModifiers, markVoidedItem, true);
	}

	public String getNameDisplay(String name, boolean includeComboModifiers, boolean markVoidedItem, boolean includeQuantity) {
		if (isTreatAsSeat()) {
			return name;
		}

		String displayName = ""; //$NON-NLS-1$

		if (getQuantity() < 0) {
			if (isVoided() && !isItemReturned()) {
				if (markVoidedItem) {
					displayName += "*Voided* "; //$NON-NLS-1$
					displayName += getItemQuantityDisplay(true);

				}
				else {
					displayName += getItemQuantityDisplay(true);
				}
			}
			else {
				displayName += getItemQuantityDisplay();
			}
		}
		else if (includeQuantity) {
			displayName += getItemQuantityDisplay();
		}
		if (isComboItem()) {
			List<TicketItem> comboItems = getComboItems();
			if (comboItems != null && !comboItems.isEmpty()) {
				displayName += name;
				displayName += "<br/>"; //$NON-NLS-1$
				for (Iterator iterator = comboItems.iterator(); iterator.hasNext();) {
					TicketItem item = (TicketItem) iterator.next();
					displayName += "&nbsp;&nbsp; #" + NumberUtil.trimDecilamIfNotNeeded(item.getQuantity() / getQuantity(), true) + " " + item.getName(); //$NON-NLS-1$ //$NON-NLS-2$
					List<TicketItemModifier> ticketItemModifiers = item.getTicketItemModifiers();
					if (iterator.hasNext() || ticketItemModifiers != null) {
						displayName += "<br/>"; //$NON-NLS-1$
					}
					if (ticketItemModifiers != null && !ticketItemModifiers.isEmpty()) {
						for (Iterator iterator2 = ticketItemModifiers.iterator(); iterator2.hasNext();) {
							TicketItemModifier modifier = (TicketItemModifier) iterator2.next();
							displayName += modifier.getNameDisplay();
							if (iterator2.hasNext()) {
								displayName += "<br/>"; //$NON-NLS-1$
							}
							else if (iterator.hasNext()) {
								displayName += "<br/>"; //$NON-NLS-1$
							}
						}
					}
				}
				//displayName += "\n"; //$NON-NLS-1$
				if (includeComboModifiers) {
					List<TicketItemModifier> ticketItemModifiers = getTicketItemModifiers();
					if (ticketItemModifiers != null && !ticketItemModifiers.isEmpty()) {
						displayName += "<br/>"; //$NON-NLS-1$
						for (Iterator itr = ticketItemModifiers.iterator(); itr.hasNext();) {
							TicketItemModifier modifier = (TicketItemModifier) itr.next();
							displayName += modifier.getNameDisplay();
							if (itr.hasNext()) {
								displayName += "<br/>"; //$NON-NLS-1$
							}
						}
					}
				}

				return displayName;
			}
		}
		else if (getSizeModifier() != null) {
			displayName += getSizeModifier().getNameDisplay().replaceAll("&nbsp;&nbsp;&nbsp;", "") + " " + getName(); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
			return displayName;
		}

		return displayName += name;
	}

	@Override
	public String getUnitPriceDisplay() {
		if (isTreatAsSeat())
			return null;

		return NumberUtil.formatNumberAcceptNegative(getUnitPrice());
	}

	@Override
	public String getItemQuantityDisplay() {
		return getItemQuantityDisplay(false);
	}

	public String getItemQuantityDisplay(boolean itemVoided) {
		if (isTreatAsSeat())
			return ""; //$NON-NLS-1$
		String code = getProperty("unit_Code", ""); //$NON-NLS-1$ //$NON-NLS-2$
		String name = getProperty("unit_display_name", ""); //$NON-NLS-1$ //$NON-NLS-2$
		String unitName = StringUtils.isBlank(name) ? code : name;
		if (StringUtils.isBlank(unitName)) {
			unitName = getUnitName();
		}
		Double quantity = itemVoided ? Math.abs(getQuantity()) : getQuantity();
		if (!isFractionalUnit() && quantity == 1 && ("ea".equals(unitName) || ("Each".equals(unitName)))) { //$NON-NLS-1$ //$NON-NLS-2$
			return ""; //$NON-NLS-1$
		}

		String itemQuantity = NumberUtil.trimDecilamIfNotNeeded(quantity, true);
		if (unitName.equals("") || "ea".equals(unitName) || "pc".equals(unitName)) { //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
			return itemQuantity + "x "; //$NON-NLS-1$
		}
		return itemQuantity + unitName + " "; //$NON-NLS-1$
	}

	public String getUnitNameDisplay() {
		String code = getProperty("unit_Code", ""); //$NON-NLS-1$ //$NON-NLS-2$
		String displayName = getProperty("unit_display_name", ""); //$NON-NLS-1$ //$NON-NLS-2$
		String unitName = StringUtils.isBlank(displayName) ? code : displayName;
		if (StringUtils.isBlank(unitName)) {
			if (getUnitName() == null) {
				return ""; //$NON-NLS-1$
			}
			unitName = getUnitName();
		}
		if ("ea".equalsIgnoreCase(unitName) || "pc".equals(unitName) || ("Each".equalsIgnoreCase(unitName))) { //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
			return ""; //$NON-NLS-1$
		}
		return unitName;
	}

	@Override
	public String getTaxAmountWithoutModifiersDisplay() {
		return NumberUtil.formatNumberAcceptNegative(getTaxAmountWithoutModifiers());
	}

	@Override
	public String getTotalAmountWithoutModifiersDisplay() {
		return NumberUtil.formatNumberAcceptNegative(getTotalAmountWithoutModifiers());
	}

	@Override
	public String getSubTotalAmountWithoutModifiersDisplay() {
		if (isTreatAsSeat())
			return null;
		return NumberUtil.formatNumberAcceptNegative(getSubtotalAmountWithoutModifiers());
	}

	@Override
	public String getItemCode() {
		return String.valueOf(getMenuItemId());
	}

	public List<Printer> getPrinters(OrderType orderType) {
		PrinterGroup printerGroup = getPrinterGroup();

		List<Printer> printerAll = new ArrayList<Printer>();
		PosPrinters printers = DataProvider.get().getPrinters();
		List<Printer> kitchenPrinters = printers.getKitchenPrinters();
		printerAll.addAll(printers.getStickerPrinters());

		if (printerGroup == null && (kitchenPrinters == null || kitchenPrinters.isEmpty()) && printers.isPrintToKds()) {
			printerAll.add(VirtualPrinter.getKdsPrinter());
		}

		if (printerGroup == null && kitchenPrinters != null && kitchenPrinters.size() > 0) {
			Printer printer = kitchenPrinters.get(0);
			if (StringUtils.isBlank(printer.getDeviceName())) {
				printerAll.add(VirtualPrinter.getKdsPrinter());
			}
			else {
				printerAll.add(printer);
			}
			return printerAll;
		}

		if (printerGroup != null) {
			List<String> printerNames = printerGroup.getPrinterNames();
			for (Printer printer : kitchenPrinters) {
				VirtualPrinter virtualPrinter = printer.getVirtualPrinter();
				if (StringUtils.isBlank(printer.getDeviceName())) {
					continue;
				}
				if (printerNames.contains(virtualPrinter.getName())) {
					printerAll.add(printer);
				}
			}
		}
		if (printerAll.isEmpty() && printers.isPrintToKds()) {
			printerAll.add(VirtualPrinter.getKdsPrinter());
		}

		return printerAll;
	}

	@Override
	public boolean canAddDiscount() {
		return true;
	}

	@Override
	public boolean canVoid() {
		return true;
	}

	@Override
	public boolean canAddAdOn() {
		return true;
	}

	@XmlTransient
	public MenuItem getMenuItem() {
		if (menuItem == null) {
			String itemId = getMenuItemId();
			if (StringUtils.isEmpty(itemId)) {
				return null;
			}
			menuItem = MenuItemDAO.getInstance().loadInitialized(itemId);
		}

		return menuItem;
	}

	public void setMenuItem(MenuItem menuItem) {
		this.menuItem = menuItem;
		//		setMenuItemId(menuItem.getId());
		String menuItemId = null;
		if (menuItem != null) {
			menuItemId = menuItem.getId();
		}
		super.setMenuItemId(menuItemId);
	}

	@Override
	public LabWorkStatus getLabWorkStatusValue() {
		return LabWorkStatus.fromString(super.getKitchenStatus());
	}

	public void setLabWorkStatusValue(LabWorkStatus labWorkStatus) {
		super.setKitchenStatus(labWorkStatus.name());
		addProcessedStatus(labWorkStatus);
	}

	@Override
	public KitchenStatus getKitchenStatusValue() {
		return KitchenStatus.fromString(super.getKitchenStatus());
	}

	public void setKitchenStatusValue(KitchenStatus kitchenStatus) {
		super.setKitchenStatus(kitchenStatus.name());
	}

	/**
	 * Set kitchen status of this item, modifier of this item and cooking instructions of this item.
	 * 
	 * @param kitchenStatus
	 */
	public void setKitchenStatusValueWithChildren(KitchenStatus kitchenStatus) {
		super.setKitchenStatus(kitchenStatus.name());
		List<TicketItemModifier> ticketItemModifiers = getTicketItemModifiers();
		if (ticketItemModifiers != null) {
			for (TicketItemModifier ticketItemModifier : ticketItemModifiers) {
				ticketItemModifier.setKitchenStatusValue(kitchenStatus);
			}
		}
		List<TicketItemCookingInstruction> cookingInstructions = getCookingInstructions();
		if (cookingInstructions != null) {
			for (TicketItemCookingInstruction ticketItemCookingInstruction : cookingInstructions) {
				ticketItemCookingInstruction.setKitchenStatusValue(kitchenStatus);
			}
			buildCoookingInstructions();
		}
	}

	@Override
	public String getUnitName() {
		if (super.getUnitName() == null) {
			return ""; //$NON-NLS-1$
		}
		return super.getUnitName();
	}

	public Double getQuantityToShip() {
		return quantityToShip == null ? 0.0 : quantityToShip;
	}

	public void setQuantityToShip(Double shipQuantity) {
		this.quantityToShip = shipQuantity;
	}

	public void setQuantity(double quantity, boolean updateComboItemsQuantity) {
		if (isComboItem() && updateComboItemsQuantity) {
			List<TicketItem> comboticketItems = getComboItems();
			if (comboticketItems != null && comboticketItems.size() > 0) {
				double ticketItemQuantity = getQuantity();
				double adjustQuantity = quantity - ticketItemQuantity;
				for (TicketItem comboTicketItem : comboticketItems) {
					comboTicketItem.setQuantity(comboTicketItem.getQuantity() + (adjustQuantity * comboTicketItem.getQuantity() / ticketItemQuantity));
				}
			}
		}
		setQuantity(quantity);
	}

	@Override
	public String getSubTotalAmountDisplay() {
		if (isTreatAsSeat())
			return null;
		if (isVoided() && !isItemReturned()) {
			return "(" + NumberUtil.format(Math.abs(getSubtotalAmount())) + ")"; //$NON-NLS-1$ //$NON-NLS-2$
		}
		return NumberUtil.format(getSubtotalAmount());
	}

	public TicketItemModifier findTicketItemModifierFor(MenuModifier menuModifier, Multiplier multiplier) {
		List<TicketItemModifier> modifiers = getTicketItemModifiers();
		if (modifiers == null) {
			return null;
		}
		for (TicketItemModifier ticketItemModifier : modifiers) {
			String itemId = ticketItemModifier.getItemId();
			String menuItemMGId = menuModifier.getMenuItemModifierSpec() == null ? null : menuModifier.getMenuItemModifierSpec().getId();
			String ticketItemMGId = ticketItemModifier.getGroupId();
			String multiplierName = ticketItemModifier.getMultiplierName();
			boolean idIsNull = itemId == null || menuItemMGId == null || ticketItemMGId == null;
			if (idIsNull) {
				return null;
			}
			if (multiplier != null && !multiplier.getId().equals(multiplierName)) {
				return null;
			}
			if (menuItemMGId.equals(ticketItemMGId) && itemId.equals(menuModifier.getId())) {
				return ticketItemModifier;
			}
		}
		return null;
	}

	public List<TicketItemModifier> findTicketItemModifiersFor(MenuModifier menuModifier, Multiplier multiplier) {
		List<TicketItemModifier> modifiers = getTicketItemModifiers();
		List<TicketItemModifier> ticketItemModifiers = new ArrayList<>();
		if (modifiers == null) {
			return ticketItemModifiers;
		}
		for (TicketItemModifier ticketItemModifier : modifiers) {
			String itemId = ticketItemModifier.getItemId();
			if (itemId != null && multiplier != null && itemId.equals(menuModifier.getId())
					&& multiplier.getId().equals(ticketItemModifier.getMultiplierName())) {
				ticketItemModifiers.add(ticketItemModifier);
			}
		}
		return ticketItemModifiers;
	}

	public double findTicketItemModifierByMenuModifier(MenuModifier menuModifier) {
		if (menuModifier == null) {
			return 0;
		}
		double totalAmount = 0;
		List<Multiplier> multiplierList = DataProvider.get().getMultiplierList();
		if (multiplierList != null) {
			for (Multiplier multiplier : multiplierList) {
				List<TicketItemModifier> ticketItemModifiers = findTicketItemModifiersFor(menuModifier, multiplier);
				if (ticketItemModifiers != null && !ticketItemModifiers.isEmpty()) {
					for (TicketItemModifier modifier : ticketItemModifiers) {
						totalAmount += modifier.getItemQuantity();
					}
				}
			}
		}
		return totalAmount;
	}

	public TicketItemModifier findTicketItemModifierFor(MenuModifier menuModifier, String sectionName) {
		return findTicketItemModifierFor(menuModifier, sectionName, null);
	}

	public TicketItemModifier findTicketItemModifierFor(MenuModifier menuModifier, String sectionName, Multiplier multiplier) {
		List<TicketItemModifier> modifiers = getTicketItemModifiers();
		if (modifiers == null) {
			return null;
		}
		for (TicketItemModifier ticketItemModifier : modifiers) {
			String itemId = ticketItemModifier.getItemId();
			if (multiplier != null) {
				if ((itemId != null && itemId.equals(menuModifier.getId())) && (sectionName != null && sectionName.equals(ticketItemModifier.getSectionName())
						&& (multiplier != null && multiplier.getId().equals(ticketItemModifier.getMultiplierName())))) {
					return ticketItemModifier;
				}
			}

			if ((itemId != null && itemId.equals(menuModifier.getId())) && (sectionName != null && sectionName.equals(ticketItemModifier.getSectionName()))) {
				return ticketItemModifier;
			}
		}
		return null;
	}

	public int countModifierFromGroup(MenuItemModifierSpec menuItemModifierSpec) {
		return TicketItemCalcFactory.getCalc(this).countModifierFromGroup(this, menuItemModifierSpec);
	}

	public boolean requiredModifiersAdded(MenuItemModifierSpec menuItemModifierSpec) {
		return TicketItemCalcFactory.getCalc(this).requiredModifiersAdded(this, menuItemModifierSpec);
	}

	public boolean deleteTicketItemModifier(TicketItemModifier ticketItemModifierToRemove) {
		return TicketItemCalcFactory.getCalc(this).deleteTicketItemModifier(this, ticketItemModifierToRemove);
	}

	public boolean deleteTicketItemModifierByName(TicketItemModifier ticketItemModifierToRemove) {
		return TicketItemCalcFactory.getCalc(this).deleteTicketItemModifierByName(this, ticketItemModifierToRemove);
	}

	public enum PIZZA_SECTION_MODE {
		FULL(1), HALF(2), QUARTER(3);

		private final int value;

		private PIZZA_SECTION_MODE(int value) {
			this.value = value;
		}

		public int getValue() {
			return value;
		}

		public static PIZZA_SECTION_MODE from(int value) {
			if (value == 2) {
				return HALF;
			}
			if (value == 3) {
				return QUARTER;
			}
			return FULL;
		}

		@Override
		public String toString() {
			return name();
		}
	}

	public void setPizzaSectionMode(PIZZA_SECTION_MODE pizzaSectionMode) {
		setPizzaSectionModeType(pizzaSectionMode.getValue());
	}

	public PIZZA_SECTION_MODE getPizzaSectionMode() {
		return PIZZA_SECTION_MODE.from(getPizzaSectionModeType());
	}

	//	public TicketItemModifier findTicketItemModifierByPageItem(MenuModifier menuModifier) {
	//		List<TicketItemModifier> modifiers = getTicketItemModifiers();
	//		if (modifiers == null) {
	//			return null;
	//		}
	//		for (TicketItemModifier ticketItemModifier : modifiers) {
	//			String itemId = ticketItemModifier.getPageItemId();
	//			if (itemId != null && itemId.equals(menuModifier.getPageItemId())) {
	//				return ticketItemModifier;
	//			}
	//		}
	//		return null;
	//	}
	//
	//	public TicketItemModifier findTicketItemComboModifierFor(MenuModifier menuModifier) {
	//		List<TicketItemModifier> modifiers = getTicketItemModifiers();
	//		if (modifiers == null) {
	//			return null;
	//		}
	//		for (TicketItemModifier ticketItemModifier : modifiers) {
	//			String itemId = ticketItemModifier.getItemId();
	//			if (itemId != null && itemId.equals(menuModifier.getId()) && ticketItemModifier.getModifierType() == TicketItemModifier.EXTRA_MODIFIER) {
	//				return ticketItemModifier;
	//			}
	//		}
	//		return null;
	//	}
	//
	//	public TicketItemModifier removeTicketItemModifierByPageItem(TicketItemModifier ticketItemModifier) {
	//		List<TicketItemModifier> ticketItemModifiers = getTicketItemModifiers();
	//		if (ticketItemModifiers == null)
	//			return ticketItemModifier;
	//
	//		for (Iterator<TicketItemModifier> iter = ticketItemModifiers.iterator(); iter.hasNext();) {
	//			TicketItemModifier oldTicketItemModifier = iter.next();
	//			String pageItemId = oldTicketItemModifier.getPageItemId();
	//			if (pageItemId != null && pageItemId.equals(ticketItemModifier.getPageItemId())
	//					&& oldTicketItemModifier.getModifierType() == ticketItemModifier.getModifierType()) {
	//				iter.remove();
	//				return oldTicketItemModifier;
	//			}
	//		}
	//		return ticketItemModifier;
	//	}

	public void removeTicketItemDiscount(TicketItemDiscount itemDiscount) {
		List<TicketItemDiscount> discounts = getDiscounts();
		if (discounts == null) {
			return;
		}

		for (Iterator<TicketItemDiscount> iterator = discounts.iterator(); iterator.hasNext();) {
			TicketItemDiscount ticketItemDiscount = iterator.next();
			if (ticketItemDiscount.getTableRowNum() == itemDiscount.getTableRowNum()) {
				iterator.remove();
				return;
			}
		}
	}

	public boolean isInventoryAdjusted() {
		if (isTreatAsSeat()) {
			return false;
		}
		return getQuantity().doubleValue() == getInventoryAdjustQty().doubleValue();
	}

	public Boolean isIncludeVoidQuantity() {
		return includeVoidQuantity == null ? false : includeVoidQuantity;
	}

	public void setIncludeVoidQuantity(Boolean includeVoidQuantity) {
		this.includeVoidQuantity = includeVoidQuantity;
	}

	public void setVoidItem(VoidItem voidItem) {
		this.voidItem = voidItem;
		if (voidItem == null) {
			removeProperty(TicketItem.JSON_PROP_VOID_REASON);
			removeProperty(TicketItem.JSON_PROP_VOID_QUANTITY);
		}
	}

	@XmlTransient
	public VoidItem getVoidItem() {
		if (voidItem != null) {
			return voidItem;
		}
		String voidReason = getProperty(JSON_PROP_VOID_REASON);
		if (StringUtils.isBlank(voidReason)) {
			return null;
		}
		double voidQuantity = POSUtil.parseDouble(getProperty(JSON_PROP_VOID_QUANTITY));
		if (voidQuantity == 0) {
			return null;
		}
		voidItem = createVoidItem(voidReason, POSUtil.getBoolean(getProperty(JSON_PROP_WASTED)), voidQuantity);
		return voidItem;
	}

	public void markVoided(String voidReason, boolean itemWasted, double quantity, boolean returned) {
		TicketItemCalcFactory.getCalc(this).markVoided(this, voidReason, itemWasted, quantity, returned);
	}

	public void setVoidProperties(String voidReason, boolean wasted, double quantity) {
		setVoidProperties(voidReason, wasted, quantity, isPrintedToKitchen());
	}

	public void setVoidProperties(String voidReason, boolean wasted, double quantity, boolean printedToKitchen) {
		User currentUser = DataProvider.get().getCurrentUser();
		TicketItemCalcFactory.getCalc(this).setVoidProperties(this, voidReason, wasted, quantity, printedToKitchen, currentUser);
	}

	public VoidItem createVoidItem(String voidReason, boolean wasted, double quantity) {
		User currentUser = DataProvider.get().getCurrentUser();
		return TicketItemCalcFactory.getCalc(this).createVoidItem(this, voidReason, wasted, quantity, currentUser,
				currentUser == null ? null : currentUser.getActiveDrawerPullReport());
	}

	@Override
	public boolean isSaved() {
		return getId() != null;
	}

	public boolean isRefundable() {
		//		List<TicketItemDiscount> discounts = getDiscounts();
		//		if (discounts == null || discounts.isEmpty()) {
		//			return true;
		//		}
		//		for (TicketItemDiscount ticketItemDiscount : discounts) {
		//			Discount discount = DiscountDAO.getInstance().get(ticketItemDiscount.getDiscountId());
		//			if (discount == null)
		//				continue;
		//			if (!discount.isRefundable())
		//				return false;
		//		}
		return true;
	}

	public java.util.List<TicketItemCookingInstruction> getCookingInstructions() {
		if (cookingInstructions == null) {
			convertCookingInstructionPropertyToList();
		}
		return cookingInstructions;
	}

	public void convertCookingInstructionPropertyToList() {
		cookingInstructions = new ArrayList<>();

		String property = super.getCookingInstructionsProperty();
		if (StringUtils.isNotEmpty(property)) {
			JsonReader jsonParser = Json.createReader(new StringReader(property));
			JsonArray jsonArray = jsonParser.readArray();
			jsonParser.close();
			for (int i = 0; i < jsonArray.size(); i++) {
				JsonObject jsonObject = (JsonObject) jsonArray.get(i);
				TicketItemCookingInstruction tic = new TicketItemCookingInstruction();
				tic.setDescription(JsonUtil.getString(jsonObject, TicketItemCookingInstruction.PROP_DESCRIPTION));
				tic.setPrintedToKitchen(JsonUtil.getBoolean(jsonObject, TicketItemCookingInstruction.PROP_PRINTED_TO_KITCHEN));
				tic.setSaved(JsonUtil.getBoolean(jsonObject, TicketItemCookingInstruction.PROP_SAVED));
				tic.setKitchenStatus(JsonUtil.getString(jsonObject, TicketItemCookingInstruction.PROP_KITCHEN_STATUS));
				tic.setParentItem(this);
				cookingInstructions.add(tic);
			}
		}
	}

	public PrinterGroup getPrinterGroup() {
		PrinterGroup printerGroup = DataProvider.get().getPrinterGroupById(getPrinterGroupId());

		if (printerGroup == null) {
			DataProvider.get().getDefaultPrinterGroup();
		}

		return printerGroup;
	}

	public void setPrinterGroup(PrinterGroup printerGroup) {
		String printerGroupId = null;
		if (printerGroup != null) {
			printerGroupId = printerGroup.getId();
		}
		super.setPrinterGroupId(printerGroupId);
	}

	public TicketItem getParentTicketItem() {
		return parentTicketItem;
	}

	public void setParentTicketItem(TicketItem parentTicketItem) {
		this.parentTicketItem = parentTicketItem;
	}

	public List<TicketItem> getComboItems() {
		return null;
	}

	public void setComboItems(List<TicketItem> comboItems) {
	}

	public void addTocomboItems(TicketItem item) {

	}

	//	public ImageIcon getCourseIcon() {
	//		if (StringUtils.isEmpty(getCourseId()))
	//			return null;
	//		Course course = DataProvider.get().getCourse(getCourseId());
	//		return course == null ? null : course.getIcon();
	//	}

	public Integer getSortOrder() {
		if (sortOrder != null) {
			return sortOrder;
		}

		if (StringUtils.isEmpty(getCourseId())) {
			return 0;
		}
		Course course = DataProvider.get().getCourse(getCourseId());
		if (course != null) {
			return course.getSortOrder();
		}
		return 0;
	}

	public void setSortOrder(Integer sortOrder) {
		this.sortOrder = sortOrder;
	}

	public boolean isSyncEdited() {
		return syncEdited;
	}

	public void setSyncEdited(boolean syncEdited) {
		this.syncEdited = syncEdited;
	}

	@Override
	public void setProperties(String properties) {
		super.setProperties(properties);
		propertiesContainer = null;
	}

	@Override
	public void addProperty(String key, String value) {
		initPropertyContainer();
		propertiesContainer.addProperty(key, value);
		super.setProperties(propertiesContainer.toString());
	}

	@Override
	public String getProperty(String key) {
		initPropertyContainer();

		if (propertiesContainer.has(key)) {
			JsonElement jsonElement = propertiesContainer.get(key);
			if (!jsonElement.isJsonNull()) {
				return jsonElement.getAsString();
			}
		}
		return null;
	}

	@Override
	public boolean hasProperty(String key) {
		return getProperty(key) != null;
	}

	public boolean isPropertyValueTrue(String propertyName) {
		String property = getProperty(propertyName);

		return POSUtil.getBoolean(property);
	}

	@Override
	public void removeProperty(String propertyName) {
		initPropertyContainer();
		propertiesContainer.remove(propertyName);
		super.setProperties(propertiesContainer.toString());
	}

	public void setUnitSelection(boolean unitSelection) {
		addProperty(AppConstants.ALLOW_UNIT_SELECTION, String.valueOf(unitSelection));
	}

	public boolean isAllowUnitSelection() {
		String isAllowUnitSelection = getProperty(AppConstants.ALLOW_UNIT_SELECTION);
		if (StringUtils.isNotEmpty(isAllowUnitSelection))
			return Boolean.parseBoolean(isAllowUnitSelection);
		return false;
	}

	public void setItemIsGiftCard(String giftCardNo, double balance) {
		setQuantity(1.0);
		setShouldPrintToKitchen(false);
		setInventoryItem(false);
		setTicketDiscountApplicable(false);
		setName(Messages.getString("GiftCardAddBalanceView.22")); //$NON-NLS-1$
		setCategoryName(Messages.getString("GiftCard")); //$NON-NLS-1$
		setGroupName(Messages.getString("GiftCard")); //$NON-NLS-1$
		setUnitPrice(balance);

		addProperty(AppConstants.IS_GIFT_CARD, String.valueOf(true));
		addProperty(AppConstants.GIFT_CARD_NO, giftCardNo);
	}

	public boolean isGiftCard() {
		return Boolean.valueOf(getProperty(AppConstants.IS_GIFT_CARD));
	}

	public String getGiftCardNo() {
		return getProperty(AppConstants.GIFT_CARD_NO);
	}

	public String getClassType() {
		return this.classType;
	}

	public void setClassType(String classType) {
		this.classType = classType;
	}

	public void setReturned(Boolean returned) {
		addProperty(JSON_PROP_RETURNED, String.valueOf(returned));
		super.setItemReturned(returned);
	}

	public boolean isReturned() {
		return Boolean.valueOf(getProperty(JSON_PROP_RETURNED));
	}

	public boolean isReturnedItemVoided() {
		return Boolean.valueOf(getProperty(JSON_PROP_RETURNED_ITEM_VOIDED));
	}

	public double getComboItemsPrice() {
		double itemsPrice = 0;
		List<TicketItem> comboItems = getComboItems();
		if (comboItems != null) {
			for (TicketItem comboItem : comboItems) {
				if (comboItem.getGroupId() == null) {
					itemsPrice += comboItem.getUnitPrice() * (comboItem.getQuantity() / getQuantity());
				}
			}
		}
		return itemsPrice;
	}

	public double getGiftCardPaidAmount() {
		return POSUtil.parseDouble(getProperty(JSON_PROP_GIFT_CARD_PAID_AMOUNT));
	}

	public void setGiftCardPaidAmount(double amount) {
		addProperty(JSON_PROP_GIFT_CARD_PAID_AMOUNT, String.valueOf(amount));
	}

	public void setSplitted(boolean split) {
		addProperty("splitted", String.valueOf(split)); //$NON-NLS-1$
	}

	public boolean isSplitted() {
		return Boolean.valueOf(getProperty("splitted")); //$NON-NLS-1$
	}

	private void initPropertyContainer() {
		if (propertiesContainer == null) {
			if (StringUtils.isBlank(super.getProperties())) {
				propertiesContainer = new com.google.gson.JsonObject();
			}
			else {
				propertiesContainer = new Gson().fromJson(super.getProperties(), com.google.gson.JsonObject.class);
			}
		}
	}

	public void setServiceChargeWithoutModifiers(double sc) {
		addProperty(JSON_PROP_SC_WITHOUT_MODOFIERS, String.valueOf(sc));
	}

	public double getServiceChargeWithoutModifiers() {
		return getDoubleProperty(JSON_PROP_SC_WITHOUT_MODOFIERS);
	}

	@Override
	public com.google.gson.JsonObject getPropertyStore() {
		initPropertyContainer();
		return propertiesContainer;
	}

	public boolean hasDiscount() {
		List<TicketItemDiscount> discounts = getDiscounts();
		if (discounts != null && discounts.size() > 0) {
			return true;
		}
		return false;
	}

	@Override
	public Boolean isShouldPrintToKitchen() {
		if (getKitchenStatusValue() == KitchenStatus.BUMP) {
			return false;
		}
		return super.isShouldPrintToKitchen();
	}

	public boolean isKitchenPrintable() {
		if (!isShouldPrintToKitchen() || isPrintedToKitchen() || isItemReturned() || isTreatAsSeat()) {
			return false;
		}
		return true;
	}

	public void setCustomerRequired(boolean customerRequired) {
		addProperty("service.customerRequired", String.valueOf(customerRequired)); //$NON-NLS-1$
	}

	@XmlTransient
	@JsonIgnore
	public boolean isCustomerRequired() {
		return getBooleanProperty("service.customerRequired", false); //$NON-NLS-1$
	}

	public void setPaymentType(ServicePaymentType paymentType) {
		addProperty("service.paymentType", paymentType == null ? null : paymentType.name()); //$NON-NLS-1$
	}

	@XmlTransient
	@JsonIgnore
	public ServicePaymentType getPaymentType() {
		String paymentType = getProperty("service.paymentType"); //$NON-NLS-1$
		if (StringUtils.isBlank(paymentType)) {
			return null;
		}
		return ServicePaymentType.valueOf(paymentType);
	}

	public void setServiceStartDate(Date serviceStartDate) {
		addProperty("service.startDate", serviceStartDate == null ? null : DateUtil.formatSyncTime(serviceStartDate)); //$NON-NLS-1$
	}

	@XmlTransient
	@JsonIgnore
	public Date getServiceStartDate() {
		String serviceStartDate = getProperty("service.startDate"); //$NON-NLS-1$
		if (StringUtils.isBlank(serviceStartDate)) {
			return null;
		}
		try {
			return DateUtil.parseSyncTime(serviceStartDate);
		} catch (ParseException e) {
			return null;
		}
	}

	public void setServiceEndDate(Date serviceEndDate) {
		addProperty("service.endDate", serviceEndDate == null ? null : DateUtil.formatSyncTime(serviceEndDate)); //$NON-NLS-1$
	}

	@XmlTransient
	@JsonIgnore
	public Date getServiceEndDate() {
		String serviceStartDate = getProperty("service.endDate"); //$NON-NLS-1$
		if (StringUtils.isBlank(serviceStartDate)) {
			return null;
		}
		try {
			return DateUtil.parseSyncTime(serviceStartDate);
		} catch (ParseException e) {
			return null;
		}
	}

	public void putUnitPriceWithoutModifierAfterDiscount(double unitPriceWithoutModifierAfterDiscount) {
		addProperty("uprice_wt_modifier_ad", String.valueOf(unitPriceWithoutModifierAfterDiscount)); //$NON-NLS-1$
	}

	@XmlTransient
	@JsonIgnore
	public Double getUnitPriceWithoutModifierAfterDiscount() {
		String property = getProperty("uprice_wt_modifier_ad"); //$NON-NLS-1$
		if (StringUtils.isBlank(property)) {
			return null;
		}
		return POSUtil.parseDouble(property);
	}

	public double getReturnableQuantity() {
		double totalReturnedCount = getQuantity();
		Ticket ticket = getTicket();
		if (ticket == null) {
			return totalReturnedCount;
		}

		List<TicketItem> ticketItems = ticket.getTicketItems();
		for (TicketItem ticketItem : ticketItems) {
			if (ticketItem.isTreatAsSeat()) {
				continue;
			}

			String voidedItemId = ticketItem.getVoidedItemId();
			if (StringUtils.isNotBlank(voidedItemId) && voidedItemId.equals(this.getId()) && ticketItem.isItemReturned()) {
				totalReturnedCount += ticketItem.getQuantity();
			}
		}

		return Math.abs(totalReturnedCount);
	}

	@XmlTransient
	@JsonIgnore
	public Double getConversionRate() {
		return getDoubleProperty("conversion_rate"); //$NON-NLS-1$
	}

	public void setConversionRate(Double conversionRate) {
		addProperty("conversion_rate", String.valueOf(conversionRate)); //$NON-NLS-1$
	}

	@XmlTransient
	@JsonIgnore
	public String getConversionRule() {
		return getProperty("conversion_rule"); //$NON-NLS-1$
	}

	public void setConversionRule(String conversionRule) {
		addProperty("conversion_rule", conversionRule); //$NON-NLS-1$
	}

	@XmlTransient
	@JsonIgnore
	public boolean isStockCountable() {
		return getBooleanProperty("stock_countable", false); //$NON-NLS-1$
	}

	public void setStockCountable(boolean stockCountable) {
		addProperty("stock_countable", String.valueOf(stockCountable)); //$NON-NLS-1$
	}

	public void putDoctorCommissionAmount(double commissionAmount) {
		addProperty("doctor.commission_amount", String.valueOf(commissionAmount)); //$NON-NLS-1$
	}

	public double getDoctorCommissionAmount() {
		return getDoubleProperty("doctor.commission_amount"); //$NON-NLS-1$
	}

	public void putLabTestId(String labTestId) {
		addProperty("lab.test.id", labTestId); //$NON-NLS-1$
	}

	public String getLabTestId() {
		return getProperty("lab.test.id", ""); //$NON-NLS-1$ //$NON-NLS-2$
	}

	public Date getDeliveryDate() {
		String property = getProperty("delivery.date", ""); //$NON-NLS-1$ //$NON-NLS-2$
		if (StringUtils.isNotBlank(property)) {
			try {
				return DateUtil.formatDateWithTime(property);
			} catch (ParseException e) {
			}
		}
		return null;
	}

	public void setDeliveryDate(Date deliveryDate) {
		if (deliveryDate != null) {
			addProperty("delivery.date", DateUtil.formatDateWithTime(deliveryDate)); //$NON-NLS-1$
		}
	}

	public void setLabDoctorRequired(boolean isLabDoctorRequired) {
		addProperty("lab.doctor.required", String.valueOf(isLabDoctorRequired)); //$NON-NLS-1$
	}

	@XmlTransient
	@JsonIgnore
	public boolean isLabDoctorRequired() {
		return getBooleanProperty("lab.doctor.required", false); //$NON-NLS-1$
	}

	public void putLabDoctorFee(double labDoctorFee) {
		addProperty("lab.doctor.fee", String.valueOf(labDoctorFee)); //$NON-NLS-1$
	}

	@XmlTransient
	@JsonIgnore
	public double getLabDoctorFee() {
		return getDoubleProperty("lab.doctor.fee"); //$NON-NLS-1$
	}

	public void putLabDoctorId(String labDoctorId) {
		addProperty("lab.doctor.id", labDoctorId); //$NON-NLS-1$
	}

	@XmlTransient
	@JsonIgnore
	public String getLabDoctorId() {
		String labDoctorId = getProperty("lab.doctor.id"); //$NON-NLS-1$
		if (StringUtils.isBlank(labDoctorId)) {
			return ""; //$NON-NLS-1$
		}
		return labDoctorId;
	}

	public void setTestItems(List<TestItem> testItems) {
		if (testItems != null) {
			Gson gson = new GsonBuilder().addSerializationExclusionStrategy(new ExclusionStrategy() {

				@Override
				public boolean shouldSkipField(FieldAttributes f) {
					if (f.getName().equals("hashCode")) { //$NON-NLS-1$
						return true;
					}
					return false;
				}

				@Override
				public boolean shouldSkipClass(Class<?> clazz) {
					return false;
				}
			}).create();
			addTestItems(gson.toJson(testItems));
		}
	}

	public void addTestItems(String testItems) {
		addProperty("test_items", testItems); //$NON-NLS-1$
	}

	@XmlTransient
	@JsonIgnore
	public List<TestItem> getTestItems() {
		List<TestItem> testItems = new ArrayList<TestItem>();
		try {
			String testItemsProperty = getProperty("test_items"); //$NON-NLS-1$
			if (StringUtils.isEmpty(testItemsProperty)) {
				return testItems;
			}
			testItems = new Gson().fromJson(testItemsProperty, new TypeToken<ArrayList<TestItem>>() {
			}.getType());
		} catch (Exception e) {
			PosLog.error(getClass(), e);
		}
		return testItems;
	}

	@JsonIgnore
	public String getTreeId() {
		if (StringUtils.isNotBlank(getName())) {
			return ""; //$NON-NLS-1$
		}
		return getTicket().getId();
	}

	public void setTreeChildren(List<TicketItem> treeChildren) {
		this.treeChildren = treeChildren;
	}

	public List<TicketItem> getTreeChildren() {
		return treeChildren;
	}

	public void addProcessedStatus(LabWorkStatus labWorkStatus) {
		try {
			List<ProcessedStatus> processedStatuses = getProcessedStatuses();
			if (LabWorkStatus.SAMPLE_COLLECTED.equals(labWorkStatus)) {
				ProcessedStatus status = ProcessedStatus.createSampleCollectedStatus(this);
				processedStatuses.add(status);
			}
			else {
				Collections.sort(processedStatuses, (ProcessedStatus p1, ProcessedStatus p2) -> p2.getProcessedKey().compareTo(p1.getProcessedKey()));

				ProcessedStatus processedStatus = processedStatuses.get(0);
				List<ProcessedStatusResult> statusResults = processedStatus.getStatusResults();
				if (statusResults != null) {
					ProcessedStatusResult createProcessedStatusResult = ProcessedStatusResult.createProcessedStatusResult(this, labWorkStatus);
					statusResults.add(createProcessedStatusResult);
				}
			}
			setProcessedStatus(processedStatuses);
		} catch (Exception e) {
		}
	}

	public void setProcessedStatus(List<ProcessedStatus> processedStatuses) {
		Gson gson = GsonUtil.createGson();
		String processedStatusGson = gson.toJson(processedStatuses);
		addProperty("processed.status", processedStatusGson); //$NON-NLS-1$
	}

	@XmlTransient
	@JsonIgnore
	public List<ProcessedStatus> getProcessedStatuses() {
		List<ProcessedStatus> processedStatuses = new ArrayList<ProcessedStatus>();
		String processedStatusProperty = getProperty("processed.status"); //$NON-NLS-1$
		if (StringUtils.isEmpty(processedStatusProperty)) {
			return processedStatuses;
		}

		Gson gson = GsonUtil.createGson();
		java.lang.reflect.Type type = new TypeToken<List<ProcessedStatus>>() {
		}.getType();
		processedStatuses = gson.fromJson(processedStatusProperty, type);

		return processedStatuses;
	}

	public void setDeclined(boolean declined) {
		addProperty("declined", String.valueOf(declined)); //$NON-NLS-1$
	}

	public boolean isDeclined() {
		return getBooleanProperty("declined", false); //$NON-NLS-1$
	}

	public void putDeclinedReason(String declinedReason) {
		addProperty("declined.reason", declinedReason); //$NON-NLS-1$
	}

	public String getDeclinedReason() {
		return getProperty("declined.reason", ""); //$NON-NLS-1$ //$NON-NLS-2$
	}
}