package com.floreantpos.model.dao;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.lang.StringUtils;
import org.hibernate.Criteria;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.ProjectionList;
import org.hibernate.criterion.Projections;
import org.hibernate.criterion.Property;
import org.hibernate.criterion.Restrictions;
import org.hibernate.transform.Transformers;

import com.floreantpos.model.Agent;
import com.floreantpos.model.BalanceType;
import com.floreantpos.model.BalanceUpdateTransaction;
import com.floreantpos.model.Customer;
import com.floreantpos.model.Doctor;
import com.floreantpos.model.GiftCard;
import com.floreantpos.model.PaymentType;
import com.floreantpos.model.PosTransaction;
import com.floreantpos.model.Ticket;
import com.floreantpos.model.TransactionSubType;
import com.floreantpos.model.TransactionType;
import com.floreantpos.model.util.DateUtil;
import com.floreantpos.report.DueReportData;
import com.floreantpos.swing.PaginationSupport;
import com.floreantpos.util.NumberUtil;
import com.floreantpos.util.POSUtil;

public class BalanceUpdateTransactionDAO extends BaseBalanceUpdateTransactionDAO {

	public BalanceUpdateTransactionDAO() {
	}

	/**
	 * Default constructor.  Can be used in place of getInstance()
	 */
	@Override
	protected Serializable save(Object obj, Session s) {
		updateTime(obj);
		return super.save(obj, s);
	}

	@Override
	protected void update(Object obj, Session s) {
		updateTime(obj);
		super.update(obj, s);
	}

	@Override
	protected void saveOrUpdate(Object obj, Session s) {
		updateTime(obj);
		super.saveOrUpdate(obj, s);
	}

	private void adjustBalanceUpdateTransaction(BalanceUpdateTransaction balanceTransaction, Session session) {
		save(balanceTransaction, session);

		BalanceType balanceType = balanceTransaction.getBalanceType();
		if (balanceType == BalanceType.GIFT_CARD) {
			GiftCard giftCard = GiftCardDAO.getInstance().findByCardNumber(session, balanceTransaction.getAccountNumber());
			if (giftCard == null) {
				//throw new PosException("No gift card found with account number " + balanceTransaction.getAccountNumber()); //$NON-NLS-1$
				return;
			}
			giftCard.setBalance(giftCard.getBalance() + balanceTransaction.getAmount());
			GiftCardDAO.getInstance().saveOrUpdate(giftCard, session);
			return;
		}

		Customer customer = CustomerDAO.getInstance().get(balanceTransaction.getAccountNumber(), session);
		if (customer == null) {
			//throw new PosException("No customer found with id " + balanceTransaction.getAccountNumber()); //$NON-NLS-1$
			return;
		}

		if (balanceType == BalanceType.CUSTOMER) {
			customer.setBalance(customer.getBalance() + balanceTransaction.getAmount());
		}
		else if (balanceType == BalanceType.LOYALTY) {
			customer.setLoyaltyPoint((int) (customer.getLoyaltyPoint() + balanceTransaction.getAmount()));
		}
		customer.setUpdateLastUpdateTime(false);
		customer.setUpdateSyncTime(false);
		CustomerDAO.getInstance().saveOrUpdate(customer, session);
	}

	public void saveBalanceUpdateTrans(BalanceType balanceType, Ticket ticket, PosTransaction transaction, TransactionType transactionType,
			String accountNumber, Double amount, Double balanceBefore, TransactionSubType subType, Session session) {
		BalanceUpdateTransaction balanceUpdateTransaction = balanceType.createTransaction(transaction == null ? PaymentType.CASH : transaction.getPaymentType(),
				transactionType, amount);
		balanceUpdateTransaction.setTransactionId(transaction == null ? null : transaction.getId());
		balanceUpdateTransaction.setTicketId(ticket == null ? null : ticket.getId());
		balanceUpdateTransaction.setAccountNumber(accountNumber);

		if (subType != null) {
			balanceUpdateTransaction.setTransactionSubType(subType.name());
			balanceUpdateTransaction.setDescription(subType.name());
		}
		if (!NumberUtil.isZero(balanceBefore)) {
			balanceBefore = NumberUtil.round(balanceBefore);
			balanceUpdateTransaction.setBalanceBefore(balanceBefore);
		}

		BalanceUpdateTransactionDAO.getInstance().save(balanceUpdateTransaction, session);
	}

	public void saveOrUpdateBalanceUpdateTransactions(List<BalanceUpdateTransaction> customerBalanceTransactions, boolean updateLastUpdateTime,
			boolean updateSyncTime) throws Exception {
		if (customerBalanceTransactions == null)
			return;

		Transaction tx = null;
		Session session = null;
		try {
			session = createNewSession();
			tx = session.beginTransaction();
			for (BalanceUpdateTransaction transaction : customerBalanceTransactions) {
				BalanceUpdateTransaction existingTransaction = get(transaction.getId());
				if (existingTransaction != null) {
					continue;
				}
				transaction.setUpdateLastUpdateTime(updateLastUpdateTime);
				transaction.setUpdateSyncTime(updateSyncTime);
				adjustBalanceUpdateTransaction(transaction, session);
			}
			tx.commit();
		} catch (Exception e) {
			tx.rollback();
			throw e;
		} finally {
			closeSession(session);
		}
	}

	public double getTicketCommissionAmount(Session session, String ticketId, BalanceType balanceType) {
		return getTicketCommissionAmount(session, ticketId, "", balanceType); //$NON-NLS-1$
	}

	public double getTicketCommissionAmount(Session session, String ticketId, String accountNo, BalanceType balanceType) {
		Criteria criteria = session.createCriteria(BalanceUpdateTransaction.class);
		criteria.setProjection(Projections.sum(BalanceUpdateTransaction.PROP_AMOUNT));

		criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_VOIDED, false));
		criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_TICKET_ID, ticketId));
		if (StringUtils.isNotBlank(accountNo)) {
			criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_ACCOUNT_NUMBER, accountNo));
		}
		criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_BALANCE_TYPE_STRING, balanceType.name()));
		double scFromItems = POSUtil.getDoubleAmount(criteria.uniqueResult());

		return scFromItems;
	}

	public void findBy(String accountNo, Date beginDate, Date currentDate, PaginationSupport tableModel) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(BalanceUpdateTransaction.class);
			criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_VOIDED, false));
			criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_ACCOUNT_NUMBER, accountNo));
			criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_BALANCE_TYPE_STRING, BalanceType.AGENT_FEE_PAYMENT.name()));
			criteria.add(Restrictions.between(BalanceUpdateTransaction.PROP_TRANSACTION_TIME, DateUtil.startOfDay(beginDate), DateUtil.endOfDay(currentDate)));
			tableModel.setNumRows(rowCount(criteria));

			criteria.setFirstResult(tableModel.getCurrentRowIndex());
			criteria.setMaxResults(tableModel.getPageSize());
			criteria.addOrder(Order.desc(BalanceUpdateTransaction.PROP_TRANSACTION_TIME));
			tableModel.setRows(criteria.list());
		}
	}

	public Map<String, Double> findSummaryBy(String accountNo, Date beginDate, Date currentDate) {
		Map<String, Double> map = new HashMap<String, Double>();
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(BalanceUpdateTransaction.class);
			criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_VOIDED, false));
			//criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_BALANCE_TYPE_STRING, BalanceType.AGENT_FEE_PAYMENT.name()));
			criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_ACCOUNT_NUMBER, accountNo));
			criteria.add(Restrictions.between(BalanceUpdateTransaction.PROP_TRANSACTION_TIME, DateUtil.startOfDay(beginDate), DateUtil.endOfDay(currentDate)));

			ProjectionList projectionList = Projections.projectionList();
			projectionList.add(Projections.groupProperty(BalanceUpdateTransaction.PROP_ACCOUNT_NUMBER), "accountNo"); //$NON-NLS-1$
			projectionList.add(Projections.groupProperty(BalanceUpdateTransaction.PROP_BALANCE_TYPE_STRING), "balanceType"); //$NON-NLS-1$
			projectionList.add(Projections.sum(BalanceUpdateTransaction.PROP_AMOUNT), "total"); //$NON-NLS-1$
			criteria.setProjection(projectionList);

			List<Object[]> list = criteria.list();
			for (Object[] obj : list) {
				String balanceTypeString = (String) obj[1];
				double amount = (double) obj[2];
				map.put(balanceTypeString, amount);
			}
		}
		return map;
	}

	public List<DueReportData> findDueReportOrders(Date beginDate, Date endDate) {
		List<DueReportData> datas = new ArrayList<DueReportData>();
		try (Session session = createNewSession()) {
			
			Criteria criteria = session.createCriteria(Customer.class);
			criteria.add(Restrictions.or(Property.forName("class").eq(Agent.class), Property.forName("class").eq(Doctor.class))); //$NON-NLS-1$ //$NON-NLS-2$
			addDeletedFilter(criteria);

			criteria.add(Restrictions.eq(Customer.PROP_ACTIVE, true));
			
			ProjectionList projectionList = Projections.projectionList();
			projectionList.add(Projections.property(Customer.PROP_NAME), "name");
			criteria.setProjection(projectionList);
			criteria.setResultTransformer(Transformers.aliasToBean(DueReportData.class));
			
//			DetachedCriteria criteria2 = DetachedCriteria.forClass(BalanceUpdateTransaction.class);
//			criteria2.add(Restrictions.in(BalanceUpdateTransaction.PROP_BALANCE_TYPE_STRING, BalanceType.AGENT_FEE_ADD.name(),
//					BalanceType.AGENT_FEE_PAYMENT.name(), BalanceType.DOCTOR_FEE_ADD.name(), BalanceType.DOCTOR_FEE_PAYMENT.name()));
//			
//			
//			criteria2.addOrder(Order.desc(BalanceUpdateTransaction.PROP_TRANSACTION_TIME));
			
			return criteria.list();
			
			
//			Criteria criteria = session.createCriteria(BalanceUpdateTransaction.class);
//			criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_VOIDED, false));
//			criteria.add(Restrictions.between(BalanceUpdateTransaction.PROP_TRANSACTION_TIME, DateUtil.startOfDay(beginDate), DateUtil.endOfDay(endDate)));
//
//			addDeletedFilter(criteria);
//
//			criteria.add(Restrictions.in(BalanceUpdateTransaction.PROP_BALANCE_TYPE_STRING, BalanceType.AGENT_FEE_ADD.name(),
//					BalanceType.AGENT_FEE_PAYMENT.name(), BalanceType.DOCTOR_FEE_ADD.name(), BalanceType.DOCTOR_FEE_PAYMENT.name()));
//
//			
//			ProjectionList projectionList = Projections.projectionList();
//			projectionList.add(Projections.property(BalanceUpdateTransaction.PROP_TRANSACTION_TIME));
//			projectionList.add(Projections.groupProperty(BalanceUpdateTransaction.PROP_ACCOUNT_NUMBER));
//			
//			criteria.addOrder(Order.desc(BalanceUpdateTransaction.PROP_TRANSACTION_TIME));
//			criteria.setProjection(projectionList);
//
//			criteria.list();

			//			ProjectionList projectionList = Projections.projectionList();
			//			projectionList.add(projectionList)
		}
		//return datas;
	}

}