package com.floreantpos.report;

import java.io.Serializable;

import org.apache.commons.lang3.StringUtils;

import com.floreantpos.model.Agent;
import com.floreantpos.model.AgentTypeEnum;
import com.floreantpos.model.Customer;
import com.floreantpos.model.Doctor;
import com.floreantpos.model.Ticket;
import com.floreantpos.model.dao.CustomerDAO;
import com.floreantpos.model.dao.DoctorDAO;
import com.floreantpos.model.util.DateUtil;
import com.floreantpos.model.util.ReferralCommissionUtil;

public class ReferralCommissionReportData implements Serializable {

	private String id;
	private String orderId;
	private String orderDate;
	private String outletId;
	private String referralType;

	private String patientName;
	private String referralId;
	private String referralBy;
	private double receivable;
	private double due;
	private double received;
	private String reportDoctor;
	private double agentCommissionOnNetSales;
	private double agentCommissionOnReport;
	private double agentCommissionTotal;
	private double labDoctorFee;

	public ReferralCommissionReportData(Ticket ticket) {
		double subtotalAmount = ticket.getSubtotalAmount() - ticket.getDiscountAmount();
		setOrderId(ticket.getId());
		setOrderDate(DateUtil.formatAsShortDate(DateUtil.convertDateToBrowserTime(ticket.getCreateDate())));
		setOutletId(ticket.getOutletId());
		setReceivable(subtotalAmount);
		setDue(ticket.getDueAmount());
		setReceived(ticket.getPaidAmount() - ticket.getRefundAmount());
		setLabDoctorFee(ticket.getLabDoctorFee());

		String agentId = ticket.getReferrerId();
		setReferralId(agentId);
		if (StringUtils.isNotBlank(agentId)) {
			Customer customer = CustomerDAO.getInstance().get(agentId);
			if (customer != null) {
				setReferralBy(customer.getName());
				if (customer instanceof Agent) {
					String agentType = customer.getAgentType();
					if (StringUtils.isNotBlank(agentType)) {
						setReferralType(AgentTypeEnum.fromString(agentType).getAgentTypeName());
					}
					else {
						setReferralType("Other");
					}
				}
				else if (customer instanceof Doctor) {
					if (customer.getIndorDoctor()) {
						setReferralType(AgentTypeEnum.INDOOR_DOCTOR.getAgentTypeName());
					}
					else {
						setReferralType(AgentTypeEnum.OUTDOOR_DOCTOR.getAgentTypeName());
					}
				}
			}
			else {
				setReferralType("Self");
				setReferralBy(""); //$NON-NLS-1$
			}

			double commissionOnReport = ReferralCommissionUtil.calculateAgentCommissionOnReport(ticket);
			double commissionOnNetSales = ReferralCommissionUtil.calculateAgentCommissionOnNetSales(ticket, commissionOnReport);
			setAgentCommissionOnNetSales(commissionOnNetSales);
			setAgentCommissionOnReport(commissionOnReport);
			setAgentCommissionTotal(commissionOnNetSales + commissionOnReport);
		}
		else {
			setReferralType("Self");
			setReferralBy(""); //$NON-NLS-1$
		}

		String doctorId = ticket.getDoctorId();
		if (StringUtils.isNotBlank(doctorId)) {
			Doctor doctor = DoctorDAO.getInstance().get(doctorId);
			if (doctor != null) {
				setReportDoctor(doctor.getName());
			}
		}

		Customer customer = ticket.getCustomer();
		setPatientName(customer.getName());
	}

	public String getId() {
		return id;
	}

	public void setId(String id) {
		this.id = id;
	}

	public String getReferralType() {
		return referralType;
	}

	public void setReferralType(String referralType) {
		this.referralType = referralType;
	}

	public String getPatientName() {
		return patientName;
	}

	public void setPatientName(String patientName) {
		this.patientName = patientName;
	}

	public String getReferralId() {
		return referralId;
	}

	public void setReferralId(String referralId) {
		this.referralId = referralId;
	}

	public String getReferralBy() {
		return referralBy;
	}

	public void setReferralBy(String referralBy) {
		this.referralBy = referralBy;
	}

	public double getReceivable() {
		return receivable;
	}

	public void setReceivable(double receivable) {
		this.receivable = receivable;
	}

	public double getDue() {
		return due;
	}

	public void setDue(double due) {
		this.due = due;
	}

	public double getReceived() {
		return received;
	}

	public void setReceived(double received) {
		this.received = received;
	}

	public String getReportDoctor() {
		return reportDoctor;
	}

	public void setReportDoctor(String reportDoctor) {
		this.reportDoctor = reportDoctor;
	}

	public double getAgentCommissionOnNetSales() {
		return agentCommissionOnNetSales;
	}

	public void setAgentCommissionOnNetSales(double agentCommissionOnNetSales) {
		this.agentCommissionOnNetSales = agentCommissionOnNetSales;
	}

	public double getAgentCommissionOnReport() {
		return agentCommissionOnReport;
	}

	public void setAgentCommissionOnReport(double agentCommissionOnReport) {
		this.agentCommissionOnReport = agentCommissionOnReport;
	}

	public double getAgentCommissionTotal() {
		return agentCommissionTotal;
	}

	public void setAgentCommissionTotal(double agentFee) {
		this.agentCommissionTotal = agentFee;
	}

	public double getLabDoctorFee() {
		return labDoctorFee;
	}

	public void setLabDoctorFee(double labDoctorFee) {
		this.labDoctorFee = labDoctorFee;
	}

	public String getOrderId() {
		return orderId;
	}

	public void setOrderId(String id) {
		this.orderId = id;
	}

	public String getOutletId() {
		return outletId;
	}

	public void setOutletId(String outletId) {
		this.outletId = outletId;
	}

	public String getOrderDate() {
		return orderDate;
	}

	public void setOrderDate(String orderDate) {
		this.orderDate = orderDate;
	}

}
