package com.floreantpos.extension.stripe;

import java.text.DecimalFormat;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.apache.commons.lang.StringUtils;

import com.floreantpos.PosException;
import com.floreantpos.PosLog;
import com.floreantpos.extension.PaymentMethod;
import com.floreantpos.extension.StripeGatewayPlugin;
import com.floreantpos.extension.cardconnect.IppMessages;
import com.floreantpos.model.CardReader;
import com.floreantpos.model.Outlet;
import com.floreantpos.model.PosTransaction;
import com.floreantpos.model.Store;
import com.floreantpos.model.Ticket;
import com.floreantpos.model.util.DataProvider;
import com.floreantpos.model.util.PaymentMethodJsonExtractor;
import com.floreantpos.ui.views.payment.CardProcessor;
import com.floreantpos.util.POSUtil;
import com.google.gson.JsonObject;
import com.orocube.workspace.subscription.StripeUtil;
import com.stripe.Stripe;
import com.stripe.exception.CardException;
import com.stripe.exception.StripeException;
import com.stripe.model.Card;
import com.stripe.model.Charge;
import com.stripe.model.Customer;
import com.stripe.model.PaymentSource;
import com.stripe.model.Refund;
import com.stripe.model.Token;
import com.stripe.param.CustomerUpdateParams;
import com.stripe.param.RefundCreateParams;

public class StripeProcessor implements CardProcessor {
	private String apiKey;
	private boolean isForOnlineOrder = Boolean.TRUE;

	public StripeProcessor(boolean isForOnlineOrder) {
		this.isForOnlineOrder = isForOnlineOrder;
		init(DataProvider.get().getOutlet());
	}

	public void init(Outlet outlet) {
		if (isForOnlineOrder) {
			this.apiKey = outlet.getProperty(Store.PROP_ONLINE_ORDER_STRIPE_API_KEY);
		}
		else {
			JsonObject paymentConfiguration = PaymentMethodJsonExtractor.getPaymentConfiguration(StripeGatewayPlugin.ID);
			this.apiKey = paymentConfiguration.get(PaymentMethodJsonExtractor.PROP_PAYMENT_METHOD_STRIPE_API_KEY).getAsString();
		}
		Stripe.apiKey = apiKey;
	}

	@Override
	public void preAuth(PosTransaction transaction) throws Exception {
	}

	@Override
	public List<PaymentMethod> getPaymentMethods(com.floreantpos.model.Customer publicCustomer) throws Exception {
		List<PaymentMethod> paymentMethods = new ArrayList<>();
		Customer stripeCustomer = getCustomer(apiKey, publicCustomer.getId());
		if (stripeCustomer != null && stripeCustomer.getSources() != null) {
			List<PaymentSource> accounts = stripeCustomer.getSources().getData();
			if (accounts != null && !accounts.isEmpty()) {
				for (Iterator<PaymentSource> iterator = accounts.iterator(); iterator.hasNext();) {
					PaymentSource account = (PaymentSource) iterator.next();
					if (account == null) {
						continue;
					}
					if (account instanceof Card) {
						Card card = (Card) account;
						PaymentMethod paymentMethod = new PaymentMethod(card.getId(), null, card.getLast4(), card.getBrand(),
								stripeCustomer.getDefaultSource() == null ? false : stripeCustomer.getDefaultSource().equals(card.getId()));
						paymentMethod.setAccountHolderName(card.getName());
						paymentMethod.setExpMonth(String.valueOf(card.getExpMonth()));
						paymentMethod.setExpYear(String.valueOf(card.getExpYear()));
						paymentMethods.add(paymentMethod);
					}
				}
			}
		}
		return paymentMethods;
	}

	@Override
	public Map<String, Object> saveCreditCardInfo(String customerId, String customerEmail, String cardHolderName, String cardNumber, String expMonth,
			String expYear, String cvv) throws Exception {
		Stripe.apiKey = apiKey;
		Map<String, Object> response = new HashMap<>();
		String cardToken = createOrUpdateCustomerInfo(apiKey, cardHolderName, customerEmail, customerId, cardNumber, POSUtil.parseInteger(expMonth),
				POSUtil.parseInteger(expYear), cvv, cardHolderName);
		response.put(StripeConstants.CARD_TOKEN, cardToken);
		return response;
	}

	public String createOrUpdateCustomerInfo(String apiKey, String customerName, String customerEmail, String customerId, String cardNumber,
			Integer cardExpMonth, Integer cardExpYear, String cvc, String holderName) {
		Customer stripeCustomer = getCustomer(apiKey, customerId);
		if (stripeCustomer == null) {
			stripeCustomer = createCustomer(apiKey, customerName, customerEmail, customerId);
			Card card = createOrUpdateCard(apiKey, stripeCustomer, null, cardNumber, cardExpMonth, cardExpYear, cvc, holderName);
			return card.getId();
		}
		else {
			if (StringUtils.isBlank(stripeCustomer.getName())) {
				try {
					stripeCustomer.setName(customerName);
					stripeCustomer.update(CustomerUpdateParams.builder().setName(customerName).build());
				} catch (StripeException e) {
					PosLog.error(getClass(), e);
				}
			}
			Card card = createOrUpdateCard(apiKey, stripeCustomer, null, cardNumber, cardExpMonth, cardExpYear, cvc, holderName);
			return card.getId();
		}
	}

	@Override
	public void removeCard(PaymentMethod paymentMethod, com.floreantpos.model.Customer customer) throws Exception {
		Stripe.apiKey = apiKey;
		try {
			Customer stripeCustomer = getCustomer(apiKey, customer.getId());
			if (stripeCustomer == null) {
				throw new PosException("No customer found.");
			}
			Card stripeCard = StripeUtil.getCardById(stripeCustomer, paymentMethod.getId());
			if (stripeCard == null) {
				throw new PosException("Credit card not found.");
			}
			stripeCard.delete();
		} catch (Exception e0) {
			throw new RuntimeException(e0);
		}
	}

	//	public Card getCard(Customer stripeCustomer, String cardNumber, int expMonth, int expYear) {
	//		if (stripeCustomer == null) {
	//			return null;
	//		}
	//		List<PaymentSource> accounts = stripeCustomer.getSources().getData();
	//		if (!accounts.isEmpty()) {
	//			for (Iterator<PaymentSource> iterator = accounts.iterator(); iterator.hasNext();) {
	//				PaymentSource account = (PaymentSource) iterator.next();
	//				if (account instanceof Card) {
	//					Card creditCard = (Card) account;
	//					if (cardNumber.endsWith(creditCard.getLast4()) && creditCard.getExpMonth().intValue() == expMonth
	//							&& creditCard.getExpYear().intValue() == expYear) {
	//						return creditCard;
	//					}
	//				}
	//			}
	//		}
	//		return null;
	//	}

	public Customer createCustomer(String apiKey, String customerName, String customerEmail, String customerId) {
		Stripe.apiKey = apiKey;
		try {
			Map<String, Object> customerParams = new HashMap<>();
			customerParams.put(StripeConstants.EMAIL, customerEmail);
			customerParams.put(StripeConstants.NAME, customerName);
			if (StringUtils.isNotEmpty(customerId)) {
				customerParams.put(StripeConstants.ID, customerId);
			}
			return Customer.create(customerParams);
		} catch (Exception e0) {
			PosLog.error(this.getClass(), e0);
			throw new RuntimeException(e0);
		}
	}

	public Token createToken(String apiKey, String cardNumber, Integer cardExpMonth, Integer cardExpYear, String cvc, String holderName) {
		Stripe.apiKey = apiKey;
		try {
			Map<String, Object> tokenParams = new HashMap<String, Object>();

			Map<String, Object> cardMetaData = new HashMap<String, Object>();

			Map<String, Object> cardParams = new HashMap<String, Object>();
			cardParams.put(StripeConstants.CARD_NUMBER, cardNumber);
			cardParams.put(StripeConstants.EXP_MONTH, cardExpMonth);
			cardParams.put(StripeConstants.EXP_YEAR, cardExpYear);
			cardParams.put(StripeConstants.CARD_CVC, cvc);
			cardParams.put(StripeConstants.NAME, holderName);
			cardParams.put(StripeConstants.META_DATA, cardMetaData);

			tokenParams.put(StripeConstants.CARD, cardParams);
			Token token = Token.create(tokenParams);
			return token;
		} catch (CardException ex) {
			PosLog.error(getClass(), ex.getMessage());
			throw new PosException(ex.getStripeError().getMessage());
		} catch (Exception e0) {
			throw new RuntimeException(e0);
		}
	}

	public Customer setSource(String apiKey, Customer customer, String tokenId) {
		Stripe.apiKey = apiKey;
		try {
			Map<String, Object> customerParams = new HashMap<>();
			customerParams.put(StripeConstants.SOURCE, tokenId);
			customer = customer.update(customerParams);
			return customer;
		} catch (CardException ex) {
			PosLog.error(getClass(), ex.getMessage());
			throw new PosException(ex.getStripeError().getMessage());
		} catch (Exception e0) {
			PosLog.error(this.getClass(), e0);
			throw new RuntimeException(e0);
		}
	}

	public Card createOrUpdateCard(String apiKey, Customer customer, Card stripeCard, String cardNumber, Integer cardExpMonth, Integer cardExpYear, String cvc,
			String holderName) {
		Stripe.apiKey = apiKey;
		try {
			Map<String, Object> cardMetaData = new HashMap<String, Object>();
			if (stripeCard == null) {
				Map<String, Object> tokenParams = new HashMap<String, Object>();
				Map<String, Object> cardParams = new HashMap<String, Object>();
				cardParams.put(StripeConstants.CARD_NUMBER, cardNumber);
				cardParams.put(StripeConstants.EXP_MONTH, cardExpMonth);
				cardParams.put(StripeConstants.EXP_YEAR, cardExpYear);
				cardParams.put(StripeConstants.CARD_CVC, cvc);
				cardParams.put(StripeConstants.NAME, holderName);
				cardParams.put(StripeConstants.META_DATA, cardMetaData);

				tokenParams.put(StripeConstants.CARD, cardParams);
				Token token = Token.create(tokenParams);

				Map<String, Object> sourceParams = new HashMap<>();
				sourceParams.put(StripeConstants.SOURCE, token.getId());
				Card card = (Card) customer.getSources().create(sourceParams);
				customer.update(CustomerUpdateParams.builder().setDefaultSource(card.getId()).build());
				return card;
			}
			else {
				Map<String, Object> cardParams = new HashMap<String, Object>();
				cardParams.put(StripeConstants.EXP_MONTH, cardExpMonth); //$NON-NLS-1$
				cardParams.put(StripeConstants.EXP_YEAR, cardExpYear); //$NON-NLS-1$
				cardParams.put(StripeConstants.META_DATA, cardMetaData); //$NON-NLS-1$
				stripeCard = (Card) stripeCard.update(cardParams);
				customer.update(CustomerUpdateParams.builder().setDefaultSource(stripeCard.getId()).build());
				return stripeCard;
			}
		} catch (CardException ex) {
			PosLog.error(getClass(), ex.getMessage());
			throw new PosException(ex.getStripeError().getMessage());
		} catch (Exception e0) {
			throw new RuntimeException(e0);
		}
	}

	public com.stripe.model.Customer getCustomer(String apiKey, String customerId) {
		return StripeUtil.getCustomerById(apiKey, customerId);
	}

	@Override
	public void captureAuthAmount(PosTransaction transaction) throws Exception {
		init(transaction.getTicket().getOutlet());
		preProcessingLog(transaction);

		Charge charge = Charge.retrieve(transaction.getCardTransactionId());
		if (charge.getCaptured()) {
			transaction.setCaptured(true);
			return;
		}
		charge = charge.capture();
		if (isNotSucceeded(charge)) {
			throw new PosException(charge.getFailureMessage());
		}
		transaction.setCaptured(true);
	}

	@Override
	public void chargeAmount(PosTransaction transaction) throws Exception {
		chargeAmount(transaction, null, false);
	}

	@Override
	public void chargeAmount(PosTransaction transaction, PaymentMethod paymentMethod, Boolean saveCard) throws Exception {
		init(transaction.getTicket().getOutlet());
		preProcessingLog(transaction);

		Ticket ticket = transaction.getTicket();
		String cardTokenId = null;
		Card stripeCard = null;
		Customer stripeCustomer = null;
		if (paymentMethod == null) {
			checkCardValidation(transaction);
			if (saveCard) {
				cardTokenId = createPaymentMethod(transaction);
				stripeCustomer = getCustomer(apiKey, ticket.getCustomerId());
				if (stripeCustomer != null) {
					stripeCard = StripeUtil.getCardById(stripeCustomer, cardTokenId);
				}
			}
			else {
				Token token = createToken(transaction);
				cardTokenId = token.getId();
				if (token.getCard() != null) {
					transaction.setCardType(token.getCard().getBrand());
				}
			}
		}
		else {
			cardTokenId = paymentMethod.getId();
			transaction.setCardType(paymentMethod.getCardType());
			stripeCustomer = getCustomer(apiKey, ticket.getCustomerId());
			if (stripeCustomer != null) {
				stripeCard = StripeUtil.getCardById(stripeCustomer, cardTokenId);
				if (stripeCard != null) {
					Map<String, Object> cardParams = new HashMap<String, Object>();
					cardParams.put(StripeConstants.EXP_MONTH, stripeCard.getExpMonth()); //$NON-NLS-1$
					cardParams.put(StripeConstants.EXP_YEAR, stripeCard.getExpYear()); //$NON-NLS-1$
					stripeCard = (Card) stripeCard.update(cardParams);
					stripeCustomer.update(CustomerUpdateParams.builder().setDefaultSource(stripeCard.getId()).build());
				}
			}
		}

		Map<String, Object> chargeParams = new HashMap<>();
		chargeParams.put(StripeConstants.AMOUNT, new DecimalFormat("000").format(transaction.getAmount() * 100)); //$NON-NLS-1$
		chargeParams.put(StripeConstants.CURRENCY, DataProvider.get().getMainCurrency().getCode());
		chargeParams.put(StripeConstants.SOURCE, cardTokenId);
		if (stripeCustomer != null && stripeCard != null) {
			chargeParams.put(StripeConstants.CUSTOMER, stripeCustomer.getId());
		}
		chargeParams.put(StripeConstants.DESCRIPTION, "Check#" + ticket.getId() + " Total: " + ticket.getTotalAmountWithTips()); //$NON-NLS-1$ //$NON-NLS-2$

		Charge charge = Charge.create(chargeParams);
		if (isNotSucceeded(charge)) {
			PosLog.error(getClass(), charge.getFailureMessage());
			throw new RuntimeException();
		}
		transaction.setCaptured(charge.getCaptured());
		transaction.setAuthorizable(false);
		transaction.setCardTransactionId(charge.getId());
		if (StringUtils.isNotBlank(charge.getAuthorizationCode())) {
			transaction.setCardAuthCode(charge.getAuthorizationCode());
		}
		else {
			transaction.setCardAuthCode(charge.getId());
		}
		if (stripeCard != null) {
			transaction.setCardExpMonth(String.valueOf(stripeCard.getExpMonth()));
			transaction.setCardExpYear(String.valueOf(stripeCard.getExpYear()));
			transaction.setCardType(stripeCard.getBrand());
			transaction.setCardCVV(stripeCard.getCvcCheck());
			transaction.setCardHolderName(stripeCard.getName());
		}

		String accountNumber = transaction.getCardNumber();
		if (StringUtils.isNotEmpty(accountNumber)) {
			accountNumber = accountNumber.substring(accountNumber.length() - 4, accountNumber.length());
			transaction.setCardNumber(accountNumber);
		}
		else {
			transaction.setCardNumber(stripeCard.getLast4());
		}
		if (StringUtils.isBlank(transaction.getCardHolderName()) && stripeCard != null) {
			transaction.setCardHolderName(stripeCard.getName());
		}
	}

	private String createPaymentMethod(PosTransaction transaction) throws Exception {
		String cardHolderName = transaction.getCardHolderName();
		String cardNumber = transaction.getCardNumber();
		String cardExpYear = transaction.getCardExpYear();
		String cardExpMonth = transaction.getCardExpMonth();
		String cvv = transaction.getCardCVV();
		String customerId = transaction.getTicket().getCustomerId();
		String customerEmail = transaction.getTicket().getCustomerEmail();
		Map<String, Object> response = saveCreditCardInfo(customerId, customerEmail, cardHolderName, cardNumber, cardExpMonth, cardExpYear, cvv);
		return (String) response.get(StripeConstants.CARD_TOKEN);
	}

	private void preProcessingLog(PosTransaction transaction) {
	}

	public boolean checkCardValidation(PosTransaction transaction) throws Exception {
		if (transaction.getCardReader().equals(CardReader.SWIPE.name()) && StringUtils.isNotBlank(transaction.getCardTrack())) {
			return Boolean.TRUE;
		}
		else if (transaction.getCardReader().equals(CardReader.MANUAL.name())) {
			if (StringUtils.isBlank(transaction.getCardNumber())) {
				throw new PosException(IppMessages.getString("SecuredProcessor.0")); //$NON-NLS-1$
			}
			if (!(StringUtils.isNotBlank(transaction.getCardExpYear()) && transaction.getCardExpYear().length() >= 2)) {
				throw new PosException(IppMessages.getString("SecuredProcessor.1")); //$NON-NLS-1$
			}
			if (StringUtils.isBlank(transaction.getCardExpMonth())) {
				throw new PosException(IppMessages.getString("SecuredProcessor.2")); //$NON-NLS-1$
			}
		}
		return Boolean.TRUE;
	}

	private Token createToken(PosTransaction transaction) {
		String cardNumber = transaction.getCardNumber();
		int expMonth = POSUtil.parseInteger(transaction.getCardExpMonth());
		int expYear = POSUtil.parseInteger(transaction.getCardExpYear());
		String cvc = transaction.getCardCVV();

		Ticket ticket = transaction.getTicket();
		try {
			Map<String, Object> tokenParams = new HashMap<String, Object>();
			Map<String, Object> cardParams = new HashMap<String, Object>();
			cardParams.put(StripeConstants.CARD_NUMBER, cardNumber);
			cardParams.put(StripeConstants.EXP_MONTH, expMonth);
			cardParams.put(StripeConstants.EXP_YEAR, expYear);
			cardParams.put(StripeConstants.CARD_CVC, cvc);
			cardParams.put(StripeConstants.ADDRESS_LINE1, ticket.getDeliveryAddress());
			cardParams.put(StripeConstants.ADDRESS_ZIP, ticket.getDeliveryZipCode());
			cardParams.put(StripeConstants.NAME, ticket.getCustomerName());
			tokenParams.put(StripeConstants.CARD, cardParams);
			Token token = Token.create(tokenParams);
			return token;
		} catch (CardException ex) {
			PosLog.error(getClass(), ex.getMessage());
			throw new PosException(ex.getMessage());
		} catch (Exception e0) {
			throw new RuntimeException(e0);
		}
	}

	private boolean isNotSucceeded(Charge charge) {
		return charge.getStatus() == null || !charge.getStatus().equals(StripeConstants.SUCCEEDED) || StringUtils.isNotBlank(charge.getFailureMessage());
	}

	private boolean isNotSucceeded(Refund refund) {
		return refund.getStatus() == null || !refund.getStatus().equals(StripeConstants.SUCCEEDED);
	}

	@Override
	public void voidTransaction(PosTransaction transaction) throws Exception {
		refundTransaction(transaction, transaction.getAmount());
	}

	@Override
	public void refundTransaction(PosTransaction transaction, double refundAmount) throws Exception {
		init(transaction.getTicket().getOutlet());
		this.preProcessingLog(transaction);

		String number = new DecimalFormat("000").format(refundAmount * 100);
		long amount = Long.valueOf(number);
		Refund refund = Refund.create(RefundCreateParams.builder().setAmount(amount).setCharge(transaction.getCardTransactionId()).build());
		if (isNotSucceeded(refund)) {
			PosLog.error(getClass(), "Refund failed. Trans. ID: " + transaction.getCardTransactionId()); //$NON-NLS-1$
			throw new RuntimeException();
		}
		transaction.setRefunded(true);
	}

	@Override
	public String getCardInformationForReceipt(PosTransaction transaction) {
		return null;
	}

	@Override
	public void cancelTransaction() {
	}

}