/**
 * ************************************************************************
 * * The contents of this file are subject to the MRPL 1.2
 * * (the  "License"),  being   the  Mozilla   Public  License
 * * Version 1.1  with a permitted attribution clause; you may not  use this
 * * file except in compliance with the License. You  may  obtain  a copy of
 * * the License at http://www.floreantpos.org/license.html
 * * Software distributed under the License  is  distributed  on  an "AS IS"
 * * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * * License for the specific  language  governing  rights  and  limitations
 * * under the License.
 * * The Original Code is FLOREANT POS.
 * * The Initial Developer of the Original Code is OROCUBE LLC
 * * All portions are Copyright (C) 2015 OROCUBE LLC
 * * All Rights Reserved.
 * ************************************************************************
 */
package com.floreantpos.model;

import java.awt.Color;

import javax.xml.bind.annotation.XmlRootElement;
import javax.xml.bind.annotation.XmlTransient;

import org.apache.commons.lang.StringUtils;

import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.floreantpos.model.base.BaseMenuGroup;
import com.floreantpos.model.dao.MenuCategoryDAO;
import com.floreantpos.model.util.DataProvider;
import com.floreantpos.util.NumberUtil;
import com.floreantpos.util.POSUtil;
import com.google.gson.Gson;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;

@JsonIgnoreProperties(ignoreUnknown = true, value = { "buttonColor", "textColor", "discounts", "menuPages" })
@XmlRootElement(name = "menu-group")
public class MenuGroup extends BaseMenuGroup implements IdContainer, TimedModel {
	private static final long serialVersionUID = 1L;

	public static String PROP_MENU_PAGES = "menuPages"; //$NON-NLS-1$

	private transient JsonObject propertiesContainer;

	private boolean updateLastUpdateTime = true;
	private boolean updateSyncTime = false;

	public boolean isUpdateSyncTime() {
		return updateSyncTime;
	}

	public void setUpdateSyncTime(boolean shouldUpdateSyncTime) {
		this.updateSyncTime = shouldUpdateSyncTime;
	}

	public boolean isUpdateLastUpdateTime() {
		return updateLastUpdateTime;
	}

	public void setUpdateLastUpdateTime(boolean shouldUpdateUpdateTime) {
		this.updateLastUpdateTime = shouldUpdateUpdateTime;
	}

	public MenuGroup() {
	}

	/**
	 * Constructor for primary key
	 */
	public MenuGroup(java.lang.String id) {
		super(id);
	}

	/**
	 * Constructor for required fields
	 */
	public MenuGroup(java.lang.String id, java.lang.String name) {

		super(id, name);
	}

	/*[CONSTRUCTOR MARKER END]*/

	private Color buttonColor;
	private Color textColor;

	private int itemCount;

	@Override
	public Integer getSortOrder() {
		return super.getSortOrder() == null ? 9999 : super.getSortOrder();
	}

	@Override
	public Integer getButtonColorCode() {
		Integer buttonColorCode = super.getButtonColorCode();
		if (buttonColorCode == null || buttonColorCode == 0 || buttonColorCode == -1316371) {
			buttonColorCode = Color.WHITE.getRGB();
		}
		return buttonColorCode;
	}

	@XmlTransient
	public Color getButtonColor() {
		if (buttonColor != null) {
			return buttonColor;
		}

		if (getButtonColorCode() == null || getButtonColorCode() == 0) {
			return null;
		}

		return buttonColor = new Color(getButtonColorCode());
	}

	public void setButtonColor(Color buttonColor) {
		this.buttonColor = buttonColor;
	}

	@XmlTransient
	public Color getTextColor() {
		if (textColor != null) {
			return textColor;
		}

		if (getTextColorCode() == null) {
			return null;
		}

		return textColor = new Color(getTextColorCode());
	}

	public void setTextColor(Color textColor) {
		this.textColor = textColor;
	}

	@JsonIgnore
	public String getDisplayName() {
		Terminal terminal = DataProvider.get().getCurrentTerminal();
		if (terminal != null && terminal.isShowTranslatedName() && StringUtils.isNotEmpty(getTranslatedName())) {
			return getTranslatedName();
		}
		return super.getName();
	}

	public void setMenuCategory(MenuCategory category) {
		if (category == null) {
			setMenuCategoryId(null);
			setMenuCategoryName(null);
			setBeverage(false);
		}
		else {
			setMenuCategoryId(category.getId());
			setMenuCategoryName(category.getDisplayName());
			setBeverage(category.isBeverage());
		}
	}

	public void setParent(MenuCategory category) {
		setMenuCategory(category);
	}

	@XmlTransient
	public MenuCategory getParent() {
		String categoryId = getMenuCategoryId();
		if (StringUtils.isNotEmpty(categoryId)) {
			return MenuCategoryDAO.getInstance().get(categoryId);
		}
		return null;
	}

	public void setItemCount(int count) {
		this.itemCount = count;
	}

	@XmlTransient
	public int getItemCount() {
		return itemCount;
	}

	@Override
	public String toString() {
		return getName();
	}

	@JsonIgnore
	public String getUniqueId() {
		return ("menu_group_" + getName() + "_" + getId()).replaceAll("\\s+", "_"); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$
	}

	@Override
	public String getProperties() {
		if (propertiesContainer != null) {
			return propertiesContainer.toString();
		}

		String properties = super.getProperties();
		if (StringUtils.isEmpty(properties)) {
			return null;
		}

		propertiesContainer = new Gson().fromJson(properties, JsonObject.class);
		return properties;
	}

	@Override
	public void setProperties(String properties) {
		super.setProperties(properties);
		propertiesContainer = new Gson().fromJson(properties, JsonObject.class);
	}

	public void addProperty(String key, String value) {
		if (propertiesContainer == null) {
			propertiesContainer = new JsonObject();
		}
		propertiesContainer.addProperty(key, value);
	}

	public String getProperty(String key) {
		if (propertiesContainer == null) {
			return null;
		}
		if (propertiesContainer.has(key)) {
			JsonElement jsonElement = propertiesContainer.get(key);
			if (!jsonElement.isJsonNull()) {
				return jsonElement.getAsString();
			}
		}
		return null;
	}

	public boolean hasProperty(String key) {
		return getProperty(key) != null;
	}

	public boolean isPropertyValueTrue(String propertyName) {
		String property = getProperty(propertyName);

		return POSUtil.getBoolean(property);
	}

	public void removeProperty(String propertyName) {
		if (propertiesContainer != null) {
			propertiesContainer.remove(propertyName);
		}
	}

	public void putRefPercentage(double refPercentage) {
		addProperty("ref.percentage", String.valueOf(refPercentage)); //$NON-NLS-1$
	}
	
	public double getRefPercentage() {
		return NumberUtil.parseOrGetZero(getProperty("ref.percentage")).doubleValue(); //$NON-NLS-1$
	}

}