/**
 * ************************************************************************
 * * The contents of this file are subject to the MRPL 1.2
 * * (the  "License"),  being   the  Mozilla   Public  License
 * * Version 1.1  with a permitted attribution clause; you may not  use this
 * * file except in compliance with the License. You  may  obtain  a copy of
 * * the License at http://www.floreantpos.org/license.html
 * * Software distributed under the License  is  distributed  on  an "AS IS"
 * * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * * License for the specific  language  governing  rights  and  limitations
 * * under the License.
 * * The Original Code is FLOREANT POS.
 * * The Initial Developer of the Original Code is OROCUBE LLC
 * * All portions are Copyright (C) 2015 OROCUBE LLC
 * * All Rights Reserved.
 * ************************************************************************
 */
package com.floreantpos.model.dao;

import java.io.Serializable;
import java.util.Collection;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.apache.commons.beanutils.PropertyUtils;
import org.apache.commons.logging.LogFactory;
import org.hibernate.Session;
import org.hibernate.Transaction;

import com.floreantpos.PosLog;
import com.floreantpos.model.ActionHistory;
import com.floreantpos.model.Discount;
import com.floreantpos.model.Gratuity;
import com.floreantpos.model.InventoryTransaction;
import com.floreantpos.model.PosTransaction;
import com.floreantpos.model.SalesArea;
import com.floreantpos.model.ShopTable;
import com.floreantpos.model.Ticket;
import com.floreantpos.model.TicketDiscount;
import com.floreantpos.model.TicketItem;
import com.floreantpos.model.TicketItemDiscount;
import com.floreantpos.model.User;
import com.floreantpos.model.util.DataProvider;
import com.floreantpos.util.NumberUtil;
import com.orocube.rest.service.server.BaseDataServiceDao;

public class ActionHistoryDAO extends BaseActionHistoryDAO {

	/**
	 * Default constructor.  Can be used in place of getInstance()
	 */
	public ActionHistoryDAO() {
	}

	@Override
	protected Serializable save(Object obj, Session s) {
		updateTime(obj);
		return super.save(obj, s);
	}

	@Override
	protected void update(Object obj, Session s) {
		updateTime(obj);
		super.update(obj, s);
	}

	@Override
	protected void saveOrUpdate(Object obj, Session s) {
		updateTime(obj);
		super.saveOrUpdate(obj, s);
	}

	public void saveOrUpdateActionHistory(List<ActionHistory> actionHistoryList, boolean updateLastUpdateTime, boolean updateSyncTime) throws Exception {
		if (actionHistoryList == null)
			return;

		Transaction tx = null;
		try (Session session = createNewSession()) {
			tx = session.beginTransaction();
			for (Iterator<ActionHistory> iterator = actionHistoryList.iterator(); iterator.hasNext();) {
				ActionHistory item = (ActionHistory) iterator.next();
				ActionHistory existingItem = get(item.getId());
				if (existingItem != null) {
					if (!BaseDataServiceDao.get().shouldSave(item.getLastUpdateTime(), existingItem.getLastUpdateTime())) {
						PosLog.info(getClass(), item.getActionName() + " already updated"); //$NON-NLS-1$
						continue;
					}
					long version = existingItem.getVersion();
					PropertyUtils.copyProperties(existingItem, item);
					existingItem.setVersion(version);
					existingItem.setUpdateLastUpdateTime(updateLastUpdateTime);
					existingItem.setUpdateSyncTime(updateSyncTime);
					update(existingItem, session);
				}
				else {
					item.setVersion(0);
					item.setUpdateLastUpdateTime(updateLastUpdateTime);
					item.setUpdateSyncTime(updateSyncTime);
					save(item, session);
				}
			}
			tx.commit();
		} catch (Exception e) {
			tx.rollback();
			throw e;
		}
	}

	public static void saveHistory(User performer, Ticket ticket, PosTransaction posTransaction, String actionName, String description) {
		Transaction transaction = null;
		try (Session session = ActionHistoryDAO.getInstance().createNewSession()) {
			transaction = session.beginTransaction();
			saveHistory(performer, ticket, posTransaction, actionName, description, session);
			transaction.commit();
		} catch (Exception e) {
			if (transaction != null) {
				transaction.rollback();
			}
			LogFactory.getLog(ActionHistoryDAO.class).error("Error occured while trying to save action history", e); //$NON-NLS-1$
		}
	}

	public static void saveHistory(User performer, Ticket ticket, String actionName, String description) {
		saveHistory(performer, ticket, null, actionName, description);
	}

	public static void saveHistory(User performer, String actionName, String description) {
		saveHistory(performer, null, null, actionName, description);
	}

	public static void saveHistory(String actionName, String description, Session session) {
		//TODO:
		saveHistory(null, null, null, actionName, description, session);
	}

	public static void saveHistory(User performer, Ticket ticket, PosTransaction transaction, String actionName, String description, Session session) {
		ActionHistory history = new ActionHistory();
		history.setActionName(actionName);
		history.setDescription(description);
		history.setPerformer(performer);
		history.setActionTime(StoreDAO.getServerTimestamp());
		history.setOutletId(DataProvider.get().getOutletId());
		
		if (ticket != null) {
			history.setTicketId(ticket.getId());
		}
		if (transaction != null) {
			history.setTransactionId(transaction.getId());
		}

		ActionHistoryDAO.getInstance().save(history, session);
	}

	public void addInventoryTransactionActionHistory(InventoryTransaction inventoryTransaction, User performer) {
		ActionHistory history = new ActionHistory();
		history.setActionTime(new Date());
		String description = String.format("Stock reason: %s, Item Id: %s, Item name: %s, Quantity: %s %s", //$NON-NLS-1$
				inventoryTransaction.getReason() != null ? inventoryTransaction.getReason() : "", //$NON-NLS-1$
				inventoryTransaction.getMenuItem().getId(), inventoryTransaction.getItemName(),
				NumberUtil.trimDecilamIfNotNeeded(inventoryTransaction.getQuantity()), inventoryTransaction.getUnit());

		history.setActionName("STOCK " + inventoryTransaction.getTransactionType().getName()); //$NON-NLS-1$
		history.setPerformer(performer);
		history.setDescription(description);
		history.setActionTime(StoreDAO.getServerTimestamp());
		history.setTicketId(inventoryTransaction.getTicketId());
		history.setOutletId(inventoryTransaction.getOutletId());
		history.setTransactionId(inventoryTransaction.getId());
		ActionHistoryDAO.getInstance().save(history);
	}

	private static void save(Map<String, ActionHistory> historyMap) {
		ActionHistoryDAO instance2 = getInstance();
		Transaction transaction = null;
		try (Session session = instance2.createNewSession()) {
			transaction = session.beginTransaction();
			for (ActionHistory actionHistory : historyMap.values()) {
				instance2.save(actionHistory, session);
			}
			transaction.commit();
		} catch (Exception e) {
			if (transaction != null) {
				transaction.rollback();
			}
		} finally {
			historyMap.clear();
		}
	}

	public void performActionHistorySaveOperation(Ticket ticket) {
		performActionHistorySaveOperation(ticket, false);
	}

	public void performActionHistorySaveOperation(Ticket ticket, boolean newOrder) {

	}

	private void populateEventsDescription(Ticket ticket) {
		if (ticket == null) {
			return;
		}
		Map<String, ActionHistory> events = ticket.getEvents();
		if (events.isEmpty()) {
			return;
		}
		Collection<ActionHistory> histories = events.values();
		histories.forEach(history -> {
			Object objectReference = history.getObjectReference();
			String description = history.getDescription();
			if (objectReference instanceof TicketDiscount) {
				TicketDiscount ticketDiscount = (TicketDiscount) objectReference;
				if (ticketDiscount.getType() == Discount.DISCOUNT_TYPE_LOYALTY) {
					description = String.format("Ticket id : %s, Loyalty points : %s", ticketDiscount.getTicket().getId(), ticketDiscount.getLoyaltyPoint()); //$NON-NLS-1$
				}
				else {
					description = String.format("Ticket id : %s, Discount id : %s, Coupon quantity : %s", ticketDiscount.getTicket().getId(), //$NON-NLS-1$
							ticketDiscount.getDiscountId(), ticketDiscount.getCouponQuantity());
				}
			}
			else if (objectReference instanceof TicketItemDiscount) {
				TicketItemDiscount itemDiscount = (TicketItemDiscount) objectReference;
				TicketItem ticketItem = itemDiscount.getTicketItem();
				description = String.format("Ticket id : %s, Item id : %s, Discount id : %s, Coupon quantity : %s", ticket != null ? ticket.getId() : "", //$NON-NLS-2$
						ticketItem != null ? ticketItem.getId() : "", itemDiscount.getDiscountId(), itemDiscount.getCouponQuantity());
			}
			else if (history.getActionName() == ActionHistory.TAX_EXEMPT && ticket.isTaxExempt()) {
				String exemptId = ticket.getProperty("tax_exempt_id");
				description = String.format("Ticket id : %s, Exempt id : %s", ticket.getId(), exemptId);
			}
			history.setDescription(description);
		});
	}

	public static void addDiscountRemovedActionHistory(Ticket ticket, TicketDiscount ticketDiscount) {
		if (ticket == null) {
			return;
		}
		String key = String.valueOf(ticketDiscount.hashCode());
		if (ticket.getEvents().containsKey(key)) {
			ticket.getEvents().remove(key);
		}
		else {
			String action = ticketDiscount.getType() == Discount.DISCOUNT_TYPE_LOYALTY ? ActionHistory.LOYALTY_REMOVED : ActionHistory.TICKET_DISCOUNT_REMOVED;
			addActionHistory(ticket, key, action, null, ticketDiscount);
		}
	}

	public static void addDiscountAddedActionHistory(Ticket ticket, TicketDiscount ticketDiscount) {
		if (ticket == null) {
			return;
		}
		String key = String.valueOf(ticketDiscount.hashCode());
		String action = ticketDiscount.getType() == Discount.DISCOUNT_TYPE_LOYALTY ? ActionHistory.LOYALTY_ADDED : ActionHistory.TICKET_DISCOUNT_ADDED;
		addActionHistory(ticket, key, action, null, ticketDiscount);
	}

	public static void addSetGratuityActionHistory(Ticket ticket, Gratuity gratuity) {
		if (ticket == null) {
			return;
		}
		if (ticket.getId() == null) {
			return;
		}
		String key, action;
		key = action = ActionHistory.GRATUITY_ADDED;
		String description = String.format("Ticket id : %s, Gratuity amount : %s", ticket.getId(), gratuity == null ? 0 : gratuity.getAmount()); //$NON-NLS-1$
		addActionHistory(ticket, key, action, description, gratuity);
	}

	public static void addTableAddedActionHistory(Ticket ticket, List<ShopTable> tables) {
		if (ticket == null) {
			return;
		}
		if (ticket.getId() == null) {
			return;
		}
		if (tables == null) {
			return;
		}
		String key, action;
		key = action = ActionHistory.TABLE_CHANGE;
		StringBuilder tableIds = new StringBuilder();
		Iterator<ShopTable> iterator = tables.iterator();
		while (iterator.hasNext()) {
			ShopTable table = iterator.next();
			tableIds.append(table.getId());
			if (iterator.hasNext()) {
				tableIds.append(", "); //$NON-NLS-1$
			}
		}
		String description = String.format("Ticket id : %s, Tables : %s", ticket.getId(), tableIds.toString()); //$NON-NLS-1$
		addActionHistory(ticket, key, action, description, tables);
	}

	public static void addSalesAreaAddedActionHistory(Ticket ticket, SalesArea salesArea) {
		if (ticket == null) {
			return;
		}
		if (ticket.getId() == null) {
			return;
		}
		if (salesArea == null) {
			return;
		}
		String key, action;
		key = action = ActionHistory.SALES_AREA_CHANGE;
		String description = String.format("Ticket id : %s, Sales area : %s", ticket.getId(), salesArea.getName()); //$NON-NLS-1$
		addActionHistory(ticket, key, action, description, salesArea);
	}

	public static void addTicketItemDiscountRemovedActionHistory(Ticket ticket, TicketItemDiscount itemDiscount) {
		if (ticket == null) {
			return;
		}
		String key = String.valueOf(itemDiscount.hashCode());
		if (ticket.getEvents().containsKey(key)) {
			ticket.getEvents().remove(key);
		}
		else {
			addActionHistory(ticket, key, ActionHistory.ITEM_DISCOUNT_REMOVED, null, itemDiscount);
		}
	}

	public static void addTicketItemDiscountAddedActionHistory(Ticket ticket, TicketItemDiscount itemDiscount) {
		if (ticket == null) {
			return;
		}
		String key = String.valueOf(itemDiscount.hashCode());
		addActionHistory(ticket, key, ActionHistory.ITEM_DISCOUNT_ADDED, null, itemDiscount);
	}

	public static void addItemVoidedActionHistory(Ticket ticket, TicketItem ticketItem, double voidQuantity) {
		if (ticket == null || ticket.getId() == null || ticketItem == null) {
			return;
		}
		String key = String.valueOf(ticketItem.hashCode());
		String action = ActionHistory.TICKET_ITEM_VOID;
		String description = String.format("Ticket id : %s, Item id : %s, Void quantity : %s", ticket.getId(), ticketItem.getId(), voidQuantity);
		addActionHistory(ticket, key, action, description, ticketItem);
	}

	public static void addActionHistory(Ticket ticket, String action, String description) {
		addActionHistory(ticket, action, action, description);
	}

	public static void addActionHistory(Ticket ticket, String key, String action, String description) {
		addActionHistory(ticket, key, action, description, null);
	}

	public static void addActionHistory(Ticket ticket, String key, String action, String description, Object objectReference) {
	}

	public static void saveActionHistory(String action, String description, User user, String outletId) {
		ActionHistoryDAO.getInstance().save(ActionHistory.create(action, description, user, outletId));
	}
}