/**
 * ************************************************************************
 * * The contents of this file are subject to the MRPL 1.2
 * * (the  "License"),  being   the  Mozilla   Public  License
 * * Version 1.1  with a permitted attribution clause; you may not  use this
 * * file except in compliance with the License. You  may  obtain  a copy of
 * * the License at http://www.floreantpos.org/license.html
 * * Software distributed under the License  is  distributed  on  an "AS IS"
 * * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * * License for the specific  language  governing  rights  and  limitations
 * * under the License.
 * * The Original Code is FLOREANT POS.
 * * The Initial Developer of the Original Code is OROCUBE LLC
 * * All portions are Copyright (C) 2015 OROCUBE LLC
 * * All Rights Reserved.
 * ************************************************************************
 */
package com.floreantpos.model.dao;

import java.io.File;
import java.net.URL;
import java.sql.Connection;
import java.util.Arrays;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.lang.StringUtils;
import org.hibernate.Criteria;
import org.hibernate.Session;
import org.hibernate.boot.Metadata;
import org.hibernate.boot.MetadataSources;
import org.hibernate.boot.SessionFactoryBuilder;
import org.hibernate.boot.registry.StandardServiceRegistry;
import org.hibernate.boot.registry.StandardServiceRegistryBuilder;
import org.hibernate.cfg.AvailableSettings;
import org.hibernate.cfg.Configuration;
import org.hibernate.criterion.DetachedCriteria;
import org.hibernate.criterion.MatchMode;
import org.hibernate.criterion.Projections;
import org.hibernate.criterion.Restrictions;
import org.hibernate.internal.CriteriaImpl;

import com.floreantpos.Database;
import com.floreantpos.PosLog;
import com.floreantpos.config.AppConfig;
import com.floreantpos.constants.AppConstants;
import com.floreantpos.model.TimedModel;
import com.floreantpos.model.User;

public abstract class _RootDAO extends com.floreantpos.model.dao._BaseRootDAO {
	public static final String LUCENE_INDEX_DIR = "/opt/menugreat_lucene_index"; //$NON-NLS-1$
	/*
	 * If you are using lazy loading, uncomment this Somewhere, you should call
	 * RootDAO.closeCurrentThreadSessions(); public void closeSession (Session
	 * session) { // do nothing here because the session will be closed later }
	 */

	/*
	 * If you are pulling the SessionFactory from a JNDI tree, uncomment this
	 * protected SessionFactory getSessionFactory(String configFile) { // If you
	 * have a single session factory, ignore the configFile parameter //
	 * Otherwise, you can set a meta attribute under the class node called
	 * "config-file" which // will be passed in here so you can tell what
	 * session factory an individual mapping file // belongs to return
	 * (SessionFactory) new
	 * InitialContext().lookup("java:/{SessionFactoryName}"); }
	 */
	private static StandardServiceRegistry standardServiceRegistry;

	public static void initialize() {
		Database database = AppConfig.getDefaultDatabase();
		String connectString = AppConfig.getConnectString();
		String databaseUser = AppConfig.getDatabaseUser();
		String databasePassword = AppConfig.getDatabasePassword();
		initialize("oropos.hibernate.cfg.xml", database, connectString, databaseUser, databasePassword); //$NON-NLS-1$
	}

	public static void initialize(String hibernanetCfgFile) {
		Database database = AppConfig.getDefaultDatabase();
		String connectString = AppConfig.getConnectString();
		String databaseUser = AppConfig.getDatabaseUser();
		String databasePassword = AppConfig.getDatabasePassword();
		initialize(hibernanetCfgFile, database, connectString, databaseUser, databasePassword); //$NON-NLS-1$
	}

	public static void initialize(String hibernanetCfgFile, Database database, String connectString, String databaseUser, String databasePassword) {
		initialize(hibernanetCfgFile, database.getHibernateDialect(), database.getHibernateConnectionDriverClass(), connectString, databaseUser,
				databasePassword);
	}

	public static void initialize(String hibernanetCfgFile, String hibernateDialectClass, String driverClass, String connectString, String databaseUser,
			String databasePassword) {
		Map<String, String> map = new HashMap<>();
		map.put("hibernate.dialect", hibernateDialectClass); //$NON-NLS-1$
		map.put("hibernate.connection.driver_class", driverClass); //$NON-NLS-1$
		map.put("hibernate.connection.url", connectString); //$NON-NLS-1$
		map.put("hibernate.connection.username", databaseUser); //$NON-NLS-1$
		map.put("hibernate.connection.password", databasePassword); //$NON-NLS-1$
		map.put("hibernate.connection.autocommit", "false"); //$NON-NLS-1$ //$NON-NLS-2$
		map.put("hibernate.max_fetch_depth", "3"); //$NON-NLS-1$ //$NON-NLS-2$
		//		map.put("hibernate.jdbc.time_zone", "UTC"); //$NON-NLS-1$ //$NON-NLS-2$
		map.put("hibernate.show_sql", "false"); //$NON-NLS-1$ //$NON-NLS-2$
		map.put("hibernate.connection.isolation", String.valueOf(Connection.TRANSACTION_READ_UNCOMMITTED)); //$NON-NLS-1$
		map.put("hibernate.cache.use_second_level_cache", "true"); //$NON-NLS-1$ //$NON-NLS-2$
		//map.put("hibernate.cache.use_query_cache", "true"); //$NON-NLS-1$ //$NON-NLS-2$
		map.put("hibernate.cache.provider_class", "org.hibernate.cache.EhCacheProvider"); //$NON-NLS-1$ //$NON-NLS-2$
		map.put("hibernate.cache.region.factory_class", "org.hibernate.cache.ehcache.SingletonEhCacheRegionFactory"); //$NON-NLS-1$ //$NON-NLS-2$

		initialize(hibernanetCfgFile, map);
	}

	public static void initialize(Map<String, String> properties) {
		initialize("cloudpos.hibernate.cfg.xml", properties); //$NON-NLS-1$
	}

	public static void initialize(String hibernanetCfgFile, Map<String, String> properties) {
		releaseConnection();

		properties.put(AvailableSettings.AUTOCOMMIT, "false"); //$NON-NLS-1$
		properties.put(AvailableSettings.MAX_FETCH_DEPTH, "3"); //$NON-NLS-1$
		properties.put(AvailableSettings.SHOW_SQL, "false"); //$NON-NLS-1$
		properties.put(AvailableSettings.USE_SECOND_LEVEL_CACHE, "false"); //$NON-NLS-1$
		properties.put("hibernate.connection.isolation", String.valueOf(Connection.TRANSACTION_READ_UNCOMMITTED)); //$NON-NLS-1$

		StandardServiceRegistryBuilder registryBuilder = new StandardServiceRegistryBuilder();
		URL resource = _RootDAO.class.getResource("/" + hibernanetCfgFile); //$NON-NLS-1$

		PosLog.debug(_RootDAO.class, resource == null ? "hibernate config file not found" : resource.toExternalForm());

		registryBuilder.configure(resource);
		registryBuilder.applySettings(properties);

		setupCconnectionPoolSettings(registryBuilder);

		standardServiceRegistry = registryBuilder.build();

		Metadata metaData = new MetadataSources(standardServiceRegistry).getMetadataBuilder().build();

		SessionFactoryBuilder sessionFactoryBuilder = metaData.getSessionFactoryBuilder();

		setSessionFactory(sessionFactoryBuilder.applyInterceptor(new PosDataInterceptor()).build());

		PosLog.debug(_RootDAO.class, "lucene index dir: " + new File(LUCENE_INDEX_DIR).getAbsolutePath());
	}

	public static void setupCconnectionPoolSettings(StandardServiceRegistryBuilder builder) {
		//min pool size
		builder.applySetting("hibernate.c3p0.min_size", "0"); //$NON-NLS-1$ //$NON-NLS-2$
		//max pool size
		builder.applySetting("hibernate.c3p0.max_size", "100"); //$NON-NLS-1$ //$NON-NLS-2$
		// When an idle connection is removed from the pool (in second)
		builder.applySetting("hibernate.c3p0.timeout", "30"); //$NON-NLS-1$ //$NON-NLS-2$
		//Number of prepared statements will be cached
		builder.applySetting("hibernate.c3p0.max_statements", "0"); //$NON-NLS-1$ //$NON-NLS-2$
		//The number of milliseconds a client calling getConnection() will wait for a Connection to be 
		//checked-in or acquired when the pool is exhausted. Zero means wait indefinitely.
		//Setting any positive value will cause the getConnection() call to time-out and break 
		//with an SQLException after the specified number of milliseconds. 
		builder.applySetting("hibernate.c3p0.checkoutTimeout", "15000"); //$NON-NLS-1$ //$NON-NLS-2$
		builder.applySetting("hibernate.c3p0.acquireRetryAttempts", "5"); //$NON-NLS-1$ //$NON-NLS-2$
		//builder.applySetting("hibernate.c3p0.acquireRetryDelay", "1000"); //$NON-NLS-1$ //$NON-NLS-2$
		builder.applySetting("hibernate.c3p0.acquireIncrement", "10"); //$NON-NLS-1$ //$NON-NLS-2$
		//builder.applySetting("hibernate.c3p0.maxIdleTime", "3000"); //$NON-NLS-1$ //$NON-NLS-2$
		builder.applySetting("testConnectionOnCheckout", "false"); //$NON-NLS-1$ //$NON-NLS-2$
		//idle time in seconds before a connection is automatically validated
		//builder.applySetting("hibernate.c3p0.idle_test_period", "3000"); //$NON-NLS-1$ //$NON-NLS-2$
		builder.applySetting("hibernate.c3p0.breakAfterAcquireFailure", "false"); //$NON-NLS-1$ //$NON-NLS-2$
	}

	public static void setupHikariCPSettings(StandardServiceRegistryBuilder builder) {
		builder.applySetting("hibernate.connection.provider_class", "org.hibernate.hikaricp.internal.HikariCPConnectionProvider"); //$NON-NLS-1$ //$NON-NLS-2$
		builder.applySetting("hibernate.hikari.idleTimeout", "30000"); //$NON-NLS-1$ //$NON-NLS-2$
		builder.applySetting("hibernate.hikari.minimumIdle", "0"); //$NON-NLS-1$ //$NON-NLS-2$
		builder.applySetting("hibernate.hikari.maximumPoolSize", "5"); //$NON-NLS-1$ //$NON-NLS-2$
	}

	public static void initialize(String configFileName, Configuration configuration) {
		releaseConnection();
		initialize();
	}

	public static void releaseConnection() {
		closeCurrentThreadSessions();

		if (sessionFactory != null && !sessionFactory.isClosed()) {
			sessionFactory.close();
		}
		if (standardServiceRegistry != null) {
			StandardServiceRegistryBuilder.destroy(standardServiceRegistry);
		}
	}

	public void refresh(Object obj) {
		try (Session session = createNewSession()) {
			super.refresh(obj, session);
		}
	}

	public int rowCount() {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.setProjection(Projections.rowCount());
			return ((Long) criteria.uniqueResult()).intValue();
		}
	}

	public int rowCount(Criteria criteria) {
		criteria.setProjection(Projections.rowCount());
		int intValue = ((Long) criteria.uniqueResult()).intValue();
		criteria.setProjection(null);
		return intValue;
	}

	public List getPageData(int pageNum, int pageSize) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.setFirstResult(pageNum * pageSize);
			criteria.setMaxResults(pageSize);

			return criteria.list();
		}
	}

	public int rowCount(Map<String, Object> restrictions) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.setProjection(Projections.rowCount());

			if (restrictions != null) {
				for (String key : restrictions.keySet()) {
					criteria.add(Restrictions.eq(key, restrictions.get(key)));
				}
			}
			Number result = (Number) criteria.uniqueResult();
			if (result != null) {
				return result.intValue();
			}
			return 0;
		}
	}

	public List getPageData(int pageNum, int pageSize, Map<String, Object> restrictions) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.setFirstResult(pageNum * pageSize);
			criteria.setMaxResults(pageSize);

			if (restrictions != null) {
				for (String key : restrictions.keySet()) {
					criteria.add(Restrictions.eq(key, restrictions.get(key)));
				}
			}
			return criteria.list();
		}
	}

	protected void updateTime(Object model) {
		if (!(model instanceof TimedModel)) {
			return;
		}
		TimedModel timedModel = (TimedModel) model;
		Date now = StoreDAO.getServerTimestamp();
		if (timedModel.isUpdateLastUpdateTime()) {
			timedModel.setLastUpdateTime(now);
		}
		if (timedModel.isUpdateSyncTime()) {
			timedModel.setLastSyncTime(now);
		}
	}

	private boolean containsDeletedField(Class modelClass) {
		try {
			if (modelClass == null) {
				modelClass = this.getReferenceClass();
			}
			return Arrays.stream(modelClass.getFields()).anyMatch(f -> f.getName().equalsIgnoreCase("PROP_DELETED")); //$NON-NLS-1$
		} catch (Exception e) {
			//PosLog.error(getReferenceClass(), e);
		}
		return false;
	}

	public void addDeletedFilter(Criteria criteria) {
		Class modelClass = null;
		try {
			if (criteria instanceof CriteriaImpl) {
				CriteriaImpl criteriaImpl = (CriteriaImpl) criteria;
				String className = criteriaImpl.getEntityOrClassName();
				if (StringUtils.isNotBlank(className)) {
					modelClass = Class.forName(className);
				}
			}
		} catch (Exception e) {
			//PosLog.error(this.getClass(), e);
		}
		this.addDeletedFilter(criteria, modelClass);
	}

	public void addDeletedFilter(Criteria criteria, Class modelClass) {
		if (this.containsDeletedField(modelClass)) {
			criteria.add(Restrictions.or(Restrictions.isNull(AppConstants.PROP_DELETED), Restrictions.eq(AppConstants.PROP_DELETED, Boolean.FALSE)));
		}
		/*else {
			PosLog.info(modelClass, "Deleted field not found in the model"); //$NON-NLS-1$
		}*/
	}

	public static void addUserWithAllRoleCriteria(Criteria criteria, User user, String fieldName) {
		if (user != null) {
			criteria.add(Restrictions.in(fieldName, user.getRoleIds()));
		}
	}

	public static void addUserWithAllRoleCriteria(Criteria criteria, User user, String fieldName, boolean isIncludeAllRole) {
		if (user != null) {
			if (isIncludeAllRole) {
				criteria.add(Restrictions.in(fieldName, user.getRoleIds()));
			}
			else {
				criteria.add(Restrictions.eq(fieldName, user.getId()));
			}
		}
	}

	public void addDetachedCriteriaProperyFilter(DetachedCriteria detachedCriteria, String fieldName, String compareKey, Object compareValue) {
		detachedCriteria.add(Restrictions.ilike(fieldName, "\"" + compareKey + "\":\"" + compareValue + "\"", MatchMode.ANYWHERE)); //$NON-NLS-1$ //$NON-NLS-2$
	}
}