/**
 * ************************************************************************
 * * The contents of this file are subject to the MRPL 1.2
 * * (the  "License"),  being   the  Mozilla   Public  License
 * * Version 1.1  with a permitted attribution clause; you may not  use this
 * * file except in compliance with the License. You  may  obtain  a copy of
 * * the License at http://www.floreantpos.org/license.html
 * * Software distributed under the License  is  distributed  on  an "AS IS"
 * * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * * License for the specific  language  governing  rights  and  limitations
 * * under the License.
 * * The Original Code is FLOREANT POS.
 * * The Initial Developer of the Original Code is OROCUBE LLC
 * * All portions are Copyright (C) 2015 OROCUBE LLC
 * * All Rights Reserved.
 * ************************************************************************
 */
package com.floreantpos.model.dao;

import java.io.Serializable;
import java.util.List;

import org.apache.commons.beanutils.PropertyUtils;
import org.hibernate.Criteria;
import org.hibernate.Session;
import org.hibernate.criterion.Criterion;
import org.hibernate.criterion.Restrictions;

import com.floreantpos.Messages;
import com.floreantpos.PosException;
import com.floreantpos.PosLog;
import com.floreantpos.model.User;
import com.floreantpos.model.UserType;
import com.orocube.rest.service.server.BaseDataServiceDao;

public class UserTypeDAO extends BaseUserTypeDAO {

	/**
	 * Default constructor.  Can be used in place of getInstance()
	 */
	public UserTypeDAO() {
	}

	@Override
	protected Serializable save(Object obj, Session s) {
		updateTime(obj);
		return super.save(obj, s);
	}

	@Override
	protected void update(Object obj, Session s) {
		updateTime(obj);
		super.update(obj, s);
	}

	@Override
	protected void delete(Object obj, Session session) {
		UserType userType = (UserType) obj;
		if (userType == null) {
			throw new PosException(Messages.getString("UserTypeDAO.0")); //$NON-NLS-1$
		}

		ensureUserTypeNotInUse(session, userType);

		userType.setDeleted(true);
		update(userType, session);
	}

	private void ensureUserTypeNotInUse(Session session, UserType userType) {
		Criteria criteria = session.createCriteria(User.class);
		addDeletedFilter(criteria);
		criteria.add(Restrictions.eq(User.PROP_USER_TYPE_ID, userType.getId()));
		List<User> users = criteria.list();
		if (users != null && users.size() > 0) {
			throw new PosException(Messages.getString("UserTypeDAO.1")); //$NON-NLS-1$
		}
	}

	@Override
	public List<UserType> findAll() {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			addDeletedFilter(criteria);
			return criteria.list();
		}
	}

	@Override
	protected void saveOrUpdate(Object obj, Session s) {
		updateTime(obj);
		super.saveOrUpdate(obj, s);
	}

	public List<UserType> findAllUnSyncUserType() {
		Session session = null;
		try {
			session = createNewSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			Criterion falseCriteria = Restrictions.eq(UserType.PROP_CLOUD_SYNCED, Boolean.FALSE);
			Criterion nullCriteria = Restrictions.isNull(UserType.PROP_CLOUD_SYNCED);
			criteria.add(Restrictions.or(falseCriteria, nullCriteria));
			return criteria.list();
		} finally {
			if (session != null) {
				closeSession(session);
			}
		}
	}

	public void saveOrUpdateUserTypes(List<UserType> userTypes, boolean updateLastUpdateTime, boolean updateSyncTime) throws Exception {
		if (userTypes == null)
			return;

		try {
			for (UserType userType : userTypes) {
				UserType existingUserType = get(userType.getId());
				if (existingUserType != null) {
					if (!BaseDataServiceDao.get().shouldSave(userType.getLastUpdateTime(), existingUserType.getLastUpdateTime())) {
						PosLog.info(getClass(), userType.getName() + " already updated"); //$NON-NLS-1$
						continue;
					}
					final String id = existingUserType.getId();
					long version = existingUserType.getVersion();
					PropertyUtils.copyProperties(existingUserType, userType);
					existingUserType.setId(id);
					existingUserType.setVersion(version);
					existingUserType.setUpdateLastUpdateTime(updateLastUpdateTime);
					existingUserType.setUpdateSyncTime(updateSyncTime);
					update(existingUserType);
				}
				else {
					userType.setUpdateLastUpdateTime(updateLastUpdateTime);
					userType.setUpdateSyncTime(updateSyncTime);
					save(userType);
				}
			}
		} catch (Exception e) {
			throw e;
		} finally {
		}

	}

	public UserType findRoleByName(String roleName) {
		try (Session session = createNewSession()) {
			return findRoleByName(roleName, session);
		}
	}

	public UserType findRoleByName(String roleName, Session session) {
		Criteria criteria = session.createCriteria(getReferenceClass());
		addDeletedFilter(criteria);
		criteria.add(Restrictions.eq(UserType.PROP_NAME, roleName).ignoreCase());
		criteria.setMaxResults(1);
		return (UserType) criteria.uniqueResult();
	}
}