/**
 * ************************************************************************
 * * The contents of this file are subject to the MRPL 1.2
 * * (the  "License"),  being   the  Mozilla   Public  License
 * * Version 1.1  with a permitted attribution clause; you may not  use this
 * * file except in compliance with the License. You  may  obtain  a copy of
 * * the License at http://www.floreantpos.org/license.html
 * * Software distributed under the License  is  distributed  on  an "AS IS"
 * * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * * License for the specific  language  governing  rights  and  limitations
 * * under the License.
 * * The Original Code is FLOREANT POS.
 * * The Initial Developer of the Original Code is OROCUBE LLC
 * * All portions are Copyright (C) 2015 OROCUBE LLC
 * * All Rights Reserved.
 * ************************************************************************
 */
package com.floreantpos.model;

import static com.floreantpos.util.NumberUtil.convertToBigDecimal;
import static com.floreantpos.util.NumberUtil.round;

import java.io.StringReader;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.List;

import javax.json.Json;
import javax.json.JsonArray;
import javax.json.JsonObject;
import javax.json.JsonReader;
import javax.xml.bind.annotation.XmlTransient;

import org.apache.commons.lang.StringUtils;
import org.json.JSONArray;
import org.json.JSONObject;

import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.floreantpos.constants.AppConstants;
import com.floreantpos.model.base.BaseTicketItemModifier;
import com.floreantpos.model.ext.KitchenStatus;
import com.floreantpos.model.util.DataProvider;
import com.floreantpos.model.util.pricecalc.TicketItemModifierCalcFactory;
import com.floreantpos.util.JsonUtil;
import com.floreantpos.util.NumberUtil;
import com.floreantpos.util.OrgJsonUtil;
import com.floreantpos.util.POSUtil;
import com.google.gson.Gson;
import com.google.gson.JsonElement;

@JsonIgnoreProperties(ignoreUnknown = true, value = { "itemQuantityDisplay" })
public class TicketItemModifier extends BaseTicketItemModifier implements ITicketItem, PropertyContainer {
	private static final long serialVersionUID = 1L;

	public final static int NORMAL_MODIFIER = 1;
	public final static int EXTRA_MODIFIER = 3;
	public final static int CRUST = 5;
	public final static int SEPERATOR = 6;

	public final static String TRANSIENT_PROP_TICKET_ITEM_QUANTITY = "ticketItemQuantity"; //$NON-NLS-1$

	public MenuModifier menuModifier;
	private boolean selected;
	private int tableRowNum;
	private double ticketItemQuantity;
	private List<TicketItemTax> taxes;
	private transient com.google.gson.JsonObject propertiesContainer;

	public TicketItemModifier() {
	}

	/**
	 * Constructor for primary key
	 */
	public TicketItemModifier(java.lang.String id) {
		super(id);
	}

	public TicketItemModifier(TicketItem ticketItem, String name, double unitPrice, double quantity) {
		setTicketItem(ticketItem);
		setName(name);
		setUnitPrice(unitPrice);
		setItemQuantity(quantity);
	}

	public TicketItemModifier(String id, TicketItem ticketItem, String name, double unitPrice, double quantity, TicketItemTax tax) {
		setId(id);
		setTicketItem(ticketItem);
		setName(name);
		setUnitPrice(unitPrice);
		setItemQuantity(quantity);
		addTotaxes(tax);
	}

	public int getTableRowNum() {
		return tableRowNum;
	}

	public void setTableRowNum(int tableRowNum) {
		this.tableRowNum = tableRowNum;
	}

	@Override
	public String toString() {
		return getName();
	}

	@Override
	public boolean canAddCookingInstruction() {
		return false;
	}

	public void calculatePrice() {
		TicketItemModifierCalcFactory.geCalc(getTicketItem()).calculatePrice(this);
	}

	public List<TicketItemDiscount> getDiscounts() {
		List<TicketItemDiscount> discounts = new ArrayList<>();

		String property = getProperty("discountDetail"); //$NON-NLS-1$
		if (StringUtils.isNotEmpty(property)) {
			JSONArray jsonArray = new JSONArray(property);
			for (int i = 0; i < jsonArray.length(); i++) {
				JSONObject jsonObject = jsonArray.getJSONObject(i);
				TicketItemDiscount tid = new TicketItemDiscount();
				tid.setTicketItem(this.getTicketItem());
				tid.setDiscountId(OrgJsonUtil.getString(jsonObject, TicketItemDiscount.PROP_DISCOUNT_ID));
				tid.setName(OrgJsonUtil.getString(jsonObject, TicketItemDiscount.PROP_NAME));
				tid.setType(OrgJsonUtil.getInt(jsonObject, TicketItemDiscount.PROP_TYPE));
				tid.setAutoApply(OrgJsonUtil.getBoolean(jsonObject, TicketItemDiscount.PROP_AUTO_APPLY));
				tid.setCouponQuantity(OrgJsonUtil.getDouble(jsonObject, TicketItemDiscount.PROP_COUPON_QUANTITY));
				tid.setMinimumAmount(OrgJsonUtil.getDouble(jsonObject, TicketItemDiscount.PROP_MINIMUM_AMOUNT));
				tid.setValue(OrgJsonUtil.getDouble(jsonObject, TicketItemDiscount.PROP_VALUE));
				tid.setAmount(OrgJsonUtil.getDouble(jsonObject, TicketItemDiscount.PROP_AMOUNT));
				discounts.add(tid);
			}
		}
		return discounts;
	}

	public BigDecimal calculateDiscount(BigDecimal subtotalAmount) {
		BigDecimal discount = convertToBigDecimal(0);
		List<TicketItemDiscount> discounts = getTicketItem().getDiscounts();
		if (discounts != null) {
			for (TicketItemDiscount ticketItemDiscount : discounts) {
				if (ticketItemDiscount.getType() == Discount.DISCOUNT_TYPE_REPRICE) {
					continue;
				}
				else if (ticketItemDiscount.getType() == Discount.DISCOUNT_TYPE_PERCENTAGE) {
					discount = discount.add(ticketItemDiscount.calculateDiscount(subtotalAmount));
				}
				else if (ticketItemDiscount.getType() == Discount.DISCOUNT_TYPE_AMOUNT) {
					double discountPercentage = (ticketItemDiscount.getValue() * 100.0) / getTicketItem().getSubtotalAmount();
					TicketItemDiscount percentDiscount = new TicketItemDiscount();
					percentDiscount.setType(Discount.DISCOUNT_TYPE_PERCENTAGE);
					percentDiscount.setValue(discountPercentage);
					percentDiscount.setTicketItem(getTicketItem());
					percentDiscount.setCouponQuantity(ticketItemDiscount.getCouponQuantity());
					discount = discount.add(percentDiscount.calculateDiscount(subtotalAmount));
				}
			}
		}
		if (discount.compareTo(subtotalAmount.abs()) > 0)
			return subtotalAmount;

		return round(discount);
	}

	public void merge(TicketItemModifier otherItem) {
		setItemQuantity(getItemQuantity() + otherItem.getItemQuantity());
		return;
	}

	public double getTotalTaxRate() {
		List<TicketItemTax> ticketItemTaxes = getTaxes();
		if (ticketItemTaxes == null || ticketItemTaxes.isEmpty())
			return 0;

		double totalTaxRate = 0;
		for (TicketItemTax tax : ticketItemTaxes) {
			totalTaxRate += tax.getRate();
		}
		return totalTaxRate;

	}

	@Override
	public String getNameDisplay() {
		return getItemQuantityDisplay() + " " + getNameDisplay(false); //$NON-NLS-1$
	}

	public String getNameDisplay(boolean kitchenReceipt) {
		if (isInfoOnly()) {
			return "&nbsp;&nbsp;&nbsp;" + getName().trim(); //$NON-NLS-1$
		}
		double itemQuantity = Math.abs(getItemQuantity());
		String itemQuantityDisplay = ""; //$NON-NLS-1$
		if (itemQuantity > 1) {
			itemQuantityDisplay = NumberUtil.trimDecilamIfNotNeeded(itemQuantity) + "x "; //$NON-NLS-1$
		}
		String display;
		if (itemQuantity > 1) {
			display = itemQuantityDisplay + getName();
		}
		else {
			display = getName().trim();
		}
		if (getModifierType() == NORMAL_MODIFIER && !kitchenReceipt) {
			display += "*"; //$NON-NLS-1$
		}
		Store store = DataProvider.get().getStore();
		boolean isShowModifierPrice = false;
		if (store != null) {
			isShowModifierPrice = store.getProperty(AppConstants.PROP_SHOW_MODIFIER_PRICE) == null ? false
					: Boolean.valueOf(store.getProperty(AppConstants.PROP_SHOW_MODIFIER_PRICE));
		}

		if (isShowModifierPrice && getUnitPrice() > 0 && !kitchenReceipt) {
			display = itemQuantityDisplay + getName() + " @" + NumberUtil.formatNumber(getUnitPrice()); //$NON-NLS-1$
		}
		if (kitchenReceipt) {
			return "&nbsp;&nbsp;&nbsp;" + "-- " + display; //$NON-NLS-1$ //$NON-NLS-2$
		}
		return "&nbsp;&nbsp;&nbsp;" + " -- " + display; //$NON-NLS-1$ //$NON-NLS-2$
	}

	@Override
	public String getUnitPriceDisplay() {
		if (isInfoOnly()) {
			return null;
		}
		return NumberUtil.formatNumberAcceptNegative(getUnitPrice());
	}

	@Override
	public String getItemQuantityDisplay() {
		/*if (isInfoOnly()) {
			return "";
		}
		Double quantity = getItemQuantity();
		if (quantity <= 1)
			return "";
		return NumberUtil.trimDecilamIfNotNeeded(quantity);*/
		return ""; //$NON-NLS-1$
	}

	@Override
	public String getTaxAmountWithoutModifiersDisplay() {
		if (isInfoOnly()) {
			return null;
		}

		return NumberUtil.formatNumberAcceptNegative(getTaxAmount());
	}

	@Override
	public String getTotalAmountWithoutModifiersDisplay() {
		if (isInfoOnly()) {
			return null;
		}

		return NumberUtil.formatNumberAcceptNegative(getTotalAmount());
	}

	@Override
	public String getSubTotalAmountWithoutModifiersDisplay() {
		if (isInfoOnly()) {
			return null;
		}

		return NumberUtil.formatNumberAcceptNegative(getSubTotalAmount());
	}

	@Override
	public String getItemCode() {
		return ""; //$NON-NLS-1$
	}

	public boolean isSelected() {
		return selected;
	}

	public void setSelected(boolean selected) {
		this.selected = selected;
	}

	@Override
	public Double getItemQuantity() {
		Double itemQuantity = super.getItemQuantity();
		if (itemQuantity == 0) {
			itemQuantity = (double) super.getItemCount();
		}

		return itemQuantity;
	}

	@Override
	public boolean canAddDiscount() {
		return false;
	}

	@Override
	public boolean canVoid() {
		return false;
	}

	@Override
	public boolean canAddAdOn() {
		return false;
	}

	@Override
	public KitchenStatus getKitchenStatusValue() {
		return KitchenStatus.fromString(super.getStatus());
	}

	public void setKitchenStatusValue(KitchenStatus kitchenStatus) {
		super.setStatus(kitchenStatus.name());
	}

	@Override
	public Double getSubtotalAmount() {
		return super.getSubTotalAmount();
	}

	@Override
	public String getSubTotalAmountDisplay() {
		return null;
	}

	@XmlTransient
	public MenuModifier getMenuModifier() {
		return menuModifier;
	}

	public void setMenuModifier(MenuModifier menuModifier) {
		this.menuModifier = menuModifier;
	}

	@Override
	public boolean isSaved() {
		return getId() != null;
	}

	public void setTaxes(List<TicketItemTax> taxes) {
		this.taxes = taxes;
		if (this.taxes == null) {
			this.taxes = new ArrayList<>();
		}
	}

	public List<TicketItemTax> getTaxes() {
		if (taxes == null) {
			taxes = new ArrayList<>();

			String property = super.getTaxesProperty();
			if (StringUtils.isNotEmpty(property)) {
				JsonReader jsonParser = Json.createReader(new StringReader(property));
				JsonArray jsonArray = jsonParser.readArray();
				jsonParser.close();
				for (int i = 0; i < jsonArray.size(); i++) {
					JsonObject tObj = (JsonObject) jsonArray.get(i);
					TicketItemTax tit = new TicketItemTax();
					tit.setId(JsonUtil.getString(tObj, TicketItemTax.PROP_ID));
					tit.setName(tObj.getString(TicketItemTax.PROP_NAME));
					tit.setRate(POSUtil.parseDouble("" + tObj.get(TicketItemTax.PROP_RATE))); //$NON-NLS-1$
					tit.setTaxAmount(POSUtil.parseDouble("" + tObj.get(TicketItemTax.PROP_TAX_AMOUNT))); //$NON-NLS-1$
					taxes.add(tit);
				}
			}
		}
		return taxes;
	}

	public void addTotaxes(TicketItemTax tax) {
		if (taxes == null) {
			taxes = new ArrayList<TicketItemTax>();
		}
		taxes = getTaxes();
		taxes.add(tax);
	}

	public void buildTaxes() {
		if (taxes == null || taxes.isEmpty()) {
			setTaxesProperty(null);
			return;
		}
		JSONArray jsonArray = new JSONArray();
		for (TicketItemTax ticketItemTax : taxes) {
			jsonArray.put(ticketItemTax.toJson());
		}
		setTaxesProperty(jsonArray.toString());
	}

	public void setTicketItemQuantity(double ticketItemQuantity) {
		this.ticketItemQuantity = ticketItemQuantity;
	}

	public double getTicketItemQuantity() {
		if (getTicketItem() != null) {
			return getTicketItem().getQuantity();
		}
		return ticketItemQuantity;
	}

	@Override
	public void setProperties(String properties) {
		super.setProperties(properties);
		propertiesContainer = null;
	}

	private void initPropertyContainer() {
		if (propertiesContainer == null) {
			if (StringUtils.isBlank(super.getProperties())) {
				propertiesContainer = new com.google.gson.JsonObject();
			}
			else {
				propertiesContainer = new Gson().fromJson(super.getProperties(), com.google.gson.JsonObject.class);
			}
		}
	}
	
	@Override
	public void addProperty(String key, String value) {
		initPropertyContainer();
		propertiesContainer.addProperty(key, value);
		super.setProperties(propertiesContainer.toString());
	}

	@Override
	public String getProperty(String key) {
		initPropertyContainer();

		if (propertiesContainer.has(key)) {
			JsonElement jsonElement = propertiesContainer.get(key);
			if (!jsonElement.isJsonNull()) {
				return jsonElement.getAsString();
			}
		}
		return null;
	}

	@Override
	public com.google.gson.JsonObject getPropertyStore() {
		initPropertyContainer();
		return propertiesContainer;
	}

	public Double getMultiplierPrice(Double groupPrice) {
		if (StringUtils.isEmpty(super.getMultiplierName())) {
			return groupPrice;
		}
		Multiplier multiplier = DataProvider.get().getMultiplierById(super.getMultiplierName());
		if (multiplier == null || multiplier.isMain()) {
			return groupPrice;
		}
		return groupPrice * multiplier.getRate() / 100;
	}

	public void setSizeAndCrust(PizzaPrice pizzaPrice) {
		setModifierType(TicketItemModifier.CRUST);
		setInfoOnly(true);
		setPizzaCrustId(pizzaPrice.getCrust().getId());
		setMenuSizeId(pizzaPrice.getSize().getId());
		setName(pizzaPrice.getSize().getName() + " " + pizzaPrice.getCrust()); //$NON-NLS-1$
	}

	public MenuItemSize getMenuSize(MenuItem menuItem) {
		List<PizzaPrice> pizzaPriceList = menuItem.getPizzaPriceList();
		if (pizzaPriceList == null) {
			return null;
		}

		String menuSizeId = getMenuSizeId();
		if (StringUtils.isNotBlank(menuSizeId)) {
			for (PizzaPrice pizzaPrice : pizzaPriceList) {
				MenuItemSize menuItemSize = pizzaPrice.getSize();

				if (menuSizeId.equals(menuItemSize.getId())) {
					return menuItemSize;
				}
			}
		}

		String sizeAndCrustName = getName();
		String sizeName = ""; //$NON-NLS-1$
		MenuItemSize itemSize = null;

		for (PizzaPrice pizzaPrice : pizzaPriceList) {
			MenuItemSize menuItemSize = pizzaPrice.getSize();

			if (sizeAndCrustName.contains(menuItemSize.getName())) {
				if (menuItemSize.getName().length() > sizeName.length()) {
					sizeName = menuItemSize.getName();
					itemSize = menuItemSize;
				}
			}
		}

		return itemSize;
	}

	public PizzaCrust getPizzaCrust(MenuItem menuItem) {
		List<PizzaPrice> pizzaPriceList = menuItem.getPizzaPriceList();
		if (pizzaPriceList == null) {
			return null;
		}

		String pizzaCrustId = getPizzaCrustId();
		if (StringUtils.isNotBlank(pizzaCrustId)) {
			for (PizzaPrice pizzaPrice : pizzaPriceList) {
				PizzaCrust pizzaCrust = pizzaPrice.getCrust();

				if (pizzaCrustId.equals(pizzaCrust.getId())) {
					return pizzaCrust;
				}
			}
		}

		String sizeAndCrustName = getName();
		String crustName = ""; //$NON-NLS-1$
		PizzaCrust crust = null;

		for (PizzaPrice pizzaPrice : pizzaPriceList) {
			PizzaCrust pizzaCrust = pizzaPrice.getCrust();

			if (sizeAndCrustName.contains(pizzaCrust.getName())) {
				if (pizzaCrust.getName().length() > crustName.length()) {
					crustName = pizzaCrust.getName();
					crust = pizzaCrust;
				}
			}
		}

		return crust;
	}

	public String getMenuSizeId() {
		return getProperty("menu_size_id"); //$NON-NLS-1$
	}

	public void setMenuSizeId(String menuSizeId) {
		addProperty("menu_size_id", menuSizeId); //$NON-NLS-1$
	}

	public String getPizzaCrustId() {
		return getProperty("pizza_crust_id"); //$NON-NLS-1$
	}

	public void setPizzaCrustId(String pizzaCrustId) {
		addProperty("pizza_crust_id", pizzaCrustId); //$NON-NLS-1$
	}

	public void calculatePriceForPizzaSection(String sectionName) {
		double unitPrice = getUnitPrice();
		switch (sectionName) {
			case "WHOLE": //$NON-NLS-1$
				setUnitPrice(unitPrice * 1);
				break;
			case "Quarter 1": //$NON-NLS-1$
			case "Quarter 2": //$NON-NLS-1$
			case "Quarter 3": //$NON-NLS-1$
			case "Quarter 4": //$NON-NLS-1$
				setUnitPrice(unitPrice * 0.25);
				break;
			case "Half 1": //$NON-NLS-1$
			case "Half 2": //$NON-NLS-1$
				setUnitPrice(unitPrice * 0.5);
				break;
		}

	}

	public void putLabDoctorFee(boolean labDoctorFee) {
		addProperty("labDoctorFee", String.valueOf(labDoctorFee)); //$NON-NLS-1$
	}
	
	public boolean isLabDoctorFee() {
		return getBooleanProperty("labDoctorFee", false); //$NON-NLS-1$
	}
	
	public void putApplyDiscountOnDoctorFee(boolean labDoctorFee) {
		addProperty("ApplyDiscountOnDoctorFee", String.valueOf(labDoctorFee)); //$NON-NLS-1$
	}
	
	public boolean isApplyDiscountOnDoctorFee() {
		return getBooleanProperty("ApplyDiscountOnDoctorFee", false); //$NON-NLS-1$
	}
}