/**
 * ************************************************************************
 * * The contents of this file are subject to the MRPL 1.2
 * * (the  "License"),  being   the  Mozilla   Public  License
 * * Version 1.1  with a permitted attribution clause; you may not  use this
 * * file except in compliance with the License. You  may  obtain  a copy of
 * * the License at http://www.floreantpos.org/license.html
 * * Software distributed under the License  is  distributed  on  an "AS IS"
 * * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * * License for the specific  language  governing  rights  and  limitations
 * * under the License.
 * * The Original Code is FLOREANT POS.
 * * The Initial Developer of the Original Code is OROCUBE LLC
 * * All portions are Copyright (C) 2015 OROCUBE LLC
 * * All Rights Reserved.
 * ************************************************************************
 */
package com.floreantpos.model.dao;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Calendar;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;

import org.apache.commons.beanutils.PropertyUtils;
import org.apache.commons.lang.StringUtils;
import org.hibernate.Criteria;
import org.hibernate.Query;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.hibernate.criterion.Criterion;
import org.hibernate.criterion.Disjunction;
import org.hibernate.criterion.MatchMode;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.Projection;
import org.hibernate.criterion.ProjectionList;
import org.hibernate.criterion.Projections;
import org.hibernate.criterion.Restrictions;
import org.hibernate.transform.Transformers;
import org.hibernate.type.StandardBasicTypes;
import org.hibernate.type.Type;

import com.floreantpos.Messages;
import com.floreantpos.PosException;
import com.floreantpos.PosLog;
import com.floreantpos.model.ActionHistory;
import com.floreantpos.model.CashDrawer;
import com.floreantpos.model.CashDropTransaction;
import com.floreantpos.model.CreditCardTransaction;
import com.floreantpos.model.Customer;
import com.floreantpos.model.CustomerAccountTransaction;
import com.floreantpos.model.GiftCertificateTransaction;
import com.floreantpos.model.Gratuity;
import com.floreantpos.model.OrderType;
import com.floreantpos.model.Outlet;
import com.floreantpos.model.Pagination;
import com.floreantpos.model.PayOutTransaction;
import com.floreantpos.model.PaymentType;
import com.floreantpos.model.PosTransaction;
import com.floreantpos.model.RefundTransaction;
import com.floreantpos.model.ReversalTransaction;
import com.floreantpos.model.StoreSession;
import com.floreantpos.model.Terminal;
import com.floreantpos.model.Ticket;
import com.floreantpos.model.TicketItem;
import com.floreantpos.model.TransactionType;
import com.floreantpos.model.User;
import com.floreantpos.model.ext.CardTypeEnum;
import com.floreantpos.model.util.DataProvider;
import com.floreantpos.model.util.DateUtil;
import com.floreantpos.model.util.FeePaymentMapper;
import com.floreantpos.model.util.TransactionSummary;
import com.floreantpos.report.EndOfDayReportData;
import com.floreantpos.report.TransactionReportModel;
import com.floreantpos.report.model.CustomerAccountTransactionItem;
import com.floreantpos.util.NumberUtil;
import com.floreantpos.util.POSUtil;
import com.floreantpos.util.StoreUtil;
import com.orocube.rest.service.server.BaseDataServiceDao;

public class PosTransactionDAO extends BasePosTransactionDAO {

	/**
	 * Default constructor.  Can be used in place of getInstance()
	 */
	public PosTransactionDAO() {
	}

	@Override
	protected Serializable save(Object obj, Session s) {
		PosTransaction transaction = (PosTransaction) obj;
		performPreSaveOperations(transaction);
		if (StringUtils.isEmpty(transaction.getStoreSessionId())) {
			StoreSession storeSession = DataProvider.get().getStoreSession();
			if (storeSession != null) {
				transaction.setStoreSessionId(storeSession.getId());
			}

		}
		return super.save(obj, s);
	}

	@Override
	protected void update(Object obj, Session s) {
		PosTransaction transaction = (PosTransaction) obj;
		performPreSaveOperations(transaction);
		if (StringUtils.isEmpty(transaction.getStoreSessionId())) {
			StoreSession storeSession = DataProvider.get().getStoreSession();
			if (storeSession != null) {
				transaction.setStoreSessionId(storeSession.getId());
			}
		}
		super.update(obj, s);
	}

	@Override
	protected void saveOrUpdate(Object obj, Session s) {
		PosTransaction transaction = (PosTransaction) obj;
		performPreSaveOperations(transaction);
		if (StringUtils.isEmpty(transaction.getStoreSessionId())) {
			transaction.setStoreSessionId(DataProvider.get().getStoreSession().getId());
		}
		super.saveOrUpdate(obj, s);
	}

	private void performPreSaveOperations(PosTransaction posTransaction) {
		if (StringUtils.isBlank(posTransaction.getUserId())) {
			PosLog.error(getClass(), "User id is null for transaction " + posTransaction.getId());
		}

		if (posTransaction.getTerminalId() == null || NumberUtil.isZero(posTransaction.getTerminalId())) {
			PosLog.error(getClass(), "User id is null for transaction " + posTransaction.getId());
		}
	}

	public List<PosTransaction> findUnauthorizedTransactions() {
		return findUnauthorizedTransactions(null);
	}

	public List<PosTransaction> findUnauthorizedTransactions(User owner) {
		List<String> selectedUserIds = new ArrayList<>();
		if (owner != null) {
			selectedUserIds = Arrays.asList(owner.getId());
		}
		return findUnauthorizedTransactions(null, null, selectedUserIds);
	}

	public List<PosTransaction> findUnauthorizedTransactions(String ticketId, Terminal terminal, List<String> userIds) {

		try (Session session = createNewSession()) {

			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.createAlias(PosTransaction.PROP_TICKET, "ticket"); //$NON-NLS-1$
			criteria.add(Restrictions.eq(PosTransaction.PROP_CAPTURED, Boolean.FALSE));
			criteria.add(Restrictions.eq(PosTransaction.PROP_MARKED_CAPTURED, Boolean.FALSE));
			criteria.add(Restrictions.or(Restrictions.isNull(PosTransaction.PROP_MARKED_CAPTURED),
					Restrictions.eq(PosTransaction.PROP_MARKED_CAPTURED, Boolean.FALSE)));
			criteria.add(Restrictions.eq(PosTransaction.PROP_AUTHORIZABLE, Boolean.TRUE));
			criteria.add(Restrictions.eq(PosTransaction.PROP_TRANSACTION_TYPE, TransactionType.CREDIT.name()));
			criteria.add(Restrictions.or(Restrictions.isNull(PosTransaction.PROP_VOIDED), Restrictions.eq(PosTransaction.PROP_VOIDED, Boolean.FALSE)));
			criteria.add(Restrictions.isNotNull(PosTransaction.PROP_TICKET));

			if (!StringUtils.isBlank(ticketId)) {
				criteria.add(Restrictions.ilike("ticket." + Ticket.PROP_ID, ticketId, MatchMode.ANYWHERE)); //$NON-NLS-1$
			}

			if (terminal != null) {
				criteria.add(Restrictions.eq(PosTransaction.PROP_TERMINAL_ID, terminal.getId()));
			}

			if (userIds != null && !userIds.isEmpty()) {
				criteria.add(Restrictions.in(PosTransaction.PROP_USER_ID, userIds));
			}

			return criteria.list();
		}
	}

	public List<PosTransaction> findCapturedTransactions(User owner) {
		List<String> selectedUserIds = new ArrayList<>();
		if (owner != null) {
			selectedUserIds = Arrays.asList(owner.getId());
		}

		return findCapturedTransactions(null, null, selectedUserIds);
	}

	@SuppressWarnings("unchecked")
	public List<PosTransaction> findCapturedTransactions(String ticketId, Terminal terminal, List<String> userIds) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(CreditCardTransaction.class);
			criteria.createAlias(PosTransaction.PROP_TICKET, "ticket"); //$NON-NLS-1$
			criteria.add(Restrictions.eq(PosTransaction.PROP_CAPTURED, Boolean.TRUE));
			criteria.add(Restrictions.or(Restrictions.isNull(PosTransaction.PROP_VOIDED), Restrictions.eq(PosTransaction.PROP_VOIDED, Boolean.FALSE)));
			criteria.add(Restrictions.isNotNull(PosTransaction.PROP_TICKET));

			if (!StringUtils.isBlank(ticketId)) {
				criteria.add(Restrictions.ilike("ticket." + Ticket.PROP_ID, ticketId, MatchMode.ANYWHERE)); //$NON-NLS-1$
			}

			if (terminal != null) {
				criteria.add(Restrictions.eq(PosTransaction.PROP_TERMINAL_ID, terminal.getId()));
			}

			if (userIds != null && !userIds.isEmpty()) {
				criteria.add(Restrictions.in(PosTransaction.PROP_USER_ID, userIds));
			}

			Calendar calendar = Calendar.getInstance();
			calendar.add(Calendar.DAY_OF_MONTH, -1);
			Date startOfDay = DateUtil.startOfDay(calendar.getTime());
			Date endOfDay = DateUtil.endOfDay(new Date());
			criteria.add(Restrictions.between(PosTransaction.PROP_TRANSACTION_TIME, startOfDay, endOfDay));
			return criteria.list();
		}
	}

	public Boolean hasUnauthorizedTransactions(User owner) {
		try (Session session = createNewSession()) {

			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.setProjection(Projections.rowCount());
			criteria.add(Restrictions.eq(PosTransaction.PROP_CAPTURED, Boolean.FALSE));
			criteria.add(Restrictions.eq(PosTransaction.PROP_MARKED_CAPTURED, Boolean.FALSE));
			criteria.add(Restrictions.eq(PosTransaction.PROP_AUTHORIZABLE, Boolean.TRUE));
			criteria.add(Restrictions.eq(PosTransaction.PROP_VOIDED, Boolean.FALSE));
			criteria.add(Restrictions.eq(PosTransaction.PROP_TRANSACTION_TYPE, TransactionType.CREDIT.name()));
			criteria.add(Restrictions.isNotNull(PosTransaction.PROP_TICKET));
			if (owner != null) {
				criteria.add(Restrictions.eq(PosTransaction.PROP_USER_ID, owner.getId()));
			}
			Number rowCount = (Number) criteria.uniqueResult();
			return rowCount != null && rowCount.intValue() > 0;
		}
	}

	public List<? extends PosTransaction> findTransactions(Terminal terminal, Class<?> transactionClass, Date from, Date to) {
		return findTransactions(terminal, transactionClass, from, to, true);
	}

	public List<? extends PosTransaction> findTransactions(Terminal terminal, Class<?> transactionClass, Date from, Date to, boolean excludeNullTicket) {
		return findTransactions(terminal, null, transactionClass, from, to, excludeNullTicket);
	}

	public List<? extends PosTransaction> findTransactions(Terminal terminal, Outlet outlet, Class<?> transactionClass, Date from, Date to,
			boolean excludeNullTicket) {
		return this.findTransactions(terminal, outlet, transactionClass, from, to, excludeNullTicket, false);
	}

	public List<PosTransaction> findTransactions(Terminal terminal, Outlet outlet, Class<?> transactionClass, Date from, Date to, boolean excludeNullTicket,
			boolean withId) {
		List<PosTransaction> transactions = findTransactions(terminal, outlet, transactionClass, from, to, excludeNullTicket, withId, false);
		if (transactions != null && transactions.size() > 0) {
			Map<String, User> userMap = new HashMap<>();
			for (Iterator<PosTransaction> iterator = transactions.iterator(); iterator.hasNext();) {
				PosTransaction transaction = (PosTransaction) iterator.next();
				if (transaction.getUserId() != null) {
					User user = userMap.get(transaction.getUserId());
					if (user == null) {
						user = DataProvider.get().getUserById(transaction.getUserId(), transaction.getOutletId());
						if (user != null) {
							userMap.put(user.getId(), user);
						}
					}
					transaction.setUserName(user == null ? "" : user.getFullName()); //$NON-NLS-1$
				}
				if (transaction.getServerId() != null) {
					User server = userMap.get(transaction.getServerId());
					if (server == null) {
						server = DataProvider.get().getUserById(transaction.getServerId(), transaction.getOutletId());
						if (server != null) {
							userMap.put(server.getId(), server);
						}
					}
					transaction.setServerName(server == null ? "" : server.getFullName()); //$NON-NLS-1$
				}
			}
		}
		return transactions;
	}

	@SuppressWarnings("unchecked")
	public List<PosTransaction> findTransactions(Terminal terminal, Outlet outlet, Class<?> transactionClass, Date from, Date to, boolean excludeNullTicket,
			boolean withId, boolean excludeVoidTransactions) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(transactionClass);
			if (excludeNullTicket) {
				criteria.add(Restrictions.isNotNull(PosTransaction.PROP_TICKET));
			}
			if (terminal != null) {
				criteria.add(Restrictions.eq(PosTransaction.PROP_TERMINAL_ID, terminal.getId()));
			}

			if (outlet != null) {
				//criteria.createAlias(PosTransaction.PROP_TERMINAL, "t");
				//TODO:
				//criteria.add(Restrictions.eq("t.outlet", outlet));
			}
			if (excludeVoidTransactions) {
				criteria.add(Restrictions.eqOrIsNull(PosTransaction.PROP_VOIDED, false));
			}
			if (from != null && to != null) {
				criteria.add(Restrictions.between(PosTransaction.PROP_TRANSACTION_TIME, from, to));
			}
			ProjectionList pList = Projections.projectionList();
			if (withId) {
				pList.add(Projections.property(PosTransaction.PROP_ID), PosTransaction.PROP_ID);
			}
			pList.add(Projections.property(PosTransaction.PROP_TICKET), PosTransaction.PROP_TICKET);
			pList.add(Projections.property(PosTransaction.PROP_PAYMENT_TYPE_STRING), PosTransaction.PROP_PAYMENT_TYPE_STRING);
			pList.add(Projections.property(PosTransaction.PROP_CARD_TYPE), PosTransaction.PROP_CARD_TYPE);
			pList.add(Projections.property(PosTransaction.PROP_CARD_READER), PosTransaction.PROP_CARD_READER);
			pList.add(Projections.property(PosTransaction.PROP_TRANSACTION_TIME), PosTransaction.PROP_TRANSACTION_TIME);
			pList.add(Projections.property(PosTransaction.PROP_USER_ID), PosTransaction.PROP_USER_ID);
			pList.add(Projections.property(PosTransaction.PROP_CARD_AUTH_CODE), PosTransaction.PROP_CARD_AUTH_CODE);
			pList.add(Projections.property(PosTransaction.PROP_TIPS_AMOUNT), PosTransaction.PROP_TIPS_AMOUNT);
			pList.add(Projections.property(PosTransaction.PROP_AMOUNT), PosTransaction.PROP_AMOUNT);
			pList.add(Projections.property(PosTransaction.PROP_TERMINAL_ID), PosTransaction.PROP_TERMINAL_ID);
			pList.add(Projections.property(PosTransaction.PROP_TRANSACTION_TYPE), PosTransaction.PROP_TRANSACTION_TYPE);
			pList.add(Projections.property(PosTransaction.PROP_CUSTOM_PAYMENT_NAME), PosTransaction.PROP_CUSTOM_PAYMENT_NAME);
			//pList.add(Projections.property(PosTransaction.PROP_TERMINAL_ID), PosTransaction.PROP_TERMINAL_ID);
			pList.add(Projections.property(CustomerAccountTransaction.PROP_CUSTOMER_ID), CustomerAccountTransaction.PROP_CUSTOMER_ID);
			criteria.setProjection(pList);

			criteria.setResultTransformer(Transformers.aliasToBean(PosTransaction.class));
			criteria.addOrder(Order.asc(PosTransaction.PROP_TRANSACTION_TIME));

			List<PosTransaction> list = criteria.list();
			return list;
		}
	}

	public List<PosTransaction> findReceived(Terminal terminal, Outlet outlet, Date from, Date to) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(PosTransaction.class);
			criteria.add(Restrictions.isNotNull(PosTransaction.PROP_TICKET));
			if (terminal != null) {
				criteria.add(Restrictions.eq(PosTransaction.PROP_TERMINAL_ID, terminal.getId()));
			}
			if (outlet != null) {
				criteria.add(Restrictions.eq(PosTransaction.PROP_OUTLET_ID, outlet.getId()));
			}

			criteria.add(Restrictions.eq(PosTransaction.PROP_TRANSACTION_TYPE, TransactionType.CREDIT.name()));
			criteria.add(Restrictions.eqOrIsNull(PosTransaction.PROP_VOIDED, Boolean.FALSE));
			criteria.add(Restrictions.between(PosTransaction.PROP_TRANSACTION_TIME, from, to));

			List<PosTransaction> list = criteria.list();
			return list;
		}
	}

	public TransactionSummary getTransactionSummary(Terminal terminal, Class transactionClass, Date from, Date to) {
		TransactionSummary summary = new TransactionSummary();
		try (Session session = createNewSession()) {

			Criteria criteria = session.createCriteria(transactionClass);
			criteria.add(Restrictions.eq(PosTransaction.PROP_DRAWER_RESETTED, Boolean.FALSE));

			if (terminal != null) {
				criteria.add(Restrictions.eq(PosTransaction.PROP_TERMINAL_ID, terminal.getId()));
			}

			if (from != null && to != null) {
				criteria.add(Restrictions.ge(PosTransaction.PROP_TRANSACTION_TIME, from));
				criteria.add(Restrictions.le(PosTransaction.PROP_TRANSACTION_TIME, to));
			}

			ProjectionList projectionList = Projections.projectionList();
			projectionList.add(Projections.count(PosTransaction.PROP_ID));
			projectionList.add(Projections.sum(PosTransaction.PROP_AMOUNT));
			projectionList.add(Projections.sum(PosTransaction.PROP_TIPS_AMOUNT));

			criteria.setProjection(projectionList);

			List list = criteria.list();

			if (list == null || list.size() == 0)
				return summary;

			Object[] o = (Object[]) list.get(0);
			int index = 0;

			summary.setCount(HibernateProjectionsUtil.getInt(o, index++));
			summary.setAmount(HibernateProjectionsUtil.getDouble(o, index++));
			summary.setTipsAmount(HibernateProjectionsUtil.getDouble(o, index++));

			return summary;
		}
	}

	public List<PosTransaction> findTransactionListByGiftCardNumber(String giftCardNumber, Date fromDate, Date toDate) {
		return findTransactionListByGiftCardNumber(giftCardNumber, fromDate, toDate, false);
	}

	public List<PosTransaction> findTransactionListByGiftCardNumber(String giftCardNumber, Date fromDate, Date toDate, boolean showOnlySales) {
		try (Session session = createNewSession()) {
			Criteria criteria;
			if (showOnlySales) {
				criteria = session.createCriteria(GiftCertificateTransaction.class);
			}
			else {
				criteria = session.createCriteria(getReferenceClass());
			}

			criteria.add(Restrictions.between(PosTransaction.PROP_TRANSACTION_TIME, DateUtil.startOfDay(fromDate), DateUtil.endOfDay(toDate)));
			if (!StringUtils.isEmpty(giftCardNumber)) {
				criteria.add(Restrictions.eq(PosTransaction.PROP_GIFT_CERT_NUMBER, giftCardNumber));
			}
			else {
				criteria.add(Restrictions.isNotNull(PosTransaction.PROP_GIFT_CERT_NUMBER));
			}
			criteria.addOrder(Order.asc(PosTransaction.PROP_GIFT_CERT_NUMBER));
			criteria.addOrder(Order.asc(PosTransaction.PROP_TRANSACTION_TIME));
			return criteria.list();
		}
	}

	public List<RefundTransaction> findRefundTransactions(Terminal terminal, Outlet outlet, Date fromDate, Date toDate) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(RefundTransaction.class);
			criteria.createAlias(PosTransaction.PROP_TICKET, "t"); //$NON-NLS-1$
			criteria.add(Restrictions.isNotNull(PosTransaction.PROP_TICKET));
			if (terminal != null) {
				criteria.add(Restrictions.eq(PosTransaction.PROP_TERMINAL_ID, terminal.getId()));
			}
			if (outlet != null) {
				criteria.add(Restrictions.eq(PosTransaction.PROP_OUTLET_ID, outlet.getId()));
			}
			criteria.add(Restrictions.ge(PosTransaction.PROP_TRANSACTION_TIME, fromDate));
			criteria.add(Restrictions.lt(PosTransaction.PROP_TRANSACTION_TIME, toDate));
			criteria.add(Restrictions.eq(PosTransaction.PROP_VOIDED, Boolean.FALSE));
			criteria.add(Restrictions.eq("t." + Ticket.PROP_VOIDED, Boolean.FALSE)); //$NON-NLS-1$

			return criteria.list();
		}
	}

	public List<PosTransaction> getStoreSessionTransactions(StoreSession storeOperationData) {
		try (Session session = createNewSession()) {
			List<String> cashDrawerIds = CashDrawerDAO.getInstance().getCashDrawerIds(storeOperationData);
			if (cashDrawerIds == null || cashDrawerIds.isEmpty())
				return null;
			Criteria criteria = session.createCriteria(getReferenceClass(), "t"); //$NON-NLS-1$
			//criteria.createAlias(PosTransaction.PROP_CASH_DRAWER, "c");
			//criteria.add(Restrictions.eqProperty("t.cashDrawer", PosTransaction.PROP_CASH_DRAWER));
			//criteria.add(Restrictions.eq("c.storeOperationData", storeOperationData));
			criteria.createAlias(PosTransaction.PROP_TICKET, "ticket"); //$NON-NLS-1$
			criteria.add(Restrictions.eq("ticket.closed", true)); //$NON-NLS-1$
			criteria.add(Restrictions.eq("ticket.voided", false)); //$NON-NLS-1$
			criteria.add(Restrictions.in(PosTransaction.PROP_CASH_DRAWER_ID, cashDrawerIds));
			return criteria.list();
		}
	}

	public void saveReversalTransaction(Ticket ticket, PosTransaction transaction, ReversalTransaction reversalTransaction) {
		Transaction tx = null;

		try (Session session = createNewSession()) {
			tx = session.beginTransaction();

			delete(transaction, session);

			performPreSaveOperations(transaction);
			saveOrUpdate(reversalTransaction, session);
			TicketDAO.getInstance().update(ticket, session);

			tx.commit();
		} catch (Exception e) {
			try {
				tx.rollback();
			} catch (Exception x) {
				PosLog.error(PosTransactionDAO.class, x);
			}
			throw e;
		}
	}

	public List<PosTransaction> findCreditTransactions(Date fromDate, Date toDate, User user) {
		Criteria criteria = null;

		try (Session session = createNewSession()) {

			criteria = session.createCriteria(PosTransaction.class);
			criteria.add(Restrictions.eq(PosTransaction.PROP_TRANSACTION_TYPE, TransactionType.CREDIT.name()));
			criteria.add(Restrictions.between(PosTransaction.PROP_TRANSACTION_TIME, fromDate, toDate));
			addMultiUserFilter(user, criteria);

			ProjectionList pList = Projections.projectionList();
			pList.add(Projections.property(PosTransaction.PROP_PAYMENT_TYPE_STRING), PosTransaction.PROP_PAYMENT_TYPE_STRING);
			pList.add(Projections.property(PosTransaction.PROP_CUSTOM_PAYMENT_NAME), PosTransaction.PROP_CUSTOM_PAYMENT_NAME);
			pList.add(Projections.property(PosTransaction.PROP_USER_ID), PosTransaction.PROP_USER_ID);
			pList.add(Projections.property(PosTransaction.PROP_ID), PosTransaction.PROP_ID);
			pList.add(Projections.property(PosTransaction.PROP_TRANSACTION_TIME), PosTransaction.PROP_TRANSACTION_TIME);
			pList.add(Projections.property(PosTransaction.PROP_AMOUNT), PosTransaction.PROP_AMOUNT);
			pList.add(Projections.property(PosTransaction.PROP_TICKET), PosTransaction.PROP_TICKET);
			criteria.setProjection(pList);
			criteria.setResultTransformer(Transformers.aliasToBean(PosTransaction.class));
			criteria.addOrder(Order.asc(PosTransaction.PROP_PAYMENT_TYPE_STRING));
			criteria.addOrder(Order.asc(PosTransaction.PROP_TRANSACTION_TIME));

			List<PosTransaction> list = criteria.list();
			return list;
		}
	}

	public List<PosTransaction> getCloudStoreSessionTransactions(StoreSession storeOperationData) {
		try (Session session = createNewSession()) {
			List<String> cashDrawerIds = CashDrawerDAO.getInstance().getCashDrawerIds(storeOperationData);
			if (cashDrawerIds == null || cashDrawerIds.isEmpty())
				return null;
			Criteria criteria = session.createCriteria(getReferenceClass(), "t"); //$NON-NLS-1$
			//criteria.createAlias(PosTransaction.PROP_CASH_DRAWER, "c");
			//criteria.add(Restrictions.eqProperty("t.cashDrawer", PosTransaction.PROP_CASH_DRAWER));
			//criteria.add(Restrictions.eq("c.storeOperationData", storeOperationData));
			criteria.add(Restrictions.in(PosTransaction.PROP_CASH_DRAWER_ID, cashDrawerIds));
			return criteria.list();
		}
	}

	private void addMultiUserFilter(User user, Criteria criteria) {
		if (user != null) {
			PosLog.info(getClass(), Messages.getString("PosTransactionDAO.5") + user.getFullName() + Messages.getString("PosTransactionDAO.6") + user.getId()); //$NON-NLS-1$ //$NON-NLS-2$
			Disjunction disjunction = Restrictions.disjunction();
			disjunction.add(Restrictions.eq(PosTransaction.PROP_USER_ID, user == null ? null : user.getId()));
			List<User> linkedUsers = user.getLinkedUser();
			if (linkedUsers != null) {
				for (User linkedUser : linkedUsers) {
					if (linkedUser.getId().equals(user.getId())) {
						continue;
					}
					PosLog.info(getClass(), Messages.getString("PosTransactionDAO.7") + linkedUser.getFullName() + Messages.getString("PosTransactionDAO.8") //$NON-NLS-1$//$NON-NLS-2$
							+ linkedUser.getId());
					disjunction.add(Restrictions.eq(PosTransaction.PROP_USER_ID, linkedUser.getId()));
				}
			}
			criteria.add(disjunction);
		}
	}

	public List<CustomerAccountTransactionItem> findCustomerAccountTransactions(Date fromDate, Date toDate, String customerIdOrName) {
		try (Session session = createNewSession()) {

			String hql = "select t.%s, t.%s, t.%s, t.%s, t.%s, t.%s, t.%s, t.%s, t.%s, c.%s" //$NON-NLS-1$
					+ " from PosTransaction as t, Customer as c" //$NON-NLS-1$
					+ " where t.%s=c.%s and t.%s between :fromDate and :toDate and t.%s=false and t.%s is not null" //$NON-NLS-2$
					+ " order by c.%s"; //$NON-NLS-2$
			hql = String.format(hql, PosTransaction.PROP_ID, PosTransaction.PROP_TICKET, PosTransaction.PROP_TRANSACTION_TIME, PosTransaction.PROP_TIPS_AMOUNT,
					PosTransaction.PROP_AMOUNT, PosTransaction.PROP_CUSTOMER_ID, PosTransaction.PROP_TRANSACTION_TYPE, PosTransaction.PROP_PAYMENT_TYPE_STRING,
					PosTransaction.PROP_CUSTOM_PAYMENT_NAME, Customer.PROP_NAME, PosTransaction.PROP_CUSTOMER_ID, Customer.PROP_ID,
					PosTransaction.PROP_TRANSACTION_TIME, PosTransaction.PROP_VOIDED, PosTransaction.PROP_TICKET, Customer.PROP_NAME);

			if (StringUtils.isNotBlank(customerIdOrName)) {
				hql += " and ( LOWER(c.name) like '%" + customerIdOrName.toLowerCase() + "%'"; //$NON-NLS-1$ //$NON-NLS-2$
				hql += " or "; //$NON-NLS-1$
				hql += " LOWER(c.id) like '%" + customerIdOrName.toLowerCase() + "%' )"; //$NON-NLS-1$ //$NON-NLS-2$
			}

			Query query = session.createQuery(hql);
			query.setTimestamp("fromDate", DateUtil.toUTC(fromDate)); //$NON-NLS-1$
			query.setTimestamp("toDate", DateUtil.toUTC(toDate)); //$NON-NLS-1$

			List<Object[]> list = query.list();
			List<CustomerAccountTransactionItem> items = new ArrayList<CustomerAccountTransactionItem>();
			for (Object[] object : list) {
				CustomerAccountTransactionItem accountTransactionItem = new CustomerAccountTransactionItem();
				accountTransactionItem.setTransactionNo(String.valueOf(object[0]));
				Ticket ticket = (Ticket) object[1];
				accountTransactionItem.setTicketNo(ticket.getId());
				accountTransactionItem.setDate((Date) object[2]);
				accountTransactionItem.setTips((double) object[3]);
				accountTransactionItem.setTotalAmount((double) object[4]);
				accountTransactionItem.setCustomerId(String.valueOf(object[5]));
				accountTransactionItem.setCustomerName(String.valueOf(object[6]));
				items.add(accountTransactionItem);
			}

			return items;

		}
	}

	public List<PosTransaction> findTransactionsForCashDrawer(String cashDrawerId, boolean voided) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass(), "t"); //$NON-NLS-1$
			criteria.add(Restrictions.eq(PosTransaction.PROP_CASH_DRAWER_ID, cashDrawerId));
			criteria.add(Restrictions.eq(PosTransaction.PROP_VOIDED, voided));
			criteria.addOrder(Order.asc(PosTransaction.PROP_PAYMENT_TYPE_STRING));
			criteria.addOrder(Order.asc(PosTransaction.PROP_TRANSACTION_TIME));

			return criteria.list();
		}
	}

	public List<PosTransaction> findTransactionsForServer(String storeSessionId, String serverId) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.eq(PosTransaction.PROP_USER_ID, serverId));
			criteria.add(Restrictions.eq(PosTransaction.PROP_STORE_SESSION_ID, storeSessionId));
			criteria.add(Restrictions.eq(PosTransaction.PROP_VOIDED, false));
			criteria.addOrder(Order.asc(PosTransaction.PROP_PAYMENT_TYPE_STRING));
			criteria.addOrder(Order.asc(PosTransaction.PROP_TRANSACTION_TIME));

			return criteria.list();
		}
	}

	public List<String> getDistinctCardMmerchantGateway(String storeSessionId, Integer terminalId) {
		List<String> gateways = null;
		Criteria criteria = null;
		try (Session session = createNewSession()) {
			criteria = session.createCriteria(getReferenceClass());
			if (StringUtils.isNotEmpty(storeSessionId)) {
				criteria.add(Restrictions.eq(PosTransaction.PROP_STORE_SESSION_ID, storeSessionId));
			}
			if (terminalId != null) {
				criteria.add(Restrictions.eq(PosTransaction.PROP_TERMINAL_ID, terminalId));
			}
			Calendar calendar = Calendar.getInstance();
			calendar.add(Calendar.DATE, -3);

			criteria.add(Restrictions.gt(PosTransaction.PROP_TRANSACTION_TIME, calendar.getTime()));
			criteria.add(Restrictions.isNotNull(PosTransaction.PROP_CARD_MERCHANT_GATEWAY));
			criteria.setProjection(Projections.distinct(Projections.property(PosTransaction.PROP_CARD_MERCHANT_GATEWAY)));

			gateways = criteria.list();
		}
		return gateways;
	}

	public List<PosTransaction> findTransactionsForSession(String storeSessionId) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.eq(PosTransaction.PROP_STORE_SESSION_ID, storeSessionId));
			criteria.add(Restrictions.eq(PosTransaction.PROP_VOIDED, false));
			criteria.addOrder(Order.asc(PosTransaction.PROP_PAYMENT_TYPE_STRING));
			criteria.addOrder(Order.asc(PosTransaction.PROP_TRANSACTION_TIME));

			return criteria.list();
		}
	}

	@Deprecated
	public String findLastTxPaymentTypeName(Ticket ticket) {
		if (StringUtils.isEmpty(ticket.getId())) {
			return null;
		}
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.eq(PosTransaction.PROP_VOIDED, false));
			criteria.add(Restrictions.eq(PosTransaction.PROP_TRANSACTION_TYPE, TransactionType.CREDIT.name()));
			criteria.add(Restrictions.eq(PosTransaction.PROP_TICKET, ticket));
			criteria.addOrder(Order.desc(PosTransaction.PROP_TRANSACTION_TIME));
			criteria.setMaxResults(1);
			criteria.setProjection(Projections.property(PosTransaction.PROP_PAYMENT_TYPE_STRING));
			String paymentTypeString = (String) criteria.uniqueResult();
			if (StringUtils.isNotEmpty(paymentTypeString)) {
				return PaymentType.valueOf(paymentTypeString).getDisplayString();
			}
			return null;
		}
	}

	public List<PosTransaction> getTransactionsByOutlet(Date from, Date to, Outlet outlet, Pagination pagination) {
		Criteria criteria = null;
		try (Session session = createNewSession()) {
			criteria = session.createCriteria(getReferenceClass());
			if (pagination != null) {
				criteria.setFirstResult(pagination.getCurrentRowIndex());
				criteria.setMaxResults(pagination.getPageSize());
			}
			updateCriteria(criteria, from, to, outlet);
			List list = criteria.list();

			criteria = session.createCriteria(getReferenceClass());
			criteria.setProjection(Projections.rowCount());
			updateCriteria(criteria, from, to, outlet);
			if (pagination != null) {
				Number uniqueResult = (Number) criteria.uniqueResult();
				pagination.setNumRows(uniqueResult.intValue());
			}
			pagination.setRows(list);
			return list;
		}
	}

	private void updateCriteria(Criteria criteria, Date from, Date to, Outlet outlet) {

		if (from != null && to != null) {
			criteria.add(Restrictions.between(PosTransaction.PROP_TRANSACTION_TIME, from, to));
		}
		if (outlet != null) {
			criteria.add(Restrictions.eq(PosTransaction.PROP_OUTLET_ID, outlet.getId()));
		}
	}

	public void savePosTransaction(PosTransaction transaction, List<PosTransaction> existingPosTransactions, Session session) {
		if (existingPosTransactions == null || existingPosTransactions.size() == 0)
			return;
		PosTransaction existingPosTransaction = null;
		int idx = existingPosTransactions.indexOf(transaction);
		if (idx != -1) {
			existingPosTransaction = existingPosTransactions.get(idx);
			if (existingPosTransaction == null) {
				save(transaction, session);
			}
			else {
				transaction.setVersion(existingPosTransaction.getVersion());
				//update(transaction, session);
			}
		}
		else {
			save(transaction, session);
		}
	}

	public void saveOrUpdatePosTransactions(List<PosTransaction> posTransactions) throws Exception {
		saveOrUpdatePosTransactions(posTransactions, false, false);
	}

	public void saveOrUpdatePosTransactions(List<PosTransaction> posTransactions, boolean updateLastUpdateTime, boolean updateSyncTime) throws Exception {
		if (posTransactions == null)
			return;

		Transaction tx = null;
		try (Session session = createNewSession()) {
			tx = session.beginTransaction();
			for (PosTransaction posTransaction : posTransactions) {
				performPreSaveOperations(posTransaction);
				PosTransaction existingPosTransaction = get(posTransaction.getId());
				if (existingPosTransaction != null) {
					if (!BaseDataServiceDao.get().shouldSave(posTransaction.getLastUpdateTime(), existingPosTransaction.getLastUpdateTime())) {
						PosLog.info(getClass(), posTransaction.getId() + " already updated"); //$NON-NLS-1$
						continue;
					}
					final String id = existingPosTransaction.getId();
					long version = existingPosTransaction.getVersion();
					PropertyUtils.copyProperties(existingPosTransaction, posTransaction);
					existingPosTransaction.setId(id);
					existingPosTransaction.setVersion(version);
					existingPosTransaction.setUpdateLastUpdateTime(updateLastUpdateTime);
					existingPosTransaction.setUpdateSyncTime(updateSyncTime);
					update(existingPosTransaction, session);
				}
				else {
					posTransaction.setUpdateLastUpdateTime(updateLastUpdateTime);
					posTransaction.setUpdateSyncTime(updateSyncTime);
					save(posTransaction, session);
				}
			}
			tx.commit();
		} catch (Exception e) {
			if (tx != null) {
				tx.rollback();
			}
			PosLog.error(getClass(), e);
			throw e;
		}
	}

	public void savePosTransactions(List<PosTransaction> transactions) throws Exception {
		if (transactions == null)
			return;
		for (PosTransaction posTransaction : transactions) {
			Transaction tx = null;
			Session session = null;
			try {
				session = createNewSession();
				tx = session.beginTransaction();

				performPreSaveOperations(posTransaction);
				PaymentType paymentType = posTransaction.getPaymentType();

				PosTransaction existPosTransaction = get(posTransaction.getId());
				if (existPosTransaction == null) {
					existPosTransaction = paymentType.createTransaction();
					PropertyUtils.copyProperties(existPosTransaction, posTransaction);
					if (StringUtils.isEmpty(existPosTransaction.getStoreSessionId())) {
						existPosTransaction.setStoreSessionId(StoreUtil.getCurrentStoreSession().getId());
					}

					save(existPosTransaction, session);
				}

				tx.commit();
			} catch (Exception e) {
				tx.rollback();
				throw e;
			} finally {
				session.close();
			}
		}
	}

	public double findTransactionsAmountByDate(Date date) {
		double netSales = 0.0;
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(PosTransaction.class);
			criteria.add(Restrictions.eq(PosTransaction.PROP_TRANSACTION_TYPE, TransactionType.CREDIT.name()));
			criteria.add(Restrictions.between(PosTransaction.PROP_TRANSACTION_TIME, DateUtil.startOfDay(date), DateUtil.endOfDay(date)));
			criteria.setProjection(Projections.sum(PosTransaction.PROP_AMOUNT));
			Object uniqueResult = criteria.uniqueResult();
			if (uniqueResult != null && (uniqueResult instanceof Number)) {
				netSales = ((Number) uniqueResult).doubleValue();
			}

			criteria = session.createCriteria(RefundTransaction.class);
			criteria.add(Restrictions.between(PosTransaction.PROP_TRANSACTION_TIME, DateUtil.startOfDay(date), DateUtil.endOfDay(date)));
			criteria.setProjection(Projections.sum(PosTransaction.PROP_AMOUNT));
			uniqueResult = criteria.uniqueResult();
			if (uniqueResult != null && (uniqueResult instanceof Number)) {
				netSales -= ((Number) uniqueResult).doubleValue();
			}
			PosLog.info(getReferenceClass(), Messages.getString("PosTransactionDAO.0") + date + Messages.getString("PosTransactionDAO.14") + netSales); //$NON-NLS-1$ //$NON-NLS-2$
			return NumberUtil.roundToTwoDigit(netSales);
		}
	}

	private Double findTransactionsAmountByHour(Session session, String outletId, Integer hourDuration, PaymentType paymentType) {
		try {
			Calendar calendar = Calendar.getInstance();
			Date toDate = calendar.getTime();
			calendar.add(Calendar.HOUR, -(hourDuration == null ? 24 : hourDuration));
			Date fromDate = calendar.getTime();
			Criteria creditCriteria = session.createCriteria(this.getReferenceClass());
			Criteria debitCriteria = session.createCriteria(this.getReferenceClass());
			creditCriteria.add(Restrictions.eq(PosTransaction.PROP_TRANSACTION_TYPE, TransactionType.CREDIT.name()));
			debitCriteria.add(Restrictions.eq(PosTransaction.PROP_TRANSACTION_TYPE, TransactionType.DEBIT.name()));
			if (paymentType != null) {
				creditCriteria.add(Restrictions.eq(PosTransaction.PROP_PAYMENT_TYPE_STRING, paymentType.name()));
				debitCriteria.add(Restrictions.eq(PosTransaction.PROP_PAYMENT_TYPE_STRING, paymentType.name()));
			}
			if (StringUtils.isNotBlank(outletId)) {
				creditCriteria.add(Restrictions.eq(PosTransaction.PROP_OUTLET_ID, outletId));
				debitCriteria.add(Restrictions.eq(PosTransaction.PROP_OUTLET_ID, outletId));
			}
			creditCriteria
					.add(Restrictions.between(PosTransaction.PROP_TRANSACTION_TIME, DateUtil.getUTCStartOfDay(fromDate), DateUtil.getUTCEndOfDay(toDate)));
			debitCriteria.add(Restrictions.between(PosTransaction.PROP_TRANSACTION_TIME, DateUtil.getUTCStartOfDay(fromDate), DateUtil.getUTCEndOfDay(toDate)));
			creditCriteria.add(Restrictions.eqOrIsNull(PosTransaction.PROP_VOIDED, false));
			ProjectionList projections = Projections.projectionList();
			projections.add(Projections.sum(PosTransaction.PROP_AMOUNT));
			creditCriteria.setProjection(projections);
			debitCriteria.setProjection(projections);
			Number creditAmount = (Number) creditCriteria.uniqueResult();
			Number debitAmount = (Number) debitCriteria.uniqueResult();
			if (creditAmount == null) {
				creditAmount = 0D;
			}
			if (debitAmount == null) {
				debitAmount = 0D;
			}
			return creditAmount.doubleValue() - debitAmount.doubleValue();
		} catch (Exception e0) {
			PosLog.error(getReferenceClass(), e0);
		}
		return 0D;
	}

	public Double findTotalTransactionsAmountByHour(Session session, String outletId, Integer hourDuration) {
		return this.findTransactionsAmountByHour(session, outletId, hourDuration, null);
	}

	public Double findCashTransactionsAmountByHour(Session session, String outletId, Integer hourDuration) {
		return this.findTransactionsAmountByHour(session, outletId, hourDuration, PaymentType.CASH);
	}

	public Double findCreditCardTransactionsAmountByHour(Session session, String outletId, Integer hourDuration) {
		return this.findTransactionsAmountByHour(session, outletId, hourDuration, PaymentType.CREDIT_CARD);
	}

	public Map<Date, Double> findTransactionsAmntGroupByDate(Date fromDate, Date toDate) {
		try (Session session = this.createNewSession()) {
			return this.findTransactionsAmntGroupByDate(fromDate, toDate, session);
		}
	}

	public Map<Date, Double> findTransactionsAmntGroupByDate(Date fromDate, Date toDate, Session session) {
		try {
			Criteria creditCriteria = session.createCriteria(PosTransaction.class);
			Criteria debitCriteria = session.createCriteria(PosTransaction.class);
			creditCriteria.add(Restrictions.eq(PosTransaction.PROP_TRANSACTION_TYPE, TransactionType.CREDIT.name()));
			debitCriteria.add(Restrictions.eq(PosTransaction.PROP_TRANSACTION_TYPE, TransactionType.DEBIT.name()));
			creditCriteria
					.add(Restrictions.between(PosTransaction.PROP_TRANSACTION_TIME, DateUtil.getUTCStartOfDay(fromDate), DateUtil.getUTCEndOfDay(toDate)));
			debitCriteria.add(Restrictions.between(PosTransaction.PROP_TRANSACTION_TIME, DateUtil.getUTCStartOfDay(fromDate), DateUtil.getUTCEndOfDay(toDate)));
			creditCriteria.add(Restrictions.eqOrIsNull(PosTransaction.PROP_VOIDED, false));
			ProjectionList projectionList = Projections.projectionList();
			Projection sqlGroupProjection = Projections.sqlGroupProjection("date(transaction_time) as transaction_time, sum(amount) as amount", //$NON-NLS-1$
					"date(transaction_time)", new String[] { "transaction_time", "amount" }, new Type[] { StandardBasicTypes.DATE, StandardBasicTypes.DOUBLE }); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
			projectionList.add(sqlGroupProjection);
			creditCriteria.setProjection(projectionList);
			debitCriteria.setProjection(projectionList);
			List<Object[]> creditList = creditCriteria.list();
			List<Object[]> debitList = debitCriteria.list();
			Map<Date, Double> dateAmntMap = new TreeMap<Date, Double>(new Comparator<Date>() {
				@Override
				public int compare(Date date1, Date date2) {
					return date1.compareTo(date2);
				}
			});
			if (creditList != null) {
				for (Object[] object : creditList) {
					if (object != null && object[0] != null && object[1] != null) {
						Date date = (Date) object[0];
						Double amnt = (Double) object[1];
						dateAmntMap.put(date, amnt);
					}
				}
			}
			if (debitList != null) {
				for (Object[] object : debitList) {
					if (object != null && object[0] != null && object[1] != null) {
						Date date = (Date) object[0];
						Double amnt = dateAmntMap.get(date);
						if (amnt == null) {
							amnt = 0D;
						}
						amnt = amnt - (Double) object[1];
						dateAmntMap.put(date, amnt);
					}
				}
			}
			return dateAmntMap;
		} catch (Exception e0) {
			PosLog.error(this.getReferenceClass(), e0);
		}
		return null;
	}

	public List<TransactionReportModel> findTransAmntGroupByDate(String outletId, Date fromDate, Date toDate, Session session) {
		//@formatter:off
		String toChar =" to_char(" + PosTransaction.PROP_TRANSACTION_TIME + ", 'YYYY-MM-DD')";//$NON-NLS-1$
		
		String hql = "select " + toChar + " as " + TransactionReportModel.PROP_DISPLAY_DATE//$NON-NLS-1$
		+ ", sum( case when transaction_type = 'CREDIT' then amount else (amount * -1) end) as " + TransactionReportModel.PROP_AMOUNT//$NON-NLS-1$
		+ " from " + PosTransaction.REF//$NON-NLS-1$
			+ " where " + PosTransaction.PROP_OUTLET_ID +" = :outletId"//$NON-NLS-1$
				+ "	and "+ PosTransaction.PROP_TRANSACTION_TIME + " between :fromDate and :toDate"//$NON-NLS-1$
				+ " and voided = false"//$NON-NLS-1$
				+ "	and transaction_type in('CREDIT'," + "'DEBIT')" //$NON-NLS-1$
			+ " group by " + toChar //$NON-NLS-1$
			+ " order by displayDate asc" //$NON-NLS-1$
			;
		//@formatter:on
		Query query = session.createQuery(hql);
		query.setString("outletId", outletId); //$NON-NLS-1$
		query.setTimestamp("fromDate", DateUtil.getUTCStartOfDay(fromDate)); //$NON-NLS-1$
		query.setTimestamp("toDate", DateUtil.getUTCEndOfDay(toDate)); //$NON-NLS-1$
		query.setResultTransformer(Transformers.aliasToBean(TransactionReportModel.class));

		return query.list();

	}

	public List<PosTransaction> findTopByPaymentTypes(Session session, String outletId, Date fromDate, Date toDate, Integer maxResult) {
		Criteria criteria = session.createCriteria(PosTransaction.class);
		criteria.add(Restrictions.eq(PosTransaction.PROP_TRANSACTION_TYPE, TransactionType.CREDIT.name()));
		if (StringUtils.isNotBlank(outletId)) {
			criteria.add(Restrictions.eq(PosTransaction.PROP_OUTLET_ID, outletId));
		}
		criteria.add(Restrictions.or(Restrictions.isNull(PosTransaction.PROP_VOIDED), Restrictions.eq(PosTransaction.PROP_VOIDED, false)));
		criteria.add(Restrictions.between(PosTransaction.PROP_TRANSACTION_TIME, DateUtil.getUTCStartOfDay(fromDate), DateUtil.getUTCEndOfDay(toDate)));
		ProjectionList projectionList = Projections.projectionList();
		projectionList.add(Projections.groupProperty(PosTransaction.PROP_PAYMENT_TYPE_STRING), PosTransaction.PROP_PAYMENT_TYPE_STRING);
		projectionList.add(Projections.sum(PosTransaction.PROP_AMOUNT), PosTransaction.PROP_AMOUNT);
		criteria.setProjection(projectionList);
		criteria.setMaxResults(maxResult == null ? 10 : maxResult);
		criteria.addOrder(Order.desc(PosTransaction.PROP_AMOUNT));
		criteria.setResultTransformer(Transformers.aliasToBean(PosTransaction.class));
		return criteria.list();
	}

	public static Disjunction createMasterCardSearchCriteria() {
		Criterion masterCard = Restrictions.eq(PosTransaction.PROP_CARD_TYPE, CardTypeEnum.MASTER_CARD.name()).ignoreCase();
		Criterion masterCard_ = Restrictions.eq(PosTransaction.PROP_CARD_TYPE, CardTypeEnum.MASTER_CARD.name().replaceAll("_", "")).ignoreCase(); //$NON-NLS-1$//$NON-NLS-2$
		Criterion masterCardSpace = Restrictions.eq(PosTransaction.PROP_CARD_TYPE, CardTypeEnum.MASTER_CARD.name().replaceAll("_", " ")).ignoreCase(); //$NON-NLS-1$//$NON-NLS-2$

		return Restrictions.or(masterCard, masterCard_, masterCardSpace);
	}

	public static Disjunction createAmexOrAmericanExpCardSearchCriteria() {
		Criterion americanExpress = Restrictions.eq(PosTransaction.PROP_CARD_TYPE, CardTypeEnum.AMERICAN_EXPRESS.name()).ignoreCase();
		Criterion americanExpress_ = Restrictions.eq(PosTransaction.PROP_CARD_TYPE, CardTypeEnum.AMERICAN_EXPRESS.name().replaceAll("_", "")).ignoreCase(); //$NON-NLS-1$//$NON-NLS-2$
		Criterion americanExprSpace = Restrictions.eq(PosTransaction.PROP_CARD_TYPE, CardTypeEnum.AMERICAN_EXPRESS.name().replaceAll("_", " ")).ignoreCase(); //$NON-NLS-1$//$NON-NLS-2$
		Criterion amex = Restrictions.eq(PosTransaction.PROP_CARD_TYPE, "AMEX").ignoreCase(); //$NON-NLS-1$

		return Restrictions.or(americanExpress, americanExpress_, americanExprSpace, amex);
	}

	@SuppressWarnings("unchecked")
	public List<EndOfDayReportData> findEndOfDayReportData(Date fromDate, Date toDate, List<OrderType> orderTypes) {
		Map<String, String> mamberIdMap = new HashMap<>();
		Map<String, EndOfDayReportData> ticketIdMap = new HashMap<>();

		List<String> orderTypeIds = POSUtil.getStringIds(orderTypes, OrderType.class);
		List<EndOfDayReportData> dataList = new ArrayList<>();

		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(Ticket.class);
			criteria.add(Restrictions.between(Ticket.PROP_CREATE_DATE, fromDate, toDate));
			criteria.add(Restrictions.eq(Ticket.PROP_VOIDED, Boolean.FALSE));
			if (orderTypes != null && !orderTypes.isEmpty()) {
				criteria.add(Restrictions.in(Ticket.PROP_ORDER_TYPE_ID, orderTypeIds)); //$NON-NLS-1$
			}
			criteria.addOrder(Order.asc(Ticket.PROP_CREATE_DATE));

			List<Ticket> list = criteria.list();
			for (Ticket fullTicket : list) {
				EndOfDayReportData tData = createEndOfDateReportData(mamberIdMap, fullTicket);

				dataList.add(tData);
				ticketIdMap.put(fullTicket.getId(), tData);
			}

			criteria = session.createCriteria(PosTransaction.class);
			criteria.createAlias("ticket", "t"); //$NON-NLS-1$ //$NON-NLS-2$
			criteria.add(Restrictions.between(PosTransaction.PROP_TRANSACTION_TIME, fromDate, toDate));
			criteria.add(Restrictions.eq("t." + Ticket.PROP_VOIDED, Boolean.FALSE));
			criteria.add(Restrictions.eq(PosTransaction.PROP_VOIDED, Boolean.FALSE));
			if (orderTypes != null && !orderTypes.isEmpty()) {
				criteria.add(Restrictions.in("t." + Ticket.PROP_ORDER_TYPE_ID, orderTypeIds)); //$NON-NLS-1$
			}
			List<PosTransaction> transactions = criteria.list();
			for (PosTransaction posTransaction : transactions) {
				String ticketId = posTransaction.getTicketId();
				EndOfDayReportData data = ticketIdMap.get(ticketId);
				if (data == null) {
					Ticket ticket = TicketDAO.getInstance().get(ticketId, posTransaction.getOutletId());
					if (ticket == null) {
						continue;
					}
					data = createEndOfDateReportData(mamberIdMap, ticket);
					data.setTotalTicketAmount(0.0);
					data.setNetAmount(0.0);
					data.setTaxAmount(0.0);
					data.setServiceCharge(0.0);
					data.setGratuityAmount(0.0);
					data.setDiscount(0.0);

					dataList.add(data);
					ticketIdMap.put(ticket.getId(), data);
				}

				if (posTransaction instanceof RefundTransaction) {
					data.setRefundPaymentCredit(posTransaction.getAmount() + data.getRefundPaymentCredit());
				}
				else {
					posTransaction.setTransactionTime(data.getTicketCreateDate());
					if (posTransaction instanceof PayOutTransaction || posTransaction instanceof CashDropTransaction) {
						continue;
					}
					else if (posTransaction.getPaymentType() == PaymentType.CASH) {
						data.setCashPaymentCredit(posTransaction.getAmount() + data.getCashPaymentCredit());
					}
					else if (posTransaction.getPaymentType() == PaymentType.CREDIT_CARD) {
						data.setCreditCardPaymentCredit(posTransaction.getAmount() + data.getCreditCardPaymentCredit());
					}
					else if (posTransaction.getPaymentType() == PaymentType.MEMBER_ACCOUNT) {
						data.setMemberChargeCredit(posTransaction.getAmount() + data.getMemberChargeCredit());
					}
					else {
						data.setOthersPaymentCredit(posTransaction.getAmount() + data.getOthersPaymentCredit());
					}
				}
			}
			Collections.sort(dataList, new Comparator<EndOfDayReportData>() {
				@Override
				public int compare(EndOfDayReportData o1, EndOfDayReportData o2) {
					return o1.getEmployeeId().compareTo(o2.getEmployeeId());
				}
			});

			return dataList;
		}
	}

	@Deprecated
	public List<EndOfDayReportData> findTicketsGroupedByEmployee(Date fromDate, Date toDate, List<OrderType> orderTypes) {

		List<String> orderTypeIds = new ArrayList<String>();
		if (orderTypes != null) {
			for (OrderType orderType : orderTypes) {
				orderTypeIds.add(orderType.getId());
			}
		}

		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.between(PosTransaction.PROP_TRANSACTION_TIME, fromDate, toDate));
			criteria.createAlias("ticket", "t"); //$NON-NLS-1$ //$NON-NLS-2$
			if (orderTypes != null) {
				criteria.add(Restrictions.in("t." + Ticket.PROP_ORDER_TYPE_ID, orderTypeIds)); //$NON-NLS-1$
			}

			ProjectionList pList = Projections.projectionList();
			pList.add(Projections.property("t." + Ticket.PROP_OWNER_ID), EndOfDayReportData.PROP_EMPLYOYEE_ID); //$NON-NLS-1$
			pList.add(Projections.property("t." + Ticket.PROP_CUSTOMER_ID), EndOfDayReportData.PROP_CUSTOMER_ID); //$NON-NLS-1$
			pList.add(Projections.property("t." + Ticket.PROP_ID), EndOfDayReportData.PROP_TICKET_ID); //$NON-NLS-1$
			pList.add(Projections.property("t." + Ticket.PROP_SUBTOTAL_AMOUNT), EndOfDayReportData.PROP_NET_AMOUNT); //$NON-NLS-1$
			pList.add(Projections.property("t." + Ticket.PROP_TAX_AMOUNT), EndOfDayReportData.PROP_TAX_AMOUNT); //$NON-NLS-1$
			pList.add(Projections.property("t." + Ticket.PROP_SERVICE_CHARGE), EndOfDayReportData.PROP_SERVICE_CHARGE); //$NON-NLS-1$
			pList.add(Projections.property("t." + Ticket.PROP_DISCOUNT_AMOUNT), EndOfDayReportData.PROP_DISCOUNT); //$NON-NLS-1$
			pList.add(Projections.property("t." + Ticket.PROP_TOTAL_AMOUNT), EndOfDayReportData.PROP_TOTAL_TICKET_AMOUNT); //$NON-NLS-1$

			pList.add(Projections.property(PosTransaction.PROP_AMOUNT), EndOfDayReportData.PROP_TOTAL_TRANACTION_AMOUNT);
			pList.add(Projections.property(PosTransaction.PROP_PAYMENT_TYPE_STRING), EndOfDayReportData.PROP_PAYMENT_TYPE);
			pList.add(Projections.property(PosTransaction.PROP_TRANSACTION_TYPE), EndOfDayReportData.PROP_TRANSACTION_TYPE);
			pList.add(Projections.property(PosTransaction.PROP_TIPS_AMOUNT), EndOfDayReportData.PROP_GRATUITY_AMOUNT);

			criteria.setProjection(pList);

			criteria.setResultTransformer(Transformers.aliasToBean(EndOfDayReportData.class));
			List list = criteria.list();

			if (list != null && list.size() > 0) {
				for (Iterator<?> iterator = list.iterator(); iterator.hasNext();) {
					EndOfDayReportData tData = (EndOfDayReportData) iterator.next();

					if (tData.getPaymentType().equalsIgnoreCase(PaymentType.CASH.name())) {
						Double totalAmount = tData.getTotalTransactionAmount();
						if (tData.getTransactionType().equalsIgnoreCase(TransactionType.DEBIT.name())) {
							tData.setRefundPaymentCredit((-1) * totalAmount);
						}
						else {
							tData.setCashPaymentCredit(totalAmount);
						}

					}
					else if (tData.getPaymentType().equalsIgnoreCase(PaymentType.CREDIT_CARD.name())) {
						Double totalAmount = tData.getTotalTransactionAmount();
						if (tData.getTransactionType().equalsIgnoreCase(TransactionType.DEBIT.name())) {
							tData.setRefundPaymentCredit((-1) * totalAmount);
						}
						else {
							tData.setCreditCardPaymentCredit(totalAmount);
						}
					}
					else if (tData.getPaymentType().equalsIgnoreCase(PaymentType.MEMBER_ACCOUNT.name())) {
						Double totalAmount = tData.getTotalTransactionAmount();
						if (tData.getTransactionType().equalsIgnoreCase(TransactionType.DEBIT.name())) {
							tData.setRefundPaymentCredit((-1) * totalAmount);
						}
						else {
							tData.setMemberChargeCredit(totalAmount);
						}
					}
					else {
						Double totalAmount = tData.getTotalTransactionAmount();
						if (tData.getTransactionType().equalsIgnoreCase(TransactionType.DEBIT.name())) {
							tData.setRefundPaymentCredit((-1) * totalAmount);
						}
						else {
							tData.setOthersPaymentCredit(totalAmount);
						}
					}
				}
			}
			//Merged transaction by ticket
			List<EndOfDayReportData> mainData = new ArrayList<>();
			Map<String, EndOfDayReportData> endOfDayReportDataMap = new HashMap<String, EndOfDayReportData>();
			for (Iterator<?> iterator = list.iterator(); iterator.hasNext();) {
				EndOfDayReportData tData = (EndOfDayReportData) iterator.next();
				EndOfDayReportData endOfDayReportData = endOfDayReportDataMap.get(tData.getTicketId());
				if (endOfDayReportData == null) {
					if (tData.getEmployeeId() != null) {
						User user = UserDAO.getInstance().get(tData.getEmployeeId(), endOfDayReportData.getOutletId());
						tData.setEmployeeName(user == null ? "" : user.getFullName()); //$NON-NLS-1$
					}

					if (tData.getCustomerId() != null) {
						Customer customer = CustomerDAO.getInstance().get(tData.getCustomerId());
						tData.setCustomerName(customer == null ? "" : customer.getName()); //$NON-NLS-1$
					}

					endOfDayReportDataMap.put(tData.getTicketId(), tData);
				}
				else {
					endOfDayReportData.setCashPaymentCredit(tData.getCashPaymentCredit() + endOfDayReportData.getCashPaymentCredit());
					endOfDayReportData.setRefundPaymentCredit(tData.getRefundPaymentCredit() + endOfDayReportData.getRefundPaymentCredit());
					endOfDayReportData.setCreditCardPaymentCredit(tData.getCreditCardPaymentCredit() + endOfDayReportData.getCreditCardPaymentCredit());
					endOfDayReportData.setMemberChargeCredit(tData.getMemberChargeCredit() + endOfDayReportData.getMemberChargeCredit());
					endOfDayReportData.setOthersPaymentCredit(tData.getOthersPaymentCredit() + endOfDayReportData.getOthersPaymentCredit());
					iterator.remove();
				}
			}
			mainData.addAll(endOfDayReportDataMap.values());
			return mainData;
		}
	}

	public static CashDrawer populateCashDrawerReportSummary(List<CashDrawer> reportList) {
		CashDrawer cashDrawersReportSummary = new CashDrawer();
		if (reportList != null) {
			for (CashDrawer report : reportList) {
				cashDrawersReportSummary.setBeginCash(cashDrawersReportSummary.getBeginCash() + report.getBeginCash());
			}
		}
		return cashDrawersReportSummary;
	}

	private EndOfDayReportData createEndOfDateReportData(Map<String, String> mamberIdMap, Ticket fullTicket) {
		EndOfDayReportData tData = new EndOfDayReportData();
		tData.setEmployeeId(fullTicket.getOwnerId());

		tData.setTicketId(fullTicket.getId());
		tData.setTicketCreateDate(fullTicket.getCreateDate());
		tData.setNetAmount(fullTicket.getSubtotalAmount());
		tData.setTaxAmount(fullTicket.getTaxAmount());
		tData.setServiceCharge(fullTicket.getServiceCharge());
		tData.setDiscount(fullTicket.getDiscountAmount());
		tData.setTotalTicketAmount(fullTicket.getTotalAmount());
		tData.setGratuityAmount(fullTicket.getGratuityAmount());

		String customerId = fullTicket.getCustomerId();
		String memberId = ""; //$NON-NLS-1$
		if (StringUtils.isNotEmpty(customerId)) {
			memberId = mamberIdMap.get(customerId);
			if (memberId == null) {
				Customer customer = CustomerDAO.getInstance().get(customerId);
				if (customer != null) {
					mamberIdMap.put(customerId, customer.getMemberId());
					memberId = customer.getMemberId();
					tData.setCustomerName(customer.getName());
				}
			}
		}
		tData.setCustomerId(customerId);
		tData.setMemberId(memberId);
		return tData;
	}

	public List<PosTransaction> findTransactionPaymentsByUser(Date startTime, Date endTime, Outlet outlet, User user, boolean isShowOnlyVoidedPayment) {
		Criteria criteria = null;

		try (Session session = createNewSession()) {

			criteria = session.createCriteria(PosTransaction.class);
			criteria.add(Restrictions.eq(PosTransaction.PROP_VOIDED, isShowOnlyVoidedPayment));

			if (isShowOnlyVoidedPayment) {
				criteria.add(Restrictions.between(PosTransaction.PROP_VOID_DATE, startTime, endTime));
			}
			else {
				criteria.add(Restrictions.between(PosTransaction.PROP_TRANSACTION_TIME, startTime, endTime));
			}

			criteria.add(Restrictions.isNotNull(PosTransaction.PROP_TICKET));
			if (outlet != null) {
				criteria.add(Restrictions.eq(PosTransaction.PROP_OUTLET_ID, outlet.getId()));
			}
			addMultiUserFilter(user, criteria);

			ProjectionList pList = Projections.projectionList();
			pList.add(Projections.property(PosTransaction.PROP_PAYMENT_TYPE_STRING), PosTransaction.PROP_PAYMENT_TYPE_STRING);
			pList.add(Projections.property(PosTransaction.PROP_CUSTOM_PAYMENT_NAME), PosTransaction.PROP_CUSTOM_PAYMENT_NAME);
			pList.add(Projections.property(PosTransaction.PROP_USER_ID), PosTransaction.PROP_USER_ID);
			pList.add(Projections.property(PosTransaction.PROP_ID), PosTransaction.PROP_ID);
			pList.add(Projections.property(PosTransaction.PROP_TRANSACTION_TIME), PosTransaction.PROP_TRANSACTION_TIME);
			pList.add(Projections.property(PosTransaction.PROP_AMOUNT), PosTransaction.PROP_AMOUNT);
			pList.add(Projections.property(PosTransaction.PROP_TICKET), PosTransaction.PROP_TICKET);
			pList.add(Projections.property(PosTransaction.PROP_TRANSACTION_TYPE), PosTransaction.PROP_TRANSACTION_TYPE);
			pList.add(Projections.property(PosTransaction.PROP_VOID_DATE), PosTransaction.PROP_VOID_DATE);
			criteria.setProjection(pList);
			criteria.setResultTransformer(Transformers.aliasToBean(PosTransaction.class));
			criteria.addOrder(Order.asc(PosTransaction.PROP_USER_ID));
			criteria.addOrder(Order.asc(PosTransaction.PROP_PAYMENT_TYPE_STRING));
			criteria.addOrder(Order.asc(PosTransaction.PROP_TRANSACTION_TIME));

			List<PosTransaction> list = criteria.list();
			return list;
		}
	}

	public double calculateTipsByUser(Date fromDate, Date toDate, User user, Outlet outlet, Boolean isCash) {
		try (Session session = createNewSession()) {
			return calculateTipsByUser(session, fromDate, toDate, user, outlet, isCash);
		}
	}

	public double calculateTipsByUser(Session session, Date fromDate, Date toDate, User user, Outlet outlet, Boolean isCash) {
		Criteria ticketCriteria = session.createCriteria(Ticket.class);
		ticketCriteria.createAlias(Ticket.PROP_GRATUITY, "g"); //$NON-NLS-1$

		ticketCriteria.add(Restrictions.eq(Ticket.PROP_VOIDED, Boolean.FALSE));
		if (user != null) {
			ticketCriteria.add(Restrictions.eq("g." + Gratuity.PROP_OWNER_ID, user.getId())); //$NON-NLS-1$
		}
		if (outlet != null) {
			ticketCriteria.add(Restrictions.eq("g." + Gratuity.PROP_OUTLET_ID, outlet.getId()));
		}
		ticketCriteria.add(Restrictions.ge(Ticket.PROP_CREATE_DATE, fromDate));
		ticketCriteria.add(Restrictions.lt(Ticket.PROP_CREATE_DATE, toDate));

		ticketCriteria.setProjection(Projections.distinct(Projections.property(Ticket.PROP_ID)));
		List<String> list = ticketCriteria.list();
		if (list.isEmpty()) {
			return 0d;
		}

		Criteria criteria = session.createCriteria(PosTransaction.class);
		criteria.createAlias(PosTransaction.PROP_TICKET, "t"); //$NON-NLS-1$

		criteria.add(Restrictions.eq(PosTransaction.PROP_VOIDED, Boolean.FALSE));

		if (isCash != null) {
			if (isCash) {
				criteria.add(Restrictions.eq(PosTransaction.PROP_PAYMENT_TYPE_STRING, PaymentType.CASH.name()));
			}
			else {
				criteria.add(Restrictions.ne(PosTransaction.PROP_PAYMENT_TYPE_STRING, PaymentType.CASH.name()));
			}
		}

		criteria.add(Restrictions.in("t." + Ticket.PROP_ID, list)); //$NON-NLS-1$
		criteria.setProjection(Projections.sum(PosTransaction.PROP_TIPS_AMOUNT));
		criteria.list();
		return POSUtil.getDoubleAmount(criteria.uniqueResult());
	}

	public double findDue(String agentId, Date fromDate, Date toDate) {

		Criteria criteria = null;

		try (Session session = createNewSession()) {
			criteria = session.createCriteria(PosTransaction.class);
			criteria.createAlias(PosTransaction.PROP_TICKET, "t"); //$NON-NLS-1$

			criteria.setProjection(Projections.sum(PosTransaction.PROP_AMOUNT));

			criteria.add(Restrictions.eq("t." + Ticket.PROP_VOIDED, Boolean.FALSE)); //$NON-NLS-1$
			criteria.add(Restrictions.lt("t." + Ticket.PROP_CREATE_DATE, fromDate)); //$NON-NLS-1$
			criteria.add(Restrictions.eq("t." + Ticket.PROP_REFERRER_ID, agentId)); //$NON-NLS-1$

			criteria.add(Restrictions.ge(PosTransaction.PROP_TRANSACTION_TIME, fromDate));
			criteria.add(Restrictions.lt(PosTransaction.PROP_TRANSACTION_TIME, toDate));
			criteria.add(Restrictions.eq(PosTransaction.PROP_VOIDED, Boolean.FALSE));

			return POSUtil.getDoubleAmount(criteria.uniqueResult());
		}

	}

	public PosTransaction doPayFeePayment(FeePaymentMapper paymentMapper) throws Exception {

		Session session = null;
		Transaction tx = null;
		try {
			session = GenericDAO.getInstance().createNewSession();
			tx = session.beginTransaction();

			Terminal terminal = paymentMapper.getTerminal();
			User user = paymentMapper.getUser();
			double tenderAmount = paymentMapper.getTenderAmount();
			if (NumberUtil.isZero(tenderAmount)) {
				throw new PosException("Tender amount should be greater than 0");
			}

			double unpaidAmount = paymentMapper.getUnpaidAmount();
			if (paymentMapper.getCustomPayment() != null && tenderAmount > unpaidAmount) {
				throw new PosException("Tender amount must not be greater than the due amount");
			}

			Customer referrer = paymentMapper.getReferrer();

			double payOutAmount = tenderAmount;
			String ticketIds = ""; //$NON-NLS-1$
			PosTransaction transaction = null;
			boolean isDoctorPayment = paymentMapper.isDoctorPayment();
			if (isDoctorPayment) {
				double totalAmount = 0;

				for (TicketItem unpaidTicketItem : paymentMapper.getUnpaidTicketItems()) {
					double labDoctorFeePaidAmount = 0;
					Ticket ticket = unpaidTicketItem.getTicket();
					Ticket fullTicket = TicketDAO.getInstance().loadFullTicket(ticket.getId(), ticket.getOutletId(), session);
					if (fullTicket == null) {
						continue;
					}

					List<TicketItem> ticketItems = fullTicket.getTicketItems();
					for (TicketItem labDoctorFeeTicketItem : ticketItems) {
						if (labDoctorFeeTicketItem.getId().equals(unpaidTicketItem.getId())) {
							if (labDoctorFeeTicketItem.isLabDoctorFeePaid()) {
								continue;
							}

							double labDoctorFee = labDoctorFeeTicketItem.getLabDoctorFee();
							Double beforeBalance = labDoctorFeeTicketItem.getLabDoctorFeePaidAmount();
							Double partialPayAmount = labDoctorFee - beforeBalance;
							if (payOutAmount >= partialPayAmount) {
								labDoctorFeeTicketItem.setLabDoctorFeePaidAmount(NumberUtil.round(labDoctorFee));
								labDoctorFeeTicketItem.setLabDoctorFeePaid(true);

								totalAmount += partialPayAmount;
								payOutAmount -= partialPayAmount;
							}
							else {
								labDoctorFeeTicketItem.setLabDoctorFeePaidAmount(NumberUtil.round(beforeBalance + payOutAmount));
								totalAmount += payOutAmount;
								payOutAmount = 0d;
							}
						}

						labDoctorFeePaidAmount += labDoctorFeeTicketItem.getLabDoctorFeePaidAmount();
					}

					fullTicket.setLabDoctorFeePaidAmount(NumberUtil.round(labDoctorFeePaidAmount));
					double dueAmount = fullTicket.getLabDoctorFee() - labDoctorFeePaidAmount;
					fullTicket.setLabDoctorFeePaid(dueAmount <= 0);

					TicketDAO.getInstance().saveOrUpdate(fullTicket, session);
					String ticketId = "'" + fullTicket.getId() + "'"; //$NON-NLS-1$ //$NON-NLS-2$
					if (!ticketIds.contains(ticketId)) {
						if (StringUtils.isNotBlank(ticketIds)) {
							ticketIds += ", "; //$NON-NLS-1$
						}
						ticketIds += ticketId;
					}

					if (payOutAmount <= 0) {
						break;
					}
				}

				if (!NumberUtil.isZero(totalAmount)) {
					CashDrawer currentCashDrawer = terminal.getActiveCurrentCashDrawer();

					FeePaymentMapper ldfPaymentMapper = new FeePaymentMapper(ticketIds, referrer, totalAmount, tenderAmount,
							paymentMapper.getCustomPayment(), paymentMapper.getCustomPaymnetRef(), user, terminal, currentCashDrawer);

					transaction = LdfPayTransactionDAO.getInstance().createLdfPayTransaction(ldfPaymentMapper, session);
				}

			}
			else {

				double amount = 0;
				for (Ticket ticket : paymentMapper.getUnpaidTickets()) {

					Ticket fullTicket = TicketDAO.getInstance().loadFullTicket(ticket.getId(), ticket.getOutletId());
					if (fullTicket.isReferrerFeePaid()) {
						continue;
					}
					double referrerFee = fullTicket.getTotalReferrerFee();
					Double beforeBalance = fullTicket.getReferrerFeePaidAmount();
					Double partialPayAmount = fullTicket.getTotalReferrerFee() - beforeBalance;
					if (payOutAmount >= partialPayAmount) {
						fullTicket.setReferrerFeePaidAmount(NumberUtil.round(referrerFee));
						fullTicket.setReferrerFeePaid(true);

						amount += partialPayAmount;
						payOutAmount -= partialPayAmount;
					}
					else {
						fullTicket.setReferrerFeePaidAmount(NumberUtil.round(beforeBalance + payOutAmount));
						amount += payOutAmount;
						payOutAmount = 0d;
					}

					TicketDAO.getInstance().saveOrUpdate(fullTicket, session);
					String ticketId = "'" + fullTicket.getId() + "'"; //$NON-NLS-1$ //$NON-NLS-2$
					if (!ticketIds.contains(ticketId)) {
						if (StringUtils.isNotBlank(ticketIds)) {
							ticketIds += ", "; //$NON-NLS-1$
						}
						ticketIds += ticketId;
					}

					if (payOutAmount <= 0) {
						break;
					}
				}
				if (!NumberUtil.isZero(amount)) {
					CashDrawer currentCashDrawer = terminal.getActiveCurrentCashDrawer();

					FeePaymentMapper rfPaymentMapper = new FeePaymentMapper(ticketIds, referrer, amount, tenderAmount,
							paymentMapper.getCustomPayment(), paymentMapper.getCustomPaymnetRef(), user, terminal, currentCashDrawer);
					transaction = RfPayTransactionDAO.getInstance().createRfPayTransaction(rfPaymentMapper, session);
				}
			}

			tx.commit();

			logJournal(transaction, isDoctorPayment, unpaidAmount, referrer);

			return transaction;

		} catch (Exception e1) {
			try {
				tx.rollback();
			} catch (Exception x) {
				PosLog.error(getClass(), x);
			}
			throw e1;
		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

	private void logJournal(PosTransaction posTransaction, boolean isDoctorPayment, double unpaidAmount, Customer referrer) {
		if (posTransaction == null) {
			return;
		}

		try {
			String actionName = isDoctorPayment ? ActionHistory.LDF_PAY : ActionHistory.RF_PAY;

			StringBuilder builder = new StringBuilder();
			builder.append("Cash drawer id: " + posTransaction.getCashDrawerId()); //$NON-NLS-1$
			builder.append((isDoctorPayment ? ", LD id: " : ", Referrer id: ") + referrer.getId()); //$NON-NLS-1$ //$NON-NLS-2$

			double paidAmount = posTransaction.getAmount();

			builder.append(String.format(", Amount before: %s, Paid amount: %s, Amount after: %s, Payment type: %s", //$NON-NLS-1$
					NumberUtil.formatNumber(unpaidAmount), NumberUtil.formatNumber(paidAmount), NumberUtil.formatNumber(unpaidAmount - paidAmount),
					posTransaction.getPaymentType().getDisplayString()));

			if (posTransaction.getPaymentType().equals(PaymentType.CUSTOM_PAYMENT)) {
				builder.append(", Name: " + posTransaction.getCustomPaymentName()); //$NON-NLS-1$
				builder.append(", Ref: " + posTransaction.getCustomPaymentRef()); //$NON-NLS-1$
			}

			String actionMessage = builder.toString();
			actionMessage += ", Transaction id: " + posTransaction.getId(); //$NON-NLS-1$
			ActionHistoryDAO.saveHistory(actionName, actionMessage);
		} catch (Exception e) {
			PosLog.error(getClass(), e);
		}
	}

}