package com.floreantpos.model.dao;

import org.apache.commons.lang.StringUtils;
import org.hibernate.Criteria;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.hibernate.criterion.DetachedCriteria;
import org.hibernate.criterion.MatchMode;
import org.hibernate.criterion.Projections;
import org.hibernate.criterion.Property;
import org.hibernate.criterion.Restrictions;

import com.floreantpos.PosLog;
import com.floreantpos.model.COAAccountType;
import com.floreantpos.model.ChartOfAccounts;
import com.floreantpos.model.DirectionType;
import com.floreantpos.model.LedgerEntry;
import com.floreantpos.model.PosTransaction;
import com.floreantpos.model.Ticket;
import com.floreantpos.swing.PaginationSupport;
import com.floreantpos.util.NumberUtil;

public class LedgerEntryDAO extends BaseLedgerEntryDAO {

	public void findByChartOfAcountName(PaginationSupport paginationSupport, String searchString) {
		try (Session session = createNewSession()) {

			Criteria criteria = session.createCriteria(LedgerEntry.class);
			addDeletedFilter(criteria);

			if (StringUtils.isNotBlank(searchString)) {
				DetachedCriteria detachedCriteria = DetachedCriteria.forClass(COAAccountType.class);
				detachedCriteria.setProjection(Property.forName(COAAccountType.PROP_ID));
				detachedCriteria.add(Restrictions.ilike(COAAccountType.PROP_NAME, searchString, MatchMode.ANYWHERE));

				criteria.add(Property.forName(LedgerEntry.PROP_ACCOUNT_ID).in(detachedCriteria)); //$NON-NLS-1$
			}
			criteria.setProjection(Projections.rowCount());

			Number rowCount = (Number) criteria.uniqueResult();
			if (rowCount != null) {
				paginationSupport.setNumRows(rowCount.intValue());
			}
			criteria.setProjection(null);
			paginationSupport.setRows(criteria.list());
		}
	}

	public void saveTicketLedgerEntry(Ticket ticket, double remainLedgerAmount) {
		ChartOfAccountsDAO instance = ChartOfAccountsDAO.getInstance();
		try (Session session = instance.createNewSession()) {
			Transaction tx = session.beginTransaction();

			ChartOfAccounts reciableChartOfAccounts = instance.findByAcountCode("5000", session);
			ChartOfAccounts salesChartOfAccounts = instance.findByAcountCode("200", session);

			LedgerEntry createDebitLedgerEntry = LedgerEntry.createLedgerEntry(ticket, reciableChartOfAccounts, DirectionType.DEBIT, remainLedgerAmount);
			LedgerEntryDAO.getInstance().save(createDebitLedgerEntry, session);
			LedgerEntry createCreditLedgerEntry = LedgerEntry.createLedgerEntry(ticket, salesChartOfAccounts, DirectionType.CREDIT, remainLedgerAmount);
			LedgerEntryDAO.getInstance().save(createCreditLedgerEntry, session);

			tx.commit();
		} catch (Exception e) {
			PosLog.error(getClass(), e);
		}
	}

	public void saveTransactionLedgerEntry(Ticket ticket, PosTransaction transaction) {
		ChartOfAccountsDAO instance = ChartOfAccountsDAO.getInstance();
		try (Session session = instance.createNewSession()) {
			Transaction tx = session.beginTransaction();

			ChartOfAccounts cashChartOfAccounts = instance.findByAcountCode("10100", session);
			ChartOfAccounts reciableChartOfAccounts = instance.findByAcountCode("5000", session);

			Double transAmount = transaction.isVoided() ? (-1) * transaction.getAmount() : transaction.getAmount();
			LedgerEntry createDebitLedgerEntry = LedgerEntry.createLedgerEntry(ticket, cashChartOfAccounts, DirectionType.DEBIT, transAmount);
			createDebitLedgerEntry.setTransactionId(transaction.getId());
			LedgerEntryDAO.getInstance().save(createDebitLedgerEntry, session);
			LedgerEntry createCreditLedgerEntry = LedgerEntry.createLedgerEntry(ticket, reciableChartOfAccounts, DirectionType.CREDIT, transAmount);
			createCreditLedgerEntry.setTransactionId(transaction.getId());
			LedgerEntryDAO.getInstance().save(createCreditLedgerEntry, session);

			tx.commit();
		} catch (Exception e) {
			PosLog.error(getClass(), e);
		}
	}

	public void saveOrderLedgerEntry(Ticket ticket, PosTransaction transaction) {
		ChartOfAccountsDAO instance = ChartOfAccountsDAO.getInstance();
		try (Session session = instance.createNewSession()) {
			Transaction	tx = session.beginTransaction();

			ChartOfAccounts productsSalesCOA = instance.findByAcountCode("200", session);
			ChartOfAccounts taxCOA = instance.findByAcountCode("506", session);
			ChartOfAccounts rfPayableCOA = instance.findByAcountCode("507", session);
			ChartOfAccounts ldfPayableCOA = instance.findByAcountCode("508", session);
			ChartOfAccounts reciableCOA = instance.findByAcountCode("5000", session);

			double remainLedgerAmount = ticket.getTotalAmount() - ticket.getLedgerAmount();
			if (!NumberUtil.isZero(remainLedgerAmount)) {

				Double taxAmount = ticket.getTaxAmount();
				Double totalReferrerFee = ticket.getTotalReferrerFee();
				Double labDoctorFee = ticket.getLabDoctorFee();
				double productSalesAmount = remainLedgerAmount - (taxAmount + totalReferrerFee + labDoctorFee);

				LedgerEntry createPSLedgerEntry = LedgerEntry.createLedgerEntry(ticket, productsSalesCOA, DirectionType.CREDIT, productSalesAmount);
				LedgerEntryDAO.getInstance().save(createPSLedgerEntry, session);

				if (!NumberUtil.isZero(taxAmount)) {
					LedgerEntry createTaxLedgerEntry = LedgerEntry.createLedgerEntry(ticket, taxCOA, DirectionType.CREDIT, taxAmount);
					LedgerEntryDAO.getInstance().save(createTaxLedgerEntry, session);
				}

				if (!NumberUtil.isZero(totalReferrerFee)) {
					LedgerEntry createRFLedgerEntry = LedgerEntry.createLedgerEntry(ticket, rfPayableCOA, DirectionType.CREDIT, totalReferrerFee);
					LedgerEntryDAO.getInstance().save(createRFLedgerEntry, session);
				}

				if (!NumberUtil.isZero(labDoctorFee)) {
					LedgerEntry createLDFLedgerEntry = LedgerEntry.createLedgerEntry(ticket, ldfPayableCOA, DirectionType.CREDIT, labDoctorFee);
					LedgerEntryDAO.getInstance().save(createLDFLedgerEntry, session);
				}

				ticket.putLedgerAmount(ticket.getTotalAmount());
				TicketDAO.getInstance().saveOrUpdate(ticket, session);
			}

			Double transAmount = transaction.getAmount();
			if (!NumberUtil.isZero(transAmount)) {
				LedgerEntry createCreditLedgerEntry = LedgerEntry.createLedgerEntry(ticket, reciableCOA, DirectionType.DEBIT, transAmount);
				createCreditLedgerEntry.setTransactionId(transaction.getId());
				LedgerEntryDAO.getInstance().save(createCreditLedgerEntry, session);
			}

			tx.commit();
		} catch (Exception e) {
			PosLog.error(getClass(), e);
		}
	}

}