/**
 * ************************************************************************
 * * The contents of this file are subject to the MRPL 1.2
 * * (the  "License"),  being   the  Mozilla   Public  License
 * * Version 1.1  with a permitted attribution clause; you may not  use this
 * * file except in compliance with the License. You  may  obtain  a copy of
 * * the License at http://www.floreantpos.org/license.html
 * * Software distributed under the License  is  distributed  on  an "AS IS"
 * * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * * License for the specific  language  governing  rights  and  limitations
 * * under the License.
 * * The Original Code is FLOREANT POS.
 * * The Initial Developer of the Original Code is OROCUBE LLC
 * * All portions are Copyright (C) 2015 OROCUBE LLC
 * * All Rights Reserved.
 * ************************************************************************
 */
package com.floreantpos.model.dao;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import java.util.Random;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.commons.lang.StringUtils;
import org.hibernate.Criteria;
import org.hibernate.Hibernate;
import org.hibernate.HibernateException;
import org.hibernate.Query;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.hibernate.criterion.Criterion;
import org.hibernate.criterion.DetachedCriteria;
import org.hibernate.criterion.Disjunction;
import org.hibernate.criterion.MatchMode;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.ProjectionList;
import org.hibernate.criterion.Projections;
import org.hibernate.criterion.Restrictions;
import org.hibernate.criterion.SimpleExpression;
import org.hibernate.criterion.Subqueries;
import org.hibernate.transform.Transformers;

import com.floreantpos.DuplicateDataException;
import com.floreantpos.Messages;
import com.floreantpos.PosException;
import com.floreantpos.constants.AppConstants;
import com.floreantpos.model.ActionHistory;
import com.floreantpos.model.Attribute;
import com.floreantpos.model.AttributeGroup;
import com.floreantpos.model.ComboGroup;
import com.floreantpos.model.ComboItem;
import com.floreantpos.model.Discount;
import com.floreantpos.model.InventoryLocation;
import com.floreantpos.model.InventoryStock;
import com.floreantpos.model.InventoryStockUnit;
import com.floreantpos.model.InventoryTransaction;
import com.floreantpos.model.InventoryTransactionType;
import com.floreantpos.model.InventoryUnit;
import com.floreantpos.model.InventoryVendor;
import com.floreantpos.model.InventoryVendorItems;
import com.floreantpos.model.MenuCategory;
import com.floreantpos.model.MenuGroup;
import com.floreantpos.model.MenuItem;
import com.floreantpos.model.MenuItemInventoryStatus;
import com.floreantpos.model.MenuItemModifierPage;
import com.floreantpos.model.MenuItemModifierSpec;
import com.floreantpos.model.MenuPageItem;
import com.floreantpos.model.OnlineItem;
import com.floreantpos.model.OrderType;
import com.floreantpos.model.Outlet;
import com.floreantpos.model.PackagingUnit;
import com.floreantpos.model.Pagination;
import com.floreantpos.model.PizzaPrice;
import com.floreantpos.model.PriceTableItem;
import com.floreantpos.model.PrinterGroup;
import com.floreantpos.model.ProductType;
import com.floreantpos.model.PurchaseOrder;
import com.floreantpos.model.PurchaseOrderItem;
import com.floreantpos.model.ReportGroup;
import com.floreantpos.model.SiiopaCustomer;
import com.floreantpos.model.Terminal;
import com.floreantpos.model.User;
import com.floreantpos.model.util.DataProvider;
import com.floreantpos.swing.BeanTableModel;
import com.floreantpos.swing.PaginatedListModel;
import com.floreantpos.swing.PaginationSupport;
import com.floreantpos.util.DefaultDataInserter;
import com.floreantpos.util.NumberUtil;
import com.floreantpos.util.POSUtil;

@SuppressWarnings({ "unchecked", "rawtypes" })
public class MenuItemDAO extends BaseMenuItemDAO {

	/**
	 * Default constructor. Can be used in place of getInstance()
	 */
	public MenuItemDAO() {
	}

	@Override
	protected Serializable save(Object obj, Session s) {
		updateTime(obj);
		Serializable serializable = super.save(obj, s);
		updateDependentModels((MenuItem) obj, s);
		return serializable;
	}

	@Override
	protected void update(Object obj, Session s) {
		updateTime(obj);
		super.update(obj, s);
		updateDependentModels((MenuItem) obj, s);
	}

	@Override
	protected void saveOrUpdate(Object obj, Session s) {
		updateTime(obj);
		super.saveOrUpdate(obj, s);
		updateDependentModels((MenuItem) obj, s);
	}

	@Override
	public void delete(Object obj, Session session) throws HibernateException {
		MenuItem menuItem = (MenuItem) obj;
		if (menuItem == null) {
			throw new PosException(Messages.getString("MenuItemDAO.0")); //$NON-NLS-1$
		}
		session.refresh(menuItem);
		checkIfItemCanbeDeleted(session, menuItem);
		if (menuItem.isComboItem()) {
			List<ComboGroup> comboGroups = menuItem.getComboGroups();
			if (comboGroups != null && !comboGroups.isEmpty()) {
				comboGroups.clear();
			}
			List<ComboItem> comboItems = menuItem.getComboItems();
			if (comboItems != null && !comboItems.isEmpty()) {
				for (ComboItem comboItem : comboItems) {
					comboItem.setDeleted(true);
					ComboItemDAO.getInstance().update(comboItem, session);
				}
			}
		}
		menuItem.setDeleted(true);
		removeFromDependentModels(menuItem, session);
		updateTime(menuItem);
		session.update(menuItem);

		//delete variants
		if (menuItem.isHasVariant()) {
			List<MenuItem> variants = menuItem.getVariants();
			if (variants != null && !variants.isEmpty()) {
				for (MenuItem variant : variants) {
					variant.setDeleted(true);
					update(variant, session);
				}
			}
		}
		removeFromDiscounts(menuItem, session);
		List<OnlineItem> onlineItems = OnlineItemDAO.getInstance().findByMenuItemId(menuItem.getId(), session);
		if (onlineItems != null && onlineItems.size() > 0) {
			for (OnlineItem onlineItem : onlineItems) {
				OnlineItemDAO.getInstance().delete(onlineItem, session);
			}
		}
	}

	private void removeFromDiscounts(MenuItem menuItem, Session session) {
		DiscountDAO discountDAO = DiscountDAO.getInstance();
		List<Discount> discounts = discountDAO.getDiscountsByMenuItem(menuItem, session);
		if (discounts != null && !discounts.isEmpty()) {
			for (Discount discount : discounts) {
				List<MenuItem> menuItems = discount.getMenuItems();
				menuItems.remove(menuItem);
				discountDAO.saveOrUpdate(discount, session);
			}
		}
	}

	/**
	 * Check if this menu item can be delete. If this item is used by combo item or combo group,
	 * it cannot be deleted.
	 * 
	 * @param s
	 * @param menuItem
	 */
	public void checkIfItemCanbeDeleted(MenuItem menuItem) {
		try (Session session = createNewSession()) {
			checkIfItemCanbeDeleted(session, menuItem);
		}
	}

	/**
	 * Check if this menu item can be delete. If this item is used by combo item or combo group,
	 * it cannot be deleted.
	 * 
	 * @param s
	 * @param menuItem
	 */
	private void checkIfItemCanbeDeleted(Session s, MenuItem menuItem) {
		String details = ""; //$NON-NLS-1$
		List<String> foreignItemNames = RecepieDAO.getInstance().getForeignDataListNames(menuItem, s);
		if (foreignItemNames != null && !foreignItemNames.isEmpty()) {
			details = Messages.getString("MenuItemDAO.7") + " " + menuItem.getName() + " " + Messages.getString("MenuItemDAO.9") //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$
					+ (foreignItemNames.size() > 1 ? Messages.getString("MenuItemDAO.4") : ":"); //$NON-NLS-1$ //$NON-NLS-2$

			int count = 1;
			for (String itemName : foreignItemNames) {
				details += "\n" + count + ". " + itemName; //$NON-NLS-1$ //$NON-NLS-2$
				count++;
			}
		}

		List<String> menuItemIdList = new ArrayList<String>();
		if (menuItem.isHasVariant()) {
			menuItemIdList = POSUtil.getStringIds(menuItem.getVariants(), MenuItem.class);
		}
		else {
			menuItemIdList = Arrays.asList(new String[] { menuItem.getId() });
		}

		List<MenuItem> menuItems = this.getMenuItemsByComboItemId(menuItemIdList, s);
		if (menuItems != null && !menuItems.isEmpty()) {
			if (StringUtils.isNotBlank(details)) {
				details += "\n"; //$NON-NLS-1$
			}
			details += Messages.getString("MenuItemDAO.7") + " " + menuItem.getName() + " " + Messages.getString("MenuItemDAO.12"); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$

			int count = 1;
			for (MenuItem item : menuItems) {
				details += "\n" + count + ". " + item.getName(); //$NON-NLS-1$ //$NON-NLS-2$
				count++;
			}
		}

		menuItems = this.getMenuItemsByComboGroupsByComboGroupItemId(menuItemIdList, s);

		if (menuItems != null && !menuItems.isEmpty()) {
			if (StringUtils.isNotBlank(details)) {
				details += "\n\n"; //$NON-NLS-1$
			}
			details += Messages.getString("MenuItemDAO.7") + " " + menuItem.getName() + " " + Messages.getString("MenuItemDAO.14"); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$

			int count = 1;
			for (MenuItem item : menuItems) {
				details += "\n" + count + ". " + item.getName(); //$NON-NLS-1$ //$NON-NLS-2$
				count++;
			}
		}

		if (StringUtils.isNotBlank(details)) {
			throw new PosException(Messages.getString("MenuItemDAO.7") + " <b>" + menuItem.getName() + "</b> " + Messages.getString("MenuItemDAO.15"), details);//$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$
		}
	}

	public List<MenuItem> getMenuItemsByComboItemId(List<String> comboItemIds) {
		try (Session session = createNewSession()) {
			return getMenuItemsByComboItemId(comboItemIds, session);
		}
	}

	private List<MenuItem> getMenuItemsByComboItemId(List<String> comboItemIds, Session session) {
		if (comboItemIds.isEmpty() || session == null) {
			return null;
		}
		Criteria criteria = session.createCriteria(this.getReferenceClass());
		this.addDeletedFilter(criteria);
		criteria.createAlias("comboItems", "i"); //$NON-NLS-1$ //$NON-NLS-2$
		criteria.add(Restrictions.isNotNull("i." + ComboItem.PROP_MENU_ITEM_ID)); //$NON-NLS-1$
		if (!comboItemIds.isEmpty()) {
			criteria.add(Restrictions.in("i." + ComboItem.PROP_ITEM_ID, comboItemIds)); //$NON-NLS-1$
		}
		criteria.setProjection(Projections.alias(Projections.distinct(Projections.property(MenuItem.PROP_NAME)), MenuItem.PROP_NAME));
		criteria.setResultTransformer(Transformers.aliasToBean(MenuItem.class));
		return criteria.list();
	}

	public List<MenuItem> getMenuItemsByComboGroupsByComboGroupItemId(List<String> menuItemIds) {
		try (Session session = createNewSession()) {
			return getMenuItemsByComboGroupsByComboGroupItemId(menuItemIds, session);
		}
	}

	private List<MenuItem> getMenuItemsByComboGroupsByComboGroupItemId(List<String> menuItemIds, Session session) {
		if (menuItemIds.isEmpty() || session == null) {
			return null;
		}
		Criteria criteria = session.createCriteria(getReferenceClass());
		this.addDeletedFilter(criteria);
		criteria.createAlias("comboGroups", "g"); //$NON-NLS-1$ //$NON-NLS-2$
		criteria.createAlias("g.items", "i"); //$NON-NLS-1$ //$NON-NLS-2$
		criteria.add(Restrictions.in("i." + MenuItem.PROP_ID, menuItemIds)); //$NON-NLS-1$
		criteria.setProjection(Projections.alias(Projections.distinct(Projections.property(MenuItem.PROP_NAME)), MenuItem.PROP_NAME));
		return criteria.setResultTransformer(Transformers.aliasToBean(this.getReferenceClass())).list();
	}

	private void removeFromDependentModels(MenuItem menuItem, Session session) {
		String menuItemId = menuItem.getId();
		PriceTableItem priceTableItem = PriceTableItemDAO.getInstance().getPriceTableItemByMenuItemId(menuItemId);
		if (priceTableItem != null) {
			PriceTableItemDAO.getInstance().delete(priceTableItem, session);
		}

		String hqlString = "delete MenuPageItem where %s=:menuItemId"; //$NON-NLS-1$

		hqlString = String.format(hqlString, MenuPageItem.PROP_MENU_ITEM_ID);
		Query query = session.createQuery(hqlString);
		query.setParameter("menuItemId", menuItemId); //$NON-NLS-1$
		query.executeUpdate();

		// DELETING MENU-ITEM-MODIFIER-SPEC and PRICE-TABLE-ITEM
		//menuItem.setMenuItemModiferSpecs(null);
	}

	private void updateDependentModels(MenuItem menuItem, Session session) {
		//save inventory stock status if new and inventory item
		//		saveInventoryStockStatus(menuItem, session);

		//update menu page item
		MenuPageItemDAO menuPageItemDAO = MenuPageItemDAO.getInstance();
		List<MenuPageItem> pageItems = menuPageItemDAO.getPageItemFor(menuItem, session);
		if (pageItems != null) {
			for (MenuPageItem menuPageItem : pageItems) {
				menuPageItem.setMenuItem(menuItem);
				menuPageItemDAO.saveOrUpdate(menuPageItem, session);
			}
		}
		List<ComboItem> comboItems = ComboItemDAO.getInstance().getByMenuItem(menuItem.getId(), session);
		for (ComboItem comboItem : comboItems) {
			comboItem.setMenuItem(menuItem);
			ComboItemDAO.getInstance().update(comboItem, session);
		}

		List<OnlineItem> onlineItems = OnlineItemDAO.getInstance().findByMenuItemId(menuItem.getId(), session);
		if (onlineItems != null) {
			for (OnlineItem onlineItem : onlineItems) {
				if (menuItem.isDeleted()) {
					OnlineItemDAO.getInstance().delete(onlineItem, session);
				}
				else {
					onlineItem.setName(menuItem.getName());
					OnlineItemDAO.getInstance().update(onlineItem, session);
				}
			}
		}

		// update Inventory Stock Model
		updateDependentStockModel((MenuItem) menuItem, session);

		if (menuItem.isHasVariant()) {
			if (menuItem.getVariants() != null && menuItem.getVariants().size() > 0) {
				for (MenuItem child : menuItem.getVariants()) {
					if (child.isDeleted()) {
						continue;
					}
					updateDependentModels(child, session);
				}
			}
		}
	}

	private void updateDependentStockModel(MenuItem menuItem, Session session) throws HibernateException {
		String hqlString = "update InventoryStock set %s=:itemName, %s=:sku, %s=:barCode where %s=:menuItemId"; //$NON-NLS-1$
		//@formatter:off
		hqlString = String.format(
				hqlString, 
				InventoryStock.PROP_ITEM_NAME, 
				InventoryStock.PROP_SKU,
				InventoryStock.PROP_BARCODE, 
				InventoryStock.PROP_MENU_ITEM_ID
				);
		
		//@formatter:on
		Query query = session.createQuery(hqlString);
		query.setParameter("itemName", menuItem.getName()); //$NON-NLS-1$
		query.setParameter("sku", menuItem.getSku()); //$NON-NLS-1$
		query.setParameter("barCode", menuItem.getBarcode()); //$NON-NLS-1$
		query.setParameter("menuItemId", menuItem.getId()); //$NON-NLS-1$
		query.executeUpdate();

	}

	public void saveQuickMenuItems(List<MenuItem> items, Outlet outlet, boolean addToOnlineItem) {
		int itemIndex = 0;

		for (MenuItem menuItem : items) {
			if (StringUtils.isBlank(menuItem.getName())) {
				continue;
			}

			String sku = menuItem.getSku();

			if (StringUtils.isNotBlank(sku)) {
				try {

					GenericDAO.getInstance().checkDifferentObjectExists(menuItem.getId(), sku, MenuItem.class, MenuItem.PROP_SKU);

				} catch (DuplicateDataException e) {
					throw new DuplicateDataException(itemIndex + "_" + e.getMessage()); //$NON-NLS-1$
				}
			}

			String menuCategoryName = menuItem.getMenuCategoryName();
			String menuGroupName = menuItem.getMenuGroupName();

			if (menuItem.getMenuCategoryId() == null && StringUtils.isNotBlank(menuCategoryName)) {
				MenuCategory menuCategory = MenuCategoryDAO.getInstance().findOrCreateCategoryByName(menuCategoryName);
				if (menuCategory != null) {
					menuItem.setMenuCategoryId(menuCategory.getId());
				}
			}

			if (menuItem.getMenuGroupId() == null && StringUtils.isNotBlank(menuGroupName)) {
				MenuGroup menuGroup = MenuGroupDAO.getInstance().findOrCreateGroupByName(menuGroupName, menuItem.getMenuCategoryId(), menuCategoryName);
				if (menuGroup != null) {
					menuItem.setMenuGroupId(menuGroup.getId());
				}
			}

			if (StringUtils.isBlank(menuItem.getUnitId())) {
				menuItem.setUnit(InventoryUnitDAO.getInstance().get("each")); //$NON-NLS-1$
			}

			Date lastUpdateTime = StoreDAO.getServerTimestamp();
			menuItem.setLastUpdateTime(lastUpdateTime);

			Session session = null;
			Transaction tx = null;
			try {
				session = createNewSession();
				tx = session.beginTransaction();

				saveOrUpdate(menuItem, session);
				OnlineItemDAO.getInstance().saveOnlineItem(menuItem, outlet, session);

				tx.commit();
			} catch (Exception e) {
				try {
					tx.rollback();
				} catch (Exception x) {
				}
				throw new RuntimeException(String.format("Error saving item %s. %s", itemIndex, menuItem.getName()), e); //$NON-NLS-1$
			} finally {
				if (session != null) {
					session.close();
				}
			}
			itemIndex++;
		}
	}

	@Override
	public void saveOrUpdate(MenuItem menuItem) throws HibernateException {
		Session session = null;
		Transaction tx = null;
		try {
			session = createNewSession();
			tx = session.beginTransaction();

			saveOrUpdate(menuItem, session);

			tx.commit();
		} catch (Exception e) {
			try {
				tx.rollback();
			} catch (Exception x) {
			}
			throw e;
		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

	public void saveOrUpdate(MenuItem menuItem, List<AttributeGroup> attributeGroups) throws HibernateException {
		Session session = null;
		Transaction tx = null;
		try {
			session = createNewSession();
			tx = session.beginTransaction();

			if (attributeGroups != null) {
				Date now = StoreDAO.getServerTimestamp();
				int sortOrder = 1;
				for (AttributeGroup attributeGroup : attributeGroups) {
					List<Attribute> attributes = attributeGroup.getAttributes();
					if (attributeGroup.getSortOrder() == 0) {
						int attSortOrder = 1;
						if (attributes != null) {
							for (Attribute attribute : attributes) {
								attribute.setLastUpdateTime(now);
								attribute.setSortOrder(attSortOrder++);
							}
						}
						attributeGroup.setSortOrder(sortOrder++);
					}
					else {
						if (attributes != null) {
							for (Attribute attribute : attributes) {
								attribute.setLastUpdateTime(now);
							}
						}
					}

					AttributeGroupDAO.getInstance().saveOrUpdate(attributeGroup, session);
				}
			}

			saveOrUpdate(menuItem, session);

			tx.commit();
		} catch (Exception e) {
			try {
				tx.rollback();
			} catch (Exception x) {
			}
			throw e;
		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

	@Override
	public void saveOrUpdate(MenuItem menuItem, Session session) throws HibernateException {
		saveOrUpdate(menuItem, session, true);
	}

	public void saveOrUpdate(MenuItem menuItem, Session session, boolean updateLastUpdateTime) {
		if (updateLastUpdateTime) {
			menuItem.setLastUpdateTime(StoreDAO.getServerTimestamp());
		}
		if (menuItem.getId() == null) {
			save(menuItem, session);
		}
		else {
			update(menuItem, session);
			List<MenuItem> variants = menuItem.getVariants();
			if (variants != null) {
				if (updateLastUpdateTime) {
					for (MenuItem variant : variants) {
						variant.setLastUpdateTime(StoreDAO.getServerTimestamp());
					}
				}
				List<MenuItem> deletedVariants = variants.stream().filter(variant -> variant.isDeleted()).collect(Collectors.toList());
				for (MenuItem variant : deletedVariants) {
					removeFromDependentModels(variant, session);
				}
			}
		}
	}

	public void initializeUnits(MenuItem menuItem) {
		Session session = null;
		try {
			session = createNewSession();
			session.refresh(menuItem);
			if (!Hibernate.isInitialized(menuItem.getStockUnits())) {
				Hibernate.initialize(menuItem.getStockUnits());
			}
		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

	public int getRowCount(String searchString) {
		Session session = null;
		Criteria criteria = null;
		try {
			session = createNewSession();
			criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.or(Restrictions.isNull(AppConstants.PROP_DELETED), Restrictions.eq(AppConstants.PROP_DELETED, Boolean.FALSE)));
			criteria.setProjection(Projections.rowCount());

			if (StringUtils.isNotEmpty(searchString)) {
				criteria.add(Restrictions.ilike(MenuItem.PROP_NAME, searchString, MatchMode.START));
			}

			Number rowCount = (Number) criteria.uniqueResult();
			if (rowCount != null) {
				return rowCount.intValue();

			}
		} finally {
			if (session != null) {
				session.close();
			}
		}
		return 0;
	}

	public void loadInventoryItems(String searchString, BeanTableModel<MenuItem> listModel) {
		Session session = null;
		Criteria criteria = null;
		try {
			session = createNewSession();
			criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.or(Restrictions.isNull(AppConstants.PROP_DELETED), Restrictions.eq(AppConstants.PROP_DELETED, Boolean.FALSE)));
			criteria.addOrder(Order.asc(MenuItem.PROP_NAME));

			if (StringUtils.isNotEmpty(searchString)) {
				criteria.add(Restrictions.ilike(MenuItem.PROP_NAME, searchString, MatchMode.START));
			}

			criteria.setFirstResult(listModel.getCurrentRowIndex());
			criteria.setMaxResults(listModel.getPageSize());
			listModel.setRows(criteria.list());
		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

	//	public int rowCount(Boolean menuItem) {
	//		Session session = null;
	//		Criteria criteria = null;
	//		try {
	//			session = createNewSession();
	//			criteria = session.createCriteria(MenuItem.class);
	//			criteria.setProjection(Projections.rowCount());
	//			if (!menuItem)
	//				criteria.add(Restrictions.eq(MenuItem.PROP_INVENTORY_ITEM, Boolean.TRUE));
	//			Number rowCount = (Number) criteria.uniqueResult();
	//			if (rowCount != null) {
	//				return rowCount.intValue();
	//			}
	//			return 0;
	//		} finally {
	//			if (session != null) { session.close(); }
	//		}
	//	}

	public int rowCount(Boolean menuItem, MenuGroup menuGroup, String itemName, Object selectedType, boolean variant) {
		return rowCount(menuItem, menuGroup, itemName, selectedType, variant, null);
	}

	public int rowCount(Boolean menuItem, MenuGroup menuGroup, String itemName, Object selectedType, boolean variant, Boolean pizzaType) {
		Session session = null;
		Criteria criteria = null;
		try {
			session = createNewSession();
			criteria = session.createCriteria(MenuItem.class);
			criteria.add(Restrictions.or(Restrictions.isNull(AppConstants.PROP_DELETED), Restrictions.eq(AppConstants.PROP_DELETED, Boolean.FALSE)));
			criteria.setProjection(Projections.rowCount());
			if (!menuItem)
				criteria.add(Restrictions.eq(MenuItem.PROP_INVENTORY_ITEM, Boolean.TRUE));

			if (pizzaType != null && pizzaType)
				criteria.add(Restrictions.eq(MenuItem.PROP_PIZZA_TYPE, pizzaType));

			if (!variant)
				criteria.add(Restrictions.or(Restrictions.isNull(MenuItem.PROP_VARIANT), Restrictions.eq(MenuItem.PROP_VARIANT, Boolean.FALSE)));
			else
				criteria.add(Restrictions.eq(MenuItem.PROP_VARIANT, Boolean.TRUE));

			if (menuGroup != null) {
				criteria.add(Restrictions.eq(MenuItem.PROP_MENU_GROUP_ID, menuGroup.getId()));
			}

			if (StringUtils.isNotEmpty(itemName)) {
				criteria.add(Restrictions.ilike(MenuItem.PROP_NAME, itemName.trim(), MatchMode.ANYWHERE));
			}
			addOrderTypeFilter(selectedType, session, criteria);
			Number rowCount = (Number) criteria.uniqueResult();
			if (rowCount != null) {
				return rowCount.intValue();
			}
			return 0;
		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

	public int rowCount(Boolean menuItem, MenuGroup menuGroup, String itemName) {
		Session session = null;
		Criteria criteria = null;
		try {
			session = createNewSession();
			criteria = session.createCriteria(MenuItem.class);
			criteria.add(Restrictions.or(Restrictions.isNull(AppConstants.PROP_DELETED), Restrictions.eq(AppConstants.PROP_DELETED, Boolean.FALSE)));
			criteria.setProjection(Projections.rowCount());
			if (!menuItem)
				criteria.add(Restrictions.eq(MenuItem.PROP_INVENTORY_ITEM, Boolean.TRUE));

			if (menuGroup != null) {
				criteria.add(Restrictions.eq(MenuItem.PROP_MENU_GROUP_ID, menuGroup.getId()));
			}

			if (StringUtils.isNotEmpty(itemName)) {
				criteria.add(Restrictions.ilike(MenuItem.PROP_NAME, itemName.trim(), MatchMode.ANYWHERE));
			}
			Number rowCount = (Number) criteria.uniqueResult();
			if (rowCount != null) {
				return rowCount.intValue();
			}
			return 0;
		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

	public int rowCount(MenuGroup menuGroup, String itemName, boolean variant, Boolean pizzaType) {
		Session session = null;
		Criteria criteria = null;
		try {
			session = createNewSession();
			criteria = session.createCriteria(MenuItem.class);
			criteria.setProjection(Projections.rowCount());
			criteria.add(Restrictions.or(Restrictions.isNull(AppConstants.PROP_DELETED), Restrictions.eq(AppConstants.PROP_DELETED, Boolean.FALSE)));

			if (pizzaType != null && pizzaType)
				criteria.add(Restrictions.eq(MenuItem.PROP_PIZZA_TYPE, pizzaType));

			if (!variant)
				criteria.add(Restrictions.or(Restrictions.isNull(MenuItem.PROP_VARIANT), Restrictions.eq(MenuItem.PROP_VARIANT, Boolean.FALSE)));
			else
				criteria.add(Restrictions.eq(MenuItem.PROP_VARIANT, Boolean.TRUE));

			if (menuGroup != null) {
				criteria.add(Restrictions.eq(MenuItem.PROP_MENU_GROUP_ID, menuGroup.getId()));
			}

			if (StringUtils.isNotEmpty(itemName)) {
				criteria.add(Restrictions.ilike(MenuItem.PROP_NAME, itemName.trim(), MatchMode.ANYWHERE));
			}
			Number rowCount = (Number) criteria.uniqueResult();
			if (rowCount != null) {
				return rowCount.intValue();
			}
			return 0;
		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

	public MenuItem loadInitialized(String menuItemId) throws HibernateException {
		try (Session session = createNewSession()) {
			MenuItem menuItem = super.get(menuItemId, session);
			initialize(menuItem, session);

			return menuItem;
		}
	}

	public MenuItem getInitialized(String menuItemId) {
		if (org.apache.commons.lang.StringUtils.isEmpty(menuItemId))
			return null;

		Session session = null;

		try {
			session = createNewSession();
			MenuItem menuItem = get(menuItemId, session);
			initialize(menuItem, session);
			return menuItem;
		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

	public void initialize(MenuItem menuItem) {
		if (menuItem == null || StringUtils.isBlank(menuItem.getId()))
			return;

		Session session = null;

		try {
			session = createNewSession();
			session.refresh(menuItem);
			initialize(menuItem, session);
		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

	public void deleteVariants(MenuItem menuItem, Session session) {
		if (menuItem == null) {
			return;
		}
		this.deleteVariants(menuItem.getVariants(), session);
	}

	public void deleteVariants(List<MenuItem> variants, Session session) {
		if (variants == null || variants.isEmpty()) {
			return;
		}
		for (MenuItem variant : variants) {
			this.delete(variant, session);
		}
	}

	public void initializeVariants(MenuItem menuItem) {
		if (menuItem == null || menuItem.getId() == null)
			return;

		//		if (Hibernate.isInitialized(menuItem.getVariants()) && Hibernate.isInitialized(menuItem.getAttributes())) {
		//			return;
		//		}

		Session session = null;

		try {
			session = createNewSession();
			session.update(menuItem);
			Hibernate.initialize(menuItem.getVariants());
			Hibernate.initialize(menuItem.getAttributes());
		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

	public void initialize(MenuItem menuItem, Session session) {
		initialize(menuItem, session, true);
	}

	public void initialize(MenuItem menuItem, Session session, boolean initializeVariants) {
		if (menuItem == null)
			return;

		Hibernate.initialize(menuItem.getStockUnits());
		Hibernate.initialize(menuItem.getMenuItemModiferSpecs());
		Hibernate.initialize(menuItem.getPizzaPriceList());
		Hibernate.initialize(menuItem.getDiscounts());
		Hibernate.initialize(menuItem.getComboGroups());
		Hibernate.initialize(menuItem.getComboItems());
		Hibernate.initialize(menuItem.getAttributes());
		if (initializeVariants) {
			Hibernate.initialize(menuItem.getVariants());
			List<MenuItem> variants = menuItem.getVariants();
			if (variants != null) {
				for (MenuItem menuItem2 : variants) {
					Hibernate.initialize(menuItem2.getAttributes());
				}
			}
		}
		List<MenuItemModifierSpec> modiferSpecs = menuItem.getMenuItemModiferSpecs();
		if (modiferSpecs != null) {
			for (MenuItemModifierSpec modifierSpec : modiferSpecs) {
				//Hibernate.initialize(modifierSpec.getDefaultModifierList());
				//Hibernate.initialize(modifierSpec.getModifierPages());

				if (modifierSpec.isUseModifierGroupSettings()) {
					modifierSpec.copyModifierGroupProperties();
				}

				Set<MenuItemModifierPage> modifierPages = modifierSpec.getModifierPages();
				for (MenuItemModifierPage modifierPage : modifierPages) {
					modifierPage.setModifierSpec(modifierSpec);
				}
			}
		}
	}

	public List<MenuItem> findByParent(Terminal terminal, MenuGroup menuGroup, boolean includeInvisibleItems) throws PosException {
		Session session = null;

		try {
			session = getSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.or(Restrictions.isNull(AppConstants.PROP_DELETED), Restrictions.eq(AppConstants.PROP_DELETED, Boolean.FALSE)));

			if (menuGroup != null) {
				criteria.add(Restrictions.eq(MenuItem.PROP_MENU_GROUP_ID, menuGroup.getId()));
			}

			criteria.addOrder(Order.asc(MenuItem.PROP_SORT_ORDER));
			criteria.addOrder(Order.asc(MenuItem.PROP_NAME));

			if (!includeInvisibleItems) {
				criteria.add(Restrictions.eq(MenuItem.PROP_VISIBLE, Boolean.TRUE));
			}

			return criteria.list();
		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

	public List<MenuItem> getVariants(MenuItem parentItem, Session session) throws PosException {
		Criteria criteria = session.createCriteria(getReferenceClass());
		criteria.add(Restrictions.or(Restrictions.isNull(AppConstants.PROP_DELETED), Restrictions.eq(AppConstants.PROP_DELETED, Boolean.FALSE)));
		if (parentItem != null) {
			criteria.add(Restrictions.eq(MenuItem.PROP_PARENT_MENU_ITEM_ID, parentItem.getId()));
		}
		criteria.add(Restrictions.eq(MenuItem.PROP_VISIBLE, Boolean.TRUE));
		criteria.addOrder(Order.asc(MenuItem.PROP_SORT_ORDER));
		criteria.addOrder(Order.asc(MenuItem.PROP_NAME));

		return criteria.list();
	}

	public List<MenuItem> getVariants(MenuItem parentItem) throws PosException {
		Session session = null;

		try {
			session = createNewSession();
			return getVariants(parentItem, session);
		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

	public int getRowCount(Terminal terminal, MenuGroup menuGroup, Object selectedOrderType) {
		Session session = null;
		Criteria criteria = null;
		try {
			session = createNewSession();
			criteria = session.createCriteria(MenuItem.class);
			criteria.add(Restrictions.or(Restrictions.isNull(AppConstants.PROP_DELETED), Restrictions.eq(AppConstants.PROP_DELETED, Boolean.FALSE)));
			// if (menuGroup != null) {
			// criteria.add(Restrictions.eq(MenuItem.PROP_MENU_GROUP_ID, menuGroup.getId()));
			// }

			addOrderTypeFilter(selectedOrderType, session, criteria);

			criteria.setProjection(Projections.rowCount());
			Number rowCount = (Number) criteria.uniqueResult();
			if (rowCount != null) {
				return rowCount.intValue();

			}
		} finally {
			if (session != null) {
				session.close();
			}
		}
		return 0;
	}

	public void loadItems(Terminal terminal, MenuGroup menuGroup, Object selectedOrderType, boolean includeInvisibleItems, PaginatedListModel listModel,
			boolean variant) {
		Session session = null;
		Criteria criteria = null;
		try {
			session = createNewSession();
			criteria = session.createCriteria(MenuItem.class);
			criteria.add(Restrictions.or(Restrictions.isNull(AppConstants.PROP_DELETED), Restrictions.eq(AppConstants.PROP_DELETED, Boolean.FALSE)));
			// if (menuGroup != null) {
			// criteria.add(Restrictions.eq(MenuItem.PROP_MENU_GROUP_ID, menuGroup.getId()));
			// }
			criteria.addOrder(Order.asc(MenuItem.PROP_SORT_ORDER));
			criteria.addOrder(Order.asc(MenuItem.PROP_NAME));

			if (!variant)
				criteria.add(Restrictions.or(Restrictions.isNull(MenuItem.PROP_VARIANT), Restrictions.eq(MenuItem.PROP_VARIANT, Boolean.FALSE)));
			else
				criteria.add(Restrictions.eq(MenuItem.PROP_VARIANT, Boolean.TRUE));

			if (!includeInvisibleItems) {
				criteria.add(Restrictions.eq(MenuItem.PROP_VISIBLE, Boolean.TRUE));
			}

			addOrderTypeFilter(selectedOrderType, session, criteria);
			criteria.setFirstResult(listModel.getCurrentRowIndex());
			criteria.setMaxResults(listModel.getPageSize());
			listModel.setData(criteria.list());
		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

	public List<MenuItemModifierSpec> findModifierGroups(MenuItem item) throws PosException {
		Session session = null;

		try {
			session = getSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.eq(MenuItem.PROP_ID, item.getId()));
			criteria.add(Restrictions.or(Restrictions.isNull(AppConstants.PROP_DELETED), Restrictions.eq(AppConstants.PROP_DELETED, Boolean.FALSE)));
			MenuItem newItem = (MenuItem) criteria.uniqueResult();
			Hibernate.initialize(newItem.getMenuItemModiferSpecs());

			return newItem.getMenuItemModiferSpecs();
		} catch (Exception e) {
			throw new PosException(""); //$NON-NLS-1$
		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

	public List<MenuItem> getMenuItems(String itemName, MenuGroup menuGroup, Object selectedType, String type) {
		Session session = null;
		Criteria criteria = null;
		try {
			session = getSession();
			criteria = session.createCriteria(MenuItem.class);
			criteria.add(Restrictions.or(Restrictions.isNull(AppConstants.PROP_DELETED), Restrictions.eq(AppConstants.PROP_DELETED, Boolean.FALSE)));
			if (StringUtils.isNotEmpty(itemName)) {
				criteria.add(Restrictions.ilike(MenuItem.PROP_NAME, itemName.trim(), MatchMode.ANYWHERE));
			}

			if (type.equals("InventoryItem")) { //$NON-NLS-1$
				criteria.add(Restrictions.eq(MenuItem.PROP_INVENTORY_ITEM, true));
			}

			addOrderTypeFilter(selectedType, session, criteria);
			return criteria.list();

		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

	public List<MenuItem> getPizzaItems(String nameOrBarcodeOrSku, MenuGroup menuGroup, Object selectedType) {
		Session session = null;
		Criteria criteria = null;
		try {
			session = createNewSession();
			criteria = session.createCriteria(MenuItem.class);
			criteria.add(Restrictions.or(Restrictions.isNull(AppConstants.PROP_DELETED), Restrictions.eq(AppConstants.PROP_DELETED, Boolean.FALSE)));
			criteria.add(Restrictions.eq(MenuItem.PROP_PIZZA_TYPE, true));

			if (StringUtils.isNotBlank(nameOrBarcodeOrSku)) {
				nameOrBarcodeOrSku = nameOrBarcodeOrSku.trim();
				Disjunction disjunction = Restrictions.disjunction();
				disjunction.add(Restrictions.eq(MenuItem.PROP_BARCODE, nameOrBarcodeOrSku));
				disjunction.add(Restrictions.eq(MenuItem.PROP_SKU, nameOrBarcodeOrSku));
				disjunction.add(Restrictions.ilike(MenuItem.PROP_NAME, nameOrBarcodeOrSku, MatchMode.ANYWHERE));
				criteria.add(disjunction);
			}
			addOrderTypeFilter(selectedType, session, criteria);
			return criteria.list();
		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

	public void releaseParent(List<MenuItem> menuItemList) {
		//		if (menuItemList == null) {
		//			return;
		//		}
		//
		//		Session session = null;
		//		Transaction tx = null;
		//
		//		try {
		//			session = createNewSession();
		//			tx = session.beginTransaction();
		//
		//			for (MenuItem menuItem : menuItemList) {
		//				menuItem.setParent(null);
		//				session.saveOrUpdate(menuItem);
		//			}
		//
		//			tx.commit();
		//		} catch (Exception e) {
		//			tx.rollback();
		//			LogFactory.getLog(ShopTableDAO.class).error(e);
		//			throw new RuntimeException(e);
		//		} finally {
		//			if (session != null) { session.close(); }
		//		}
	}

	public void releaseParentAndDelete(MenuItem item) {
		if (item == null) {
			return;
		}

		Session session = null;
		Transaction tx = null;

		try {
			session = createNewSession();
			tx = session.beginTransaction();

			String queryStringForDiscount = "delete from MENUITEM_DISCOUNT where MENUITEM_ID='%s'"; //$NON-NLS-1$
			queryStringForDiscount = String.format(queryStringForDiscount, item.getId());
			Query query = session.createSQLQuery(queryStringForDiscount);
			query.executeUpdate();

			String queryString = "delete from MENU_PAGE_ITEM where MENU_ITEM_ID='%s'"; //$NON-NLS-1$
			queryString = String.format(queryString, item.getId());
			Query queryForPageItem = session.createSQLQuery(queryString);
			queryForPageItem.executeUpdate();

			String queryStringReleaseRecipe = "delete from RECIPE_TABLE where MENU_ITEM_ID='%s'"; //$NON-NLS-1$
			queryStringReleaseRecipe = String.format(queryStringReleaseRecipe, item.getId());
			Query queryReleaseRecipe = session.createSQLQuery(queryStringReleaseRecipe);
			queryReleaseRecipe.executeUpdate();

			session.delete(item);

			tx.commit();
		} catch (Exception e) {
			tx.rollback();
			throw e;
		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

	public MenuItem getMenuItemByBarcodeOrSKU(String barcode) {
		MenuItem menuItem = getMenuItemByBarcode(barcode);
		if (menuItem == null) {
			menuItem = getMenuItemBySKU(barcode);
		}
		return menuItem;
	}

	public MenuItem getMenuItemByBarcodeOrSKU(OrderType orderType, String barcode) {
		List<String> catIdList = MenuCategoryDAO.getInstance().findActiveCategoryIds(orderType);

		try (Session session = MenuItemDAO.getInstance().createNewSession()) {
			MenuItem menuItem = getMenuItemByBarcode(orderType, barcode, catIdList, session);
			if (menuItem == null) {
				menuItem = getMenuItemBySKU(orderType, barcode, catIdList, session);
			}
			return menuItem;
		}
	}

	public MenuItem getMenuItemByBarcode(String barcode) {
		return getMenuItemByBarcode(barcode, true);
	}

	public MenuItem getMenuItemByBarcode(String barcode, boolean showVisibleItems) {
		if (StringUtils.isBlank(barcode)) {
			return null;
		}
		Session session = null;
		Criteria criteria = null;
		try {
			session = createNewSession();
			criteria = session.createCriteria(MenuItem.class);
			criteria.add(Restrictions.or(Restrictions.isNull(AppConstants.PROP_DELETED), Restrictions.eq(AppConstants.PROP_DELETED, Boolean.FALSE)));

			if (showVisibleItems) {
				criteria.add(Restrictions.eq(MenuItem.PROP_VISIBLE, showVisibleItems));
			}
			criteria.add(Restrictions.like(MenuItem.PROP_BARCODE, barcode));
			criteria.add(Restrictions.eq(MenuItem.PROP_HAS_VARIANT, Boolean.FALSE));

			List<MenuItem> result = criteria.list();
			if (result == null || result.isEmpty()) {
				return null;
			}
			return (MenuItem) result.get(0);
		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

	public MenuItem getMenuItemByBarcode(OrderType orderType, String barcode, List<String> catIdList, Session session) {

		Criteria criteria = session.createCriteria(MenuItem.class);
		criteria.add(Restrictions.or(Restrictions.isNull(AppConstants.PROP_DELETED), Restrictions.eq(AppConstants.PROP_DELETED, Boolean.FALSE)));

		criteria.add(Restrictions.eq(MenuItem.PROP_VISIBLE, true));
		criteria.add(Restrictions.eq(MenuItem.PROP_BARCODE, barcode));
		criteria.add(Restrictions.eq(MenuItem.PROP_HAS_VARIANT, Boolean.FALSE));

		Criterion catIdExp = Restrictions.or(Restrictions.in(MenuItem.PROP_MENU_CATEGORY_ID, catIdList), Restrictions.isNull(MenuItem.PROP_MENU_CATEGORY_ID));
		if (catIdList == null || catIdList.isEmpty()) {
			catIdExp = Restrictions.isNull(MenuItem.PROP_MENU_CATEGORY_ID);
		}
		criteria.add(catIdExp);

		List<MenuItem> result = criteria.list();
		if (result == null || result.isEmpty()) {
			return null;
		}
		return (MenuItem) result.get(0);
	}

	public MenuItem getMenuItemBySKU(String skuNo) {
		if (StringUtils.isBlank(skuNo)) {
			return null;
		}
		Session session = null;
		try {
			session = createNewSession();
			return getMenuItemBySKU(skuNo, session, false, true);
		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

	public MenuItem getMenuItemBySKU(String skuNo, Session session, boolean includeInactiveItems) {
		return getMenuItemBySKU(skuNo, session, includeInactiveItems, false);
	}

	public MenuItem getMenuItemBySKU(String skuNo, Session session, boolean includeInactiveItems, boolean filterByHasVariant) {
		Criteria criteria = session.createCriteria(MenuItem.class);
		criteria.add(Restrictions.or(Restrictions.isNull(AppConstants.PROP_DELETED), Restrictions.eq(AppConstants.PROP_DELETED, Boolean.FALSE)));

		if (!includeInactiveItems) {
			criteria.add(Restrictions.eq(MenuItem.PROP_VISIBLE, Boolean.TRUE));
		}
		criteria.add(Restrictions.like(MenuItem.PROP_SKU, skuNo));
		if (filterByHasVariant) {
			criteria.add(Restrictions.eq(MenuItem.PROP_HAS_VARIANT, Boolean.FALSE));
		}

		List<MenuItem> result = criteria.list();
		if (result == null || result.isEmpty()) {
			return null;
		}
		return (MenuItem) result.get(0);
	}

	public MenuItem getMenuItemBySKU(OrderType orderType, String skuNo, List<String> catIdList, Session session) {
		Criteria criteria = session.createCriteria(MenuItem.class);
		criteria.add(Restrictions.or(Restrictions.isNull(AppConstants.PROP_DELETED), Restrictions.eq(AppConstants.PROP_DELETED, Boolean.FALSE)));

		criteria.add(Restrictions.eq(MenuItem.PROP_VISIBLE, Boolean.TRUE));
		criteria.add(Restrictions.eq(MenuItem.PROP_SKU, skuNo));
		criteria.add(Restrictions.eq(MenuItem.PROP_HAS_VARIANT, Boolean.FALSE));

		Criterion catIdExp = Restrictions.or(Restrictions.in(MenuItem.PROP_MENU_CATEGORY_ID, catIdList), Restrictions.isNull(MenuItem.PROP_MENU_CATEGORY_ID));
		if (catIdList == null || catIdList.isEmpty()) {
			catIdExp = Restrictions.isNull(MenuItem.PROP_MENU_CATEGORY_ID);
		}
		criteria.add(catIdExp);

		List<MenuItem> result = criteria.list();
		if (result == null || result.isEmpty()) {
			return null;
		}
		return (MenuItem) result.get(0);
	}

	public MenuItem getMenuItemByName(String itemName, Session session) {
		Criteria criteria = session.createCriteria(MenuItem.class);
		criteria.add(Restrictions.or(Restrictions.isNull(AppConstants.PROP_DELETED), Restrictions.eq(AppConstants.PROP_DELETED, Boolean.FALSE)));

		criteria.add(Restrictions.eq(MenuItem.PROP_VISIBLE, true));
		criteria.add(Restrictions.ilike(MenuItem.PROP_NAME, itemName, MatchMode.EXACT));
		criteria.setMaxResults(1);
		return (MenuItem) criteria.uniqueResult();
	}

	public List<MenuItem> getMenuItemByName(String itemName) {
		return getMenuItemByName(itemName, true);
	}

	public List<MenuItem> getMenuItemByName(String itemName, boolean showVisibleItems) {
		Session session = null;
		Criteria criteria = null;
		try {
			session = createNewSession();
			criteria = session.createCriteria(MenuItem.class);
			criteria.add(Restrictions.or(Restrictions.isNull(AppConstants.PROP_DELETED), Restrictions.eq(AppConstants.PROP_DELETED, Boolean.FALSE)));

			if (showVisibleItems) {
				criteria.add(Restrictions.eq(MenuItem.PROP_VISIBLE, showVisibleItems));
			}
			criteria.add(Restrictions.ilike(MenuItem.PROP_NAME, itemName, MatchMode.ANYWHERE));
			List<MenuItem> result = criteria.list();
			if (result == null || result.isEmpty()) {
				return null;
			}
			return result;
		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

	public List<MenuItem> getMenuItemsByNameOrBarcode(MenuGroup menuGroup, String searchString) {
		return getMenuItemsByNameOrBarcode(menuGroup, searchString, true, false);
	}

	public List<MenuItem> getMenuItemsByNameOrBarcode(MenuGroup menuGroup, String searchString, boolean hideInvisibleItems, boolean includeVariantParent) {
		Session session = null;
		Criteria criteria = null;
		try {
			session = createNewSession();
			criteria = session.createCriteria(MenuItem.class);
			criteria.add(Restrictions.or(Restrictions.isNull(AppConstants.PROP_DELETED), Restrictions.eq(AppConstants.PROP_DELETED, Boolean.FALSE)));
			if (hideInvisibleItems) {
				criteria.add(Restrictions.eq(MenuItem.PROP_VISIBLE, Boolean.TRUE));
			}
			if (!includeVariantParent) {
				criteria.add(Restrictions.eq(MenuItem.PROP_HAS_VARIANT, Boolean.FALSE));
			}
			criteria.add(Restrictions.or(Restrictions.eq(MenuItem.PROP_BARCODE, searchString),
					Restrictions.ilike(MenuItem.PROP_NAME, searchString, MatchMode.ANYWHERE)));
			if (menuGroup != null) {
				criteria.add(Restrictions.eq(MenuItem.PROP_MENU_GROUP_ID, menuGroup.getId()));
			}
			criteria.addOrder(Order.asc(MenuItem.PROP_NAME).ignoreCase());
			List<MenuItem> result = criteria.list();
			if (result == null || result.isEmpty()) {
				return null;
			}
			return result;
		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

	public List<MenuItem> getMenuItemsByNameOrBarcode(MenuCategory menuCategory, String searchString, boolean hideInvisibleItems,
			boolean includeVariantParent) {
		Session session = null;
		Criteria criteria = null;
		try {
			session = createNewSession();
			criteria = session.createCriteria(MenuItem.class);
			criteria.add(Restrictions.or(Restrictions.isNull(AppConstants.PROP_DELETED), Restrictions.eq(AppConstants.PROP_DELETED, Boolean.FALSE)));
			if (hideInvisibleItems) {
				criteria.add(Restrictions.eq(MenuItem.PROP_VISIBLE, Boolean.TRUE));
			}
			criteria.add(Restrictions.or(Restrictions.eq(MenuItem.PROP_BARCODE, searchString),
					Restrictions.ilike(MenuItem.PROP_NAME, searchString, MatchMode.ANYWHERE)));

			if (!includeVariantParent) {
				criteria.add(Restrictions.eq(MenuItem.PROP_HAS_VARIANT, Boolean.FALSE));
			}

			if (menuCategory != null) {
				criteria.add(Restrictions.eq(MenuItem.PROP_MENU_CATEGORY_ID, menuCategory.getId()));
			}
			criteria.addOrder(Order.asc(MenuItem.PROP_NAME).ignoreCase());
			List<MenuItem> result = criteria.list();
			if (result == null || result.isEmpty()) {
				return null;
			}
			return result;
		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

	public List<MenuItem> getMenuItemsByNameOrBarcode(OrderType orderType, MenuCategory menuCategory, MenuGroup menuGroup, String itemName,
			boolean hideInvisibleItems) {
		return getMenuItemsByNameOrBarcode(orderType, menuCategory, menuGroup, itemName, hideInvisibleItems, false);
	}

	/**
	 * 
	 * @param orderType
	 * @param menuCategory
	 * @param menuGroup
	 * @param searchString
	 * @param hideInvisibleItems If true, item with property visible=false will be filtered out
	 * @param includeVariantParent If true
	 * @return
	 */
	public List<MenuItem> getMenuItemsByNameOrBarcode(OrderType orderType, MenuCategory menuCategory, MenuGroup menuGroup, String searchString,
			boolean hideInvisibleItems, boolean includeVariantParent) {
		return getMenuItemsByNameOrBarcode(orderType, menuCategory, menuGroup, searchString, hideInvisibleItems, includeVariantParent, false);

	}

	/**
	 * 
	 * @param orderType
	 * @param menuCategory
	 * @param menuGroup
	 * @param searchString
	 * @param hideInvisibleItems If true, item with property visible=false will be filtered out
	 * @param includeVariantParent If true
	 * @param hideRawMaterial default false
	 * @return
	 */
	public List<MenuItem> getMenuItemsByNameOrBarcode(OrderType orderType, MenuCategory menuCategory, MenuGroup menuGroup, String searchString,
			boolean hideInvisibleItems, boolean includeVariantParent, boolean hideRawMaterial) {
		if (menuGroup != null) {
			return getMenuItemsByNameOrBarcode(menuGroup, searchString, hideInvisibleItems, includeVariantParent);
		}
		if (menuCategory != null) {
			return getMenuItemsByNameOrBarcode(menuCategory, searchString, hideInvisibleItems, includeVariantParent);
		}
		List<String> catIdList = MenuCategoryDAO.getInstance().findActiveCategoryIds(orderType);

		Session session = null;
		Criteria criteria = null;
		try {
			session = createNewSession();
			criteria = session.createCriteria(MenuItem.class);
			criteria.add(Restrictions.or(Restrictions.isNull(AppConstants.PROP_DELETED), Restrictions.eq(AppConstants.PROP_DELETED, Boolean.FALSE)));
			if (hideInvisibleItems) {
				criteria.add(Restrictions.eq(MenuItem.PROP_VISIBLE, Boolean.TRUE));
			}

			if (hideRawMaterial) {
				criteria.add(Restrictions.eq(MenuItem.PROP_RAW_MATERIAL, false));
			}

			if (!includeVariantParent) {
				criteria.add(Restrictions.eq(MenuItem.PROP_HAS_VARIANT, Boolean.FALSE));
			}
			criteria.add(Restrictions.or(Restrictions.eq(MenuItem.PROP_BARCODE, searchString),
					Restrictions.ilike(MenuItem.PROP_NAME, searchString, MatchMode.ANYWHERE)));

			Disjunction disjunction = Restrictions.disjunction();
			if (catIdList != null && !catIdList.isEmpty()) {
				disjunction.add(Restrictions.in(MenuItem.PROP_MENU_CATEGORY_ID, catIdList));
			}
			disjunction.add(Restrictions.isNull(MenuItem.PROP_MENU_CATEGORY_ID));
			disjunction.add(Restrictions.eq(MenuItem.PROP_MENU_CATEGORY_ID, "")); //$NON-NLS-1$
			criteria.add(disjunction);

			criteria.addOrder(Order.asc(MenuItem.PROP_NAME).ignoreCase());

			List<MenuItem> result = criteria.list();
			if (result == null || result.isEmpty()) {
				return null;
			}
			return result;
		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

	public List<MenuItem> getMenuItems(String sku, String name, String barcode, ReportGroup reportGroup) {
		Session session = null;
		Criteria criteria = null;
		try {
			session = createNewSession();
			criteria = session.createCriteria(MenuItem.class);
			criteria.add(Restrictions.or(Restrictions.isNull(AppConstants.PROP_DELETED), Restrictions.eq(AppConstants.PROP_DELETED, Boolean.FALSE)));
			if (StringUtils.isNotEmpty(barcode))
				criteria.add(Restrictions.eq(MenuItem.PROP_BARCODE, barcode));
			if (StringUtils.isNotEmpty(sku))
				criteria.add(Restrictions.eq(MenuItem.PROP_SKU, sku));
			if (StringUtils.isNotEmpty(name)) {
				criteria.add(Restrictions.ilike(MenuItem.PROP_NAME, name, MatchMode.START));
			}
			if (reportGroup != null) {
				criteria.add(Restrictions.eq(MenuItem.PROP_REPORT_GROUP_ID, reportGroup.getId()));
			}

			return criteria.list();
		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

	public List<MenuItem> findMenuItemsForStockCount(String searchString, MenuGroup group, InventoryVendor vendor, InventoryLocation location) {
		try (Session session = createNewSession()) {
			Criteria criteria = null;
			List<String> commonMenuItemsId = null;
			if (vendor != null) {
				criteria = session.createCriteria(InventoryVendorItems.class);
				criteria.add(Restrictions.eq(InventoryVendorItems.PROP_VENDOR, vendor));
				criteria.createAlias(InventoryVendorItems.PROP_ITEM, "item"); //$NON-NLS-1$
				criteria.setProjection(Projections.property("item.id")); //$NON-NLS-1$
				commonMenuItemsId = criteria.list();
			}
			if (location != null) {
				criteria = session.createCriteria(InventoryStock.class);
				criteria.add(Restrictions.eq(InventoryStock.PROP_LOCATION_ID, location.getId()));
				criteria.setProjection(Projections.property(InventoryStock.PROP_MENU_ITEM_ID));
				if (commonMenuItemsId == null) {
					commonMenuItemsId = criteria.list();
				}
				else {
					commonMenuItemsId.retainAll(criteria.list());
				}
			}
			// menu item
			criteria = session.createCriteria(MenuItem.class);
			criteria.add(Restrictions.or(Restrictions.isNull(AppConstants.PROP_DELETED), Restrictions.eq(AppConstants.PROP_DELETED, Boolean.FALSE)));
			criteria.add(Restrictions.eq(MenuItem.PROP_INVENTORY_ITEM, Boolean.TRUE));
			criteria.add(Restrictions.eq(MenuItem.PROP_HAS_VARIANT, Boolean.FALSE));
			criteria.add(Restrictions.isNotNull(MenuItem.PROP_UNIT_ID));

			if (StringUtils.isNotEmpty(searchString)) {
				criteria.add(Restrictions.or(Restrictions.eq(MenuItem.PROP_BARCODE, searchString), Restrictions
						.or(Restrictions.eq(MenuItem.PROP_SKU, searchString), Restrictions.ilike(MenuItem.PROP_NAME, searchString, MatchMode.START))));
			}

			if (group != null) {
				criteria.add(Restrictions.eq(MenuItem.PROP_MENU_GROUP_ID, group.getId()));
			}

			if (commonMenuItemsId != null && !commonMenuItemsId.isEmpty()) {
				criteria.add(Restrictions.in(MenuItem.PROP_ID, commonMenuItemsId));
			}
			criteria.setMaxResults(10);
			return criteria.list();
		}
	}

	public List<MenuItem> getPizzaItems() {
		Session session = null;
		Criteria criteria = null;
		try {
			session = createNewSession();
			criteria = session.createCriteria(MenuItem.class);
			criteria.add(Restrictions.or(Restrictions.isNull(AppConstants.PROP_DELETED), Restrictions.eq(AppConstants.PROP_DELETED, Boolean.FALSE)));
			criteria.add(Restrictions.eq(MenuItem.PROP_PIZZA_TYPE, true));

			List<MenuItem> result = criteria.list();

			return result;
		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

	public List<MenuItem> getMenuItems() {
		Session session = null;
		Criteria criteria = null;
		try {
			session = createNewSession();
			criteria = session.createCriteria(MenuItem.class);
			criteria.add(Restrictions.or(Restrictions.isNull(AppConstants.PROP_DELETED), Restrictions.eq(AppConstants.PROP_DELETED, Boolean.FALSE)));
			criteria.add(Restrictions.or(Restrictions.eq(MenuItem.PROP_PIZZA_TYPE, false), Restrictions.isNull(MenuItem.PROP_PIZZA_TYPE)));

			List<MenuItem> result = criteria.list();

			return result;
		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

	public List<MenuItem> getInventortItems() {
		Session session = null;
		Criteria criteria = null;
		try {
			session = createNewSession();
			criteria = session.createCriteria(MenuItem.class);
			criteria.add(Restrictions.or(Restrictions.isNull(AppConstants.PROP_DELETED), Restrictions.eq(AppConstants.PROP_DELETED, Boolean.FALSE)));
			criteria.add(Restrictions.eq(MenuItem.PROP_INVENTORY_ITEM, true));

			List<MenuItem> result = criteria.list();

			return result;
		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

	public MenuItem getReplenishedMenuItem(Integer id, Session session) {
		Criteria criteria = null;
		/*criteria = session.createCriteria(MenuItem.class);
		criteria.add(Restrictions.eq(MenuItem.PROP_ID, id));
		criteria.createAlias(MenuItem.PROP_STOCK_STATUS, "status");
		criteria.add(Restrictions.ltProperty("status.availableUnit", MenuItem.PROP_REORDER_LEVEL));
		*/
		criteria = session.createCriteria(MenuItem.class);
		criteria.add(Restrictions.or(Restrictions.isNull(AppConstants.PROP_DELETED), Restrictions.eq(AppConstants.PROP_DELETED, Boolean.FALSE)));
		criteria.add(Restrictions.eq(MenuItem.PROP_ID, id));
		MenuItem menuItem = (MenuItem) criteria.uniqueResult();

		Double reorderLevel = 0.0;
		if (menuItem != null)
			reorderLevel = menuItem.getReorderLevel();

		if (reorderLevel == 0)
			return null;

		criteria = session.createCriteria(MenuItemInventoryStatus.class);
		criteria.add(Restrictions.eq(MenuItemInventoryStatus.PROP_ID, id));
		criteria.add(Restrictions.lt(MenuItemInventoryStatus.PROP_AVAILABLE_UNIT, reorderLevel));
		MenuItemInventoryStatus inventoryStatus = (MenuItemInventoryStatus) criteria.uniqueResult();
		if (inventoryStatus == null)
			return null;
		menuItem.setStockStatus(inventoryStatus);
		return menuItem;
	}

	public List<MenuItem> getReOrderedMenuItems() {
		Session session = null;
		Criteria criteria = null;
		try {
			session = createNewSession();
			criteria = session.createCriteria(getReferenceClass(), "item"); //$NON-NLS-1$
			criteria.add(Restrictions.or(Restrictions.isNull(AppConstants.PROP_DELETED), Restrictions.eq(AppConstants.PROP_DELETED, Boolean.FALSE)));
			criteria.addOrder(Order.asc(MenuItem.PROP_ID));
			//criteria.createAlias(MenuItem.PROP_STOCK_STATUS, "status");
			//criteria.add(Restrictions.or(Restrictions.isNull("status.unitOnHand"), Restrictions.leProperty("status.unitOnHand", MenuItem.PROP_REORDER_LEVEL)));
			criteria.add(Subqueries.exists(getDetachedCriteriaForStockStatus("item.id"))); //$NON-NLS-1$
			criteria.add(Restrictions.eq(MenuItem.PROP_INVENTORY_ITEM, true));
			List list = criteria.list();
			return list;
		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

	private DetachedCriteria getDetachedCriteriaForStockStatus(String propItemId) {
		DetachedCriteria stockStatusSubCriteria = DetachedCriteria.forClass(MenuItemInventoryStatus.class, "status") //$NON-NLS-1$
				.setProjection(Projections.property(MenuItemInventoryStatus.PROP_MENU_ITEM_ID))
				.add(Restrictions.eqProperty(MenuItemInventoryStatus.PROP_MENU_ITEM_ID, propItemId))
				.add(Restrictions.or(Restrictions.isNull("status." + MenuItemInventoryStatus.PROP_UNIT_ON_HAND), //$NON-NLS-1$
						Restrictions.leProperty("status." + MenuItemInventoryStatus.PROP_UNIT_ON_HAND, "item." + MenuItem.PROP_REORDER_LEVEL))); //$NON-NLS-1$ //$NON-NLS-2$
		return stockStatusSubCriteria;
	}

	@Deprecated
	public int rowReOrderedItemCount(MenuGroup menuGroup, String nameOrBarcodeOrSku, boolean variant, Boolean pizzaType) {
		Session session = null;
		Criteria criteria = null;
		try {
			session = createNewSession();
			criteria = session.createCriteria(MenuItem.class, "item"); //$NON-NLS-1$
			criteria.add(Restrictions.or(Restrictions.isNull(AppConstants.PROP_DELETED), Restrictions.eq(AppConstants.PROP_DELETED, Boolean.FALSE)));
			criteria.setProjection(Projections.rowCount());
			//criteria.createAlias(MenuItem.PROP_STOCK_STATUS, "status");
			//criteria.add(Restrictions.leProperty("status.unitOnHand", MenuItem.PROP_REORDER_LEVEL));
			criteria.add(Subqueries.exists(getDetachedCriteriaForStockStatus("item.id"))); //$NON-NLS-1$
			criteria.add(Restrictions.eq(MenuItem.PROP_INVENTORY_ITEM, true));

			if (pizzaType != null && pizzaType)
				criteria.add(Restrictions.eq(MenuItem.PROP_PIZZA_TYPE, pizzaType));

			if (!variant)
				criteria.add(Restrictions.or(Restrictions.isNull(MenuItem.PROP_VARIANT), Restrictions.eq(MenuItem.PROP_VARIANT, Boolean.FALSE)));
			else
				criteria.add(Restrictions.eq(MenuItem.PROP_VARIANT, Boolean.TRUE));

			if (menuGroup != null) {
				criteria.add(Restrictions.eq(MenuItem.PROP_MENU_GROUP_ID, menuGroup.getId()));
			}

			if (StringUtils.isNotEmpty(nameOrBarcodeOrSku)) {
				Disjunction disjunction = Restrictions.disjunction();
				disjunction.add(Restrictions.eq(MenuItem.PROP_BARCODE, nameOrBarcodeOrSku));
				disjunction.add(Restrictions.eq(MenuItem.PROP_SKU, nameOrBarcodeOrSku));
				disjunction.add(Restrictions.ilike(MenuItem.PROP_NAME, nameOrBarcodeOrSku.trim(), MatchMode.ANYWHERE));
				criteria.add(disjunction);
			}

			Number rowCount = (Number) criteria.uniqueResult();
			if (rowCount != null) {
				return rowCount.intValue();
			}
			return 0;
		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

	public void loadReorderedMenuItems(PaginationSupport model, MenuGroup menuGroup, String nameOrSku, boolean variant, Boolean pizzaType) {
		Session session = null;
		Criteria criteria = null;
		try {
			session = createNewSession();
			criteria = session.createCriteria(MenuItem.class, "item"); //$NON-NLS-1$
			criteria.add(Restrictions.or(Restrictions.isNull(AppConstants.PROP_DELETED), Restrictions.eq(AppConstants.PROP_DELETED, Boolean.FALSE)));
			//criteria.createAlias(MenuItem.PROP_STOCK_STATUS, "status");
			//criteria.add(Restrictions.leProperty("status.unitOnHand", MenuItem.PROP_REORDER_LEVEL));
			criteria.add(Subqueries.exists(getDetachedCriteriaForStockStatus("item.id"))); //$NON-NLS-1$
			criteria.add(Restrictions.eq(MenuItem.PROP_INVENTORY_ITEM, true));

			if (pizzaType != null && pizzaType) {
				criteria.add(Restrictions.eq(MenuItem.PROP_PIZZA_TYPE, pizzaType));
			}

			if (!variant)
				criteria.add(Restrictions.or(Restrictions.isNull(MenuItem.PROP_VARIANT), Restrictions.eq(MenuItem.PROP_VARIANT, Boolean.FALSE)));
			else
				criteria.add(Restrictions.eq(MenuItem.PROP_VARIANT, Boolean.TRUE));

			if (StringUtils.isNotEmpty(nameOrSku)) {
				Disjunction disjunction = Restrictions.disjunction();
				disjunction.add(Restrictions.eq(MenuItem.PROP_SKU, nameOrSku));
				disjunction.add(Restrictions.ilike(MenuItem.PROP_NAME, nameOrSku.trim(), MatchMode.ANYWHERE));
				criteria.add(disjunction);
			}
			if (menuGroup != null) {
				criteria.add(Restrictions.eq(MenuItem.PROP_MENU_GROUP_ID, menuGroup.getId()));
			}
			model.setNumRows(super.rowCount(criteria));
			criteria.addOrder(Order.asc(MenuItem.PROP_ID));
			criteria.setFirstResult(model.getCurrentRowIndex());
			criteria.setMaxResults(model.getPageSize());
			List<MenuItem> result = criteria.list();
			model.setRows(result);
		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

	public List<MenuItem> getMenuItems(String sku, String name, String barcode, MenuGroup menuGroup, Pagination pagination) {
		return this.getMenuItems(sku, name, barcode, menuGroup, pagination, null);
	}

	public List<MenuItem> getMenuItems(String sku, String name, String barcode, MenuGroup menuGroup, Pagination pagination, Boolean filterParentMenuItem) {
		return getMenuItems(sku, name, barcode, menuGroup, null, pagination, filterParentMenuItem);
	}

	public List<MenuItem> getMenuItems(String sku, String name, String barcode, MenuGroup menuGroup, MenuCategory menuCategory, Pagination pagination,
			Boolean filterParentMenuItem) {

		Session session = null;
		Criteria criteria = null;
		try {
			session = this.createNewSession();
			criteria = session.createCriteria(this.getReferenceClass());
			this.updateCriteria(criteria, sku, name, barcode, menuGroup, menuCategory, pagination, filterParentMenuItem);
			List list = criteria.list();
			if (pagination != null) {
				criteria = session.createCriteria(this.getReferenceClass());
				criteria.setProjection(Projections.rowCount());
				this.updateCriteria(criteria, sku, name, barcode, menuGroup, menuCategory, null, filterParentMenuItem);
				Number uniqueResult = (Number) criteria.uniqueResult();
				pagination.setNumRows(uniqueResult.intValue());
			}

			return list;
		} finally {
			if (session != null) {
				session.close();
			}
		}

	}

	private void updateCriteria(Criteria criteria, String sku, String name, String barcode, MenuGroup menuGroup, MenuCategory menuCategory,
			Pagination pagination, Boolean filterParentMenuItem) {
		criteria.add(Restrictions.or(Restrictions.isNull(MenuItem.PROP_DELETED), Restrictions.eq(MenuItem.PROP_DELETED, Boolean.FALSE)));
		if (StringUtils.isNotEmpty(barcode)) {
			criteria.add(Restrictions.eq(MenuItem.PROP_BARCODE, barcode));
		}
		if (StringUtils.isNotEmpty(sku)) {
			criteria.add(Restrictions.eq(MenuItem.PROP_SKU, sku));
		}
		if (StringUtils.isNotEmpty(name)) {
			criteria.add(Restrictions.ilike(MenuItem.PROP_NAME, name, MatchMode.ANYWHERE));
		}
		if (menuGroup != null) {
			criteria.add(Restrictions.eq(MenuItem.PROP_MENU_GROUP_ID, menuGroup.getId()));
		}
		if (menuCategory != null) {
			criteria.add(Restrictions.eq(MenuItem.PROP_MENU_CATEGORY_ID, menuCategory.getId()));
		}
		if (pagination != null) {
			criteria.setFirstResult(pagination.getCurrentRowIndex());
			criteria.setMaxResults(pagination.getPageSize());
		}
		if (filterParentMenuItem != null) {
			//False condition will be added if necessary in future
			if (filterParentMenuItem) {
				criteria.add(Restrictions.or(Restrictions.eq(MenuItem.PROP_HAS_VARIANT, Boolean.FALSE), Restrictions.eq(MenuItem.PROP_VARIANT, Boolean.TRUE)));
			}
		}
	}

	public void loadMenuItems(PaginationSupport model, Boolean menuItem) {
		loadMenuItems(model, menuItem, null, null, null, false);
	}

	public void loadMenuItems(PaginationSupport model, Boolean menuItem, MenuGroup menuGroup, String itemName, MenuCategory selectedType, boolean variant) {
		loadMenuItems(model, menuItem, menuGroup, itemName, selectedType, variant, null, null);
	}

	public List<MenuItem> loadInventoryMenuItem() {
		Session session = null;
		Criteria criteria = null;
		try {
			session = createNewSession();
			criteria = session.createCriteria(MenuItem.class);
			criteria.add(Restrictions.or(Restrictions.isNull(AppConstants.PROP_DELETED), Restrictions.eq(AppConstants.PROP_DELETED, Boolean.FALSE)));
			criteria.add(Restrictions.eq(MenuItem.PROP_INVENTORY_ITEM, Boolean.TRUE));
			criteria.addOrder(Order.asc(MenuItem.PROP_ID));
			List result = criteria.list();
			return result;
		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

	public List<MenuItem> loadInventoryMenuItem(String nameOrSku, MenuGroup menuGroup, MenuCategory menuCategory) {
		Session session = null;
		Criteria criteria = null;
		try {
			session = createNewSession();
			criteria = session.createCriteria(MenuItem.class);
			criteria.add(Restrictions.or(Restrictions.isNull(AppConstants.PROP_DELETED), Restrictions.eq(AppConstants.PROP_DELETED, Boolean.FALSE)));
			criteria.add(Restrictions.eq(MenuItem.PROP_INVENTORY_ITEM, Boolean.TRUE));
			criteria.addOrder(Order.asc(MenuItem.PROP_MENU_GROUP_ID));
			criteria.addOrder(Order.asc(MenuItem.PROP_NAME));
			if (menuGroup != null) {
				criteria.add(Restrictions.eq(MenuItem.PROP_MENU_GROUP_ID, menuGroup.getId()));
			}
			if (menuCategory != null) {
				criteria.add(Restrictions.eq(MenuItem.PROP_MENU_CATEGORY_ID, menuCategory.getId()));
			}
			if (StringUtils.isNotEmpty(nameOrSku)) {
				criteria.add(
						Restrictions.or(Restrictions.ilike(MenuItem.PROP_NAME, nameOrSku, MatchMode.ANYWHERE), Restrictions.eq(MenuItem.PROP_SKU, nameOrSku)));
			}

			List result = criteria.list();
			return result;
		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

	public void loadMenuItems(PaginationSupport model, Boolean menuItem, MenuGroup menuGroup, String itemName, MenuCategory menuCategory, boolean variant,
			Boolean pizzaType) {
		loadMenuItems(model, menuItem, menuGroup, itemName, menuCategory, variant, pizzaType, null);
	}

	public void loadMenuItems(PaginationSupport model, Boolean menuItem, MenuGroup menuGroup, String itemName, MenuCategory menuCategory, boolean variant,
			Boolean pizzaType, Boolean activeStatus) {
		loadMenuItems(model, menuItem, menuGroup, itemName, menuCategory, variant, pizzaType, activeStatus, null);
	}

	/**
	 * @param model
	 * @param menuItem
	 * @param menuGroup
	 * @param itemName
	 * @param menuCategory
	 * @param variant
	 * @param pizzaType
	 * @param activeStatus
	 * @param isCombo
	 */
	public void loadMenuItems(PaginationSupport model, Boolean menuItem, MenuGroup menuGroup, String itemName, MenuCategory menuCategory, Boolean variant,
			Boolean pizzaType, Boolean activeStatus, Boolean isCombo) {
		Session session = null;
		Criteria criteria = null;
		try {
			session = createNewSession();
			criteria = session.createCriteria(MenuItem.class);
			criteria.add(Restrictions.or(Restrictions.isNull(AppConstants.PROP_DELETED), Restrictions.eq(AppConstants.PROP_DELETED, Boolean.FALSE)));
			if (!menuItem)
				criteria.add(Restrictions.eq(MenuItem.PROP_INVENTORY_ITEM, Boolean.TRUE));

			if (pizzaType != null && pizzaType) {
				criteria.add(Restrictions.eq(MenuItem.PROP_PIZZA_TYPE, pizzaType));
			}
			if (variant != null) {
				if (!variant)
					criteria.add(Restrictions.or(Restrictions.isNull(MenuItem.PROP_VARIANT), Restrictions.eq(MenuItem.PROP_VARIANT, Boolean.FALSE)));
				else {
					criteria.add(Restrictions.eq(MenuItem.PROP_VARIANT, Boolean.TRUE));
					criteria.add(Restrictions.isNotNull(MenuItem.PROP_PARENT_MENU_ITEM_ID));
				}
			}

			if (isCombo != null) {
				criteria.add(Restrictions.eq(MenuItem.PROP_COMBO_ITEM, isCombo));
			}

			if (StringUtils.isNotEmpty(itemName)) {
				Disjunction or = Restrictions.disjunction();
				or.add(Restrictions.ilike(MenuItem.PROP_NAME, itemName.trim(), MatchMode.ANYWHERE));
				or.add(Restrictions.ilike(MenuItem.PROP_BARCODE, itemName.trim(), MatchMode.ANYWHERE));
				or.add(Restrictions.ilike(MenuItem.PROP_SKU, itemName.trim(), MatchMode.ANYWHERE));
				criteria.add(or);
			}
			if (menuGroup != null) {
				criteria.add(Restrictions.eq(MenuItem.PROP_MENU_GROUP_ID, menuGroup.getId()));
			}

			if (menuCategory != null) {
				criteria.add(Restrictions.eq(MenuItem.PROP_MENU_CATEGORY_ID, menuCategory.getId()));
			}

			if (activeStatus != null) {
				criteria.add(Restrictions.eq(MenuItem.PROP_VISIBLE, activeStatus));
			}

			criteria.setProjection(Projections.rowCount());
			Number uniqueResult = (Number) criteria.uniqueResult();
			model.setNumRows(uniqueResult.intValue());
			criteria.setProjection(null);

			criteria.addOrder(Order.asc(MenuItem.PROP_SORT_ORDER));
			criteria.addOrder(Order.asc(MenuItem.PROP_NAME).ignoreCase());
			criteria.setFirstResult(model.getCurrentRowIndex());
			criteria.setMaxResults(model.getPageSize());
			List<MenuItem> result = criteria.list();
			model.setRows(result);
		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

	/*	public void loadQuickMenuItems(PaginationSupport model, Boolean menuItem, MenuGroup menuGroup, String itemName, MenuCategory menuCategory, Boolean variant,
				Boolean pizzaType, Boolean activeStatus, Boolean isCombo) {
	
		}*/

	public void loadQuickMenuItems(PaginationSupport model, Boolean menuItem, MenuGroup menuGroup, String itemName, MenuCategory menuCategory, Boolean variant,
			Boolean pizzaType, Boolean activeStatus, Boolean isCombo, Boolean isService) {
		loadQuickMenuItems(model, menuItem, menuGroup, itemName, menuCategory, variant, pizzaType, activeStatus, isCombo, isService, ""); //$NON-NLS-1$
	}

	public void loadQuickMenuItems(PaginationSupport model, Boolean menuItem, MenuGroup menuGroup, String itemName, MenuCategory menuCategory, Boolean variant,
			Boolean pizzaType, Boolean activeStatus, Boolean isCombo, Boolean isService, String sortKey) {
		loadQuickMenuItems(model, menuItem, menuGroup, itemName, menuCategory, variant, pizzaType, activeStatus, isCombo, isService, sortKey, null);
	}

	public void loadQuickMenuItems(PaginationSupport model, Boolean menuItem, MenuGroup menuGroup, String itemName, MenuCategory menuCategory, Boolean variant,
			Boolean pizzaType, Boolean activeStatus, Boolean isCombo, Boolean isService, String sortKey, ProductType productType) {
		Session session = null;
		Criteria criteria = null;
		try {
			session = createNewSession();
			criteria = session.createCriteria(MenuItem.class);
			criteria.add(Restrictions.or(Restrictions.isNull(AppConstants.PROP_DELETED), Restrictions.eq(AppConstants.PROP_DELETED, Boolean.FALSE)));
			if (!menuItem)
				criteria.add(Restrictions.eq(MenuItem.PROP_INVENTORY_ITEM, Boolean.TRUE));

			if (pizzaType != null && pizzaType) {
				criteria.add(Restrictions.eq(MenuItem.PROP_PIZZA_TYPE, pizzaType));
			}
			if (variant != null) {
				SimpleExpression isHasVariantCriteria = Restrictions.eq(MenuItem.PROP_HAS_VARIANT, variant);
				criteria.add(isHasVariantCriteria);
			}
			else {
				criteria.add(Restrictions.or(Restrictions.isNull(MenuItem.PROP_VARIANT), Restrictions.eq(MenuItem.PROP_VARIANT, Boolean.FALSE)));
			}

			if (productType != null) {
				criteria.add(Restrictions.eq(MenuItem.PROP_PRODUCT_TYPE, productType.name()));
			}

			if (isCombo != null) {
				criteria.add(Restrictions.eq(MenuItem.PROP_COMBO_ITEM, isCombo));
			}
			if (isService != null) {
				if (isService) {
					criteria.add(Restrictions.eq(MenuItem.PROP_SERVICE, Boolean.TRUE));
				}
				else {
					criteria.add(Restrictions.or(Restrictions.isNull(MenuItem.PROP_SERVICE), Restrictions.eq(MenuItem.PROP_SERVICE, Boolean.FALSE)));
				}
			}

			if (StringUtils.isNotEmpty(itemName)) {
				Disjunction or = Restrictions.disjunction();
				or.add(Restrictions.ilike(MenuItem.PROP_NAME, itemName.trim(), MatchMode.ANYWHERE));
				or.add(Restrictions.ilike(MenuItem.PROP_BARCODE, itemName.trim(), MatchMode.ANYWHERE));
				or.add(Restrictions.ilike(MenuItem.PROP_SKU, itemName.trim(), MatchMode.ANYWHERE));
				criteria.add(or);
			}
			if (menuGroup != null) {
				criteria.add(Restrictions.eq(MenuItem.PROP_MENU_GROUP_ID, menuGroup.getId()));
			}

			if (menuCategory != null) {
				criteria.add(Restrictions.eq(MenuItem.PROP_MENU_CATEGORY_ID, menuCategory.getId()));
			}

			if (activeStatus != null) {
				criteria.add(Restrictions.eq(MenuItem.PROP_VISIBLE, activeStatus));
			}

			criteria.setProjection(Projections.rowCount());
			Number uniqueResult = (Number) criteria.uniqueResult();
			model.setNumRows(uniqueResult.intValue());
			criteria.setProjection(null);

			if (StringUtils.isNotBlank(sortKey)) {
				String[] part = sortKey.split("_"); //$NON-NLS-1$
				String fieldName = part[0];
				boolean sort = Boolean.valueOf(part[1]);
				if (!sort) {
					criteria.addOrder(Order.asc(fieldName).ignoreCase());
				}
				else {
					criteria.addOrder(Order.desc(fieldName).ignoreCase());
				}
			}
			else {
				criteria.addOrder(Order.asc(MenuItem.PROP_NAME).ignoreCase());
			}
			criteria.setFirstResult(model.getCurrentRowIndex());
			criteria.setMaxResults(model.getPageSize());
			List<MenuItem> result = criteria.list();
			model.setRows(result);
		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

	public List<MenuItem> loadLabTestForCharges(ProductType productType) {
		try (Session session = createNewSession()) {

			Criteria criteria = session.createCriteria(MenuItem.class);
			addDeletedFilter(criteria);

			if (productType != null) {
				criteria.add(Restrictions.eq(MenuItem.PROP_PRODUCT_TYPE, productType.name()));
			}

			criteria.addOrder(Order.asc(MenuItem.PROP_MENU_GROUP_ID).ignoreCase());
			criteria.addOrder(Order.asc(MenuItem.PROP_NAME).ignoreCase());

			List<MenuItem> result = criteria.list();

			return result;
		}
	}

	private void addOrderTypeFilter(Object selectedType, Session session, Criteria criteria) {
		if (selectedType instanceof OrderType) {
			OrderType orderType = (OrderType) selectedType;
			Criteria criteria2 = session.createCriteria(MenuCategory.class);
			criteria2.createAlias("orderTypes", "ot"); //$NON-NLS-1$ //$NON-NLS-2$
			criteria2.add(Restrictions.in("ot." + OrderType.PROP_ID, Arrays.asList(orderType.getId()))); //$NON-NLS-1$
			List<MenuCategory> list = criteria2.list();
			if (list != null && !list.isEmpty()) {
				List<String> idList = new ArrayList<>();
				for (MenuCategory object : list) {
					idList.add(object.getId());
				}
				criteria.add(Restrictions.in(MenuItem.PROP_MENU_CATEGORY_ID, idList));
			}
		}
	}

	public void findByBarcodeOrName(PaginationSupport paginationSupport, Boolean inventoryItemOnly, MenuGroup menuGroup, String itemNameOrBarcode,
			Boolean includeVariantItem, Boolean pizzaType, Boolean showInvisibleItems) {
		findByBarcodeOrName(paginationSupport, inventoryItemOnly, menuGroup, itemNameOrBarcode, includeVariantItem, pizzaType, showInvisibleItems, true, true,
				(String[]) null);
	}

	public void findByBarcodeOrName(PaginationSupport paginationSupport, Boolean inventoryItemOnly, MenuGroup menuGroup, String itemNameOrBarcode,
			Boolean includeVariantParent, Boolean pizzaType, Boolean showInvisibleItems, boolean showVariant, boolean showComboItem, String... fields) {
		Session session = null;
		Criteria criteria = null;
		try {
			session = createNewSession();
			criteria = session.createCriteria(MenuItem.class);
			criteria.setFirstResult(paginationSupport.getCurrentRowIndex());
			criteria.setMaxResults(paginationSupport.getPageSize());

			criteria.add(Restrictions.eq(MenuItem.PROP_VISIBLE, Boolean.TRUE));
			criteria.add(Restrictions.or(Restrictions.isNull(AppConstants.PROP_DELETED), Restrictions.eq(AppConstants.PROP_DELETED, Boolean.FALSE)));

			if (StringUtils.isNotEmpty(itemNameOrBarcode)) {
				Criterion name = Restrictions.ilike(MenuItem.PROP_NAME, itemNameOrBarcode.trim(), MatchMode.ANYWHERE);
				Criterion barcode = Restrictions.ilike(MenuItem.PROP_BARCODE, itemNameOrBarcode.trim(), MatchMode.ANYWHERE);
				Criterion sku = Restrictions.ilike(MenuItem.PROP_SKU, itemNameOrBarcode.trim(), MatchMode.ANYWHERE);
				Disjunction or = Restrictions.or(name, barcode, sku);
				criteria.add(or);
			}
			if (inventoryItemOnly)
				criteria.add(Restrictions.eq(MenuItem.PROP_INVENTORY_ITEM, Boolean.TRUE));

			if (pizzaType != null && pizzaType) {
				criteria.add(Restrictions.eq(MenuItem.PROP_PIZZA_TYPE, pizzaType));
			}
			if (!includeVariantParent) {
				criteria.add(Restrictions.eq(MenuItem.PROP_HAS_VARIANT, includeVariantParent));
			}
			if (!showVariant) {
				criteria.add(Restrictions.eq(MenuItem.PROP_VARIANT, showVariant));
			}
			if (!showComboItem) {
				criteria.add(Restrictions.eq(MenuItem.PROP_COMBO_ITEM, showComboItem));
			}
			if (menuGroup != null) {
				criteria.add(Restrictions.eq(MenuItem.PROP_MENU_GROUP_ID, menuGroup.getId()));
			}
			if (showInvisibleItems) {
				criteria.add(Restrictions.eq(MenuItem.PROP_VISIBLE, showInvisibleItems));
			}
			criteria.setProjection(Projections.rowCount());
			Number rowCount = (Number) criteria.uniqueResult();
			if (rowCount != null) {
				paginationSupport.setNumRows(rowCount.intValue());
			}

			criteria.setProjection(null);
			criteria.addOrder(Order.asc(MenuItem.PROP_SORT_ORDER));
			criteria.addOrder(Order.asc(MenuItem.PROP_NAME));
			if (fields != null && fields.length > 0) {
				ProjectionList projectionList = Projections.projectionList();
				for (String field : fields) {
					projectionList.add(Projections.property(field), field);
				}
				criteria.setProjection(projectionList);
				criteria.setResultTransformer(Transformers.aliasToBean(MenuItem.class));
				paginationSupport.setRows(criteria.list());
			}
			else {
				paginationSupport.setRows(criteria.list());
			}

		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

	public void findPriceTableItemByBarcodeOrName(PaginationSupport paginationSupport, MenuGroup menuGroup, String itemNameOrBarcode,
			Boolean includeVariantParent, Boolean pizzaType, Boolean showInvisibleItems, boolean showVariant, boolean showComboItem, String... fields) {
		Session session = null;
		Criteria criteria = null;
		try {
			session = createNewSession();
			criteria = session.createCriteria(MenuItem.class);
			criteria.setFirstResult(paginationSupport.getCurrentRowIndex());
			criteria.setMaxResults(paginationSupport.getPageSize());

			criteria.add(Restrictions.eq(MenuItem.PROP_VISIBLE, Boolean.TRUE));
			criteria.add(Restrictions.or(Restrictions.isNull(AppConstants.PROP_DELETED), Restrictions.eq(AppConstants.PROP_DELETED, Boolean.FALSE)));

			if (StringUtils.isNotEmpty(itemNameOrBarcode)) {
				Criterion name = Restrictions.ilike(MenuItem.PROP_NAME, itemNameOrBarcode.trim(), MatchMode.ANYWHERE);
				Criterion barcode = Restrictions.ilike(MenuItem.PROP_BARCODE, itemNameOrBarcode.trim(), MatchMode.ANYWHERE);
				Criterion sku = Restrictions.ilike(MenuItem.PROP_SKU, itemNameOrBarcode.trim(), MatchMode.ANYWHERE);
				Disjunction or = Restrictions.or(name, barcode, sku);
				criteria.add(or);
			}

			if (pizzaType != null && pizzaType) {
				criteria.add(Restrictions.eq(MenuItem.PROP_PIZZA_TYPE, pizzaType));
			}
			if (!includeVariantParent) {
				criteria.add(Restrictions.eq(MenuItem.PROP_HAS_VARIANT, includeVariantParent));
			}
			if (!showVariant) {
				criteria.add(Restrictions.eq(MenuItem.PROP_VARIANT, showVariant));
			}
			if (!showComboItem) {
				criteria.add(Restrictions.eq(MenuItem.PROP_COMBO_ITEM, showComboItem));
			}
			if (menuGroup != null) {
				criteria.add(Restrictions.eq(MenuItem.PROP_MENU_GROUP_ID, menuGroup.getId()));
			}
			if (showInvisibleItems) {
				criteria.add(Restrictions.eq(MenuItem.PROP_VISIBLE, showInvisibleItems));
			}
			criteria.setProjection(Projections.rowCount());
			Number rowCount = (Number) criteria.uniqueResult();
			if (rowCount != null) {
				paginationSupport.setNumRows(rowCount.intValue());
			}

			criteria.setProjection(null);
			criteria.addOrder(Order.asc(MenuItem.PROP_SORT_ORDER));
			criteria.addOrder(Order.asc(MenuItem.PROP_NAME));

			List resultList;
			if (fields != null && fields.length > 0) {
				ProjectionList projectionList = Projections.projectionList();
				for (String field : fields) {
					projectionList.add(Projections.property(field), field);
				}
				criteria.setProjection(projectionList);
				criteria.setResultTransformer(Transformers.aliasToBean(MenuItem.class));
				resultList = criteria.list();
			}
			else {
				resultList = criteria.list();
			}
			List<PriceTableItem> priceTableItems = new ArrayList<>();
			for (Object object : resultList) {
				MenuItem menuItem = (MenuItem) object;
				PriceTableItem item = new PriceTableItem();
				item.setMenuItem(menuItem);
				item.setMenuItemId(menuItem.getId());
				item.setItemName(menuItem.getDisplayName());
				item.setItemBarcode(menuItem.getBarcode());
				item.setRegularPrice(menuItem.getPrice());
				item.setPrice(menuItem.getPrice());
				priceTableItems.add(item);
			}
			paginationSupport.setRows(priceTableItems);

		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

	public MenuItem findByName(String menuItemName) {
		Session session = null;
		Criteria criteria = null;
		try {
			session = createNewSession();
			criteria = session.createCriteria(MenuItem.class);
			criteria.add(Restrictions.or(Restrictions.isNull(AppConstants.PROP_DELETED), Restrictions.eq(AppConstants.PROP_DELETED, Boolean.FALSE)));
			criteria.add(Restrictions.eq(MenuItem.PROP_NAME, menuItemName));
			List result = criteria.list();
			if (result.size() == 0) {
				return null;
			}
			return (MenuItem) result.get(0);
		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

	/**
	 * This method for sync menu item with all variants
	 * 
	 * @param menuItemId
	 * 
	 * did not check deleted item for Sync
	 * 
	 * @return MenuItem list with parent and variant items
	 */
	public List<MenuItem> findMenuItemsById(String menuItemId) {
		try (Session session = createNewSession();) {
			Criteria criteria = session.createCriteria(MenuItem.class);
			criteria.add(Restrictions.eq(MenuItem.PROP_ID, menuItemId));
			criteria.setMaxResults(1);

			MenuItem menuItem = (MenuItem) criteria.uniqueResult();

			List<MenuItem> menuItems = new ArrayList<>();
			if (menuItem.isVariant()) {
				MenuItem parentMenuItem = menuItem.getParentMenuItem();
				menuItems.add(parentMenuItem);
				menuItems.addAll(getAllVariants(parentMenuItem));
			}
			else if (menuItem.isHasVariant()) {
				menuItems.add(menuItem);
				menuItems.addAll(getAllVariants(menuItem));
			}
			else {
				menuItems.add(menuItem);
			}
			return menuItems;
		}
	}

	public List<MenuItem> getAllVariants(MenuItem parentItem) throws PosException {
		try (Session session = createNewSession();) {
			Criteria criteria = session.createCriteria(MenuItem.class);
			if (parentItem != null) {
				criteria.add(Restrictions.eq(MenuItem.PROP_PARENT_MENU_ITEM_ID, parentItem.getId()));
			}
			return criteria.list();
		}
	}

	public MenuItem findByBarcode(String barcode, Session session) {
		if (StringUtils.isEmpty(barcode)) {
			return null;
		}
		Criteria criteria = session.createCriteria(MenuItem.class);
		criteria.add(Restrictions.or(Restrictions.isNull(AppConstants.PROP_DELETED), Restrictions.eq(AppConstants.PROP_DELETED, Boolean.FALSE)));
		criteria.add(Restrictions.eq(MenuItem.PROP_BARCODE, barcode.trim()));
		List list = criteria.list();
		if (list.size() > 0) {
			return (MenuItem) list.get(0);
		}

		return null;
	}

	public void loadMenuItems(BeanTableModel model, Boolean menuItem, MenuGroup menuGroup, String itemName) {

		Session session = null;
		Criteria criteria = null;
		try {
			session = createNewSession();
			criteria = session.createCriteria(MenuItem.class);
			criteria.add(Restrictions.or(Restrictions.isNull(AppConstants.PROP_DELETED), Restrictions.eq(AppConstants.PROP_DELETED, Boolean.FALSE)));
			if (!menuItem)
				criteria.add(Restrictions.eq(MenuItem.PROP_INVENTORY_ITEM, Boolean.TRUE));

			if (StringUtils.isNotEmpty(itemName)) {
				Disjunction or = Restrictions.disjunction();
				or.add(Restrictions.ilike(MenuItem.PROP_NAME, itemName.trim(), MatchMode.ANYWHERE));
				or.add(Restrictions.ilike(MenuItem.PROP_BARCODE, itemName.trim(), MatchMode.ANYWHERE));
				or.add(Restrictions.ilike(MenuItem.PROP_SKU, itemName.trim(), MatchMode.ANYWHERE));
				criteria.add(or);
			}
			if (menuGroup != null) {
				criteria.add(Restrictions.eq(MenuItem.PROP_MENU_GROUP_ID, menuGroup.getId()));
			}

			criteria.setProjection(Projections.rowCount());
			Number uniqueResult = (Number) criteria.uniqueResult();
			model.setNumRows(uniqueResult.intValue());
			criteria.setProjection(null);

			String modelSortColumn = model.getSortBy();
			if (org.apache.commons.lang.StringUtils.isNotEmpty(modelSortColumn)) {
				criteria.addOrder(model.isAscOrder() ? Order.asc(modelSortColumn) : Order.desc(modelSortColumn));
			}
			else {
				criteria.addOrder(Order.asc(MenuItem.PROP_SORT_ORDER));
				criteria.addOrder(Order.asc(MenuItem.PROP_NAME));
			}
			criteria.setFirstResult(model.getCurrentRowIndex());
			criteria.setMaxResults(model.getPageSize());
			List<MenuItem> result = criteria.list();
			model.setRows(result);
		} finally {
			if (session != null) {
				session.close();
			}
		}

	}

	public void loadMenuItems(PaginationSupport model, MenuGroup menuGroup, String itemName, boolean variant, Boolean pizzaType) {
		Session session = null;
		Criteria criteria = null;
		try {
			session = createNewSession();
			criteria = session.createCriteria(MenuItem.class);
			criteria.add(Restrictions.or(Restrictions.isNull(AppConstants.PROP_DELETED), Restrictions.eq(AppConstants.PROP_DELETED, Boolean.FALSE)));

			if (pizzaType != null && pizzaType) {
				criteria.add(Restrictions.eq(MenuItem.PROP_PIZZA_TYPE, pizzaType));
			}

			if (!variant)
				criteria.add(Restrictions.or(Restrictions.isNull(MenuItem.PROP_VARIANT), Restrictions.eq(MenuItem.PROP_VARIANT, Boolean.FALSE)));
			else
				criteria.add(Restrictions.eq(MenuItem.PROP_VARIANT, Boolean.TRUE));

			if (StringUtils.isNotEmpty(itemName)) {
				criteria.add(Restrictions.ilike(MenuItem.PROP_NAME, itemName.trim(), MatchMode.ANYWHERE));
			}
			if (menuGroup != null) {
				criteria.add(Restrictions.eq(MenuItem.PROP_MENU_GROUP_ID, menuGroup.getId()));
			}

			criteria.addOrder(Order.asc(MenuItem.PROP_ID));
			criteria.setFirstResult(model.getCurrentRowIndex());
			criteria.setMaxResults(model.getPageSize());
			List<MenuItem> result = criteria.list();
			model.setRows(result);
		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

	public void loadInventoryItems(PaginationSupport model, MenuGroup menuGroup, String itemName) {
		Session session = null;
		Criteria criteria = null;
		try {
			session = createNewSession();
			criteria = session.createCriteria(MenuItem.class);
			criteria.add(Restrictions.or(Restrictions.isNull(AppConstants.PROP_DELETED), Restrictions.eq(AppConstants.PROP_DELETED, Boolean.FALSE)));
			criteria.add(Restrictions.eq(MenuItem.PROP_INVENTORY_ITEM, true));
			if (StringUtils.isNotEmpty(itemName)) {
				//@formatter:off
				criteria.add(Restrictions.or(
								Restrictions.ilike(MenuItem.PROP_NAME, itemName.trim(), MatchMode.ANYWHERE),
								Restrictions.ilike(MenuItem.PROP_SKU, itemName.trim(), MatchMode.START),
								Restrictions.ilike(MenuItem.PROP_BARCODE, itemName.trim(), MatchMode.START)));
				//@formatter:on
			}
			if (menuGroup != null) {
				criteria.add(Restrictions.eq(MenuItem.PROP_MENU_GROUP_ID, menuGroup.getId()));
			}

			criteria.setProjection(Projections.rowCount());
			Number rowCount = (Number) criteria.uniqueResult();
			model.setNumRows(rowCount.intValue());

			criteria.setProjection(null);
			criteria.addOrder(Order.asc(MenuItem.PROP_ID));
			criteria.setFirstResult(model.getCurrentRowIndex());
			criteria.setMaxResults(model.getPageSize());
			List<MenuItem> result = criteria.list();
			model.setRows(result);
		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

	public List<MenuItem> findAllInventoryItems() {
		Session session = null;
		Criteria criteria = null;
		try {
			session = createNewSession();
			criteria = session.createCriteria(MenuItem.class);
			criteria.add(Restrictions.or(Restrictions.isNull(AppConstants.PROP_DELETED), Restrictions.eq(AppConstants.PROP_DELETED, Boolean.FALSE)));
			criteria.add(Restrictions.eq(MenuItem.PROP_INVENTORY_ITEM, true));
			return criteria.list();
		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

	public void updateStockQuantity(String menuItemId, String outletId, double unitQuantity, boolean updateAvailableUnit, boolean updateOnHandUnit,
			Session session) {
		/*String sql = null;
		if (updateAvailableUnit && updateOnHandUnit) {
			sql = String.format("update MenuItem set availableUnit = (availableUnit + %s),unitOnHand = (unitOnHand + %s) where id = '%s'", unitQuantity,
					unitQuantity, menuItemId);
		}
		else if (updateOnHandUnit) {
			sql = String.format("update MenuItem set unitOnHand = (unitOnHand + %s) where id = '%s'", unitQuantity, menuItemId);
		}
		else {
			sql = String.format("update MenuItem set availableUnit = (availableUnit + %s) where id = '%s'", unitQuantity, menuItemId);
		}
		session.createQuery(sql).executeUpdate();*/
		MenuItemInventoryStatus status = MenuItemInventoryStatusDAO.getInstance().get(menuItemId, outletId, session);
		if (status == null) {
			status = new MenuItemInventoryStatus();
			status.setMenuItemId(menuItemId);
			status.setOutletId(outletId);
		}
		if (updateAvailableUnit) {
			status.setAvailableUnit(status.getAvailableUnit() + unitQuantity);
		}
		if (updateOnHandUnit) {
			status.setUnitOnHand(status.getUnitOnHand() + unitQuantity);
		}
		MenuItemInventoryStatusDAO.getInstance().saveOrUpdate(status, session);
	}

	public void updateStockQuantity(String menuItemId, double unitQuantity, double unitCost, boolean updateAvailableUnit, boolean updateOnHandUnit,
			Session session) {
		String sql = null;
		if (updateAvailableUnit && updateOnHandUnit) {
			sql = String.format("update MenuItem set availableUnit = (availableUnit + %s),unitOnHand = (unitOnHand + %s),cost = %s where id = '%s'", //$NON-NLS-1$
					unitQuantity, unitQuantity, unitCost, menuItemId);
		}
		else if (updateOnHandUnit) {
			sql = String.format("update MenuItem set unitOnHand = (unitOnHand + %s),cost = %s where id = '%s'", unitQuantity, unitCost, menuItemId); //$NON-NLS-1$
		}
		else {
			sql = String.format("update MenuItem set availableUnit = (availableUnit + %s),cost = %s where id = '%s'", unitQuantity, unitCost, menuItemId); //$NON-NLS-1$
		}
		session.createQuery(sql).executeUpdate();
	}

	public boolean hasMenuItem() {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(MenuItem.class);
			addDeletedFilter(criteria);
			criteria.setProjection(Projections.rowCount());
			Number rowCount = (Number) criteria.uniqueResult();
			return rowCount != null && rowCount.intValue() > 0;
		}
	}

	public boolean existsMenuItem(MenuGroup menuGroup) {
		Session session = null;
		Criteria criteria = null;
		try {
			session = createNewSession();
			criteria = session.createCriteria(MenuItem.class);
			criteria.add(Restrictions.or(Restrictions.isNull(AppConstants.PROP_DELETED), Restrictions.eq(AppConstants.PROP_DELETED, Boolean.FALSE)));
			criteria.setProjection(Projections.rowCount());
			if (menuGroup != null) {
				criteria.add(Restrictions.eq(MenuItem.PROP_MENU_GROUP_ID, menuGroup.getId()));
			}

			Number rowCount = (Number) criteria.uniqueResult();
			return rowCount != null && rowCount.intValue() > 0;
		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

	public boolean existsMenuItem(MenuCategory menuCategory) {
		Session session = null;
		Criteria criteria = null;
		try {
			session = createNewSession();
			criteria = session.createCriteria(MenuItem.class);
			criteria.add(Restrictions.or(Restrictions.isNull(AppConstants.PROP_DELETED), Restrictions.eq(AppConstants.PROP_DELETED, Boolean.FALSE)));
			criteria.setProjection(Projections.rowCount());
			if (menuCategory != null) {
				criteria.add(Restrictions.eq(MenuItem.PROP_MENU_CATEGORY_ID, menuCategory.getId()));
			}

			Number rowCount = (Number) criteria.uniqueResult();
			return rowCount != null && rowCount.intValue() > 0;
		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

	public void updateLastPurchaseCost(String menuItemId, double cost, boolean updateLastPurchasePrice, Session session) {

		String sql = null;
		double avgCost = InventoryTransactionDAO.getInstance().findItemAvgCost(menuItemId, session);
		double actualCost = updateLastPurchasePrice ? cost : avgCost;
		sql = String.format("update MenuItem set lastPurchasedCost = %s,avgCost=%s, cost = %s where id = '%s'", cost, avgCost, actualCost, menuItemId); //$NON-NLS-1$

		session.createQuery(sql).executeUpdate();
	}

	public void updateLastPurchaseCost(PurchaseOrder order, boolean updateLastPurchasePrice) {
		Session session = null;
		Transaction tx = null;
		try {
			session = createNewSession();
			tx = session.beginTransaction();
			List<PurchaseOrderItem> orderItems = order.getOrderItems();
			for (PurchaseOrderItem purchaseOrderItem : orderItems) {
				MenuItem menuItem = purchaseOrderItem.getMenuItem();
				double baseUnitQuantity = menuItem.getBaseUnitQuantity(purchaseOrderItem.getItemUnitName());
				double unitPrice = purchaseOrderItem.getUnitPrice() / baseUnitQuantity;
				updateLastPurchaseCost(menuItem.getId(), unitPrice, updateLastPurchasePrice, session);
			}

			session.saveOrUpdate(order);
			tx.commit();
		} catch (Exception e) {
			if (tx != null) {
				tx.rollback();
			}
			throwException(e);
		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

	///This method is only for fetch specific fields..

	public MenuItem getMenuItemWithFields(String menuItemId, String... fields) throws Exception {
		Session session = null;
		try {
			session = createNewSession();
			return getMenuItemWithFields(session, menuItemId, fields);
		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

	public MenuItem getMenuItemWithFields(Session session, String menuItemId, String... fields) throws Exception {
		Criteria criteria = null;
		criteria = session.createCriteria(MenuItem.class);
		criteria.add(Restrictions.or(Restrictions.isNull(AppConstants.PROP_DELETED), Restrictions.eq(AppConstants.PROP_DELETED, Boolean.FALSE)));
		criteria.add(Restrictions.eq(MenuItem.PROP_ID, menuItemId));

		ProjectionList projectionList = Projections.projectionList();
		projectionList.add(Projections.property(MenuItem.PROP_ID), MenuItem.PROP_ID);
		for (String field : fields) {
			projectionList.add(Projections.property(field), field);
		}
		criteria.setProjection(projectionList);
		criteria.setResultTransformer(Transformers.aliasToBean(MenuItem.class));
		MenuItem menuItem = (MenuItem) criteria.uniqueResult();
		if (menuItem != null) {
			Hibernate.initialize(menuItem.getStockUnits());
		}

		return menuItem;
	}

	public void loadMenuItems(PaginationSupport model, String searchItem) {
		loadMenuItems(model, searchItem, false);
	}

	public void loadMenuItems(PaginationSupport model, String searchItem, Boolean inventoryItemsOnly) {
		Session session = null;
		Criteria criteria = null;
		try {
			session = createNewSession();
			criteria = session.createCriteria(MenuItem.class);
			criteria.add(Restrictions.or(Restrictions.isNull(AppConstants.PROP_DELETED), Restrictions.eq(AppConstants.PROP_DELETED, Boolean.FALSE)));

			if (StringUtils.isNotEmpty(searchItem)) {
				criteria.add(Restrictions.ilike(MenuItem.PROP_NAME, searchItem.trim(), MatchMode.ANYWHERE));
			}
			criteria.add(Restrictions.eq(MenuItem.PROP_HAS_VARIANT, false));
			criteria.setProjection(Projections.rowCount());
			if (inventoryItemsOnly) {
				criteria.add(Restrictions.eq(MenuItem.PROP_INVENTORY_ITEM, Boolean.TRUE));
			}
			Number uniqueResult = (Number) criteria.uniqueResult();
			model.setNumRows(uniqueResult.intValue());
			criteria.setProjection(null);

			criteria.addOrder(Order.asc(MenuItem.PROP_SORT_ORDER));
			criteria.addOrder(Order.asc(MenuItem.PROP_NAME));
			criteria.setFirstResult(model.getCurrentRowIndex());
			criteria.setMaxResults(model.getPageSize());
			List<MenuItem> result = criteria.list();
			model.setRows(result);
		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

	public List<MenuItem> findAllUnSyncMenuItems() {
		Session session = null;
		try {
			session = createNewSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.or(Restrictions.isNull(AppConstants.PROP_DELETED), Restrictions.eq(AppConstants.PROP_DELETED, Boolean.FALSE)));
			Criterion falseCriteria = Restrictions.eq(MenuItem.PROP_CLOUD_SYNCED, Boolean.FALSE);
			Criterion nullCriteria = Restrictions.isNull(MenuItem.PROP_CLOUD_SYNCED);
			criteria.add(Restrictions.or(falseCriteria, nullCriteria));
			return criteria.list();
		} finally {
			if (session != null) {
				if (session != null) {
					session.close();
				}
			}
		}
	}

	public void updateMenuItemSync(List<String> ids, Boolean synced) {
		if (ids == null || ids.isEmpty())
			return;

		String whereCondition = "("; //$NON-NLS-1$
		for (Iterator iterator = ids.iterator(); iterator.hasNext();) {
			String id = (String) iterator.next();
			whereCondition += "'" + id + "'"; //$NON-NLS-1$ //$NON-NLS-2$
			if (iterator.hasNext())
				whereCondition += ","; //$NON-NLS-1$
		}
		whereCondition += ")"; //$NON-NLS-1$
		//@formatter:off
		Transaction tx=null;
		Session session = null;
		try {
			session = getSession();
			tx=session.beginTransaction();
		String hqlString = "update MenuItem set %s=:synced where %s in" + whereCondition; //$NON-NLS-1$
		hqlString = String.format(hqlString, MenuItem.PROP_CLOUD_SYNCED, MenuItem.PROP_ID);
		//@formatter:on
			Query query = session.createQuery(hqlString);
			query.setParameter("synced", synced); //$NON-NLS-1$
			query.executeUpdate();
			tx.commit();
		} catch (Exception e) {
			tx.rollback();
			throw e;
		} finally {
			if (session != null) {
				session.close();
			}
		}

	}

	public boolean hasByBarcodeOrSKU(String barcode, String sku) {
		Session session = null;
		Criteria criteria = null;
		try {
			session = createNewSession();
			criteria = session.createCriteria(MenuItem.class);
			criteria.add(Restrictions.or(Restrictions.isNull(AppConstants.PROP_DELETED), Restrictions.eq(AppConstants.PROP_DELETED, Boolean.FALSE)));

			Disjunction or = Restrictions.disjunction();
			if (StringUtils.isNotEmpty(barcode)) {
				or.add(Restrictions.eq(MenuItem.PROP_BARCODE, barcode));
			}
			if (StringUtils.isNotEmpty(sku)) {
				or.add(Restrictions.eq(MenuItem.PROP_SKU, sku));
			}
			criteria.add(or);
			List result = criteria.list();
			if (result.size() == 0) {
				return false;
			}
			return true;
		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

	public MenuItem getMenuItemByBarcodeAndSKU(String barcode, String sku) {
		if (StringUtils.isBlank(barcode) && StringUtils.isBlank(sku)) {
			return null;
		}

		Session session = null;
		Criteria criteria = null;
		try {
			session = createNewSession();
			criteria = session.createCriteria(MenuItem.class);
			criteria.add(Restrictions.or(Restrictions.isNull(AppConstants.PROP_DELETED), Restrictions.eq(AppConstants.PROP_DELETED, Boolean.FALSE)));

			if (StringUtils.isNotBlank(barcode)) {
				criteria.add(Restrictions.like(MenuItem.PROP_BARCODE, barcode));
			}
			if (StringUtils.isNotBlank(sku)) {
				criteria.add(Restrictions.like(MenuItem.PROP_SKU, sku));
			}
			List<MenuItem> result = criteria.list();
			if (result == null || result.isEmpty()) {
				return null;
			}
			return (MenuItem) result.get(0);
		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

	//	public MenuItem hasByBarcode(MenuItem item) {
	//
	//		Session session = null;
	//		Criteria criteria = null;
	//		try {
	//			session = createNewSession();
	//			criteria = session.createCriteria(MenuItem.class);
	//			criteria.add(Restrictions.or(Restrictions.isNull(AppConstants.PROP_DELETED), Restrictions.eq(AppConstants.PROP_DELETED, Boolean.FALSE)));
	//			criteria.add(Restrictions.eq(MenuItem.PROP_BARCODE, item.getBarcode()));
	//
	//			List result = criteria.list();
	//			if (result.size() == 0) {
	//				return null;
	//			}
	//			else {
	//				return (MenuItem) result.get(0);
	//			}
	//
	//		} finally {
	//			if (session != null) {
	//				session.close();
	//			}
	//		}
	//	}
	//
	//	public MenuItem hasBySKU(MenuItem item) {
	//
	//		Session session = null;
	//		Criteria criteria = null;
	//		try {
	//			session = createNewSession();
	//			criteria = session.createCriteria(MenuItem.class);
	//			criteria.add(Restrictions.or(Restrictions.isNull(AppConstants.PROP_DELETED), Restrictions.eq(AppConstants.PROP_DELETED, Boolean.FALSE)));
	//			criteria.add(Restrictions.eq(MenuItem.PROP_SKU, item.getSku()));
	//
	//			List result = criteria.list();
	//			if (result.size() == 0) {
	//				return null;
	//			}
	//			else {
	//				return (MenuItem) result.get(0);
	//			}
	//
	//		} finally {
	//			if (session != null) {
	//				session.close();
	//			}
	//		}
	//	}

	/**
	 * This method is used to check Menu item name is already exist or not id database.
	 * 
	 * @param menuItemIdToExclude Id of the menu item.
	 * @param menuItemName Name of the menu item.
	 * 
	  */
	public boolean isNameExist(String menuItemIdToExclude, String menuItemName) {
		if (StringUtils.isBlank(menuItemName)) {
			return false;
		}

		try (Session session = this.createNewSession()) {
			Criteria criteria = session.createCriteria(MenuItem.class);
			if (StringUtils.isNotBlank(menuItemIdToExclude)) {
				criteria.add(Restrictions.ne(MenuItem.PROP_ID, menuItemIdToExclude));
			}
			criteria.add(Restrictions.or(Restrictions.isNull(AppConstants.PROP_DELETED), Restrictions.eq(AppConstants.PROP_DELETED, Boolean.FALSE)));
			criteria.add(Restrictions.eq(MenuItem.PROP_NAME, menuItemName));
			List list = criteria.list();
			return list != null && !list.isEmpty();
		}
	}

	public void removePizzaPrices(List<String> listOfPriceId, Session session) {
		Criteria criteria = session.createCriteria(getReferenceClass());
		criteria.createAlias("pizzaPriceList", "priceList"); //$NON-NLS-1$ //$NON-NLS-2$
		criteria.add(Restrictions.in("priceList.id", listOfPriceId)); //$NON-NLS-1$
		criteria.setResultTransformer(Criteria.DISTINCT_ROOT_ENTITY);
		List<MenuItem> list = criteria.list();
		if (list != null && !list.isEmpty()) {
			for (MenuItem menuItem : list) {
				List<PizzaPrice> pizzaPriceList = menuItem.getPizzaPriceList();
				for (Iterator iterator = pizzaPriceList.iterator(); iterator.hasNext();) {
					PizzaPrice pizzaPrice = (PizzaPrice) iterator.next();
					if (listOfPriceId.contains(pizzaPrice.getId())) {
						iterator.remove();
					}
				}
				session.update(menuItem);
			}
		}
	}

	public List<MenuItem> getMenuItemsByPackagingUnit(PackagingUnit packagingUnit, Session session) {
		Criteria criteria = session.createCriteria(InventoryStockUnit.class);
		criteria.add(Restrictions.eq(InventoryStockUnit.PROP_PACKAGING_UNIT, packagingUnit));
		criteria.setProjection(Projections.distinct(Projections.property(InventoryStockUnit.PROP_MENU_ITEM_ID)));
		List menuItemIds = criteria.list();
		if (menuItemIds != null && !menuItemIds.isEmpty()) {
			criteria = session.createCriteria(this.getReferenceClass());
			this.addDeletedFilter(criteria);
			criteria.add(Restrictions.in(MenuItem.PROP_ID, menuItemIds));
			criteria.setProjection(Projections.alias(Projections.property(MenuItem.PROP_NAME), MenuItem.PROP_NAME));
			List<MenuItem> menuItems = criteria.setResultTransformer(Transformers.aliasToBean(this.getReferenceClass())).list();
			return menuItems;
		}
		return null;
	}

	public List<MenuItem> getMenuItemsByPrinterGroup(PrinterGroup printerGroup, Session session) {
		if (printerGroup == null) {
			return null;
		}
		Criteria criteria = session.createCriteria(this.getReferenceClass());
		this.addDeletedFilter(criteria);
		criteria.add(Restrictions.eq(MenuItem.PROP_PRINTER_GROUP_ID, printerGroup.getId()));
		criteria.setProjection(Projections.alias(Projections.property(MenuItem.PROP_NAME), MenuItem.PROP_NAME));
		return criteria.setResultTransformer(Transformers.aliasToBean(this.getReferenceClass())).list();
	}

	public void doCopyModifiersToOtherItems(MenuItem sourceMenuItem, List<MenuItem> menuItemList) throws Exception {
		List<MenuItemModifierSpec> menuItemModiferSpecs = sourceMenuItem.getMenuItemModiferSpecs();
		if (menuItemModiferSpecs == null || menuItemModiferSpecs.isEmpty()) {
			return;
		}
		Transaction transaction = null;
		try (Session session = createNewSession()) {
			transaction = session.beginTransaction();
			for (MenuItem menuItem : menuItemList) {
				session.refresh(menuItem);
				for (MenuItemModifierSpec menuItemModifierSpec : menuItemModiferSpecs) {
					if (menuItemModifierSpec != null && menuItemModifierSpec.getId() != null) {
						session.refresh(menuItemModifierSpec);
					}

					List<MenuItemModifierSpec> modifierSpecs = menuItem.getMenuItemModiferSpecs();
					for (Iterator<MenuItemModifierSpec> iterator = modifierSpecs.iterator(); iterator.hasNext();) {
						MenuItemModifierSpec spec = (MenuItemModifierSpec) iterator.next();
						if (spec.getModifierGroupId().equals(menuItemModifierSpec.getModifierGroupId())) {
							iterator.remove();
						}
					}
					menuItem.addTomenuItemModiferSpecs(menuItemModifierSpec.deepClone());
				}
			}
			transaction.commit();
		}
	}

	public void saveOrUpdateMenuItems(List<MenuItem> menuItemList) throws Exception {
		Transaction transaction = null;
		try (Session session = createNewSession()) {
			transaction = session.beginTransaction();
			for (MenuItem menuItem : menuItemList) {
				saveOrUpdate(menuItem, session);
			}
			transaction.commit();
		}
	}

	public void deleteMenuItems(List<MenuItem> menuItemList) {
		try (Session session = createNewSession()) {
			Transaction transaction = session.beginTransaction();
			for (MenuItem menuItem : menuItemList) {
				delete(menuItem, session);
			}
			transaction.commit();
		}
	}

	public List<MenuItem> findMenuItemByCategoryId(String categoryId) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(MenuItem.class);
			if (!StringUtils.isBlank(categoryId)) {
				criteria.add(Restrictions.eq(MenuItem.PROP_MENU_CATEGORY_ID, categoryId));
			}

			addDeletedFilter(criteria);
			return criteria.list();
		}
	}

	public List<MenuItem> findMenuItemByTaxGroupId(String taxGroupId) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(MenuItem.class);
			if (!StringUtils.isBlank(taxGroupId)) {
				criteria.add(Restrictions.eq(MenuItem.PROP_TAX_GROUP_ID, taxGroupId));
			}

			addDeletedFilter(criteria);
			return criteria.list();
		}
	}

	@Override
	public List<MenuItem> findAll() {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			addDeletedFilter(criteria);
			criteria.addOrder(Order.asc(MenuItem.PROP_SORT_ORDER));
			return criteria.list();
		}
	}

	public int getMenuItemsCountByGroupId(String menuGroupId) {
		if (menuGroupId == null) {
			return 0;
		}
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(MenuItem.class);
			this.addDeletedFilter(criteria, MenuItem.class);
			criteria.add(Restrictions.eq(MenuItem.PROP_MENU_GROUP_ID, menuGroupId));
			criteria.add(Restrictions.eq(MenuItem.PROP_VARIANT, false));
			criteria.setProjection(Projections.rowCount());
			Number rowCount = (Number) criteria.uniqueResult();
			if (rowCount != null) {
				return rowCount.intValue();
			}
		}
		return 0;
	}

	public List<MenuItem> findAllServiceItems() {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(MenuItem.class);
			this.addDeletedFilter(criteria, MenuItem.class);
			criteria.add(Restrictions.eq(MenuItem.PROP_SERVICE, Boolean.TRUE));
			return criteria.list();
		}
	}

	public List<MenuItem> getMenuItemsByGroupId(String menuGroupId, Session s) {
		Criteria criteria = s.createCriteria(MenuItem.class);
		this.addDeletedFilter(criteria, MenuItem.class);
		criteria.add(Restrictions.eq(MenuItem.PROP_MENU_GROUP_ID, menuGroupId));
		criteria.add(Restrictions.eq(MenuItem.PROP_VARIANT, false));
		return criteria.list();
	}

	public boolean hasByBarcodeOrSKUOrMenuItemName(String barcode, String sku, String menuItemName) {
		Session session = null;
		Criteria criteria = null;
		try {
			session = createNewSession();
			criteria = session.createCriteria(MenuItem.class);
			criteria.add(Restrictions.or(Restrictions.isNull(AppConstants.PROP_DELETED), Restrictions.eq(AppConstants.PROP_DELETED, Boolean.FALSE)));

			Disjunction orDisjunction = Restrictions.disjunction();
			if (StringUtils.isNotEmpty(barcode)) {
				orDisjunction.add(Restrictions.eq(MenuItem.PROP_BARCODE, barcode));
			}
			if (StringUtils.isNotEmpty(sku)) {
				orDisjunction.add(Restrictions.eq(MenuItem.PROP_SKU, sku));
			}
			if (StringUtils.isNotEmpty(menuItemName)) {
				orDisjunction.add(Restrictions.eq(MenuItem.PROP_NAME, menuItemName));
			}
			criteria.add(orDisjunction);
			List result = criteria.list();
			if (result.size() == 0) {
				return false;
			}
			return true;
		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

	public List getSyncableMenuItems(Date lastUpdateTime, String outletId, int offset, int limit) {
		try (Session session = this.createNewSession()) {
			List<MenuItem> unSyncItems = getSyncableMenuItems(lastUpdateTime, offset, limit, session);
			for (Iterator iterator = unSyncItems.iterator(); iterator.hasNext();) {
				MenuItem menuItem = (MenuItem) iterator.next();
				MenuItemDAO.getInstance().initialize(menuItem, session, false);
				if (menuItem.isComboItem()) {
					List<ComboGroup> comboGroups = menuItem.getComboGroups();
					if (comboGroups != null && comboGroups.size() > 0) {
						for (ComboGroup comboGroup : comboGroups) {
							List<MenuItem> comboGroupItems = comboGroup.getItems();
							if (comboGroupItems != null && comboGroupItems.size() > 0) {
								for (MenuItem comboGroupItem : comboGroupItems) {
									MenuItemDAO.getInstance().initialize(comboGroupItem, session, false);
								}
							}
						}
					}
				}
			}
			return unSyncItems;
		}
	}

	private List<MenuItem> getSyncableMenuItems(Date lastUpdateTime, int offset, int limit, Session session) {
		Criteria criteria = session.createCriteria(MenuItem.class);
		if (lastUpdateTime != null) {
			criteria.add(Restrictions.or(Restrictions.isNull("lastUpdateTime"), Restrictions.gt("lastUpdateTime", lastUpdateTime))); //$NON-NLS-1$ //$NON-NLS-2$
		}
		if (limit > 0 && offset >= 0) {
			criteria.setFirstResult(offset);
			criteria.setMaxResults(limit);
		}
		criteria.addOrder(Order.asc(MenuItem.PROP_COMBO_ITEM));
		criteria.addOrder(Order.asc(MenuItem.PROP_VARIANT));
		criteria.addOrder(Order.desc(MenuItem.PROP_HAS_VARIANT));
		criteria.addOrder(Order.asc(MenuItem.PROP_ID));

		criteria.setResultTransformer(Criteria.DISTINCT_ROOT_ENTITY);
		return criteria.list();
	}

	public int getNextSkuNumber() {
		return getNextSkuNumber(1001);
	}

	public int getNextSkuNumber(int skuNo) {
		try {
			MenuItem menuItem = getMenuItemBySKU(String.valueOf(skuNo));
			if (menuItem == null) {
				return skuNo;
			}

			Random r = new Random();
			String randomNumber = String.format("%04d", r.nextInt(skuNo)); //$NON-NLS-1$

			int rNumber = Integer.parseInt(randomNumber) + 1001;
			return getNextSkuNumber(rNumber);
		} catch (Exception e) {
		}
		return 1001;
	}

	public void updateLastUpdateTime(String menuItemId, Date lastUpdateTime) {
		if (StringUtils.isBlank(menuItemId)) {
			return;
		}
		Transaction tx = null;
		try (Session session = createNewSession()) {
			tx = session.beginTransaction();
			String hqlString = "update " + MenuItem.REF + " set %s=:lastUpdateTime where %s=:menuItemId"; //$NON-NLS-1$ //$NON-NLS-2$
			//@formatter:off
					hqlString = String.format(
							hqlString, 
							MenuItem.PROP_LAST_UPDATE_TIME, 
							MenuItem.PROP_ID
							);
					
			//@formatter:on
			Query query = session.createQuery(hqlString);
			query.setParameter("lastUpdateTime", lastUpdateTime); //$NON-NLS-1$
			query.setParameter("menuItemId", menuItemId); //$NON-NLS-1$
			query.executeUpdate();

			tx.commit();
		} catch (Exception e) {
			if (tx != null) {
				tx.rollback();
			}

			// ignore
		}

	}

	public void adjustInventoryOnHand(MenuItem menuItem, double quantity, Outlet selectedOutlet, SiiopaCustomer siiopaCustomer) {
		double unitOnHand = menuItem.getUnitOnHand();
		if (NumberUtil.format6DigitNumber(unitOnHand).equals(NumberUtil.format6DigitNumber(quantity))) {
			return;
		}

		InventoryUnit unit = menuItem.getUnit();
		if (unit == null) {
			throw new PosException(Messages.getString("MenuItemDAO.16")); //$NON-NLS-1$
		}

		if (quantity < 0) {
			throw new PosException(Messages.getString("MenuItemDAO.17")); //$NON-NLS-1$
		}
		Double stockDifference = quantity - unitOnHand;
		InventoryLocation location;
		boolean inTransaction = false;
		if (stockDifference > 0) {
			inTransaction = true;
			location = InventoryLocationDAO.getInstance().getDefaultInInventoryLocation(selectedOutlet.getId());
		}
		else {
			inTransaction = false;
			location = InventoryLocationDAO.getInstance().getDefaultOutInventoryLocation(selectedOutlet.getId());
		}

		String outletId = selectedOutlet.getId();
		if (location == null) {
			location = new DefaultDataInserter().createDefaultInventoryLocation(outletId, inTransaction, !inTransaction);
		}

		InventoryTransaction inventoryTransaction = new InventoryTransaction();
		inventoryTransaction.setOutletId(outletId);

		inventoryTransaction.setMenuItem(menuItem);
		inventoryTransaction.setQuantity(Math.abs(stockDifference));
		String uniqueId = unit.getId();
		inventoryTransaction.setUnit(uniqueId);
		inventoryTransaction.setUnitCost(menuItem.calculateBaseUnitCost(unit));
		Double baseUnitQuantity = menuItem.getBaseUnitQuantity(uniqueId);
		inventoryTransaction.setTotal(baseUnitQuantity * inventoryTransaction.getQuantity() * menuItem.getCost());
		inventoryTransaction.setTransactionDate(new Date());
		String strReason = (stockDifference > 0) ? InventoryTransaction.REASON_ADJUST_IN : InventoryTransaction.REASON_ADJUST_OUT;
		inventoryTransaction.setReason(strReason);

		if (inTransaction) {
			inventoryTransaction.setType(InventoryTransactionType.IN.getType());
			inventoryTransaction.setToInventoryLocation(location);
		}
		else {
			inventoryTransaction.setType(InventoryTransactionType.OUT.getType());
			inventoryTransaction.setFromInventoryLocation(location);
		}
		InventoryTransactionDAO.getInstance().performInventoryTransaction(inventoryTransaction);

		StringBuilder sb = new StringBuilder();
		sb.append("Menu item id: " + menuItem.getId()); //$NON-NLS-1$
		sb.append(", Menu item name " + menuItem.getName()); //$NON-NLS-1$
		sb.append(", Unit on hand: " + NumberUtil.formatNumber(unitOnHand)); //$NON-NLS-1$
		sb.append(", Actual Unit: " + NumberUtil.formatNumber(quantity)); //$NON-NLS-1$

		User currentUser = DataProvider.get().getCurrentUser();
		if (currentUser == null) {
			if (siiopaCustomer != null) {
				sb.append(", Performed by siiopa customer: " + siiopaCustomer.getName()); //$NON-NLS-1$
				sb.append(", and Siiopa customer id: " + siiopaCustomer.getId()); //$NON-NLS-1$
			}
		}
		else {
			sb.append(", Performed by " + currentUser.getFullName()); //$NON-NLS-1$
			sb.append(", and user id: " + currentUser.getId()); //$NON-NLS-1$
		}

		ActionHistoryDAO.saveActionHistory(ActionHistory.ADJUST_INVENTORY_ON_HAND, sb.toString(), currentUser, outletId);
	}

}