package com.floreantpos.model.dao;

import java.util.List;

import org.apache.commons.lang.StringUtils;
import org.hibernate.Criteria;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.hibernate.criterion.DetachedCriteria;
import org.hibernate.criterion.MatchMode;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.ProjectionList;
import org.hibernate.criterion.Projections;
import org.hibernate.criterion.Property;
import org.hibernate.criterion.Restrictions;
import org.hibernate.type.DoubleType;
import org.hibernate.type.Type;

import com.floreantpos.PosLog;
import com.floreantpos.model.COAAccountType;
import com.floreantpos.model.ChartOfAccounts;
import com.floreantpos.model.DirectionType;
import com.floreantpos.model.LedgerEntry;
import com.floreantpos.model.PosTransaction;
import com.floreantpos.model.RefundTransaction;
import com.floreantpos.model.Ticket;
import com.floreantpos.model.TicketItem;
import com.floreantpos.swing.PaginationSupport;
import com.floreantpos.util.NumberUtil;

public class LedgerEntryDAO extends BaseLedgerEntryDAO {

	public void findByChartOfAcountName(PaginationSupport paginationSupport, String searchString, String orderId) {
		try (Session session = createNewSession()) {

			Criteria criteria = session.createCriteria(LedgerEntry.class);
			addDeletedFilter(criteria);

			if (StringUtils.isNotBlank(orderId)) {
				criteria.add(Restrictions.eq(LedgerEntry.PROP_TICKET_ID, orderId));
			}
			if (StringUtils.isNotBlank(searchString)) {
				DetachedCriteria detachedCriteria = DetachedCriteria.forClass(COAAccountType.class);
				detachedCriteria.setProjection(Property.forName(COAAccountType.PROP_ID));
				detachedCriteria.add(Restrictions.ilike(COAAccountType.PROP_NAME, searchString, MatchMode.ANYWHERE));

				criteria.add(Property.forName(LedgerEntry.PROP_ACCOUNT_ID).in(detachedCriteria)); //$NON-NLS-1$
			}
			criteria.setProjection(Projections.rowCount());

			Number rowCount = (Number) criteria.uniqueResult();
			if (rowCount != null) {
				paginationSupport.setNumRows(rowCount.intValue());
			}
			criteria.setProjection(null);
			criteria.addOrder(Order.desc(LedgerEntry.PROP_LAST_UPDATE_TIME));
			paginationSupport.setRows(criteria.list());
		}
	}

	public void saveTicketLedgerEntry(Ticket ticket, double remainLedgerAmount) {
		ChartOfAccountsDAO instance = ChartOfAccountsDAO.getInstance();
		try (Session session = instance.createNewSession()) {
			Transaction tx = session.beginTransaction();

			ChartOfAccounts reciableChartOfAccounts = instance.findByAcountCode("5000", session);
			ChartOfAccounts salesChartOfAccounts = instance.findByAcountCode("200", session);

			LedgerEntry createDebitLedgerEntry = LedgerEntry.createLedgerEntry(ticket, reciableChartOfAccounts, DirectionType.DEBIT, remainLedgerAmount);
			LedgerEntryDAO.getInstance().save(createDebitLedgerEntry, session);
			LedgerEntry createCreditLedgerEntry = LedgerEntry.createLedgerEntry(ticket, salesChartOfAccounts, DirectionType.CREDIT, remainLedgerAmount);
			LedgerEntryDAO.getInstance().save(createCreditLedgerEntry, session);

			tx.commit();
		} catch (Exception e) {
			PosLog.error(getClass(), e);
		}
	}

	public void saveTransactionLedgerEntry(Ticket ticket, PosTransaction transaction) {
		ChartOfAccountsDAO instance = ChartOfAccountsDAO.getInstance();
		try (Session session = instance.createNewSession()) {
			Transaction tx = session.beginTransaction();

			ChartOfAccounts cashChartOfAccounts = instance.findByAcountCode("10100", session);
			ChartOfAccounts reciableChartOfAccounts = instance.findByAcountCode("5000", session);

			Double transAmount = transaction.isVoided() ? (-1) * transaction.getAmount() : transaction.getAmount();
			LedgerEntry createDebitLedgerEntry = LedgerEntry.createLedgerEntry(ticket, cashChartOfAccounts, DirectionType.DEBIT, transAmount);
			createDebitLedgerEntry.setTransactionId(transaction.getId());
			LedgerEntryDAO.getInstance().save(createDebitLedgerEntry, session);
			LedgerEntry createCreditLedgerEntry = LedgerEntry.createLedgerEntry(ticket, reciableChartOfAccounts, DirectionType.CREDIT, transAmount);
			createCreditLedgerEntry.setTransactionId(transaction.getId());
			LedgerEntryDAO.getInstance().save(createCreditLedgerEntry, session);

			tx.commit();
		} catch (Exception e) {
			PosLog.error(getClass(), e);
		}
	}

	public void saveOrderLedgerEntry(Ticket ticket, PosTransaction transaction) {
		double maxRFFee = ticket.getReferrerFeeOnNetSales();
		List<TicketItem> ticketItems = ticket.getTicketItems();
		if (ticketItems != null) {
			for (TicketItem ticketItem : ticketItems) {
				maxRFFee += ticketItem.getReferrerFeeOnReport();
			}
		}

		ChartOfAccountsDAO instance = ChartOfAccountsDAO.getInstance();
		try (Session session = instance.createNewSession()) {
			Transaction tx = session.beginTransaction();

			ChartOfAccounts productsSalesCOA = instance.findByAcountCode("200", session);
			ChartOfAccounts taxCOA = instance.findByAcountCode("506", session);
			ChartOfAccounts rfPayableCOA = instance.findByAcountCode("507", session);
			ChartOfAccounts ldfPayableCOA = instance.findByAcountCode("508", session);
			ChartOfAccounts reciableCOA = instance.findByAcountCode("5000", session);
			ChartOfAccounts cashChartOfAccounts = instance.findByAcountCode("10100", session);

			String ticketId = ticket.getId();
			double leProductSalesAmount = getLedgerEntryAmount(productsSalesCOA.getId(), ticketId);
			double leTaxAmount = getLedgerEntryAmount(taxCOA.getId(), ticketId);
			double leRFAmount = getLedgerEntryAmount(rfPayableCOA.getId(), ticketId);
			double leLDFAmount = getLedgerEntryAmount(ldfPayableCOA.getId(), ticketId);

			double previousProcessAmount = leProductSalesAmount + leTaxAmount + leRFAmount + leLDFAmount;
			double remainLedgerAmount = ticket.getTotalAmount() - previousProcessAmount;
			if (!NumberUtil.isZero(remainLedgerAmount)) {

				double maxTaxAmount = ticket.getTaxAmount();
				double maxLabDoctorFee = ticket.getLabDoctorFee();
				double otherAmount = maxTaxAmount + maxRFFee + maxLabDoctorFee;
				double maxSalesAmount = ticket.getTotalAmount() - otherAmount;

				double remainProductSalesAmount = maxSalesAmount - leProductSalesAmount;
				double remainTaxAmount = maxTaxAmount - leTaxAmount;
				double remainRFAmount = maxRFFee - leRFAmount;
				double remainLDFAmount = maxLabDoctorFee - leLDFAmount;

				if (!NumberUtil.isZero(remainProductSalesAmount)) {
					LedgerEntry createPSLedgerEntry = LedgerEntry.createLedgerEntry(ticket, productsSalesCOA, DirectionType.CREDIT, remainProductSalesAmount);
					LedgerEntryDAO.getInstance().save(createPSLedgerEntry, session);
				}

				if (!NumberUtil.isZero(remainTaxAmount)) {
					LedgerEntry createTaxLedgerEntry = LedgerEntry.createLedgerEntry(ticket, taxCOA, DirectionType.CREDIT, remainTaxAmount);
					LedgerEntryDAO.getInstance().save(createTaxLedgerEntry, session);
				}

				if (!NumberUtil.isZero(remainRFAmount)) {
					LedgerEntry createRFLedgerEntry = LedgerEntry.createLedgerEntry(ticket, rfPayableCOA, DirectionType.CREDIT, remainRFAmount);
					LedgerEntryDAO.getInstance().save(createRFLedgerEntry, session);
				}

				if (!NumberUtil.isZero(remainLDFAmount)) {
					LedgerEntry createLDFLedgerEntry = LedgerEntry.createLedgerEntry(ticket, ldfPayableCOA, DirectionType.CREDIT, remainLDFAmount);
					LedgerEntryDAO.getInstance().save(createLDFLedgerEntry, session);
				}

				if (!NumberUtil.isZero(remainLedgerAmount)) {
					LedgerEntry createCreditLedgerEntry = LedgerEntry.createLedgerEntry(ticket, reciableCOA, DirectionType.DEBIT, remainLedgerAmount);
					LedgerEntryDAO.getInstance().save(createCreditLedgerEntry, session);
				}

				if (ticket.isPaid()) {
					ticket.setAccountProcessed(true);
				}
				TicketDAO.getInstance().saveOrUpdate(ticket, session);
			}

			Double transAmount = transaction.getAmount();
			if (transaction instanceof RefundTransaction || transaction.isVoided()) {
				transAmount = (-1) * transAmount;
			}
			if (!NumberUtil.isZero(transAmount)) {
				LedgerEntry createCashLedgerEntry = LedgerEntry.createLedgerEntry(ticket, cashChartOfAccounts, DirectionType.DEBIT, transAmount);
				createCashLedgerEntry.setTransactionId(transaction.getId());
				LedgerEntryDAO.getInstance().save(createCashLedgerEntry, session);

				LedgerEntry createCreditLedgerEntry = LedgerEntry.createLedgerEntry(ticket, reciableCOA, DirectionType.CREDIT, transAmount);
				createCreditLedgerEntry.setTransactionId(transaction.getId());
				LedgerEntryDAO.getInstance().save(createCreditLedgerEntry, session);
			}

			tx.commit();
		} catch (Exception e) {
			PosLog.error(getClass(), e);
		}
	}

	public double getLedgerEntryAmount(String accountId, String ticketId) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(LedgerEntry.class);

			criteria.add(Restrictions.eq(LedgerEntry.PROP_ACCOUNT_ID, accountId));
			criteria.add(Restrictions.eq(LedgerEntry.PROP_TICKET_ID, ticketId));

			ProjectionList projectionList = Projections.projectionList();
			projectionList.add(Projections.sqlProjection("sum(direction*amount) AS " + LedgerEntry.PROP_AMOUNT, new String[] { LedgerEntry.PROP_AMOUNT },
					new Type[] { new DoubleType() }));
			criteria.setProjection(projectionList);
			Number number = (Number) criteria.uniqueResult();
			if (number != null) {
				return number.doubleValue();
			}
		}
		return 0;
	}

}