package com.floreantpos.model.dao;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.lang.StringUtils;
import org.hibernate.Criteria;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.hibernate.criterion.DetachedCriteria;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.ProjectionList;
import org.hibernate.criterion.Projections;
import org.hibernate.criterion.Property;
import org.hibernate.criterion.Restrictions;
import org.hibernate.transform.Transformers;
import org.hibernate.type.DoubleType;
import org.hibernate.type.Type;

import com.floreantpos.PosLog;
import com.floreantpos.model.COAAccountType;
import com.floreantpos.model.ChartOfAccounts;
import com.floreantpos.model.DirectionType;
import com.floreantpos.model.LedgerEntry;
import com.floreantpos.model.PosTransaction;
import com.floreantpos.model.RefundTransaction;
import com.floreantpos.model.Ticket;
import com.floreantpos.model.TicketItem;
import com.floreantpos.report.model.BalanceSheetLedgerEntryModel;
import com.floreantpos.swing.PaginationSupport;
import com.floreantpos.util.NumberUtil;

public class LedgerEntryDAO extends BaseLedgerEntryDAO {

	public void findByChartOfAcountName(PaginationSupport paginationSupport, Date fromDate, Date toDate, String accountId, String orderId) {
		try (Session session = createNewSession()) {

			Criteria criteria = session.createCriteria(LedgerEntry.class);
			addDeletedFilter(criteria);

			if (StringUtils.isNotBlank(orderId)) {
				criteria.add(Restrictions.eq(LedgerEntry.PROP_TICKET_ID, orderId));
			}

			if (fromDate != null) {
				criteria.add(Restrictions.ge(LedgerEntry.PROP_CREATE_DATE, fromDate));
			}
			if (toDate != null) {
				criteria.add(Restrictions.lt(LedgerEntry.PROP_CREATE_DATE, toDate));
			}

			if (StringUtils.isNotBlank(accountId)) {
				criteria.add(Restrictions.eq(LedgerEntry.PROP_ACCOUNT_ID, accountId));
			}

			//			if (StringUtils.isNotBlank(searchString)) {
			//				DetachedCriteria detachedCriteria = DetachedCriteria.forClass(COAAccountType.class);
			//				detachedCriteria.setProjection(Property.forName(COAAccountType.PROP_ID));
			//				detachedCriteria.add(Restrictions.ilike(COAAccountType.PROP_NAME, searchString, MatchMode.ANYWHERE));
			//
			//				criteria.add(Property.forName(LedgerEntry.PROP_ACCOUNT_ID).in(detachedCriteria)); //$NON-NLS-1$
			//			}
			criteria.setProjection(Projections.rowCount());

			Number rowCount = (Number) criteria.uniqueResult();
			if (rowCount != null) {
				paginationSupport.setNumRows(rowCount.intValue());
			}
			criteria.setProjection(null);
			criteria.addOrder(Order.desc(LedgerEntry.PROP_CREATE_DATE));
			paginationSupport.setRows(criteria.list());
		}
	}

	public void saveTicketLedgerEntry(Ticket ticket, double remainLedgerAmount) {
		ChartOfAccountsDAO instance = ChartOfAccountsDAO.getInstance();
		try (Session session = instance.createNewSession()) {
			Transaction tx = session.beginTransaction();

			ChartOfAccounts receivableChartOfAccounts = instance.findByAcountCode("5000", session);
			ChartOfAccounts salesChartOfAccounts = instance.findByAcountCode("200", session);

			LedgerEntry createDebitLedgerEntry = LedgerEntry.createLedgerEntry(ticket, receivableChartOfAccounts, DirectionType.DEBIT, remainLedgerAmount,
					"ORDER/" + ticket.getId());
			LedgerEntryDAO.getInstance().save(createDebitLedgerEntry, session);
			LedgerEntry createCreditLedgerEntry = LedgerEntry.createLedgerEntry(ticket, salesChartOfAccounts, DirectionType.CREDIT, remainLedgerAmount,
					"ORDER/" + ticket.getId());
			LedgerEntryDAO.getInstance().save(createCreditLedgerEntry, session);

			tx.commit();
		} catch (Exception e) {
			PosLog.error(getClass(), e);
		}
	}

	public void saveTransactionLedgerEntry(Ticket ticket, PosTransaction transaction) {
		ChartOfAccountsDAO instance = ChartOfAccountsDAO.getInstance();
		try (Session session = instance.createNewSession()) {
			Transaction tx = session.beginTransaction();

			ChartOfAccounts cashChartOfAccounts = instance.findByAcountCode("10100", session);
			ChartOfAccounts receivableChartOfAccounts = instance.findByAcountCode("5000", session);

			Double transAmount = transaction.isVoided() ? (-1) * transaction.getAmount() : transaction.getAmount();
			LedgerEntry createDebitLedgerEntry = LedgerEntry.createLedgerEntry(ticket, cashChartOfAccounts, DirectionType.DEBIT, transAmount,
					"TRANS/" + transaction.getId());
			createDebitLedgerEntry.setTransactionId(transaction.getId());
			LedgerEntryDAO.getInstance().save(createDebitLedgerEntry, session);
			LedgerEntry createCreditLedgerEntry = LedgerEntry.createLedgerEntry(ticket, receivableChartOfAccounts, DirectionType.CREDIT, transAmount,
					"TRANS/" + transaction.getId());
			createCreditLedgerEntry.setTransactionId(transaction.getId());
			LedgerEntryDAO.getInstance().save(createCreditLedgerEntry, session);

			tx.commit();
		} catch (Exception e) {
			PosLog.error(getClass(), e);
		}
	}

	public void saveOrderLedgerEntry(Ticket ticket, PosTransaction transaction) {
		double maxRFFee = ticket.getReferrerFeeOnNetSales();
		double totalCost = 0;
		List<TicketItem> ticketItems = ticket.getTicketItems();
		if (ticketItems != null) {
			for (TicketItem ticketItem : ticketItems) {
				maxRFFee += ticketItem.getReferrerFeeOnReport();
				if (ticketItem.isVoided() || ticketItem.isReturned() || ticketItem.isReturnedSource()) {
					continue;
				}
				totalCost += ticketItem.getUnitCost();
			}
		}

		ChartOfAccountsDAO instance = ChartOfAccountsDAO.getInstance();
		try (Session session = instance.createNewSession()) {
			Transaction tx = session.beginTransaction();

			ChartOfAccounts productsSalesCOA = instance.findByAcountCode("200", session);
			ChartOfAccounts taxCOA = instance.findByAcountCode("831", session);
			ChartOfAccounts rfPayableCOA = instance.findByAcountCode("832", session);
			ChartOfAccounts ldfPayableCOA = instance.findByAcountCode("833", session);
//			ChartOfAccounts taxCOA = instance.findByAcountCode("506", session);
//			ChartOfAccounts rfPayableCOA = instance.findByAcountCode("507", session);
//			ChartOfAccounts ldfPayableCOA = instance.findByAcountCode("508", session);
			ChartOfAccounts receivableCOA = instance.findByAcountCode("5000", session);
			ChartOfAccounts ownerADrawings = instance.findByAcountCode("880", session);
			ChartOfAccounts cashChartOfAccounts = instance.findByAcountCode("10100", session);

			String ticketId = ticket.getId();
			double leProductSalesAmount = getLedgerEntryAmount(productsSalesCOA.getId(), ticketId);
			double leTaxAmount = getLedgerEntryAmount(taxCOA.getId(), ticketId);
			double leRFAmount = getLedgerEntryAmount(rfPayableCOA.getId(), ticketId);
			double leLDFAmount = getLedgerEntryAmount(ldfPayableCOA.getId(), ticketId);

			double previousProcessAmount = leProductSalesAmount + leTaxAmount + leRFAmount + leLDFAmount;
			double remainLedgerAmount = ticket.getTotalAmount() + previousProcessAmount;
			if (!NumberUtil.isZero(remainLedgerAmount)) {
				double maxTaxAmount = ticket.getTaxAmount();
				double maxLabDoctorFee = ticket.getLabDoctorFee();
				double otherAmount = maxTaxAmount + maxRFFee + maxLabDoctorFee;
				double maxSalesAmount = ticket.getTotalAmount() - otherAmount;
				//remainLedgerAmount = remainLedgerAmount - otherAmount;
				//double maxSalesAmount = ticket.getTotalAmount();
				double remainProductSalesAmount = maxSalesAmount + leProductSalesAmount;
				double remainTaxAmount = maxTaxAmount + leTaxAmount;
				double remainRFAmount = maxRFFee + leRFAmount;
				double remainLDFAmount = maxLabDoctorFee + leLDFAmount;

				String groupId = SequenceNumberDAO.getInstance().getNextLedgerRef("ORDER");

				if (!NumberUtil.isZero(remainProductSalesAmount)) {
					LedgerEntry createPSLedgerEntry = LedgerEntry.createLedgerEntry(ticket, productsSalesCOA, DirectionType.CREDIT, remainProductSalesAmount,
							"ORDER/" + ticket.getId());
					createPSLedgerEntry.setGroupId(groupId);
					LedgerEntryDAO.getInstance().save(createPSLedgerEntry, session);
				}

				if (!NumberUtil.isZero(remainTaxAmount)) {
					LedgerEntry createTaxLedgerEntry = LedgerEntry.createLedgerEntry(ticket, taxCOA, DirectionType.CREDIT, remainTaxAmount,
							"ORDER/" + ticket.getId());
					createTaxLedgerEntry.setGroupId(groupId);
					LedgerEntryDAO.getInstance().save(createTaxLedgerEntry, session);
				}

				if (!NumberUtil.isZero(remainRFAmount)) {
					LedgerEntry createRFLedgerEntry = LedgerEntry.createLedgerEntry(ticket, rfPayableCOA, DirectionType.CREDIT, remainRFAmount,
							"ORDER/" + ticket.getId());
					createRFLedgerEntry.setGroupId(groupId);
					LedgerEntryDAO.getInstance().save(createRFLedgerEntry, session);
				}

				if (!NumberUtil.isZero(remainLDFAmount)) {
					LedgerEntry createLDFLedgerEntry = LedgerEntry.createLedgerEntry(ticket, ldfPayableCOA, DirectionType.CREDIT, remainLDFAmount,
							"ORDER/" + ticket.getId());
					createLDFLedgerEntry.setGroupId(groupId);
					LedgerEntryDAO.getInstance().save(createLDFLedgerEntry, session);
				}

				if (!NumberUtil.isZero(remainLedgerAmount)) {
					LedgerEntry createDebitLedgerEntry = LedgerEntry.createLedgerEntry(ticket, receivableCOA, DirectionType.DEBIT, remainLedgerAmount,
							"ORDER/" + ticket.getId());
					createDebitLedgerEntry.setGroupId(groupId);
					LedgerEntryDAO.getInstance().save(createDebitLedgerEntry, session);
				}

				if (ticket.isPaid()) {
					ticket.setAccountProcessed(true);
				}
				TicketDAO.getInstance().saveOrUpdate(ticket, session);
				saveInventoryAdjLedgerEntry(instance, session, groupId, ticket, totalCost);
			}

			Double transAmount = transaction.getAmount();
			if (transaction instanceof RefundTransaction || transaction.isVoided()) {
				transAmount = (-1) * transAmount;
			}
			if (!NumberUtil.isZero(transAmount)) {
				String groupId = SequenceNumberDAO.getInstance().getNextLedgerRef("TRANS");
				LedgerEntry createCashLedgerEntry = LedgerEntry.createLedgerEntry(ticket, cashChartOfAccounts, DirectionType.CREDIT, transAmount,
						"TRANS/" + transaction.getId());
				createCashLedgerEntry.setTransactionId(transaction.getId());
				createCashLedgerEntry.setGroupId(groupId);
				LedgerEntryDAO.getInstance().save(createCashLedgerEntry, session);

				LedgerEntry createReceivableTransLedgerEntry = LedgerEntry.createLedgerEntry(ticket, ownerADrawings, DirectionType.DEBIT, transAmount,
						"TRANS/" + transaction.getId());
				createReceivableTransLedgerEntry.setTransactionId(transaction.getId());
				createReceivableTransLedgerEntry.setGroupId(groupId);
				LedgerEntryDAO.getInstance().save(createReceivableTransLedgerEntry, session);
			}

			tx.commit();
		} catch (Exception e) {
			PosLog.error(getClass(), e);
		}
	}

	public void saveInventoryAdjLedgerEntry(ChartOfAccountsDAO instance, Session session, String groupId, Ticket ticket, double totalCost) {
		ChartOfAccounts costOfGoodsSoldCOA = instance.findByAcountCode("310", session);
		ChartOfAccounts inventoryCOA = instance.findByAcountCode("630", session);

		double leCostOfGoodsSoldAmount = getLedgerEntryAmount(costOfGoodsSoldCOA.getId(), ticket.getId());
		double remainCostOfGoodsSoldAmount = totalCost - leCostOfGoodsSoldAmount;

		if (!NumberUtil.isZero(remainCostOfGoodsSoldAmount)) {
			LedgerEntry createCostOfGoodsSoldLedgerEntry = LedgerEntry.createLedgerEntry(ticket, costOfGoodsSoldCOA, DirectionType.DEBIT,
					remainCostOfGoodsSoldAmount, "ORDER/" + ticket.getId());
			createCostOfGoodsSoldLedgerEntry.setGroupId(groupId);
			LedgerEntryDAO.getInstance().save(createCostOfGoodsSoldLedgerEntry, session);

			LedgerEntry createInventoryLedgerEntry = LedgerEntry.createLedgerEntry(ticket, inventoryCOA, DirectionType.CREDIT, remainCostOfGoodsSoldAmount,
					"ORDER/" + ticket.getId());
			createInventoryLedgerEntry.setGroupId(groupId);
			LedgerEntryDAO.getInstance().save(createInventoryLedgerEntry, session);
		}
	}

	public double getLedgerEntryAmount(String accountId, String ticketId) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(LedgerEntry.class);

			criteria.add(Restrictions.eq(LedgerEntry.PROP_ACCOUNT_ID, accountId));
			criteria.add(Restrictions.eq(LedgerEntry.PROP_TICKET_ID, ticketId));

			ProjectionList projectionList = Projections.projectionList();
			projectionList.add(Projections.sqlProjection("sum(direction*amount) AS " + LedgerEntry.PROP_AMOUNT, new String[] { LedgerEntry.PROP_AMOUNT },
					new Type[] { new DoubleType() }));
			criteria.setProjection(projectionList);
			Number number = (Number) criteria.uniqueResult();
			if (number != null) {
				return number.doubleValue();
			}
		}
		return 0;
	}

	public void saveRFPayLedgerEntry(Map<Ticket, Double> ticketsMap, String transactionId, Session session) {
		Set<Ticket> keySet = ticketsMap.keySet();
		for (Ticket ticket : keySet) {
			ChartOfAccountsDAO instance = ChartOfAccountsDAO.getInstance();

			ChartOfAccounts cashChartOfAccounts = instance.findByAcountCode("10100", session);
			//ChartOfAccounts rfPayableCOA = instance.findByAcountCode("507", session);
			ChartOfAccounts rfPayableCOA = instance.findByAcountCode("832", session);

			double transAmount = ticketsMap.get(ticket);
			String groupId = SequenceNumberDAO.getInstance().getNextLedgerRef("RF");

			LedgerEntry createCashLedgerEntry = LedgerEntry.createLedgerEntry(ticket, cashChartOfAccounts, DirectionType.CREDIT, transAmount,
					"TRANS/" + transactionId);
			createCashLedgerEntry.setTransactionId(transactionId);
			createCashLedgerEntry.setGroupId(groupId);
			LedgerEntryDAO.getInstance().save(createCashLedgerEntry, session);
			LedgerEntry createRFLedgerEntry = LedgerEntry.createLedgerEntry(ticket, rfPayableCOA, DirectionType.DEBIT, transAmount, "TRANS/" + transactionId);
			createRFLedgerEntry.setTransactionId(transactionId);
			createRFLedgerEntry.setGroupId(groupId);
			LedgerEntryDAO.getInstance().save(createRFLedgerEntry, session);
		}

	}

	public void saveLDFPayLedgerEntry(Map<Ticket, Double> ticketsMap, String transactionId, Session session) {
		Set<Ticket> keySet = ticketsMap.keySet();
		for (Ticket ticket : keySet) {
			ChartOfAccountsDAO instance = ChartOfAccountsDAO.getInstance();

			ChartOfAccounts cashChartOfAccounts = instance.findByAcountCode("10100", session);
			//ChartOfAccounts ldfPayableCOA = instance.findByAcountCode("508", session);
			ChartOfAccounts ldfPayableCOA = instance.findByAcountCode("833", session);

			double transAmount = ticketsMap.get(ticket);
			String groupId = SequenceNumberDAO.getInstance().getNextLedgerRef("LDF");

			LedgerEntry createCashLedgerEntry = LedgerEntry.createLedgerEntry(ticket, cashChartOfAccounts, DirectionType.CREDIT, transAmount,
					"TRANS/" + transactionId);
			createCashLedgerEntry.setTransactionId(transactionId);
			createCashLedgerEntry.setGroupId(groupId);
			LedgerEntryDAO.getInstance().save(createCashLedgerEntry, session);
			LedgerEntry createLDFPayLedgerEntry = LedgerEntry.createLedgerEntry(ticket, ldfPayableCOA, DirectionType.DEBIT, transAmount,
					"TRANS/" + transactionId);
			createLDFPayLedgerEntry.setTransactionId(transactionId);
			createLDFPayLedgerEntry.setGroupId(groupId);
			LedgerEntryDAO.getInstance().save(createLDFPayLedgerEntry, session);
		}
	}

	public List getLedgerEntries2(List<String> coaAccountTypeGroupIds) {
		try (Session session = createNewSession()) {

			DetachedCriteria coaTypeDetachedCriteria = DetachedCriteria.forClass(COAAccountType.class);
			coaTypeDetachedCriteria.setProjection(Property.forName(COAAccountType.PROP_ID));
			if (!coaAccountTypeGroupIds.isEmpty()) {
				coaTypeDetachedCriteria.add(Restrictions.in(COAAccountType.PROP_COA_ACCOUNT_TYPE_GROUP_ID, coaAccountTypeGroupIds));
			}

			DetachedCriteria coaDetachedCriteria = DetachedCriteria.forClass(ChartOfAccounts.class);
			coaDetachedCriteria.setProjection(Property.forName(ChartOfAccounts.PROP_ID));
			coaDetachedCriteria.add(Property.forName(ChartOfAccounts.PROP_COA_ACCOUNT_TYPE_ID).in(coaTypeDetachedCriteria)); //$NON-NLS-1$

			Criteria criteria = session.createCriteria(LedgerEntry.class);

			ProjectionList projectionList = Projections.projectionList();
			projectionList.add(Projections.property(LedgerEntry.PROP_ACCOUNT_ID), "coaId");
			projectionList.add(Projections.sqlProjection("sum(direction*amount) AS " + "amount", new String[] { "amount" }, new Type[] { new DoubleType() }));

			projectionList.add(Projections.groupProperty(LedgerEntry.PROP_ACCOUNT_ID));
			criteria.setProjection(projectionList);

			criteria.add(Property.forName(LedgerEntry.PROP_ACCOUNT_ID).in(coaDetachedCriteria)); //$NON-NLS-1$

			criteria.setResultTransformer(Transformers.aliasToBean(BalanceSheetLedgerEntryModel.class));
			return criteria.list();

		}
	}

	public List getLedgerEntries(List<String> coaAccountTypeGroupIds) {
		try (Session session = createNewSession()) {

			DetachedCriteria coaTypeDetachedCriteria = DetachedCriteria.forClass(COAAccountType.class);
			coaTypeDetachedCriteria.setProjection(Property.forName(COAAccountType.PROP_ID));
			if (!coaAccountTypeGroupIds.isEmpty()) {
				coaTypeDetachedCriteria.add(Restrictions.in(COAAccountType.PROP_COA_ACCOUNT_TYPE_GROUP_ID, coaAccountTypeGroupIds));
			}

			DetachedCriteria coaDetachedCriteria = DetachedCriteria.forClass(ChartOfAccounts.class);
			coaDetachedCriteria.setProjection(Property.forName(ChartOfAccounts.PROP_ID));
			coaDetachedCriteria.add(Property.forName(ChartOfAccounts.PROP_COA_ACCOUNT_TYPE_ID).in(coaTypeDetachedCriteria)); //$NON-NLS-1$

			Criteria criteria = session.createCriteria(LedgerEntry.class);

			ProjectionList projectionList = Projections.projectionList();
			projectionList.add(Projections.property(LedgerEntry.PROP_ACCOUNT_ID), "coaId");
			projectionList.add(Projections.sqlProjection("sum(direction*amount) AS " + "amount", new String[] { "amount" }, new Type[] { new DoubleType() }));

			projectionList.add(Projections.groupProperty(LedgerEntry.PROP_ACCOUNT_ID));
			criteria.setProjection(projectionList);

			criteria.add(Property.forName(LedgerEntry.PROP_ACCOUNT_ID).in(coaDetachedCriteria)); //$NON-NLS-1$

			criteria.setResultTransformer(Transformers.aliasToBean(BalanceSheetLedgerEntryModel.class));
			return criteria.list();

		}
	}

	public BalanceSheetLedgerEntryModel getRetainedEarning() {
		try (Session session = createNewSession()) {
			List<String> coaAccountTypeGroupIds = Arrays.asList("revenue", "expense");

			DetachedCriteria coaTypeDetachedCriteria = DetachedCriteria.forClass(COAAccountType.class);
			coaTypeDetachedCriteria.setProjection(Property.forName(COAAccountType.PROP_ID));
			if (!coaAccountTypeGroupIds.isEmpty()) {
				coaTypeDetachedCriteria.add(Restrictions.in(COAAccountType.PROP_COA_ACCOUNT_TYPE_GROUP_ID, coaAccountTypeGroupIds));
			}

			DetachedCriteria coaDetachedCriteria = DetachedCriteria.forClass(ChartOfAccounts.class);
			coaDetachedCriteria.setProjection(Property.forName(ChartOfAccounts.PROP_ID));
			coaDetachedCriteria.add(Property.forName(ChartOfAccounts.PROP_COA_ACCOUNT_TYPE_ID).in(coaTypeDetachedCriteria)); //$NON-NLS-1$

			Criteria criteria = session.createCriteria(LedgerEntry.class);

			ProjectionList projectionList = Projections.projectionList();
			projectionList.add(Projections.sqlProjection("sum(direction*amount) AS " + "amount", new String[] { "amount" }, new Type[] { new DoubleType() }));
			criteria.setProjection(projectionList);

			criteria.add(Property.forName(LedgerEntry.PROP_ACCOUNT_ID).in(coaDetachedCriteria)); //$NON-NLS-1$
			BalanceSheetLedgerEntryModel retainedEarning = new BalanceSheetLedgerEntryModel();
			retainedEarning.setName("Retained earning");
			Number number = (Number) criteria.uniqueResult();
			if (number != null) {
				retainedEarning.setAmount(number.doubleValue());
			}
			return retainedEarning;

		}
	}

	public List getSumOfLedgerEntries(Date startDate, Date endDate) {
		try (Session session = createNewSession()) {
			Map<String, BalanceSheetLedgerEntryModel> map = new HashMap<String, BalanceSheetLedgerEntryModel>();
			Criteria criteria = session.createCriteria(LedgerEntry.class);

			if (startDate != null) {
				criteria.add(Restrictions.ge(LedgerEntry.PROP_CREATE_DATE, startDate));
			}
			if (endDate != null) {
				criteria.add(Restrictions.lt(LedgerEntry.PROP_CREATE_DATE, endDate));
			}

			ProjectionList projectionList = Projections.projectionList();
			projectionList.add(Projections.property(LedgerEntry.PROP_ACCOUNT_ID), "coaId");
			projectionList.add(Projections.property(LedgerEntry.PROP_DIRECTION), LedgerEntry.PROP_DIRECTION);
			projectionList.add(Projections.sqlProjection("sum(amount) AS " + "amount", new String[] { "amount" }, new Type[] { new DoubleType() }));

			projectionList.add(Projections.groupProperty(LedgerEntry.PROP_ACCOUNT_ID));
			projectionList.add(Projections.groupProperty(LedgerEntry.PROP_DIRECTION));
			criteria.setProjection(projectionList);

			criteria.setResultTransformer(Transformers.aliasToBean(BalanceSheetLedgerEntryModel.class));

			List<BalanceSheetLedgerEntryModel> list = criteria.list();
			if (list != null) {

				for (BalanceSheetLedgerEntryModel entryModel : list) {
					String coaId = entryModel.getCoaId();

					BalanceSheetLedgerEntryModel model = map.get(coaId);
					if (model == null) {
						model = new BalanceSheetLedgerEntryModel();
						model.setCoaId(coaId);
						model.setDirection(null);
						map.put(coaId, model);
					}

					DirectionType directionType = DirectionType.getByTypeNo(entryModel.getDirection());
					if (DirectionType.DEBIT == directionType) {
						model.setDebitAmount(entryModel.getAmount());
					}
					else {
						model.setCreditAmount(entryModel.getAmount());
					}
				}

			}
			return new ArrayList(map.values());

		}
	}

	public double getTaxLedgerEntryAmount(Date startDate, Date endDate) {
		return getLedgerEntriesAmountByTypeId("expense", "506", startDate, endDate);
	}

	public double getLedgerEntriesAmountByTypeId(String coaAccountTypeId, Date startDate, Date endDate) {
		return getLedgerEntriesAmountByTypeId(coaAccountTypeId, "", startDate, endDate);
	}

	private double getLedgerEntriesAmountByTypeId(String coaAccountTypeId, String coaCode, Date startDate, Date endDate) {
		try (Session session = createNewSession()) {

			DetachedCriteria coaDetachedCriteria = DetachedCriteria.forClass(ChartOfAccounts.class);
			coaDetachedCriteria.setProjection(Property.forName(ChartOfAccounts.PROP_ID));
			if (StringUtils.isNotBlank(coaAccountTypeId)) {
				coaDetachedCriteria.add(Restrictions.eq(ChartOfAccounts.PROP_COA_ACCOUNT_TYPE_ID, coaAccountTypeId));
			}
			if (StringUtils.isNotBlank(coaCode)) {
				coaDetachedCriteria.add(Restrictions.eq(ChartOfAccounts.PROP_ACCOUNT_CODE, coaCode));
			}
			Criteria criteria = session.createCriteria(LedgerEntry.class);

			if (startDate != null) {
				criteria.add(Restrictions.ge(LedgerEntry.PROP_CREATE_DATE, startDate));
			}
			if (endDate != null) {
				criteria.add(Restrictions.lt(LedgerEntry.PROP_CREATE_DATE, endDate));
			}

			ProjectionList projectionList = Projections.projectionList();
			projectionList.add(Projections.sqlProjection("sum(direction*amount) AS " + "amount", new String[] { "amount" }, new Type[] { new DoubleType() }));
			criteria.setProjection(projectionList);

			criteria.add(Property.forName(LedgerEntry.PROP_ACCOUNT_ID).in(coaDetachedCriteria)); //$NON-NLS-1$

			Number number = (Number) criteria.uniqueResult();
			if (number != null) {
				return number.doubleValue();
			}

		}
		return 0;
	}

	/**
	 * @param startDate
	 * @param endDate
	 *  find all account type IDs with group id "expense"
	 *  find all ChartOfAccounts IDs with COAAccountType IDs without "direct_costs" account type
	 * 
	 * @return Expense amount by account type without cost
	 */
	public List getExpenseLedgerEntries(Date startDate, Date endDate) {
		return getCustomLedgerEntries(startDate, endDate, Arrays.asList("expense"), "direct_costs", "506");
	}

	public List getCustomLedgerEntries(Date startDate, Date endDate, List<String> accountTypeGroupIds, String withoutAccountTypeId, String withoutAccountCode) {
		return getCustomLedgerEntries(startDate, endDate, accountTypeGroupIds, Arrays.asList(withoutAccountTypeId), withoutAccountCode);
	}

	public List getCustomLedgerEntries(Date startDate, Date endDate, List<String> accountTypeGroupIds, List<String> withoutAccountTypeIds,
			String withoutAccountCode) {
		try (Session session = createNewSession()) {

			DetachedCriteria coaTypeDetachedCriteria = DetachedCriteria.forClass(COAAccountType.class);
			coaTypeDetachedCriteria.setProjection(Property.forName(COAAccountType.PROP_ID));

			if (accountTypeGroupIds != null) {
				coaTypeDetachedCriteria.add(Restrictions.in(COAAccountType.PROP_COA_ACCOUNT_TYPE_GROUP_ID, accountTypeGroupIds));
			}

			DetachedCriteria coaDetachedCriteria = DetachedCriteria.forClass(ChartOfAccounts.class);
			coaDetachedCriteria.setProjection(Property.forName(ChartOfAccounts.PROP_ID));
			coaDetachedCriteria.add(Property.forName(ChartOfAccounts.PROP_COA_ACCOUNT_TYPE_ID).in(coaTypeDetachedCriteria)); //$NON-NLS-1$
			if (!withoutAccountTypeIds.isEmpty()) {
				coaDetachedCriteria.add(Restrictions.not(Restrictions.in(ChartOfAccounts.PROP_COA_ACCOUNT_TYPE_ID, withoutAccountTypeIds)));
			}
			if (StringUtils.isNotBlank(withoutAccountCode)) {
				coaDetachedCriteria.add(Restrictions.ne(ChartOfAccounts.PROP_ACCOUNT_CODE, withoutAccountCode));
			}
			Criteria criteria = session.createCriteria(LedgerEntry.class);

			if (startDate != null) {
				criteria.add(Restrictions.ge(LedgerEntry.PROP_CREATE_DATE, startDate));
			}
			if (endDate != null) {
				criteria.add(Restrictions.lt(LedgerEntry.PROP_CREATE_DATE, endDate));
			}

			ProjectionList projectionList = Projections.projectionList();
			projectionList.add(Projections.property(LedgerEntry.PROP_ACCOUNT_ID), "coaId");
			projectionList.add(Projections.sqlProjection("sum(direction*amount) AS " + "amount", new String[] { "amount" }, new Type[] { new DoubleType() }));

			projectionList.add(Projections.groupProperty(LedgerEntry.PROP_ACCOUNT_ID));
			criteria.setProjection(projectionList);

			criteria.add(Property.forName(LedgerEntry.PROP_ACCOUNT_ID).in(coaDetachedCriteria)); //$NON-NLS-1$

			criteria.setResultTransformer(Transformers.aliasToBean(BalanceSheetLedgerEntryModel.class));
			return criteria.list();

		}
	}

	public List getLedgerEntriesByTypeId(String coaAccountTypeId, Date startDate, Date endDate) {
		return getLedgerEntriesByTypeId(Arrays.asList(coaAccountTypeId), startDate, endDate);
	}

	public List getLedgerEntriesByTypeId(List<String> coaAccountTypeId, Date startDate, Date endDate) {
		try (Session session = createNewSession()) {

			DetachedCriteria coaDetachedCriteria = DetachedCriteria.forClass(ChartOfAccounts.class);
			coaDetachedCriteria.setProjection(Property.forName(ChartOfAccounts.PROP_ID));
			coaDetachedCriteria.add(Property.forName(ChartOfAccounts.PROP_COA_ACCOUNT_TYPE_ID).in(coaAccountTypeId)); //$NON-NLS-1$
			Criteria criteria = session.createCriteria(LedgerEntry.class);

			if (startDate != null) {
				criteria.add(Restrictions.ge(LedgerEntry.PROP_CREATE_DATE, startDate));
			}
			if (endDate != null) {
				criteria.add(Restrictions.lt(LedgerEntry.PROP_CREATE_DATE, endDate));
			}

			ProjectionList projectionList = Projections.projectionList();
			projectionList.add(Projections.property(LedgerEntry.PROP_ACCOUNT_ID), "coaId");
			projectionList.add(Projections.sqlProjection("sum(direction*amount) AS " + "amount", new String[] { "amount" }, new Type[] { new DoubleType() }));

			projectionList.add(Projections.groupProperty(LedgerEntry.PROP_ACCOUNT_ID));
			criteria.setProjection(projectionList);

			criteria.add(Property.forName(LedgerEntry.PROP_ACCOUNT_ID).in(coaDetachedCriteria)); //$NON-NLS-1$

			criteria.setResultTransformer(Transformers.aliasToBean(BalanceSheetLedgerEntryModel.class));
			criteria.addOrder(Order.desc("coaId"));
			return criteria.list();

		}
	}

}