package com.floreantpos.model.dao;

import java.io.Serializable;
import java.util.List;

import org.apache.commons.lang.StringUtils;
import org.hibernate.Criteria;
import org.hibernate.Session;
import org.hibernate.criterion.Criterion;
import org.hibernate.criterion.MatchMode;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.Projections;
import org.hibernate.criterion.Restrictions;

import com.floreantpos.Messages;
import com.floreantpos.PosException;
import com.floreantpos.model.ChartOfAccounts;
import com.floreantpos.swing.PaginationSupport;

public class ChartOfAccountsDAO extends BaseChartOfAccountsDAO {

	public ChartOfAccountsDAO() {
	}

	@Override
	protected void delete(Object obj, Session session) {
		ChartOfAccounts chartOfAccounts = (ChartOfAccounts) obj;
		if (chartOfAccounts == null) {
			throw new PosException("Chart of accounts is not found");
		}

		chartOfAccounts.setDeleted(Boolean.TRUE);
		saveOrUpdate(chartOfAccounts, session);
	}

	@Override
	protected Serializable save(Object obj, Session s) {
		doCheckValidation(obj);
		updateTime(obj);
		return super.save(obj, s);
	}

	@Override
	protected void update(Object obj, Session s) {
		doCheckValidation(obj);
		updateTime(obj);
		super.update(obj, s);
	}

	@Override
	protected void saveOrUpdate(Object obj, Session s) {
		doCheckValidation(obj);
		updateTime(obj);
		super.saveOrUpdate(obj, s);
	}

	protected void doCheckValidation(Object obj) {
		ChartOfAccounts chartOfAccounts = (ChartOfAccounts) obj;

		String accountName = chartOfAccounts.getName();
		if (StringUtils.isNotBlank(accountName) && accountName.length() > 255) {
			throw new PosException(String.format(Messages.getString("value_too_long"), "account name", 255)); //$NON-NLS-1$
		}

		String accountCode = chartOfAccounts.getAccountCode();
		if (StringUtils.isNotBlank(accountCode) && accountCode.length() > 255) {
			throw new PosException(String.format(Messages.getString("value_too_long"), "account code", 255)); //$NON-NLS-1$
		}

		String description = chartOfAccounts.getDescription();
		if (StringUtils.isNotBlank(description) && description.length() > 512) {
			throw new PosException(String.format(Messages.getString("value_too_long"), "description", 512)); //$NON-NLS-1$
		}
	}

	@Override
	public List<ChartOfAccounts> findAll() {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(ChartOfAccounts.class);
			addDeletedFilter(criteria);
			criteria.addOrder(Order.asc(ChartOfAccounts.PROP_COA_ACCOUNT_TYPE_ID));
			criteria.addOrder(Order.asc(ChartOfAccounts.PROP_NAME));
			return criteria.list();
		}
	}

	public void findByAcountNameOrCode(PaginationSupport paginationSupport, String searchString) {
		Session session = null;
		try {
			session = createNewSession();
			Criteria criteria = session.createCriteria(ChartOfAccounts.class);
			addDeletedFilter(criteria);

			if (StringUtils.isNotEmpty(searchString)) {
				Criterion name = Restrictions.ilike(ChartOfAccounts.PROP_NAME, searchString.trim(), MatchMode.ANYWHERE);
				Criterion barcode = Restrictions.ilike(ChartOfAccounts.PROP_ACCOUNT_CODE, searchString.trim(), MatchMode.ANYWHERE);
				criteria.add(Restrictions.or(name, barcode));
			}
			criteria.setProjection(Projections.rowCount());

			Number rowCount = (Number) criteria.uniqueResult();
			if (rowCount != null) {
				paginationSupport.setNumRows(rowCount.intValue());
			}
			criteria.setProjection(null);
			criteria.addOrder(Order.asc(ChartOfAccounts.PROP_COA_ACCOUNT_TYPE_ID).ignoreCase());
			criteria.addOrder(Order.asc(ChartOfAccounts.PROP_NAME).ignoreCase());
			criteria.setFirstResult(paginationSupport.getCurrentRowIndex());
			criteria.setMaxResults(paginationSupport.getPageSize());
			paginationSupport.setRows(criteria.list());
		} finally {
			closeSession(session);
		}
	}

	public ChartOfAccounts findByAcountCode(String accountCode, Session session) {
		if (StringUtils.isEmpty(accountCode)) {
			return null;
		}
		Criteria criteria = session.createCriteria(ChartOfAccounts.class);
		criteria.add(Restrictions.eq(ChartOfAccounts.PROP_ACCOUNT_CODE, accountCode.trim()));
		criteria.setMaxResults(1);
		return (ChartOfAccounts) criteria.uniqueResult();
	}

}