package com.floreantpos.model.dao;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Date;
import java.util.Iterator;
import java.util.List;

import org.apache.commons.lang3.StringUtils;
import org.hibernate.Criteria;
import org.hibernate.Hibernate;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.hibernate.criterion.Projections;
import org.hibernate.criterion.Restrictions;

import com.floreantpos.PosLog;
import com.floreantpos.model.InventoryTransaction;
import com.floreantpos.model.InventoryTransactionType;
import com.floreantpos.model.MenuItem;
import com.floreantpos.model.StockCount;
import com.floreantpos.model.StockCountItem;
import com.orocube.rest.service.server.BaseDataServiceDao;

public class StockCountDAO extends BaseStockCountDAO {

	/**
	 * Default constructor.  Can be used in place of getInstance()
	 */
	public StockCountDAO() {
	}

	@Override
	protected Serializable save(Object obj, Session s) {
		updateTime(obj);
		return super.save(obj, s);
	}

	@Override
	protected void update(Object obj, Session s) {
		updateTime(obj);
		super.update(obj, s);
	}

	@Override
	protected void saveOrUpdate(Object obj, Session s) {
		updateTime(obj);
		super.saveOrUpdate(obj, s);
	}

	public void saveOrUpdate(StockCount stockCount, boolean updateStock) {
		Session session = null;
		Transaction tx = null;
		try {
			session = createNewSession();
			tx = session.beginTransaction();

			saveOrUpdate(stockCount, updateStock, session);
			tx.commit();
		} catch (Exception e) {
			if (tx != null) {
				tx.rollback();
			}
			throwException(e);
		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

	public void saveOrUpdate(StockCount stockCount, boolean updateStock, Session session) {
		if (stockCount.getId() == null) {
			stockCount.setCreatedDate(new Date());
		}
		stockCount.setLastModifiedDate(new Date());
		if (updateStock) {
			adjustInventoryItems(session, stockCount);
		}
		session.saveOrUpdate(stockCount);
	}

	private void adjustInventoryItems(Session session, StockCount stockCount) {
		List<StockCountItem> orderItems = stockCount.getCountItems();
		if (orderItems == null) {
			return;
		}
		for (StockCountItem orderItem : orderItems) {
			if (orderItem.isAdjusted())
				continue;
			orderItem.setAdjusted(Boolean.TRUE);

			Double stockDifference = orderItem.getActualUnit() - orderItem.getUnitOnHand();

			if (stockDifference == 0)
				continue;

			InventoryTransaction stockInTrans = new InventoryTransaction();
			MenuItem menuItem = MenuItemDAO.getInstance().get(orderItem.getItemId(), session);
			if (menuItem == null) {
				menuItem = MenuItemDAO.getInstance().getMenuItemBySKU(orderItem.getSku(), session, true);
				if (menuItem != null) {
					orderItem.setItemId(menuItem.getId());
				}
			}
			Hibernate.initialize(menuItem.getStockUnits());
			stockInTrans.setMenuItem(menuItem);
			stockInTrans.setQuantity(Math.abs(stockDifference));
			stockInTrans.setUnit(orderItem.getUnit());
			double baseUnitQuantity = menuItem.getBaseUnitQuantity(orderItem.getUnit());
			stockInTrans.setUnitCost(baseUnitQuantity * menuItem.getCost());
			stockInTrans.setTotal(baseUnitQuantity * stockInTrans.getQuantity() * menuItem.getCost());
			stockInTrans.setTransactionDate(new Date());
			String reason = orderItem.getReason();

			if (StringUtils.isEmpty(reason)) {
				String strReason = (stockDifference > 0) ? InventoryTransaction.REASON_ADJUST_IN : InventoryTransaction.REASON_ADJUST_OUT;
				stockInTrans.setReason(strReason);
			}

			else {
				stockInTrans.setReason(reason);
			}

			if (stockDifference > 0) {
				stockInTrans.setType(InventoryTransactionType.IN.getType());
				stockInTrans.setToInventoryLocation(orderItem.getInventoryLocation());
			}
			else {
				stockInTrans.setType(InventoryTransactionType.OUT.getType());
				stockInTrans.setFromInventoryLocation(orderItem.getInventoryLocation());
			}
			InventoryTransactionDAO.getInstance().adjustInventoryStock(stockInTrans, session);
		}
	}

	public String getNextStockCountSequenceNumber() {
		Session session = null;
		try {
			session = getSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.setProjection(Projections.max(StockCount.PROP_REF_NUMBER));
			Object maxNumber = criteria.uniqueResult();
			if (maxNumber == null)
				return "10001";
			try {
				return String.valueOf(Integer.parseInt((String) maxNumber) + 1);
			} catch (Exception e) {
				return "";
			}
		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

	public List findBy(String ref, Date from, Date to, Boolean showVerified) {
		Session session = null;
		try {
			session = getSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			if (StringUtils.isNotEmpty(ref)) {
				criteria.add(Restrictions.ilike(StockCount.PROP_REF_NUMBER, ref));
			}
			if (from != null && to != null) {
				criteria.add(Restrictions.ge(StockCount.PROP_CREATED_DATE, from));
				criteria.add(Restrictions.le(StockCount.PROP_CREATED_DATE, to));
			}
			if (!showVerified) {
				criteria.add(Restrictions.isNull(StockCount.PROP_VERIFIED_BY));
			}
			return criteria.list();
		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

	//download and upload
	public void saveOrUpdateStockCounts(List<StockCount> dataList, boolean updateLastUpdateTime, boolean updateSyncTime) throws Exception {
		if (dataList == null)
			return;

		Transaction tx = null;
		try (Session session = createNewSession()) {
			tx = session.beginTransaction();
			for (Iterator<StockCount> iterator = dataList.iterator(); iterator.hasNext();) {
				StockCount item = (StockCount) iterator.next();
				List<StockCountItem> items = item.getCountItems();
				item.setCountItems(null);
				item.setUpdateLastUpdateTime(updateLastUpdateTime);
				item.setUpdateSyncTime(updateSyncTime);

				StockCountDAO dao = StockCountDAO.getInstance();
				StockCount existingItem = dao.get(item.getId());
				if (existingItem != null) {
					if (!BaseDataServiceDao.get().shouldSave(item.getLastUpdateTime(), existingItem.getLastUpdateTime())) {
						PosLog.info(getClass(), item.getId() + " already updated"); //$NON-NLS-1$
						continue;
					}
					long version = existingItem.getVersion();
					item.setVersion(version);
				}
				else {
					save(item, session);
				}
				List<StockCountItem> existingItems = null;
				if (existingItem != null && existingItem.getCountItems() != null) {
					existingItems = existingItem.getCountItems();
				}
				if (existingItems == null) {
					existingItems = new ArrayList<>();
				}
				if (items != null && items.size() > 0) {
					for (StockCountItem stockCountItem : items) {
						stockCountItem.setStockCount(item);
						int idx = existingItems.indexOf(stockCountItem);
						StockCountItem existingStockCountItem = null;
						if (idx != -1) {
							existingStockCountItem = existingItems.get(idx);
							if (existingStockCountItem == null) {
								stockCountItem.setVersion(0);
								StockCountItemDAO.getInstance().save(stockCountItem, session);
							}
							else {
								stockCountItem.setVersion(existingStockCountItem.getVersion());
							}
						}
						else {
							stockCountItem.setVersion(0);
							StockCountItemDAO.getInstance().save(stockCountItem, session);
						}
					}
				}
				item.setCountItems(items);
				saveOrUpdate(item, session);
			}
			tx.commit();
		} catch (Exception e) {
			tx.rollback();
			throw e;
		}
	}

}