/**
 * ************************************************************************
 * * The contents of this file are subject to the MRPL 1.2
 * * (the  "License"),  being   the  Mozilla   Public  License
 * * Version 1.1  with a permitted attribution clause; you may not  use this
 * * file except in compliance with the License. You  may  obtain  a copy of
 * * the License at http://www.floreantpos.org/license.html
 * * Software distributed under the License  is  distributed  on  an "AS IS"
 * * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * * License for the specific  language  governing  rights  and  limitations
 * * under the License.
 * * The Original Code is FLOREANT POS.
 * * The Initial Developer of the Original Code is OROCUBE LLC
 * * All portions are Copyright (C) 2015 OROCUBE LLC
 * * All Rights Reserved.
 * ************************************************************************
 */
package com.floreantpos.model.dao;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import org.apache.commons.beanutils.PropertyUtils;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.hibernate.Criteria;
import org.hibernate.Query;
import org.hibernate.Session;
import org.hibernate.StaleStateException;
import org.hibernate.Transaction;
import org.hibernate.criterion.Junction;
import org.hibernate.criterion.MatchMode;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.ProjectionList;
import org.hibernate.criterion.Projections;
import org.hibernate.criterion.Restrictions;
import org.hibernate.transform.ResultTransformer;
import org.hibernate.transform.Transformers;

import com.floreantpos.Messages;
import com.floreantpos.POSConstants;
import com.floreantpos.PosException;
import com.floreantpos.PosLog;
import com.floreantpos.constants.AppConstants;
import com.floreantpos.main.Application;
import com.floreantpos.model.ActionHistory;
import com.floreantpos.model.AttendenceHistory;
import com.floreantpos.model.CashDrawer;
import com.floreantpos.model.DrawerType;
import com.floreantpos.model.EmployeeInOutHistory;
import com.floreantpos.model.Outlet;
import com.floreantpos.model.PosTransaction;
import com.floreantpos.model.Shift;
import com.floreantpos.model.Terminal;
import com.floreantpos.model.Ticket;
import com.floreantpos.model.User;
import com.floreantpos.model.UserPermission;
import com.floreantpos.model.UserType;
import com.floreantpos.model.util.DataProvider;
import com.floreantpos.model.util.DateUtil;
import com.floreantpos.services.report.CashDrawerReportService;
import com.floreantpos.swing.PaginationSupport;
import com.floreantpos.util.AESencrp;
import com.floreantpos.util.StoreUtil;
import com.floreantpos.util.UserNotFoundException;

public class UserDAO extends BaseUserDAO {
	public final static UserDAO instance = new UserDAO();

	/**
	 * Default constructor. Can be used in place of getInstance()
	 */
	public UserDAO() {
	}

	@Override
	protected Serializable save(Object obj, Session s) {
		updateTime(obj);
		return super.save(obj, s);
	}

	@Override
	protected void update(Object obj, Session s) {
		updateTime(obj);
		super.update(obj, s);
	}

	@Override
	protected void delete(Object obj, Session session) {
		User user = (User) obj;
		if (user == null) {
			throw new PosException(Messages.getString("UserDAO.1")); //$NON-NLS-1$
		}
		user.setDeleted(true);
		if (user.isRoot()) {
			List<User> linkedUsers = user.getLinkedUser();
			if (linkedUsers != null && !linkedUsers.isEmpty()) {
				for (User linkedUser : linkedUsers) {
					if (linkedUser.equals(user)) {
						continue;
					}
					delete(linkedUser, session);
				}
			}
		}
		update(user, session);
	}

	public void saveOrUpdateUserWithRole(User user, UserType userType) {
		Transaction tx = null;
		try (Session session = createNewSession()) {
			tx = session.beginTransaction();
			saveOrUpdateUserWithRole(user, userType, session);
			tx.commit();
		} catch (Exception e) {
			if (tx != null) {
				tx.rollback();
			}
			throw e;
		}
	}

	public void saveOrUpdateUserWithRole(User user, UserType userType, Session session) {
		if (userType != null) {
			UserType existingUserType = UserTypeDAO.getInstance().findRoleByName(userType.getName());
			if (existingUserType != null) {
				Set<UserPermission> userPermissions = userType.getPermissions();
				if (userPermissions == null) {
					userPermissions = new HashSet<>();
				}
				Set<UserPermission> existingUserPermissions = existingUserType.getPermissions();
				if (existingUserPermissions == null) {
					existingUserPermissions = new HashSet<>();
				}
				if (!CollectionUtils.isEqualCollection(userPermissions, existingUserPermissions)) {
					existingUserType.setPermissions(userPermissions);
					UserTypeDAO.getInstance().saveOrUpdate(existingUserType, session);
				}
				user.setType(existingUserType);
			}
			else {
				UserTypeDAO.getInstance().saveOrUpdate(userType, session);
				user.setType(userType);
			}
		}
		else {
			user.setType(userType);
			UserTypeDAO.getInstance().saveOrUpdate(userType, session);
		}
		saveOrUpdate(user, session);
	}

	@Override
	protected void saveOrUpdate(Object obj, Session s) {
		updateTime(obj);
		super.saveOrUpdate(obj, s);
	}

	public int rowCount(String filterItem) {
		Session session = null;
		Criteria criteria = null;
		try {
			session = createNewSession();
			criteria = session.createCriteria(User.class);
			addDeletedFilter(criteria);
			criteria.setProjection(Projections.rowCount());
			if (StringUtils.isNotEmpty(filterItem)) {
				criteria.add(Restrictions.or(Restrictions.ilike(User.PROP_ID, filterItem, MatchMode.ANYWHERE),
						Restrictions.or(Restrictions.ilike(User.PROP_FIRST_NAME, filterItem, MatchMode.ANYWHERE),
								Restrictions.ilike(User.PROP_LAST_NAME, filterItem, MatchMode.ANYWHERE))));
			}
			criteria.add(Restrictions.or(Restrictions.isNull(User.PROP_ROOT), Restrictions.eq(User.PROP_ROOT, true)));
			Number rowCount = (Number) criteria.uniqueResult();
			if (rowCount != null) {
				return rowCount.intValue();
			}
			return 0;
		} finally {
			closeSession(session);
		}
	}

	public int getActiveUserCount() {
		try (Session session = this.createNewSession()) {
			Criteria criteria = session.createCriteria(this.getReferenceClass());
			this.addDeletedFilter(criteria);
			criteria.setProjection(Projections.rowCount());
			criteria.add(Restrictions.eq(User.PROP_ACTIVE, Boolean.TRUE));
			criteria.add(Restrictions.ne(User.PROP_ID, "online_order_manager")); //$NON-NLS-1$
			Number rowCount = (Number) criteria.uniqueResult();
			if (rowCount != null) {
				return rowCount.intValue();
			}
			return 0;
		}
	}

	public int getActivePosLiveUserCount() {
		return getActivePosLiveUserCount(null);
	}

	public int getActivePosLiveUserCount(List<String> skipIds) {
		try (Session session = this.createNewSession()) {
			return getActivePosLiveUserCount(skipIds, session);
		}
	}

	public int getActivePosLiveUserCount(List<String> skipIds, Session session) {
		Criteria criteria = session.createCriteria(this.getReferenceClass());
		this.addDeletedFilter(criteria);
		criteria.setProjection(Projections.rowCount());
		if (skipIds != null && skipIds.size() > 0) {
			criteria.add(Restrictions.not(Restrictions.in(User.PROP_ID, skipIds)));
		}
		criteria.add(Restrictions.eq(User.PROP_ACTIVE, Boolean.TRUE));
		criteria.add(Restrictions.eq(User.PROP_POSLIVE_ENABLE, Boolean.TRUE));
		criteria.add(Restrictions.ne(User.PROP_ID, "online_order_manager")); //$NON-NLS-1$
		Number rowCount = (Number) criteria.uniqueResult();
		if (rowCount != null) {
			return rowCount.intValue();
		}
		return 0;
	}

	public List<PosTransaction> findTopSellers(Session session, String outletId, Date fromDate, Date toDate, Integer maxResult) {

		//@formatter:off
		String hql = "select " //$NON-NLS-1$
								+ PosTransaction.PROP_USER_ID + " as " + PosTransaction.PROP_USER_ID//$NON-NLS-1$
								+ ", sum( case when transaction_type = 'CREDIT' then amount else (amount * -1) end) as " + PosTransaction.PROP_AMOUNT//$NON-NLS-1$
						+ " from " + PosTransaction.REF//$NON-NLS-1$
							+ " where " + " outlet_id = " +"'"+ outletId+"'" //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$
								+ "	and transaction_time between " +"'"+ DateUtil.getUTCStartOfDay(fromDate)+"'"  //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
										+ " and " +"'"+ DateUtil.getUTCEndOfDay(toDate)+"'"//$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
								+ " and voided = false"//$NON-NLS-1$
								+ " and user_id is not null"//$NON-NLS-1$
								+ "	and transaction_type in('CREDIT'," + "'DEBIT')" //$NON-NLS-1$ //$NON-NLS-2$
							+ " group by" + " user_id" //$NON-NLS-1$ //$NON-NLS-2$
							+ " order by " + PosTransaction.PROP_AMOUNT + " desc" //$NON-NLS-1$ //$NON-NLS-2$
							;
		Query query = session.createQuery(hql);
		query.setMaxResults(maxResult);
		query.setResultTransformer(Transformers.aliasToBean(PosTransaction.class));
		return query.list();

		//@formatter:on

	}

	public User get(String userId, String outletId, Session session) {
		return get(new User(userId, outletId), session);
	}

	public List<User> loadUsers(PaginationSupport model, String filterItem) {
		Session session = null;
		Criteria criteria = null;
		try {
			session = createNewSession();
			criteria = session.createCriteria(User.class);
			addDeletedFilter(criteria);
			if (StringUtils.isNotEmpty(filterItem)) {
				criteria.add(Restrictions.or(Restrictions.ilike(User.PROP_ID, filterItem, MatchMode.ANYWHERE),
						Restrictions.or(Restrictions.ilike(User.PROP_FIRST_NAME, filterItem, MatchMode.ANYWHERE),
								Restrictions.ilike(User.PROP_LAST_NAME, filterItem, MatchMode.ANYWHERE))));
			}
			criteria.add(Restrictions.or(Restrictions.isNull(User.PROP_ROOT), Restrictions.eq(User.PROP_ROOT, true)));
			criteria.setFirstResult(model.getCurrentRowIndex());
			criteria.setMaxResults(model.getPageSize());
			List<User> result = criteria.list();
			model.setRows(result);
			return result;
		} finally {
			closeSession(session);
		}
	}

	public User getRandomUser() {
		Session session = null;
		try {
			session = createNewSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			addDeletedFilter(criteria);
			criteria.add(Restrictions.or(Restrictions.isNull(User.PROP_ROOT), Restrictions.eq(User.PROP_ROOT, true)));
			criteria.add(Restrictions.eq(User.PROP_ACTIVE, true));
			criteria.setMaxResults(1);
			List<User> users = criteria.list();
			if (users.size() > 0) {
				return users.get(0);
			}
			return null;
		} finally {
			if (session != null) {
				closeSession(session);
			}
		}
	}

	@Override
	public List<User> findAll() {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			addDeletedFilter(criteria);
			return criteria.list();
		}
	}

	public List<User> findClockedInUsers() {
		return findClockedInUsers("");
	}

	public List<User> findClockedInUsers(String outletId) {
		Session session = null;
		try {
			session = createNewSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			addDeletedFilter(criteria);
			if (StringUtils.isNotBlank(outletId)) {
				criteria.add(Restrictions.eq(User.PROP_OUTLET_ID, outletId));
			}
			criteria.add(Restrictions.or(Restrictions.isNull(User.PROP_ACTIVE), Restrictions.eq(User.PROP_ACTIVE, true)));
			criteria.add(Restrictions.eq(User.PROP_CLOCKED_IN, true));
			return criteria.list();
		} finally {
			if (session != null) {
				closeSession(session);
			}
		}
	}

	public List<User> findAllActive() {
		return findAllActive(null);
	}

	public List<User> findAllActive(String outletId) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			addDeletedFilter(criteria);
			if (StringUtils.isNotBlank(outletId)) {
				criteria.add(Restrictions.eq(User.PROP_OUTLET_ID, outletId));
			}
			Junction activeUserCriteria = Restrictions.disjunction().add(Restrictions.isNull(User.PROP_ACTIVE))
					.add(Restrictions.eq(User.PROP_ACTIVE, Boolean.TRUE));
			criteria.add(Restrictions.or(Restrictions.isNull(User.PROP_ROOT), Restrictions.eq(User.PROP_ROOT, true)));
			criteria.add(activeUserCriteria);
			//criteria.add(Restrictions.eq(User.PROP_CLOCKED_IN, Boolean.TRUE));
			return criteria.list();
		}
	}

	public List<User> findLabStuff(String outletId) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(User.class);
			addDeletedFilter(criteria);
			if (StringUtils.isNotBlank(outletId)) {
				criteria.add(Restrictions.eq(User.PROP_OUTLET_ID, outletId));
			}
			Junction activeUserCriteria = Restrictions.disjunction().add(Restrictions.isNull(User.PROP_ACTIVE))
					.add(Restrictions.eq(User.PROP_ACTIVE, Boolean.TRUE));
			criteria.add(Restrictions.or(Restrictions.isNull(User.PROP_ROOT), Restrictions.eq(User.PROP_ROOT, true)));
			criteria.add(activeUserCriteria);

			addCriteriaProperyFilter(criteria, User.PROP_PROPERTIES, "user.lab_stuff", true);
			List<User> result = criteria.list();
			if (result == null || result.isEmpty()) {
				return null;
			}
			return result;
		}
	}

	public List<User> findActiveUsersForPayroll() {
		Session session = null;

		try {
			session = createNewSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			addDeletedFilter(criteria);
			Junction activeUserCriteria = Restrictions.disjunction().add(Restrictions.isNull(User.PROP_ACTIVE))
					.add(Restrictions.eq(User.PROP_ACTIVE, Boolean.TRUE));
			criteria.add(Restrictions.or(Restrictions.isNull(User.PROP_ROOT), Restrictions.eq(User.PROP_ROOT, true)));
			criteria.add(activeUserCriteria);
			criteria.addOrder(Order.asc(User.PROP_FIRST_NAME));

			return criteria.list();
		} finally {
			if (session != null) {
				closeSession(session);
			}
		}
	}

	public List<User> findDrivers() {
		Session session = null;

		try {
			session = getSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			addDeletedFilter(criteria);
			criteria.add(Restrictions.eq(User.PROP_DRIVER, Boolean.TRUE));
			criteria.add(Restrictions.or(Restrictions.isNull(User.PROP_ROOT), Restrictions.eq(User.PROP_ROOT, true)));

			return criteria.list();
		} finally {
			if (session != null) {
				closeSession(session);
			}
		}
	}

	public User findUser(String userId) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			addDeletedFilter(criteria);
			criteria.add(Restrictions.eq(User.PROP_ID, userId));
			criteria.add(Restrictions.or(Restrictions.isNull(User.PROP_ROOT), Restrictions.eq(User.PROP_ROOT, true)));

			Object result = criteria.uniqueResult();
			if (result != null) {
				return (User) result;
			}
			else {
				throw new UserNotFoundException(String.format(Messages.getString("UserDAO.0"), userId)); //$NON-NLS-1$ //$NON-NLS-2$
			}
		}
	}

	public User findUserByEmail(String userEmail) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			addDeletedFilter(criteria);
			criteria.add(Restrictions.eq(User.PROP_EMAIL, userEmail));
			criteria.add(Restrictions.or(Restrictions.isNull(User.PROP_ROOT), Restrictions.eq(User.PROP_ROOT, true)));
			criteria.setMaxResults(1);

			return (User) criteria.uniqueResult();
		}
	}

	public User findUserBySecretKey(String secretKey) {
		return findUserBySecretKey(secretKey, null);
	}

	public User findUserBySecretKey(String secretKey, String outletId) {
		Session session = null;
		try {
			session = getSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			addDeletedFilter(criteria);
			criteria.add(Restrictions.eq(User.PROP_PASSWORD, getEncrypedPassword(secretKey)));
			if (StringUtils.isNotBlank(outletId)) {
				criteria.add(Restrictions.eq(User.PROP_OUTLET_ID, outletId));
			}
			criteria.add(Restrictions.or(Restrictions.isNull(User.PROP_ROOT), Restrictions.eq(User.PROP_ROOT, true)));
			criteria.add(Restrictions.or(Restrictions.isNull(User.PROP_ACTIVE), Restrictions.eq(User.PROP_ACTIVE, true)));

			List list = criteria.list();
			if (list != null && list.size() > 0) {
				return (User) list.get(0);
			}
			return null;
		} finally {
			if (session != null) {
				closeSession(session);
			}
		}
	}

	public User findPosliveUserBySecretKey(String secretKey, String outletId) {
		Session session = null;
		try {
			session = getSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			addDeletedFilter(criteria);
			criteria.add(Restrictions.eq(User.PROP_PASSWORD, getEncrypedPassword(secretKey)));
			if (StringUtils.isNotBlank(outletId)) {
				criteria.add(Restrictions.eq(User.PROP_OUTLET_ID, outletId));
			}
			criteria.add(Restrictions.or(Restrictions.isNull(User.PROP_ROOT), Restrictions.eq(User.PROP_ROOT, true)));
			criteria.add(Restrictions.or(Restrictions.isNull(User.PROP_ACTIVE), Restrictions.eq(User.PROP_ACTIVE, true)));
			criteria.add(Restrictions.eq(User.PROP_POSLIVE_ENABLE, true));

			List list = criteria.list();
			if (list != null && list.size() > 0) {
				return (User) list.get(0);
			}
			return null;
		} finally {
			if (session != null) {
				closeSession(session);
			}
		}
	}

	private String getEncrypedPassword(String secretKey) {
		if (StringUtils.isNotEmpty(secretKey)) {
			try {
				secretKey = AESencrp.encrypt(secretKey);
			} catch (Exception e) {
				PosLog.error(getClass(), e);
			}
		}
		return secretKey;
	}

	public boolean isUserExist(String userId) {
		try {
			User user = findUser(userId);

			return user != null;

		} catch (UserNotFoundException x) {
			return false;
		}
	}

	public Integer findUserWithMaxId() {
		Session session = null;

		try {
			session = getSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			addDeletedFilter(criteria);
			criteria.setProjection(Projections.max(User.PROP_ID));
			criteria.add(Restrictions.or(Restrictions.isNull(User.PROP_ROOT), Restrictions.eq(User.PROP_ROOT, true)));

			List list = criteria.list();
			if (list != null && list.size() > 0) {
				return (Integer) list.get(0);
			}

			return null;
		} finally {
			if (session != null) {
				closeSession(session);
			}
		}
	}

	public List<User> getClockedInUser(Terminal terminal, boolean includeStaffBank) {
		Session session = null;

		try {
			session = getSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			addDeletedFilter(criteria);
			criteria.add(Restrictions.eq(User.PROP_CLOCKED_IN, Boolean.TRUE));
			//criteria.add(Restrictions.or(Restrictions.isNull(User.PROP_ROOT), Restrictions.eq(User.PROP_ROOT, true)));

			if (!includeStaffBank) {
				criteria.add(Restrictions.eq(User.PROP_STAFF_BANK, Boolean.FALSE));
			}
			Junction activeUserCriteria = Restrictions.disjunction().add(Restrictions.isNull(User.PROP_ACTIVE))
					.add(Restrictions.eq(User.PROP_ACTIVE, Boolean.TRUE));
			criteria.add(activeUserCriteria);
			//			if (terminal != null)
			//				criteria.add(Restrictions.eq(User.PROP_CURRENT_TERMINAL, terminal));

			return criteria.list();
		} finally {
			if (session != null) {
				closeSession(session);
			}
		}

	}

	public void saveClockIn(User user, AttendenceHistory attendenceHistory, Shift shift, Calendar currentTime) {
		Session session = null;
		Transaction tx = null;

		ActionHistory actionHistory = new ActionHistory();
		actionHistory.setActionName("CLOCK IN"); //$NON-NLS-1$
		actionHistory.setActionTime(attendenceHistory.getClockInTime());
		String actionMessage = String.format("User %s clocks in", user.getId()); //$NON-NLS-1$
		actionHistory.setDescription(actionMessage);
		actionHistory.setPerformer(user);

		try {
			session = getSession();
			tx = session.beginTransaction();

			session.saveOrUpdate(user);
			session.saveOrUpdate(attendenceHistory);
			session.save(actionHistory);

			//			saveOrUpdate(user, session);
			//			AttendenceHistoryDAO.getInstance().saveOrUpdate(attendenceHistory, session);
			//			ActionHistoryDAO.getInstance().save(actionHistory, session);

			tx.commit();
		} catch (Exception e) {
			PosLog.error(getClass(), e);

			if (tx != null) {
				try {
					tx.rollback();
				} catch (Exception x) {
				}
			}
			throw new PosException(Messages.getString("UserDAO.2"), e); //$NON-NLS-1$

		} finally {
			if (session != null) {
				closeSession(session);
			}
		}
	}

	public void saveClockOut(User user, AttendenceHistory attendenceHistory, Shift shift) {
		ActionHistory actionHistory = new ActionHistory();
		actionHistory.setActionName("CLOCK OUT"); //$NON-NLS-1$
		actionHistory.setActionTime(attendenceHistory.getClockOutTime());
		String actionMessage = String.format("User %s clocks out", user.getId()); //$NON-NLS-1$
		actionHistory.setDescription(actionMessage);
		actionHistory.setPerformer(user);

		Session session = null;
		Transaction tx = null;

		try {
			session = getSession();
			tx = session.beginTransaction();

			UserDAO.getInstance().saveOrUpdate(user, session);
			AttendenceHistoryDAO.getInstance().saveOrUpdate(attendenceHistory, session);
			ActionHistoryDAO.getInstance().save(actionHistory, session);

			tx.commit();
		} catch (StaleStateException e) {
			throw new PosException(Messages.getString("UserDAO.5") + Messages.getString("UserDAO.15"), e); //$NON-NLS-1$ //$NON-NLS-2$
		} catch (Exception e) {
			if (tx != null) {
				try {
					tx.rollback();
				} catch (Exception x) {
				}
			}
			throw e;// PosException(Messages.getString("UserDAO.3"), e); //$NON-NLS-1$

		} finally {
			if (session != null) {
				closeSession(session);
			}
		}
	}

	public void saveDriverOut(User user, EmployeeInOutHistory attendenceHistory, Shift shift, Calendar currentTime) {
		Session session = null;
		Transaction tx = null;

		try {
			session = getSession();
			tx = session.beginTransaction();

			session.saveOrUpdate(user);
			session.saveOrUpdate(attendenceHistory);

			tx.commit();
		} catch (Exception e) {
			PosLog.error(getClass(), e);

			if (tx != null) {
				try {
					tx.rollback();
				} catch (Exception x) {
				}
			}
			throw new PosException(Messages.getString("UserDAO.2"), e); //$NON-NLS-1$

		} finally {
			if (session != null) {
				closeSession(session);
			}
		}
	}

	public void saveDriverIn(User user, EmployeeInOutHistory attendenceHistory, Shift shift, Calendar currentTime) {
		Session session = null;
		Transaction tx = null;

		try {
			session = getSession();
			tx = session.beginTransaction();

			session.saveOrUpdate(user);
			session.saveOrUpdate(attendenceHistory);

			tx.commit();
		} catch (Exception e) {
			if (tx != null) {
				try {
					tx.rollback();
				} catch (Exception x) {
				}
			}
			throw new PosException(Messages.getString("UserDAO.3"), e); //$NON-NLS-1$

		} finally {
			if (session != null) {
				closeSession(session);
			}
		}
	}

	private boolean validate(User user, boolean editMode) throws PosException {
		String hql = "from User u where u." + User.PROP_ID + "=:userId"; //$NON-NLS-1$
		Session session = getSession();
		Query query = session.createQuery(hql);
		query = query.setParameter("userId", user.getId()); //$NON-NLS-1$

		if (query.list().size() > 0) {
			throw new PosException(Messages.getString("UserDAO.7")); //$NON-NLS-1$
		}

		return true;
	}

	public void saveOrUpdate(User user, boolean editMode) {
		Session session = null;

		try {
			if (!editMode) {
				validate(user, editMode);
			}
			super.saveOrUpdate(user);
		} catch (Exception x) {
			throw new PosException(x.getMessage(), x);
		} finally {
			closeSession(session);
		}

	}

	// public User findByPassword(String password) throws PosException {
	// Session session = null;
	// Transaction tx = null;
	//		
	// String hql = "from User u where u.password=:password";
	//		
	// try {
	// session = getSession();
	// tx = session.beginTransaction();
	// Query query = session.createQuery(hql);
	// query = query.setParameter("password", password);
	// User user = (User) query.uniqueResult();
	// tx.commit();
	// if(user == null) {
	// throw new PosException("User not found");
	// }
	// return user;
	// } catch(PosException x) {
	// throw x;
	// } catch (Exception e) {
	// try {
	// if(tx != null) {
	// tx.rollback();
	// }
	// }catch(Exception e2) {}
	// throw new PosException("Unnable to find user", e);
	// } finally {
	// if(session != null) {
	// session.close();
	// }
	// }
	// }

	public int findNumberOfOpenTickets(User user) throws PosException {
		Session session = null;
		Transaction tx = null;

		String hql = "select count(*) from Ticket ticket where ticket.owner=:owner and ticket." //$NON-NLS-1$
				+ Ticket.PROP_CLOSED + "settled=false"; //$NON-NLS-1$
		int count = 0;
		try {
			session = getSession();
			tx = session.beginTransaction();
			Query query = session.createQuery(hql);
			query = query.setEntity("owner", user); //$NON-NLS-1$
			Iterator iterator = query.iterate();
			if (iterator.hasNext()) {
				count = ((Integer) iterator.next()).intValue();
			}
			tx.commit();
			return count;
		} catch (Exception e) {
			try {
				if (tx != null) {
					tx.rollback();
				}
			} catch (Exception e2) {
			}
			throw new PosException(Messages.getString("UserDAO.12"), e); //$NON-NLS-1$
		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

	public boolean isClockedIn(User user) {
		Session session = getSession();
		Criteria criteria = session.createCriteria(getReferenceClass());
		addDeletedFilter(criteria);
		criteria.add(Restrictions.eq(User.PROP_ID, user.getId()));
		criteria.add(Restrictions.eq(User.PROP_OUTLET_ID, user.getOutletId()));
		criteria.setMaxResults(1);

		ProjectionList projectionList = Projections.projectionList();
		projectionList.add(Projections.property(User.PROP_CLOCKED_IN));

		ResultTransformer transformer = new ResultTransformer() {
			public Object transformTuple(Object[] row, String[] arg1) {
				return row[0];
			}

			public List transformList(List arg0) {
				return arg0;
			}
		};
		criteria.setProjection(projectionList).setResultTransformer(transformer);
		Boolean result = (Boolean) criteria.uniqueResult();
		return result == null ? Boolean.FALSE : result;
	}

	public User getRandomUser(Session session) {
		Criteria criteria = session.createCriteria(getReferenceClass());
		addDeletedFilter(criteria);
		criteria.add(Restrictions.eq(User.PROP_ACTIVE, Boolean.TRUE));
		criteria.setMaxResults(1);
		List list = criteria.list();
		if (list.size() > 0) {
			return (User) list.get(0);
		}

		return null;
	}

	public void performForceCloseStaffBank(User staffBankOwner, User closedByUser, Session session) throws Exception {
		CashDrawer staffBank = staffBankOwner.getActiveDrawerPullReport();
		CashDrawerReportService reportService = new CashDrawerReportService(staffBank);
		reportService.populateReport(session);

		staffBank.setClosedBy(closedByUser);
		staffBank.setReportTime(StoreDAO.getServerTimestamp());

		CashDrawerDAO.getInstance().saveOrUpdate(staffBank, session);

		staffBankOwner.setStaffBankStarted(false);
		staffBankOwner.setCurrentCashDrawer(null);
	}

	public void doForceClockOutUser(User userToClockOut, User actionPerformer) throws Exception {
		Session session = null;
		Transaction tx = null;
		try {
			if (!actionPerformer.isManager() && !actionPerformer.isAdministrator() && !actionPerformer.hasPermission(UserPermission.OPEN_CLOSE_STORE)) {
				throw new PosException(Messages.getString("UserDAO.18")); //$NON-NLS-1$
			}

			session = createNewSession();
			tx = session.beginTransaction();

			if (userToClockOut.isStaffBankStarted()) {
				performForceCloseStaffBank(userToClockOut, actionPerformer, session);
			}

			saveAttendenceHistory(userToClockOut, session);

			userToClockOut.setClockedIn(false);
			userToClockOut.setCurrentShift(null);
			userToClockOut.setLastClockInTime(null);
			userToClockOut.setLastClockOutTime(null);
			userToClockOut.setAvailableForDelivery(false);

			session.evict(userToClockOut);
			update(userToClockOut, session);

			tx.commit();
			logJournal(userToClockOut, actionPerformer);
		} catch (Exception e) {
			if (tx != null) {
				tx.rollback();
			}
			throw e;
		} finally {
			closeSession(session);
		}
	}

	private void logJournal(User userToClockOut, User actionPerformer) {
		if (userToClockOut.isStaffBankStarted()) {
			StringBuilder sb = new StringBuilder();
			sb.append(" Staff id: " + userToClockOut.getId()); //NON-NLS-1$
			sb.append(" Staff bank id: " + userToClockOut.getActiveDrawerPullReport().getId()); //NON-NLS-1$
			ActionHistoryDAO.saveHistory(actionPerformer, null, null, ActionHistory.STAFF_BANK_FORCE_CLOSE, sb.toString());
		}
		ActionHistoryDAO.saveHistory(ActionHistory.FORCED_CLOCK_OUT, "Staff id: " + userToClockOut.getId());
	}

	private void saveAttendenceHistory(User userToClockOut, Session session) {
		AttendenceHistory attendenceHistory = AttendenceHistoryDAO.getInstance().findByLastClockInTime(userToClockOut, session);
		Calendar currentCalendar = Calendar.getInstance();
		currentCalendar.setTime(StoreDAO.getServerTimestamp());
		if (attendenceHistory == null) {
			attendenceHistory = AttendenceHistoryDAO.createNewHistory(userToClockOut, DataProvider.get().getCurrentTerminal(),
					userToClockOut.getCurrentShift());
		}
		attendenceHistory.setClockedOut(true);
		attendenceHistory.setClockOutTime(StoreDAO.getServerTimestamp());
		attendenceHistory.setClockOutHour(Short.valueOf((short) currentCalendar.get(Calendar.HOUR_OF_DAY)));

		AttendenceHistoryDAO.getInstance().saveOrUpdate(attendenceHistory, session);
	}

	public List<String> findUsersPasswords() {
		Session session = null;
		try {
			session = createNewSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			addDeletedFilter(criteria);
			criteria.add(Restrictions.eq(User.PROP_ROOT, true));
			criteria.setProjection(Projections.property(User.PROP_PASSWORD));
			return criteria.list();
		} finally {
			if (session != null) {
				closeSession(session);
			}
		}
	}

	public User findUserById(String userId) {
		Session session = null;

		try {
			session = getSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			addDeletedFilter(criteria);
			criteria.add(Restrictions.eq(User.PROP_ID, userId));

			Object result = criteria.uniqueResult();
			if (result != null) {
				return (User) result;
			}
			else {
				throw new UserNotFoundException(String.format(Messages.getString("UserDAO.0"), userId)); //$NON-NLS-1$ //$NON-NLS-2$
			}
		} finally {
			if (session != null) {
				closeSession(session);
			}
		}
	}

	public void startStaffBank(User assignToUser) {
		Session session = null;
		Transaction tx = null;

		try {
			CashDrawer staffBankReport = new CashDrawer();
			staffBankReport.setStartTime(new Date());
			staffBankReport.setAssignedUser(assignToUser);
			staffBankReport.setTerminal(Application.getInstance().getTerminal());
			staffBankReport.setStoreSession(StoreUtil.getCurrentStoreSession());
			staffBankReport.setDrawerType(DrawerType.STAFF_BANK);
			staffBankReport.setAssignedBy(assignToUser);
			//TODO
			//staffBankReport.setOutletId(DataProvider.get().getCurrentOutletId());

			session = createNewSession();
			tx = session.beginTransaction();

			CashDrawerDAO.getInstance().save(staffBankReport, session);

			assignToUser.setCurrentCashDrawer(staffBankReport);
			assignToUser.setStaffBankStarted(true);

			saveOrUpdate(assignToUser, session);
			tx.commit();
		} catch (Exception e) {
			try {
				tx.rollback();
			} catch (Exception x) {
			}
			throw e;
		} finally {
			closeSession(session);
		}
	}

	@Deprecated
	/**
	 * Use BaseDataServiceDao saveOrUpdateUsers methord.
	 */
	public void saveOrUpdateAllUsers(List<User> dataList, boolean updateLastUpdateTime, boolean updateSyncTime) throws Exception {
		if (dataList == null || dataList.isEmpty())
			return;

		List<User> parentUser = new ArrayList<User>();
		List<User> childUser = new ArrayList<User>();

		for (Iterator<User> iterator = dataList.iterator(); iterator.hasNext();) {
			User user = (User) iterator.next();
			if (user.isRoot()) {
				parentUser.add(user);
			}
			else {
				childUser.add(user);
			}
		}

		saveOrUpdateUsers(parentUser, updateLastUpdateTime, updateSyncTime);
		saveOrUpdateUsers(childUser, updateLastUpdateTime, updateSyncTime);
	}

	private void saveOrUpdateUsers(List<User> users, boolean updateLastUpdateTime, boolean updateSyncTime) throws Exception {
		if (users == null)
			return;

		Transaction tx = null;
		Session session = null;
		try {
			session = createNewSession();
			tx = session.beginTransaction();
			for (User user : users) {
				User existingUser = get(user.getId(), user.getOutletId());
				if (existingUser != null) {
					CashDrawer sourceCashDrawer = StringUtils.isBlank(user.getCurrentCashDrawerId()) ? null
							: CashDrawerDAO.getInstance().get(user.getCurrentCashDrawerId());
					CashDrawer existingCashDrawer = StringUtils.isBlank(existingUser.getCurrentCashDrawerId()) ? null
							: CashDrawerDAO.getInstance().get(existingUser.getCurrentCashDrawerId());

					final String id = existingUser.getId();
					long version = existingUser.getVersion();
					PropertyUtils.copyProperties(existingUser, user);
					existingUser.setId(id);
					existingUser.setVersion(version);
					existingUser.setUpdateLastUpdateTime(updateLastUpdateTime);
					existingUser.setUpdateSyncTime(updateSyncTime);

					if (sourceCashDrawer == null && existingCashDrawer != null && existingCashDrawer.getReportTime() == null) {
						existingUser.setCurrentCashDrawerId(existingCashDrawer.getId());
					}
					if (sourceCashDrawer != null && existingCashDrawer != null && !sourceCashDrawer.getId().equals(existingCashDrawer.getId())) {
						if (existingCashDrawer.getStartTime().after(sourceCashDrawer.getStartTime())) {
							existingUser.setCurrentCashDrawerId(existingCashDrawer.getId());
						}
					}
					update(existingUser, session);
				}
				else {
					user.setUpdateLastUpdateTime(updateLastUpdateTime);
					user.setUpdateSyncTime(updateSyncTime);
					save(user, session);
				}
			}
			tx.commit();
		} catch (Exception e) {
			tx.rollback();
			throw e;
		} finally {
			closeSession(session);
		}
	}

	public User get(String id, String outletId) {
		return get(new User(id, outletId));
	}

	public List<String> getIdListForType(UserType userType, Session session) {
		Criteria userIdCrit = session.createCriteria(User.class);
		userIdCrit.add(Restrictions.or(Restrictions.isNull(AppConstants.PROP_DELETED), Restrictions.eq(AppConstants.PROP_DELETED, Boolean.FALSE)));
		userIdCrit.setProjection(Projections.property(User.PROP_ID));
		userIdCrit.add(Restrictions.eq(User.PROP_USER_TYPE_ID, userType.getId()));
		return userIdCrit.list();
	}

	public List<User> findUsersExcept(String id) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			if (id != null) {
				criteria.add(Restrictions.ne(User.PROP_ID, id));
			}
			addDeletedFilter(criteria);
			return criteria.list();
		}
	}

	public boolean isEmailExist(String email) {
		if (StringUtils.isEmpty(email)) {
			return false;
		}
		Session session = null;
		try {
			session = createNewSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.eq(User.PROP_EMAIL, email));
			criteria.setProjection(Projections.rowCount());
			Number number = (Number) criteria.uniqueResult();
			return number != null && number.intValue() > 0;
		} finally {
			if (session != null) {
				closeSession(session);
			}
		}
	}

	public boolean isEmailExistExceptUser(String email, User customer) {
		if (StringUtils.isEmpty(email)) {
			return false;
		}
		Session session = null;
		try {
			session = createNewSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.eq(User.PROP_EMAIL, email).ignoreCase());
			if (customer.getId() != null) {
				criteria.add(Restrictions.ne(User.PROP_ID, customer.getId()));
			}
			criteria.setProjection(Projections.rowCount());
			Number number = (Number) criteria.uniqueResult();
			return number != null && number.intValue() > 0;

		} finally {
			if (session != null) {
				closeSession(session);
			}
		}
	}

	public User getOnlineOrderCreator(String outletId) {
		User user = UserDAO.getInstance().get("online_order_manager", outletId); //$NON-NLS-1$
		if (user == null) {
			UserType userType = UserTypeDAO.getInstance().get("online_order_manager"); //$NON-NLS-1$
			if (userType == null) {
				userType = new UserType("online_order_manager"); //$NON-NLS-1$
				userType.setName("Online Order"); //$NON-NLS-1$
				userType.addTopermissions(UserPermission.CREATE_TICKET);
				userType.addTopermissions(UserPermission.ADD_DISCOUNT);
				userType.addTopermissions(UserPermission.EDIT_OTHER_USERS_TICKETS);
				userType.addTopermissions(UserPermission.REFUND);
				userType.addTopermissions(UserPermission.SETTLE_TICKET);

				UserTypeDAO.getInstance().save(userType);
			}

			user = new User("online_order_manager", outletId); //$NON-NLS-1$
			user.setFirstName("Online"); //$NON-NLS-1$
			user.setLastName("Order"); //$NON-NLS-1$
			user.setActive(true);
			user.setRoot(true);
			user.setUserTypeId(userType.getId());
			UserDAO.getInstance().save(user);
		}
		return user;
	}

	public List<User> getUserByOutletId(String outletId) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			if (outletId != null) {
				criteria.add(Restrictions.eq(User.PROP_OUTLET_ID, outletId));
			}
			addDeletedFilter(criteria);
			return criteria.list();
		}
	}

	/**
	 * This method is used to get the user if there is only one user in database.
	 * If there is more than one user in database, this method will return null.
	 * 
	 * @return User if there is only one user in database, otherwise returns null.
	 */
	public User findPosliveSingleUser(String outletId) {
		try (Session session = getSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			addDeletedFilter(criteria);
			criteria.add(Restrictions.or(Restrictions.isNull(User.PROP_ROOT), Restrictions.eq(User.PROP_ROOT, true)));
			criteria.add(Restrictions.or(Restrictions.isNull(User.PROP_ACTIVE), Restrictions.eq(User.PROP_ACTIVE, true)));
			criteria.add(Restrictions.eq(User.PROP_POSLIVE_ENABLE, true));
			criteria.add(Restrictions.eq(User.PROP_OUTLET_ID, outletId));

			criteria.setMaxResults(2);

			List list = criteria.list();
			if (list.size() > 1) {
				return null;
			}
			return (User) list.get(0);
		}
	}

	@SuppressWarnings("unchecked")
	public List<User> getUsers(Outlet outlet, String searchString, boolean posLiveEnable, List<String> skipIds, boolean isActive) {
		String userActiveStatus = null;
		if (isActive) {
			userActiveStatus = POSConstants.ACTIVE;
		}
		else {
			userActiveStatus = POSConstants.DEACTIVE;
		}
		return getUsers(outlet, searchString, posLiveEnable, skipIds, userActiveStatus);
	}

	@SuppressWarnings("unchecked")
	public List<User> getUsers(Outlet outlet, String searchString, boolean posLiveEnable, List<String> skipIds, String userActiveStatus) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			if (outlet != null) {
				criteria.add(Restrictions.eq(User.PROP_OUTLET_ID, outlet.getId()));
			}
			if (posLiveEnable) {
				criteria.add(Restrictions.eq(User.PROP_POSLIVE_ENABLE, Boolean.TRUE));
			}
			else {
				criteria.add(Restrictions.or(Restrictions.isNull(User.PROP_POSLIVE_ENABLE), Restrictions.eq(User.PROP_POSLIVE_ENABLE, Boolean.FALSE)));
			}
			if (StringUtils.isNotEmpty(searchString)) {
				criteria.add(Restrictions.or(Restrictions.ilike(User.PROP_FIRST_NAME, searchString, MatchMode.ANYWHERE),
						Restrictions.ilike(User.PROP_LAST_NAME, searchString, MatchMode.ANYWHERE)));
			}
			if (skipIds != null && skipIds.size() > 0) {
				criteria.add(Restrictions.not(Restrictions.in(User.PROP_ID, skipIds)));
			}

			if (StringUtils.isNotBlank(userActiveStatus)) {
				if (userActiveStatus.equals(POSConstants.ACTIVE)) {
					criteria.add(Restrictions.eq(User.PROP_ACTIVE, true));
				}
				else if (userActiveStatus.equals(POSConstants.DEACTIVE)) {
					criteria.add(Restrictions.eq(User.PROP_ACTIVE, false));
				}
			}

			criteria.add(Restrictions.or(Restrictions.isNull(User.PROP_ROOT), Restrictions.eq(User.PROP_ROOT, true)));
			addDeletedFilter(criteria);
			return criteria.list();
		}
	}
}