/**
 * ************************************************************************
 * * The contents of this file are subject to the MRPL 1.2
 * * (the  "License"),  being   the  Mozilla   Public  License
 * * Version 1.1  with a permitted attribution clause; you may not  use this
 * * file except in compliance with the License. You  may  obtain  a copy of
 * * the License at http://www.floreantpos.org/license.html
 * * Software distributed under the License  is  distributed  on  an "AS IS"
 * * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * * License for the specific  language  governing  rights  and  limitations
 * * under the License.
 * * The Original Code is FLOREANT POS.
 * * The Initial Developer of the Original Code is OROCUBE LLC
 * * All portions are Copyright (C) 2015 OROCUBE LLC
 * * All Rights Reserved.
 * ************************************************************************
 */
package com.floreantpos.model.dao;

import java.io.Serializable;
import java.util.Date;
import java.util.Iterator;
import java.util.List;

import org.apache.commons.lang.StringUtils;
import org.hibernate.Criteria;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.hibernate.criterion.Criterion;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.Projections;
import org.hibernate.criterion.Restrictions;

import com.floreantpos.PosLog;
import com.floreantpos.model.CashBreakdown;
import com.floreantpos.model.CashDrawer;
import com.floreantpos.model.DrawerType;
import com.floreantpos.model.StoreSession;
import com.floreantpos.model.Terminal;
import com.floreantpos.model.User;
import com.floreantpos.services.report.CashDrawerReportService;
import com.floreantpos.swing.PaginationSupport;
import com.orocube.rest.service.server.BaseDataServiceDao;

public class CashDrawerDAO extends BaseCashDrawerDAO {

	/**
	 * Default constructor.  Can be used in place of getInstance()
	 */
	public CashDrawerDAO() {
	}

	@Override
	protected Serializable save(Object obj, Session s) {
		updateTime(obj);
		return super.save(obj, s);
	}

	@Override
	protected void update(Object obj, Session s) {
		updateTime(obj);
		super.update(obj, s);
	}

	@Override
	protected void saveOrUpdate(Object obj, Session s) {
		updateTime(obj);
		super.saveOrUpdate(obj, s);
	}

	public void loadCashDrawerReport(PaginationSupport model, Date fromDate, Date toDate) {
		Session session = null;
		try {
			session = createNewSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.between(CashDrawer.PROP_START_TIME, fromDate, toDate));
			criteria.setProjection(Projections.rowCount());
			Number rowCount = (Number) criteria.uniqueResult();
			model.setNumRows(rowCount.intValue());

			criteria.setProjection(null);
			criteria.setFirstResult(model.getCurrentRowIndex());
			criteria.setMaxResults(model.getPageSize());

			List<CashDrawer> cashDrawers = criteria.list();
			for (CashDrawer cashDrawer : cashDrawers) {
				if (cashDrawer.getReportTime() == null) {
					CashDrawerReportService cashDrawerReportService = new CashDrawerReportService(cashDrawer);
					cashDrawerReportService.populateReport();
				}
			}
			model.setRows(cashDrawers);
		} finally {
			closeSession(session);
		}
	}

	public List<CashDrawer> findReports(Date start, Date end) {
		Session session = null;

		try {
			session = getSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.ge(CashDrawer.PROP_REPORT_TIME, start));
			criteria.add(Restrictions.le(CashDrawer.PROP_REPORT_TIME, end));

			return criteria.list();
		} finally {
			closeSession(session);
		}
	}

	public List<CashDrawer> findByStoreOperationData(StoreSession data, Boolean openDrawer) {
		if (data == null)
			return null;
		Session session = null;
		try {
			session = getSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.eq(CashDrawer.PROP_STORE_SESSION_ID, data.getId()));
			if (openDrawer != null)
				criteria.add(openDrawer ? Restrictions.isNull(CashDrawer.PROP_REPORT_TIME) : Restrictions.isNotNull(CashDrawer.PROP_REPORT_TIME));
			return criteria.list();
		} finally {
			closeSession(session);
		}
	}

	public List<CashDrawer> findByStoreOperationData(StoreSession data) {
		if (data == null)
			return null;
		Session session = null;
		try {
			session = getSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.eq(CashDrawer.PROP_STORE_SESSION_ID, data.getId()));
			return criteria.list();
		} finally {
			closeSession(session);
		}
	}

	public List<String> getCashDrawerIds(StoreSession data) {
		if (data == null)
			return null;
		Session session = null;
		try {
			session = getSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.setProjection(Projections.property(CashDrawer.PROP_ID));
			criteria.add(Restrictions.eq(CashDrawer.PROP_STORE_SESSION_ID, data.getId()));
			List list = criteria.list();
			return list;
		} finally {
			closeSession(session);
		}
	}

	//	public CashDrawer loadFullCashDrawer(String id) {
	//		Session session = null;
	//		try {
	//			session = createNewSession();
	//			CashDrawer cashDrawer = (CashDrawer) session.get(getReferenceClass(), id);
	//			Hibernate.initialize(cashDrawer.getTransactions());
	//			return cashDrawer;
	//		} finally {
	//			closeSession(session);
	//		}
	//	}

	//	public CashDrawer loadFullCashDrawer(CashDrawer cashDrawer) {
	//		Session session = null;
	//		try {
	//			session = createNewSession();
	//			session.refresh(cashDrawer);
	//			Hibernate.initialize(cashDrawer.getTransactions());
	//			return cashDrawer;
	//		} finally {
	//			closeSession(session);
	//		}
	//	}

	public double getSumOfOpeningBalance(StoreSession storeSession) {
		Session session = null;
		try {
			session = getSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.eq(CashDrawer.PROP_STORE_SESSION_ID, storeSession.getId()));
			criteria.setProjection(Projections.sum(CashDrawer.PROP_BEGIN_CASH));

			Double openingBalance = (Double) criteria.uniqueResult();
			if (openingBalance == null) {
				return 0.0;
			}
			return openingBalance;
		} finally {
			closeSession(session);
		}
	}

	public List<CashDrawer> findByUnSyncStoreOperationData(StoreSession data) {
		if (data == null)
			return null;
		Session session = null;
		try {
			session = getSession();
			Criteria criteria = session.createCriteria(getReferenceClass());

			Criterion nullUpdateTime = Restrictions.isNull(CashDrawer.PROP_LAST_UPDATE_TIME);
			Criterion nullSyncTime = Restrictions.isNull(CashDrawer.PROP_LAST_SYNC_TIME);
			Criterion gtQuery = Restrictions.gtProperty(CashDrawer.PROP_LAST_UPDATE_TIME, CashDrawer.PROP_LAST_SYNC_TIME);
			criteria.add(Restrictions.or(nullUpdateTime, nullSyncTime, gtQuery));

			criteria.add(Restrictions.eq(CashDrawer.PROP_STORE_SESSION_ID, data.getId()));

			return criteria.list();
		} finally {
			closeSession(session);
		}
	}

	//	public void updateCashDrawerSync(List<String> ids) {
	//		if (ids == null || ids.isEmpty())
	//			return;
	//
	//		String whereCondition = "(";
	//		for (Iterator iterator = ids.iterator(); iterator.hasNext();) {
	//			String id = (String) iterator.next();
	//			whereCondition += "'" + id + "'";
	//			if (iterator.hasNext())
	//				whereCondition += ",";
	//		}
	//		whereCondition += ")";
//		//@formatter:off
//		Transaction tx=null;
//		Session session = null;
//		try {
//			session = getSession();
//			tx=session.beginTransaction();
//		String hqlString = "update CashDrawer set %s=:synced where %s in"+whereCondition;
//		hqlString = String.format(hqlString, CashDrawer.PROP_CLOUD_SYNCED, CashDrawer.PROP_ID);
//		//@formatter:on
	//			Query query = session.createQuery(hqlString);
	//			query.setParameter("synced", true);
	//			query.executeUpdate();
	//			tx.commit();
	//		} catch (Exception e) {
	//			tx.rollback();
	//			throw e;
	//		} finally {
	//			closeSession(session);
	//		}
	//
	//	}

	public void saveOrUpdateCashDrawer(List<CashDrawer> dataList, boolean updateLastUpdateTime, boolean updateSyncTime) throws Exception {
		if (dataList == null)
			return;

		Transaction tx = null;
		Session session = null;
		try {
			session = createNewSession();
			tx = session.beginTransaction();
			for (Iterator iterator = dataList.iterator(); iterator.hasNext();) {
				CashDrawer item = (CashDrawer) iterator.next();
				List<CashBreakdown> cashBreakdowns = item.getCashBreakdownList();
				item.setCashBreakdownList(null);
				item.setUpdateLastUpdateTime(updateLastUpdateTime);
				item.setUpdateSyncTime(updateSyncTime);

				CashDrawerDAO dao = CashDrawerDAO.getInstance();
				CashDrawer existingItem = dao.get(item.getId());
				if (existingItem != null) {
					if (!BaseDataServiceDao.get().shouldSave(item.getLastUpdateTime(), existingItem.getLastUpdateTime())) {
						PosLog.info(getClass(), item.getId() + " already updated"); //$NON-NLS-1$
						continue;
					}
					long version = existingItem.getVersion();
					item.setVersion(version);
				}
				else {
					save(item, session);
				}
				if (cashBreakdowns != null && cashBreakdowns.size() > 0) {
					for (CashBreakdown breakdown : cashBreakdowns) {
						CashBreakdownDAO breakdownDAO = CashBreakdownDAO.getInstance();
						CashBreakdown existingCashBreakdown = breakdownDAO.get(breakdown.getId());
						if (existingCashBreakdown == null) {
							dao.save(breakdown, session);
						}
						else {
							breakdown.setVersion(existingCashBreakdown.getVersion());
						}
					}
				}
				item.setCashBreakdownList(cashBreakdowns);
				saveOrUpdate(item, session);
			}
			tx.commit();
		} catch (Exception e) {
			tx.rollback();
			throw e;
		} finally {
			closeSession(session);
		}
	}

	public List<CashDrawer> findStaffBank(Date fromDate, Date toDate, User staff) {
		Session session = null;
		try {
			session = getSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.eq(CashDrawer.PROP_TYPE, DrawerType.STAFF_BANK.getTypeNumber()));
			criteria.add(Restrictions.between(CashDrawer.PROP_START_TIME, fromDate, toDate));
			if (staff != null)
				criteria.add(Restrictions.eq(CashDrawer.PROP_ASSIGNED_USER_ID, staff.getId()));

			List<CashDrawer> cashDrawerList = criteria.list();
			if (!cashDrawerList.isEmpty()) {
				for (CashDrawer cashDrawer : cashDrawerList) {
					CashDrawerReportService cashDrawerReportService = new CashDrawerReportService(cashDrawer);
					cashDrawerReportService.populateReport();
				}
			}
			return cashDrawerList;
		} finally {
			closeSession(session);
		}
	}

	public void saveDrawerAssignment(CashDrawer cashDrawer, Terminal terminal) {
		Transaction tx = null;
		try (Session session = createNewSession()) {
			tx = session.beginTransaction();

			if (cashDrawer != null) {
				saveOrUpdate(cashDrawer);
			}
			if (terminal != null) {
				TerminalDAO.getInstance().saveOrUpdate(terminal);
			}

			tx.commit();
		} catch (Exception e) {
			try {
				tx.rollback();
			} catch (Exception x) {
			}
			throw e;
		}
	}

	public boolean isExist(String drawerId) {
		if (StringUtils.isEmpty(drawerId)) {
			return false;
		}
		Session session = null;
		try {
			session = createNewSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.eq(CashDrawer.PROP_ID, drawerId));
			criteria.setProjection(Projections.rowCount());
			Number number = (Number) criteria.uniqueResult();
			return number != null && number.intValue() > 0;
		} finally {
			if (session != null) {
				closeSession(session);
			}
		}
	}

	public void transferUser(CashDrawer cashDrawer, User user) {
		Transaction transaction = null;
		try (Session session = createNewSession()) {
			transaction = session.beginTransaction();
			cashDrawer.setAssignedUser(user);
			update(cashDrawer, session);
			transaction.commit();
		} catch (Exception e) {
			transaction.rollback();
			throw e;
		}
	}

	public CashDrawer findCashDrawerByTerminalId(Integer terminalId) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(CashDrawer.class);
			criteria.add(Restrictions.eq(CashDrawer.PROP_TERMINAL_ID, terminalId));
			criteria.add(Restrictions.isNull(CashDrawer.PROP_REPORT_TIME));
			criteria.addOrder(Order.asc(CashDrawer.PROP_START_TIME));
			criteria.setMaxResults(1);
			return (CashDrawer) criteria.uniqueResult();
		}
	}

}