/**
 * ************************************************************************
 * * The contents of this file are subject to the MRPL 1.2
 * * (the  "License"),  being   the  Mozilla   Public  License
 * * Version 1.1  with a permitted attribution clause; you may not  use this
 * * file except in compliance with the License. You  may  obtain  a copy of
 * * the License at http://www.floreantpos.org/license.html
 * * Software distributed under the License  is  distributed  on  an "AS IS"
 * * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * * License for the specific  language  governing  rights  and  limitations
 * * under the License.
 * * The Original Code is FLOREANT POS.
 * * The Initial Developer of the Original Code is OROCUBE LLC
 * * All portions are Copyright (C) 2015 OROCUBE LLC
 * * All Rights Reserved.
 * ************************************************************************
 */
package com.floreantpos.model.dao;

import java.io.Serializable;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.apache.commons.lang.StringUtils;
import org.hibernate.Criteria;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.hibernate.criterion.MatchMode;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.Projections;
import org.hibernate.criterion.Restrictions;
import org.hibernate.transform.Transformers;

import com.floreantpos.PosException;
import com.floreantpos.PosLog;
import com.floreantpos.model.IUnit;
import com.floreantpos.model.InventoryLocation;
import com.floreantpos.model.InventoryTransaction;
import com.floreantpos.model.InventoryTransactionType;
import com.floreantpos.model.InventoryUnit;
import com.floreantpos.model.InventoryVendor;
import com.floreantpos.model.MenuItem;
import com.floreantpos.model.PurchaseOrder;
import com.floreantpos.model.PurchaseOrderItem;
import com.floreantpos.model.util.DateUtil;
import com.floreantpos.services.InventoryService;
import com.floreantpos.swing.PaginatedListModel;
import com.floreantpos.util.POSUtil;
import com.orocube.rest.service.server.BaseDataServiceDao;

public class PurchaseOrderDAO extends BasePurchaseOrderDAO {

	/**
	 * Default constructor.  Can be used in place of getInstance()
	 */
	public PurchaseOrderDAO() {
	}

	@Override
	protected Serializable save(Object obj, Session s) {
		updateTime(obj);
		return super.save(obj, s);
	}

	@Override
	protected void update(Object obj, Session s) {
		updateTime(obj);
		super.update(obj, s);
	}

	@Override
	protected void saveOrUpdate(Object obj, Session s) {
		updateTime(obj);
		super.saveOrUpdate(obj, s);
	}

	@Override
	protected void delete(Object obj, Session s) {
		if (obj instanceof PurchaseOrder) {
			PurchaseOrder purchaseOrder = (PurchaseOrder) obj;
			purchaseOrder.setDeleted(Boolean.TRUE);
			super.update(purchaseOrder, s);
		}
		else {
			super.delete(obj, s);
		}
	}

	public synchronized void saveOrUpdate(PurchaseOrder order, boolean updateStock) {
		Session session = null;
		Transaction tx = null;
		try {
			session = createNewSession();
			tx = session.beginTransaction();

			saveOrUpdate(order, updateStock, session);
			tx.commit();
		} catch (Exception e) {
			if (tx != null) {
				tx.rollback();
			}
			throwException(e);
		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

	public synchronized void saveOrUpdateWithItemAvailBalance(PurchaseOrder order) {
		Session session = null;
		Transaction tx = null;
		try {
			session = createNewSession();
			tx = session.beginTransaction();
			updateAvailableUnit(order, session);
			tx.commit();
		} catch (Exception e) {
			if (tx != null) {
				tx.rollback();
			}
			throwException(e);
		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

	private void updateAvailableUnit(PurchaseOrder purchaseOrder, Session session) {
		List<PurchaseOrderItem> orderItems = purchaseOrder.getOrderItems();
		if (orderItems == null) {
			return;
		}
		for (PurchaseOrderItem purchaseOrderItem : orderItems) {

			MenuItem menuItem = purchaseOrderItem.getMenuItem();
			Double quantityToReceive = purchaseOrderItem.getItemQuantity();

			MenuItemDAO.getInstance().updateStockQuantity(menuItem.getId(), purchaseOrder.getOutletId(), quantityToReceive, true, false, session);
		}
		session.saveOrUpdate(purchaseOrder);
	}

	public void saveOrUpdate(PurchaseOrder order, boolean updateStock, Session session) {
		if (order.getId() == null) {
			order.setCreatedDate(new Date());
			order.setStatus(PurchaseOrder.ORDER_PENDING);
		}
		if (updateStock) {
			adjustInventoryItems(session, order);
		}
		session.saveOrUpdate(order);
	}

	public void saveOrUpdate(PurchaseOrder order, boolean updateStock, boolean isAvailableUnitUpdate) {

		Transaction tx = null;
		try (final Session session = createNewSession()) {
			tx = session.beginTransaction();

			if (order.getId() == null) {
				if (order.getCreatedDate() == null) {
					order.setCreatedDate(new Date());
				}
				order.setStatus(PurchaseOrder.ORDER_PENDING);
			}
			if (updateStock) {
				List<PurchaseOrderItem> orderItems = order.getOrderItems();
				if (orderItems != null) {
					Map<String, MenuItem> itemMap = new HashMap<String, MenuItem>();
					orderItems.forEach(orderItem -> {
						MenuItem menuItem = itemMap.get(orderItem.getMenuItemId());
						if (menuItem == null) {
							menuItem = orderItem.getMenuItem();
							if (menuItem != null) {
								MenuItemDAO.getInstance().refresh(menuItem, session);
								itemMap.put(menuItem.getId(), menuItem);
							}
						}
						String selectedUnitID = orderItem.getItemUnitName();
						IUnit iUnit = InventoryUnitDAO.getInstance().findByCode(selectedUnitID);
						InventoryUnit selectedUnit = null;
						if (iUnit instanceof InventoryUnit) {
							selectedUnit = (InventoryUnit) iUnit;
						}
						InventoryService.getInstance().adjustMenuItemCost(menuItem, orderItem.getUnitPrice(), orderItem.getItemQuantity(), selectedUnit,
								session);
					});
					if (itemMap.size() > 0) {
						for (MenuItem item : itemMap.values()) {
							MenuItemDAO.getInstance().saveOrUpdate(item, session);
						}
					}
				}

				adjustInventoryItems(session, order, isAvailableUnitUpdate);
			}
			session.saveOrUpdate(order);
			tx.commit();
		} catch (Exception e) {
			if (tx != null) {
				tx.rollback();
			}
			throwException(e);
		}
	}

	private void adjustInventoryItems(Session session, PurchaseOrder order) {
		adjustInventoryItems(session, order, true);
	}

	private void adjustInventoryItems(Session session, PurchaseOrder order, boolean isAvailableUnitUpdate) {
		List<PurchaseOrderItem> orderItems = order.getOrderItems();
		if (orderItems == null) {
			return;
		}
		boolean fullyReceived = true;
		for (PurchaseOrderItem orderItem : orderItems) {
			orderItem.setQuantityReceived(orderItem.getQuantityReceived() + orderItem.getQuantityToReceive());
			InventoryTransaction stockInTrans = new InventoryTransaction();
			MenuItem menuItem = orderItem.getMenuItem();
			stockInTrans.setMenuItem(menuItem);
			stockInTrans.setQuantity(orderItem.getQuantityToReceive());
			Double unitPrice = orderItem.getUnitPrice();
			stockInTrans.setUnitCost(unitPrice);
			stockInTrans.setToInventoryLocation(order.getInventoryLocation());
			stockInTrans.setTransactionDate(order.getReceivingDate());
			stockInTrans.setOutletId(order.getOutletId());
			stockInTrans.setVendor(order.getVendor());
			stockInTrans.setUnit(orderItem.getItemUnitName());
			stockInTrans.setTotal(stockInTrans.getQuantity() * unitPrice);
			if (order.getType().equals(PurchaseOrder.DEBIT)) {
				stockInTrans.setReason(InventoryTransaction.REASON_PURCHASE);
				stockInTrans.setTransactionType(InventoryTransactionType.IN);
			}
			else {
				stockInTrans.setReason(InventoryTransaction.REASON_RETURN);
				stockInTrans.setTransactionType(InventoryTransactionType.OUT);
			}
			InventoryTransactionDAO.getInstance().adjustInventoryStock(stockInTrans, session, isAvailableUnitUpdate, true);
			if (orderItem.getQuantityReceived().doubleValue() != orderItem.getItemQuantity().doubleValue()) {
				fullyReceived = false;
			}
			orderItem.setQuantityToReceive(0.0);
		}
		if (fullyReceived) {
			order.setStatus(PurchaseOrder.ORDER_FULLY_RECEIVED);
		}
		else {
			order.setStatus(PurchaseOrder.ORDER_PARTIALLY_RECEIVED);
		}
	}

	public String getNextOrderSequenceNumber() {
		Session session = null;
		try {
			session = getSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.setProjection(Projections.max(PurchaseOrder.PROP_ORDER_ID));
			addDeletedFilter(criteria);
			Object maxNumber = criteria.uniqueResult();
			if (maxNumber == null)
				return "10001";
			try {
				return String.valueOf(Integer.parseInt((String) maxNumber) + 1);
			} catch (Exception e) {
				return "";
			}
		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

	public boolean exists(String purchaseOrderNo, String id) {
		if (StringUtils.isBlank(purchaseOrderNo)) {
			return false;
		}
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.setProjection(Projections.rowCount());
			criteria.add(Restrictions.eq(PurchaseOrder.PROP_ORDER_ID, purchaseOrderNo));
			addDeletedFilter(criteria);
			if (StringUtils.isNotBlank(id)) {
				criteria.add(Restrictions.ne(PurchaseOrder.PROP_ID, id));
			}
			Number rowCount = (Number) criteria.uniqueResult();
			return rowCount != null && rowCount.intValue() > 0;
		}
	}

	@SuppressWarnings("unchecked")
	public List<PurchaseOrder> findBy(String poNumber, InventoryVendor vendor, Date from, Date to) {
		Session session = null;
		try {
			session = getSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			if (StringUtils.isNotEmpty(poNumber)) {
				criteria.add(Restrictions.ilike(PurchaseOrder.PROP_ORDER_ID, poNumber.trim(), MatchMode.START));
			}
			if (vendor != null) {
				criteria.add(Restrictions.eq(PurchaseOrder.PROP_VENDOR, vendor));
			}
			if (from != null && to != null) {
				criteria.add(Restrictions.ge(PurchaseOrder.PROP_CREATED_DATE, from));
				criteria.add(Restrictions.le(PurchaseOrder.PROP_CREATED_DATE, to));
			}
			addDeletedFilter(criteria);
			return criteria.list();
		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

	//download and upload
	public void saveOrUpdatePurchaseOrders(List<PurchaseOrder> dataList, boolean updateLastUpdateTime, boolean updateSyncTime) throws Exception {
		if (dataList == null)
			return;

		Map<String, Object> existingItemMap = new HashMap<>();
		try (Session session = createNewSession()) {
			for (PurchaseOrder purchaseOrder : dataList) {
				PurchaseOrder existingOrder = PurchaseOrderDAO.getInstance().get(purchaseOrder.getId(), session);
				if (existingOrder != null) {
					existingItemMap.put(purchaseOrder.getId(), purchaseOrder);

					List<PurchaseOrderItem> orderItems = existingOrder.getOrderItems();
					for (PurchaseOrderItem purchaseOrderItem : orderItems) {
						existingItemMap.put(purchaseOrderItem.getId(), purchaseOrderItem);
					}
				}
			}
		}

		Transaction tx = null;
		try (Session session = createNewSession()) {
			for (Iterator<PurchaseOrder> iterator = dataList.iterator(); iterator.hasNext();) {
				PurchaseOrder purchaseOrder = (PurchaseOrder) iterator.next();

				try {
					tx = session.beginTransaction();

					List<PurchaseOrderItem> purchaseOrderItems = POSUtil.copyList(purchaseOrder.getOrderItems());
					POSUtil.clear(purchaseOrder.getOrderItems());
					purchaseOrder.setUpdateLastUpdateTime(updateLastUpdateTime);
					purchaseOrder.setUpdateSyncTime(updateSyncTime);

					PurchaseOrder existingPurchaseOrder = (PurchaseOrder) existingItemMap.get(purchaseOrder.getId());
					if (existingPurchaseOrder != null) {
						if (!BaseDataServiceDao.get().shouldSave(purchaseOrder.getLastUpdateTime(), existingPurchaseOrder.getLastUpdateTime())) {
							PosLog.debug(getClass(), purchaseOrder.getId() + " already updated"); //$NON-NLS-1$
							continue;
						}
						long version = existingPurchaseOrder.getVersion();
						purchaseOrder.setVersion(version);
					}
					else {
						save(purchaseOrder, session);
					}
					for (PurchaseOrderItem purchaseOrderItem : purchaseOrderItems) {
						purchaseOrderItem.setPurchaseOrder(purchaseOrder);

						PurchaseOrderItem existingPurchaseOrderItem = (PurchaseOrderItem) existingItemMap.get(purchaseOrderItem.getId());
						if (existingPurchaseOrderItem == null) {
							purchaseOrderItem.setVersion(0);
							PurchaseOrderItemDAO.getInstance().save(purchaseOrderItem, session);
						}
						else {
							purchaseOrderItem.setVersion(existingPurchaseOrderItem.getVersion());
							PurchaseOrderItemDAO.getInstance().update(purchaseOrderItem, session);
						}
					}
					if (purchaseOrder.getOrderItems() != null) {
						purchaseOrder.getOrderItems().addAll(purchaseOrderItems);
					}
					else {
						purchaseOrder.setOrderItems(purchaseOrderItems);
					}

					update(purchaseOrder, session);

					tx.commit();
				} catch (Exception e) {
					PosLog.error(getReferenceClass(), "Error saving purchase order " + purchaseOrder.getId());
					if (tx != null) {
						tx.rollback();
					}
					throw e;
				}
			}
		}
	}

	@SuppressWarnings("unchecked")
	public List<PurchaseOrder> getPurchaseOrderByInvLocation(InventoryLocation inventoryLocation, Session session) {
		Criteria criteria = session.createCriteria(getReferenceClass());
		criteria.add(Restrictions.eq(PurchaseOrder.PROP_INVENTORY_LOCATION, inventoryLocation));
		criteria.add(Restrictions.ne(PurchaseOrder.PROP_STATUS, PurchaseOrder.ORDER_FULLY_RECEIVED));
		criteria.setProjection(Projections.alias(Projections.property(PurchaseOrder.PROP_ID), PurchaseOrder.PROP_ID));
		addDeletedFilter(criteria);
		return criteria.setResultTransformer(Transformers.aliasToBean(getReferenceClass())).list();
	}

	public void loadPurchaseOrder(PaginatedListModel listModel) throws PosException {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(PurchaseOrder.class);
			addDeletedFilter(criteria);

			listModel.setNumRows(rowCount(criteria));

			criteria.setFirstResult(listModel.getCurrentRowIndex());
			criteria.setMaxResults(listModel.getPageSize());

			criteria.addOrder(Order.desc(PurchaseOrder.PROP_CREATED_DATE));
			listModel.setData(criteria.list());
		}
	}

	public int countPurchaseOrderByCreatedDate(Date createdDate) throws PosException {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(PurchaseOrder.class);
			addDeletedFilter(criteria);

			if (createdDate != null) {
				criteria.add(Restrictions.between(PurchaseOrder.PROP_CREATED_DATE, DateUtil.startOfDay(createdDate), DateUtil.endOfDay(createdDate)));
			}
			return rowCount(criteria);
		}
	}
}