package com.floreantpos.model.dao;

import java.io.Serializable;
import java.util.Iterator;
import java.util.List;

import org.apache.commons.beanutils.PropertyUtils;
import org.apache.commons.lang.StringUtils;
import org.hibernate.Criteria;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.hibernate.criterion.Restrictions;

import com.floreantpos.Messages;
import com.floreantpos.PosException;
import com.floreantpos.PosLog;
import com.floreantpos.model.Department;
import com.floreantpos.model.OrderType;
import com.floreantpos.model.PriceRule;
import com.floreantpos.model.SalesArea;
import com.floreantpos.model.ShopTable;
import com.floreantpos.model.Terminal;
import com.orocube.rest.service.server.BaseDataServiceDao;

public class SalesAreaDAO extends BaseSalesAreaDAO {

	/**
	 * Default constructor.  Can be used in place of getInstance()
	 * @param department 
	 */

	@Override
	protected Serializable save(Object obj, Session s) {
		updateTime(obj);
		return super.save(obj, s);
	}

	@Override
	protected void update(Object obj, Session s) {
		updateTime(obj);
		super.update(obj, s);
	}

	@Override
	protected void saveOrUpdate(Object obj, Session s) {
		updateTime(obj);
		super.saveOrUpdate(obj, s);
	}

	@Override
	protected void delete(Object obj, Session session) {
		SalesArea bean = (SalesArea) obj;
		if (bean == null) {
			throw new PosException(Messages.getString("SalesAreaDAO.0")); //$NON-NLS-1$
		}
		checkForeignRelation(bean);
		bean.setDeleted(Boolean.TRUE);
		update(bean, session);
	}

	private void checkForeignRelation(SalesArea bean) throws PosException {
		String id = bean.getId();
		StringBuilder ref = new StringBuilder();
		StringBuilder details = new StringBuilder();
		try (Session session = createNewSession()) {
			GenericDAO dao = GenericDAO.getInstance();
			append(dao.getForeignDataListNames(session, OrderType.class, OrderType.PROP_SALES_AREA_ID, id), Messages.getString("SalesAreaDAO.1"), ref, details); //$NON-NLS-1$
			append(dao.getForeignDataListNames(session, PriceRule.class, PriceRule.PROP_SALES_AREA_ID, id), Messages.getString("SalesAreaDAO.2"), ref, details); //$NON-NLS-1$
			append(dao.getForeignDataListNames(session, ShopTable.class, ShopTable.PROP_SALES_AREA_ID, id), Messages.getString("SalesAreaDAO.3"), ref, details); //$NON-NLS-1$
			append(dao.getForeignDataListNames(session, Terminal.class, Terminal.PROP_SALES_AREA_ID, id), Messages.getString("SalesAreaDAO.4"), ref, details); //$NON-NLS-1$
		}
		String foreignItemDetails = details.toString();
		if (StringUtils.isNotBlank(foreignItemDetails)) {
			String foreignReferences = ref.toString();
			if (foreignReferences.endsWith(", ")) { //$NON-NLS-1$
				foreignReferences = foreignReferences.substring(0, foreignReferences.length() - 2);
			}
			throw new PosException(Messages.getString("SalesAreaDAO.6") + foreignReferences + Messages.getString("SalesAreaDAO.7") //$NON-NLS-1$ //$NON-NLS-2$
					+ foreignReferences + ".", foreignItemDetails); //$NON-NLS-1$
		}
	}

	private void append(List<String> foreignDataListNames, String ref, StringBuilder refDetails, StringBuilder details) {
		if (foreignDataListNames == null || foreignDataListNames.isEmpty()) {
			return;
		}
		details.append(Messages.getString("SalesAreaDAO.9") + ref + ":"); //$NON-NLS-1$ //$NON-NLS-2$
		refDetails.append(ref + ", "); //$NON-NLS-1$
		int count = 1;
		for (String itemName : foreignDataListNames) {
			details.append("\n" + count + ". " + itemName); //$NON-NLS-1$ //$NON-NLS-2$
			count++;
		}
		details.append("\n"); //$NON-NLS-1$
	}

	@Override
	public List<SalesArea> findAll() {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			addDeletedFilter(criteria);
			return criteria.list();
		}
	}

	public List<SalesArea> findSalesAreaByDept(Department department) {
		Session session = null;

		try {
			session = getSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.eq(SalesArea.PROP_DEPARTMENT_ID, department.getId()));
			addDeletedFilter(criteria);
			return criteria.list();
		} catch (Exception e) {
			PosLog.error(SalesAreaDAO.class, e);
		} finally {
			if (session != null) {
				session.close();
			}
		}
		return null;
	}

	public SalesArea findById(String id) {
		Session session = null;

		try {
			session = getSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.eq(SalesArea.PROP_ID, id));
			addDeletedFilter(criteria);
			return (SalesArea) criteria.uniqueResult();

		} finally {
			if (session != null) {
				closeSession(session);
			}
		}
	}

	public SalesArea find(SalesArea salesArea) {
		Session session = null;

		try {
			if (salesArea == null) {
				return null;
			}
			session = getSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.eq(SalesArea.PROP_ID, salesArea.getId()));
			addDeletedFilter(criteria);
			return (SalesArea) criteria.uniqueResult();

		} finally {
			if (session != null) {
				closeSession(session);
			}
		}
	}

	public SalesArea findByName(String name) {
		Session session = null;

		try {
			session = getSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.eq(SalesArea.PROP_NAME, name));
			addDeletedFilter(criteria);
			return (SalesArea) criteria.uniqueResult();

		} finally {
			if (session != null) {
				closeSession(session);
			}
		}
	}

	public void saveOrUpdateSalesArea(List<SalesArea> salesAreaList, boolean updateLastUpdateTime, boolean updateSyncTime) throws Exception {
		if (salesAreaList == null)
			return;

		Transaction tx = null;
		Session session = null;
		try {
			session = createNewSession();
			tx = session.beginTransaction();
			for (Iterator iterator = salesAreaList.iterator(); iterator.hasNext();) {
				SalesArea item = (SalesArea) iterator.next();
				SalesAreaDAO dao = SalesAreaDAO.getInstance();
				SalesArea existingItem = dao.get(item.getId());
				if (existingItem != null) {
					if (!BaseDataServiceDao.get().shouldSave(item.getLastUpdateTime(), existingItem.getLastUpdateTime())) {
						PosLog.info(getClass(), item.getName() + " already updated"); //$NON-NLS-1$
						continue;
					}
					long version = existingItem.getVersion();
					PropertyUtils.copyProperties(existingItem, item);
					existingItem.setVersion(version);
					existingItem.setUpdateLastUpdateTime(updateLastUpdateTime);
					existingItem.setUpdateSyncTime(updateSyncTime);
					dao.update(existingItem);
				}
				else {
					item.setVersion(0);
					item.setUpdateLastUpdateTime(updateLastUpdateTime);
					item.setUpdateSyncTime(updateSyncTime);
					dao.save(item);
				}
			}
			tx.commit();
		} catch (Exception e) {
			tx.rollback();
			throw e;
		} finally {
			closeSession(session);
		}

	}

}