package com.floreantpos.model.dao;

import java.io.Serializable;
import java.util.Iterator;
import java.util.List;

import org.hibernate.Criteria;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.hibernate.criterion.Restrictions;

import com.floreantpos.constants.AppConstants;
import com.floreantpos.model.BookingInfo;
import com.floreantpos.model.ShopTable;
import com.floreantpos.model.ShopTableStatus;
import com.floreantpos.model.ShopTableTicket;
import com.floreantpos.model.TableStatus;
import com.floreantpos.model.Ticket;
import com.floreantpos.util.CopyUtil;

public class ShopTableStatusDAO extends BaseShopTableStatusDAO {

	/**
	 * Default constructor.  Can be used in place of getInstance()
	 */
	public ShopTableStatusDAO() {
	}

	@Override
	protected Serializable save(Object obj, Session s) {
		ShopTableStatus shopTableStatus = (ShopTableStatus) obj;
		updateTime(shopTableStatus);
		Serializable save = super.save(obj, s);
		publishOnMqtt(shopTableStatus);
		return save;
	}

	@Override
	protected void update(Object obj, Session s) {
		ShopTableStatus shopTableStatus = (ShopTableStatus) obj;
		updateTime(shopTableStatus);
		super.update(obj, s);
		publishOnMqtt(shopTableStatus);
	}

	@Override
	protected void saveOrUpdate(Object obj, Session s) {
		ShopTableStatus shopTableStatus = (ShopTableStatus) obj;
		updateTime(shopTableStatus);
		super.saveOrUpdate(obj, s);
		publishOnMqtt(shopTableStatus);
	}

	private void publishOnMqtt(ShopTableStatus shopTableStatus) {
		if (!shopTableStatus.isShouldPublishMqtt()) {
			return;
		}
	}

	public void addTicketsToShopTableStatus(List<Integer> tableNumbers, List<Ticket> tickets, Session session) {
		if (tableNumbers == null || tableNumbers.isEmpty() || tickets == null || tickets.isEmpty())
			return;
		//TODO:
		String outletId = tickets.get(0).getOutletId();
		for (Integer tableNumber : tableNumbers) {
			ShopTableStatus shopTableStatus = get(tableNumber, outletId);
			if (shopTableStatus == null) {
				shopTableStatus = new ShopTableStatus();
				shopTableStatus.setId(tableNumber);
			}
			shopTableStatus.setTableStatus(TableStatus.Serving);
			shopTableStatus.addToTableTickets(tickets);
			if (session == null)
				saveOrUpdate(shopTableStatus);
			else
				saveOrUpdate(shopTableStatus, session);
		}
	}

	public void removeTicketFromShopTableStatus(Ticket ticket, Session session) {
		if (ticket == null)
			return;

		List<Integer> tableNumbers = ticket.getTableNumbers();
		if (tableNumbers == null || tableNumbers.isEmpty())
			return;

		for (Integer tableNumber : tableNumbers) {
			ShopTableStatus shopTableStatus = session == null ? get(tableNumber, ticket.getOutletId()) : get(tableNumber, ticket.getOutletId(), session);
			if (shopTableStatus == null)
				return;
			List<ShopTableTicket> ticketNumbers = shopTableStatus.getTicketNumbers();
			if (ticketNumbers != null) {
				for (Iterator iterator = ticketNumbers.iterator(); iterator.hasNext();) {
					ShopTableTicket shopTableTicket = (ShopTableTicket) iterator.next();
					if (shopTableTicket.getTicketId().equals(ticket.getId())) {
						iterator.remove();
					}
				}
			}
			shopTableStatus.setTicketNumbers(ticketNumbers);
			if (ticketNumbers == null || ticketNumbers.isEmpty()) {
				shopTableStatus.setTicketNumbers(null);
				shopTableStatus.setTableStatus(TableStatus.Available);
				shopTableStatus.removeProperty(ShopTableStatus.SEAT_TIME);
			}

			// Shop table and booking info update
			ShopTableDAO shopTableDAO = ShopTableDAO.getInstance();
			ShopTable shopTable = shopTableDAO.get(tableNumber, ticket.getOutletId());
			if (shopTable != null) {
				if (shopTable.getCurrentBookingId() != null) {
					BookingInfoDAO bookingInfoDAO = BookingInfoDAO.getInstance();
					BookingInfo bookingInfo = bookingInfoDAO.get(shopTable.getCurrentBookingId());
					if (bookingInfo != null) {
						bookingInfo.setStatus(BookingInfo.STATUS_CLOSE);
						bookingInfo.setClosed(true);
						bookingInfoDAO.saveOrUpdate(bookingInfo);
					}
				}
				shopTable.setCurrentBookingId(null);
				shopTable.setCustomerName(null);
				shopTable.removeProperty(AppConstants.SHOP_TABLE_CUSTOMER_ID);
				shopTable.removeProperty(ShopTable.RESERVATION_NUMBER);
				shopTableDAO.update(shopTable, session);
			}

			shopTableStatus.setShouldPublishMqtt(ticket.isShouldPublishMqtt());
			if (session == null) {
				ShopTableStatusDAO.getInstance().saveOrUpdate(shopTableStatus);
			}
			else {
				ShopTableStatusDAO.getInstance().saveOrUpdate(shopTableStatus, session);
			}
		}
	}

	public ShopTableStatus get(Integer tableNumber, String outletId, Session session) {
		return get(new ShopTableStatus(tableNumber, outletId), session);
	}

	//Upload & Download to new DB
	public void saveShopTableStatus(ShopTableStatus sTableStatus, String clientOutletId, boolean updateLastUpdateTime, boolean updateSyncTime)
			throws Exception {
		if (sTableStatus == null)
			return;

		Transaction tx = null;
		Session session = null;
		try {
			session = createNewSession();
			tx = session.beginTransaction();

			ShopTableStatus existingShopTableStatus = get(sTableStatus.getId(), clientOutletId);
			if (existingShopTableStatus == null) {
				sTableStatus.setUpdateLastUpdateTime(updateLastUpdateTime);
				sTableStatus.setUpdateSyncTime(updateSyncTime);
				save(sTableStatus, session);
			}
			tx.commit();
		} catch (Exception e) {
			tx.rollback();
			throw e;
		} finally {
			session.close();
		}
	}

	public ShopTableStatus updateVersionOfShopTableStatus(ShopTableStatus shopTableStatus) throws Exception {

		ShopTableStatus existingItem = ShopTableStatusDAO.getInstance().get(shopTableStatus.getId(), shopTableStatus.getOutletId());
		if (existingItem != null) {
			long versionItemModifier = existingItem.getVersion();
			CopyUtil.deepCopy(shopTableStatus);
			shopTableStatus.setVersion(versionItemModifier);
		}
		return shopTableStatus;
	}

	public void saveOrUpdateShopTableStatus(List<ShopTableStatus> dataList, boolean updateLastUpdateTime, boolean updateSyncTime) throws Exception {

		if (dataList == null)
			return;

		Transaction tx = null;
		Session session = null;
		try {
			session = createNewSession();
			tx = session.beginTransaction();

			for (Iterator<ShopTableStatus> iterator = dataList.iterator(); iterator.hasNext();) {
				ShopTableStatus item = (ShopTableStatus) iterator.next();
				item.setShouldPublishMqtt(false);

				ShopTableStatus existingItem = get(item.getId(), item.getOutletId());
				if (existingItem != null) {
					item.setVersion(existingItem.getVersion());
					item.setUpdateLastUpdateTime(updateLastUpdateTime);
					item.setUpdateSyncTime(updateSyncTime);
					update(item, session);
				}
				else {
					item.setUpdateLastUpdateTime(updateLastUpdateTime);
					item.setUpdateSyncTime(updateSyncTime);
					save(item, session);
				}
			}
			tx.commit();
		} catch (Exception e) {
			tx.rollback();
			throw e;
		} finally {
			closeSession(session);
		}
	}

	public List<Ticket> findOpenTickets(ShopTableStatus status) {
		return findTickets(status, false);
	}

	public List<Ticket> findTickets(ShopTableStatus status, Boolean includeClosedTickets) {
		if (status == null)
			return null;

		List<String> ticketIds = status.getListOfTicketNumbers();
		if (ticketIds == null || ticketIds.isEmpty())
			return null;

		Session session = null;
		try {
			session = createNewSession();
			Criteria criteria = session.createCriteria(Ticket.class);
			criteria.add(Restrictions.in(Ticket.PROP_ID, ticketIds));
			if (!includeClosedTickets) {
				criteria.add(Restrictions.eq(Ticket.PROP_CLOSED, false));
			}
			return criteria.list();
		} finally {
			closeSession(session);
		}
	}

	public ShopTableStatus get(int selectedTable, String outletId) {
		return get(new ShopTableStatus(selectedTable, outletId));
	}
}