package com.floreantpos.model.dao;

import java.io.InputStream;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.apache.commons.beanutils.PropertyUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.LogFactory;
import org.hibernate.Criteria;
import org.hibernate.NullPrecedence;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.hibernate.criterion.MatchMode;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.ProjectionList;
import org.hibernate.criterion.Projections;
import org.hibernate.criterion.Restrictions;
import org.hibernate.transform.Transformers;

import com.floreantpos.DuplicateDataException;
import com.floreantpos.Messages;
import com.floreantpos.PosLog;
import com.floreantpos.model.ImageResource;
import com.floreantpos.model.ImageResource.IMAGE_CATEGORY;
import com.floreantpos.model.MenuItem;
import com.floreantpos.model.Outlet;
import com.floreantpos.swing.PaginationSupport;
import com.orocube.rest.service.server.BaseDataServiceDao;

public class ImageResourceDAO extends BaseImageResourceDAO {

	/**
	 * Default constructor.  Can be used in place of getInstance()
	 */
	public ImageResourceDAO() {
	}

	@Override
	protected Serializable save(Object obj, Session s) {
		updateTime(obj);
		return super.save(obj, s);
	}

	@Override
	protected void update(Object obj, Session s) {
		updateTime(obj);
		super.update(obj, s);
	}

	@Override
	protected void saveOrUpdate(Object obj, Session s) {
		updateTime(obj);
		super.saveOrUpdate(obj, s);
	}

	public int rowCount() {
		Session session = null;
		Criteria criteria = null;
		try {
			session = createNewSession();
			criteria = session.createCriteria(ImageResource.class);
			criteria.setProjection(Projections.rowCount());
			Number rowCount = (Number) criteria.uniqueResult();
			if (rowCount != null) {
				return rowCount.intValue();

			}
			return 0;
		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

	public ImageResource getImageByDescription(String description) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(ImageResource.class);
			criteria.add(Restrictions.eq(ImageResource.PROP_DESCRIPTION, description));
			criteria.add(Restrictions.ne(ImageResource.PROP_IMAGE_CATEGORY_NUM, IMAGE_CATEGORY.DELETED.getType()));
			criteria.setMaxResults(1);
			return (ImageResource) criteria.uniqueResult();
		}
	}

	public void getImages(PaginationSupport model, IMAGE_CATEGORY imgCategory, String description) {
		Session session = null;
		Criteria criteria = null;
		try {
			session = createNewSession();
			criteria = session.createCriteria(ImageResource.class);
			if (imgCategory != null) {
				criteria.add(Restrictions.eq(ImageResource.PROP_IMAGE_CATEGORY_NUM, imgCategory.getType()));
			}
			if (StringUtils.isNotEmpty(description)) {
				criteria.add(Restrictions.ilike(ImageResource.PROP_DESCRIPTION, description, MatchMode.START));
			}
			if (imgCategory != IMAGE_CATEGORY.DELETED) {
				criteria.add(Restrictions.ne(ImageResource.PROP_IMAGE_CATEGORY_NUM, IMAGE_CATEGORY.DELETED.getType()));
			}

			model.setNumRows(rowCount(criteria));

			criteria.addOrder(Order.desc(ImageResource.PROP_LAST_UPDATE_TIME).nulls(NullPrecedence.LAST));

			int currentRowIndex = model.getCurrentRowIndex();
			criteria.setFirstResult(currentRowIndex);
			criteria.setMaxResults(model.getPageSize());
			List<ImageResource> result = criteria.list();
			model.setRows(result);
		} catch (Exception e) {
			model.setRows(new ArrayList<>());
		} finally {
			closeSession(session);
		}
	}

	public void loadImageProperties(PaginationSupport model, IMAGE_CATEGORY imgCategory, String description) {
		Session session = null;
		Criteria criteria = null;
		try {
			session = createNewSession();
			criteria = session.createCriteria(ImageResource.class);
			if (imgCategory != null) {
				criteria.add(Restrictions.eq(ImageResource.PROP_IMAGE_CATEGORY_NUM, imgCategory.getType()));
			}
			if (StringUtils.isNotEmpty(description)) {
				criteria.add(Restrictions.ilike(ImageResource.PROP_DESCRIPTION, description, MatchMode.START));
			}
			if (imgCategory != IMAGE_CATEGORY.DELETED) {
				criteria.add(Restrictions.ne(ImageResource.PROP_IMAGE_CATEGORY_NUM, IMAGE_CATEGORY.DELETED.getType()));
			}

			model.setNumRows(rowCount(criteria));

			ProjectionList projectionList = Projections.projectionList();
			projectionList.add(Projections.property(ImageResource.PROP_ID), ImageResource.PROP_ID);
			projectionList.add(Projections.property(ImageResource.PROP_DESCRIPTION), ImageResource.PROP_DESCRIPTION);
			projectionList.add(Projections.property(ImageResource.PROP_IMAGE_CATEGORY_NUM), ImageResource.PROP_IMAGE_CATEGORY_NUM);

			criteria.setProjection(projectionList);
			criteria.setResultTransformer(Transformers.aliasToBean(ImageResource.class));
			criteria.addOrder(Order.desc(ImageResource.PROP_LAST_UPDATE_TIME).nulls(NullPrecedence.LAST));

			int currentRowIndex = model.getCurrentRowIndex();
			criteria.setFirstResult(currentRowIndex);
			criteria.setMaxResults(model.getPageSize());
			List<ImageResource> result = criteria.list();
			model.setRows(result);
		} catch (Exception e) {
			model.setRows(new ArrayList<>());
		} finally {
			closeSession(session);
		}
	}

	public ImageResource getDefaultFloorImage(InputStream inputStream) throws Exception {
		Session session = null;
		Criteria criteria = null;
		try {
			session = createNewSession();
			criteria = session.createCriteria(ImageResource.class);
			criteria.addOrder(Order.asc(ImageResource.PROP_ID));
			criteria.add(Restrictions.eq(ImageResource.PROP_IMAGE_CATEGORY_NUM, IMAGE_CATEGORY.FLOORPLAN.getType()));
			criteria.add(Restrictions.eq(ImageResource.PROP_DESCRIPTION, "default_floor_image")); //$NON-NLS-1$
			List list = criteria.list();
			if (list.size() > 0) {
				return (ImageResource) list.get(0);
			}
			else {
				if (inputStream != null) {
					ImageResource imageResource = new ImageResource();
					byte[] byteArray = IOUtils.toByteArray(inputStream);
					imageResource.setImageBytes(byteArray);
					imageResource.setDescription("default_floor_image"); //$NON-NLS-1$
					imageResource.setImageCategoryNum(IMAGE_CATEGORY.FLOORPLAN.getType());
					save(imageResource);
					return imageResource;
				}
			}
		} finally {
			if (session != null) {
				session.close();
			}
			if (inputStream != null) {
				inputStream.close();
			}
		}
		return null;
	}

	public ImageResource findById(String id) {
		try (Session session = createNewSession()) {
			return findById(id, session);
		}
	}

	public ImageResource findById(String id, Session session) {
		if (StringUtils.isBlank(id)) {
			return null;
		}
		Criteria criteria = session.createCriteria(ImageResource.class);
		criteria.add(Restrictions.eq(ImageResource.PROP_ID, id));
		criteria.add(Restrictions.ne(ImageResource.PROP_IMAGE_CATEGORY_NUM, IMAGE_CATEGORY.DELETED.getType()));
		criteria.setMaxResults(1);
	
		return (ImageResource) criteria.uniqueResult();
	}

	public void saveOrUpdate(List<ImageResource> resourceList) {
		Session session = null;
		Transaction tx = null;
		try {
			session = createNewSession();
			tx = session.beginTransaction();

			for (ImageResource resource : resourceList) {
				saveOrUpdate(resource, session);
			}

			tx.commit();
		} catch (Exception e) {
			tx.rollback();
			LogFactory.getLog(ImageResourceDAO.class).error(e);
			throw new RuntimeException(e);
		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

	public void saveOrUpdateImageResourceList(List<ImageResource> dataList, boolean updateLastUpdateTime, boolean updateSyncTime) throws Exception {
		if (dataList == null)
			return;

		Transaction tx = null;
		try (Session session = createNewSession()) {
			tx = session.beginTransaction();

			for (Iterator<ImageResource> iterator = dataList.iterator(); iterator.hasNext();) {
				ImageResource item = (ImageResource) iterator.next();
				saveOrUpdateImageResource(session, item, updateLastUpdateTime, updateSyncTime);
			}
			tx.commit();
		} catch (Exception e) {
			tx.rollback();
			throw e;
		}
	}

	public void saveOrUpdateImageResource(Session session, ImageResource item, boolean updateLastUpdateTime, boolean updateSyncTime) throws Exception {
		if (item.getImageBytes() == null) {
			item.setImageBytes(new byte[0]);
		}
		ImageResource existingItem = get(item.getId());
		if (existingItem != null) {
			if (!BaseDataServiceDao.get().shouldSave(item.getLastUpdateTime(), existingItem.getLastUpdateTime())) {
				PosLog.info(getClass(), item.getId() + " already updated"); //$NON-NLS-1$
				return;
			}
			long version = existingItem.getVersion();
			PropertyUtils.copyProperties(existingItem, item);
			existingItem.setVersion(version);
			existingItem.setUpdateLastUpdateTime(updateLastUpdateTime);
			existingItem.setUpdateSyncTime(updateSyncTime);
			if (session == null) {
				update(existingItem);
			}
			else {
				update(existingItem, session);
			}
		}
		else {
			item.setUpdateLastUpdateTime(updateLastUpdateTime);
			item.setUpdateSyncTime(updateSyncTime);
			if (session == null) {
				save(item);
			}
			else {
				save(item, session);
			}
		}
	}

	public List<String> getMenuItemNamesByImageId(String imageId) {
		try (Session session = this.createNewSession()) {
			Criteria criteria = session.createCriteria(MenuItem.class);
			this.addDeletedFilter(criteria, MenuItem.class);
			criteria.add(Restrictions.eq(MenuItem.PROP_IMAGE_ID, imageId));
			criteria.add(Restrictions.eq(MenuItem.PROP_VARIANT, false));

			criteria.setProjection(Projections.property(MenuItem.PROP_NAME));
			criteria.setMaxResults(50);

			return criteria.list();
		}
	}

	@SuppressWarnings("unchecked")
	public List<String> getAllOutletNamesByImageId(String imageId) {
		try (Session session = this.createNewSession()) {
			Criteria criteria = session.createCriteria(Outlet.class);
			this.addDeletedFilter(criteria);

			//check in properties
			if (imageId != null) {
				criteria.add(Restrictions.like(Outlet.PROP_EXTRA_PROPERTIES, imageId, MatchMode.ANYWHERE));
				criteria.setProjection(Projections.property(Outlet.PROP_NAME));
			}
			return criteria.list();
		}
	}

	public void checkDifferentImageResourceExists(String fileName) {
		checkDifferentImageResourceExists("", "", fileName); //$NON-NLS-1$ //$NON-NLS-2$
	}

	public void checkDifferentImageResourceExists(String existingId, String outletId, String fileName) {
		try (Session session = this.createNewSession()) {
			Criteria criteria = session.createCriteria(ImageResource.class);
			addDeletedFilter(criteria);
			if (StringUtils.isNotEmpty(existingId)) {
				criteria.add(Restrictions.ne("id", existingId)); //$NON-NLS-1$
			}
			if (StringUtils.isNotBlank(outletId)) {
				criteria.add(Restrictions.eq("outletId", outletId)); //$NON-NLS-1$
			}
			criteria.add(Restrictions.ilike(ImageResource.PROP_PROPERTIES, "'" + fileName + "'", MatchMode.ANYWHERE)); //$NON-NLS-1$ //$NON-NLS-2$

			criteria.setProjection(Projections.rowCount());

			Number rowCount = (Number) criteria.uniqueResult();
			if (rowCount != null && rowCount.intValue() > 0) {
				throw new DuplicateDataException(String.format("%s" + " %s " + Messages.getString("GenericDAO.22"), "File ", fileName)); //$NON-NLS-1$ //$NON-NLS-2$//$NON-NLS-3$
			}
		}
	}
}