/**
 * ************************************************************************
 * * The contents of this file are subject to the MRPL 1.2
 * * (the  "License"),  being   the  Mozilla   Public  License
 * * Version 1.1  with a permitted attribution clause; you may not  use this
 * * file except in compliance with the License. You  may  obtain  a copy of
 * * the License at http://www.floreantpos.org/license.html
 * * Software distributed under the License  is  distributed  on  an "AS IS"
 * * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * * License for the specific  language  governing  rights  and  limitations
 * * under the License.
 * * The Original Code is FLOREANT POS.
 * * The Initial Developer of the Original Code is OROCUBE LLC
 * * All portions are Copyright (C) 2015 OROCUBE LLC
 * * All Rights Reserved.
 * ************************************************************************
 */
package com.floreantpos.model.dao;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collection;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import org.apache.commons.beanutils.PropertyUtils;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.logging.LogFactory;
import org.hibernate.Criteria;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.hibernate.criterion.MatchMode;
import org.hibernate.criterion.Projections;
import org.hibernate.criterion.Restrictions;

import com.floreantpos.PosLog;
import com.floreantpos.model.BookingInfo;
import com.floreantpos.model.Customer;
import com.floreantpos.model.ShopTable;
import com.floreantpos.model.ShopTableStatus;
import com.floreantpos.model.TableStatus;
import com.orocube.rest.service.server.BaseDataServiceDao;

import liquibase.util.StringUtils;

public class BookingInfoDAO extends BaseBookingInfoDAO {

	/**
	 * Default constructor.  Can be used in place of getInstance()
	 */
	public BookingInfoDAO() {
	}

	@Override
	protected void delete(Object obj, Session s) {
		if (obj instanceof BookingInfo) {
			BookingInfo bookingInfo = (BookingInfo) obj;
			bookingInfo.setDeleted(Boolean.TRUE);
			super.update(bookingInfo, s);
		}
		else {
			super.delete(obj, s);
		}
	}

	@SuppressWarnings("unchecked")
	@Override
	public List<BookingInfo> findAll() {
		try (Session session = this.createNewSession()) {
			Criteria criteria = session.createCriteria(this.getReferenceClass());
			this.addDeletedFilter(criteria);
			return criteria.list();
		}
	}

	@Override
	protected Serializable save(Object obj, Session s) {
		updateTime(obj);
		return super.save(obj, s);
	}

	@Override
	protected void update(Object obj, Session s) {
		updateTime(obj);
		super.update(obj, s);
	}

	@Override
	protected void saveOrUpdate(Object obj, Session s) {
		updateTime(obj);
		super.saveOrUpdate(obj, s);
	}

	@SuppressWarnings("unchecked")
	public List<BookingInfo> getBookedTables(Date startDate, Date endDate) {
		try (Session session = this.createNewSession()) {
			Criteria criteria = session.createCriteria(this.getReferenceClass());
			this.addDeletedFilter(criteria);
			criteria.add(Restrictions.or(Restrictions.ge(BookingInfo.PROP_FROM_DATE, startDate), Restrictions.le(BookingInfo.PROP_TO_DATE, endDate)));
			return criteria.list();
		} catch (Exception e) {
			PosLog.error(getClass(), e);
		}
		return null;
	}

	@SuppressWarnings("unchecked")
	public List<BookingInfo> getAllOpenBooking() {
		try (Session session = this.createNewSession()) {
			Criteria criteria = session.createCriteria(this.getReferenceClass());
			this.addDeletedFilter(criteria);
			criteria.add(Restrictions.ne(BookingInfo.PROP_STATUS, BookingInfo.STATUS_CLOSE));
			return criteria.list();
		} catch (Exception e) {
			PosLog.error(getClass(), e);
		}
		return null;

	}

	@SuppressWarnings("unchecked")
	public void setUrgentBookingInfoInTables(Set<ShopTable> tables, Session session) {
		Transaction tx = null;
		try {
			tx = session.beginTransaction();
			Criteria criteria = session.createCriteria(getReferenceClass());
			this.addDeletedFilter(criteria);
			criteria.add(Restrictions.eq(BookingInfo.PROP_CLOSED, false));
			criteria.add(Restrictions.eq(BookingInfo.PROP_STATUS, BookingInfo.STATUS_OPEN));

			Calendar cal = Calendar.getInstance();
			Date start = cal.getTime();
			cal.add(Calendar.MINUTE, 30);
			Date end = cal.getTime();

			criteria.add(Restrictions.between(BookingInfo.PROP_FROM_DATE, start, end));

			List<Integer> tableIds = new ArrayList<Integer>();
			for (ShopTable shopTable : tables) {
				tableIds.add(shopTable.getId());
			}
			criteria.createAlias("tables", "tableIds");
			criteria.add(Restrictions.in("tableIds." + ShopTable.PROP_ID, tableIds));

			List<BookingInfo> list = criteria.list();
			List<ShopTable> updatedTableList = new ArrayList<ShopTable>();
			List<BookingInfo> updatedBookingInfoList = new ArrayList<BookingInfo>();
			if (!list.isEmpty()) {
				for (BookingInfo bookingInfo : list) {
					List<ShopTable> bookedTables = bookingInfo.getTables();
					Collection<ShopTable> commonTables = CollectionUtils.intersection(tables, bookedTables);
					for (ShopTable shopTable : commonTables) {
						TableStatus tableStatus = shopTable.getShopTableStatus().getTableStatus();
						if (tableStatus.equals(TableStatus.Available)) {
							shopTable.setCurrentBookingId(bookingInfo.getId());
							shopTable.addProperty(ShopTable.RESERVATION_NUMBER, bookingInfo.getBookingId());
							shopTable.setTableStatus(TableStatus.Booked);
							if (bookingInfo.getCustomer() != null) {
								shopTable.setCustomerName(bookingInfo.getCustomer().getName());
							}
							updatedTableList.add(shopTable);
						}
					}
				}
			}
			if (!updatedTableList.isEmpty()) {
				ShopTableDAO.getInstance().updateTableList(updatedTableList, session);
			}
			if (!updatedBookingInfoList.isEmpty()) {
				for (BookingInfo bookingInfo : updatedBookingInfoList) {
					update(bookingInfo, session);
				}
			}
			tx.commit();
		} catch (Exception e) {
			if (tx != null) {
				tx.rollback();
			}
			PosLog.error(getClass(), e);
		}
	}

	public void setBookingStatus(BookingInfo bookingInfo, String bookingStatus, List<ShopTableStatus> tableStatusList) {

		Session session = null;
		Transaction tx = null;

		try {
			session = createNewSession();
			tx = session.beginTransaction();

			bookingInfo.setStatus(bookingStatus);
			saveOrUpdate(bookingInfo);

			if (bookingStatus.equals(BookingInfo.STATUS_SEAT) || bookingStatus.equals(BookingInfo.STATUS_DELAY)) {
				ShopTableDAO.getInstance().bookedTables(tableStatusList);
			}

			if (bookingStatus.equals(BookingInfo.STATUS_CANCEL) || bookingStatus.equals(BookingInfo.STATUS_NO_APR)
					|| bookingStatus.equals(BookingInfo.STATUS_CLOSE)) {
				ShopTableDAO.getInstance().freeTables(tableStatusList);
			}

			tx.commit();

		} catch (Exception e) {
			tx.rollback();
			LogFactory.getLog(BookingInfo.class).error(e);
			throw new RuntimeException(e);
		} finally {
			closeSession(session);
		}

	}

	public List<BookingInfo> getTodaysBooking() {
		return getTodaysBooking(null, null);
	}

	@SuppressWarnings("unchecked")
	public List<BookingInfo> getTodaysBooking(String bookingStatus, String searchTxt) {
		try (Session session = this.createNewSession()) {
			Calendar startDate = Calendar.getInstance();
			startDate.setLenient(false);
			startDate.setTime(new Date());
			startDate.set(Calendar.HOUR_OF_DAY, 0);
			startDate.set(Calendar.MINUTE, 0);
			startDate.set(Calendar.SECOND, 0);
			startDate.set(Calendar.MILLISECOND, 0);

			Calendar endDate = Calendar.getInstance();
			endDate.setLenient(false);
			endDate.setTime(new Date());
			endDate.set(Calendar.HOUR_OF_DAY, 23);
			endDate.set(Calendar.MINUTE, 59);
			endDate.set(Calendar.SECOND, 59);

			Criteria criteria = null;
			List<String> customerIds = null;

			if (StringUtils.isNotEmpty(searchTxt)) {
				criteria = session.createCriteria(Customer.class);
				this.addDeletedFilter(criteria);
				// @formatter:off
				criteria.add(
						Restrictions.or(
								Restrictions.ilike(Customer.PROP_NAME, searchTxt, MatchMode.ANYWHERE),
								Restrictions.ilike(Customer.PROP_MOBILE_NO,  searchTxt, MatchMode.START)
								)
						);
				// @formatter:on
				criteria.setProjection(Projections.property(Customer.PROP_ID));
				customerIds = criteria.list();
			}
			criteria = session.createCriteria(BookingInfo.class);
			this.addDeletedFilter(criteria);
			if (bookingStatus == null) {
				criteria.add(Restrictions.ge(BookingInfo.PROP_FROM_DATE, startDate.getTime()))
						.add(Restrictions.le(BookingInfo.PROP_FROM_DATE, endDate.getTime()));
			}
			else {
				criteria.add(Restrictions.ge(BookingInfo.PROP_FROM_DATE, startDate.getTime()))
						.add(Restrictions.le(BookingInfo.PROP_FROM_DATE, endDate.getTime())).add(Restrictions.eq(BookingInfo.PROP_STATUS, bookingStatus));
			}

			if (customerIds != null && !customerIds.isEmpty()) {
				criteria.add(Restrictions.in(BookingInfo.PROP_CUSTOMER_ID, customerIds));
			}
			return criteria.list();
		} catch (Exception e) {
			PosLog.error(getClass(), e);
		}
		return null;
	}

	@SuppressWarnings("unchecked")
	public List<ShopTable> getAllBookedTablesByDate(Date startDate, Date endDate) {
		try (Session session = this.createNewSession()) {
			Criteria criteria = session.createCriteria(BookingInfo.class);
			this.addDeletedFilter(criteria);
			criteria.add(Restrictions.and(Restrictions.gt(BookingInfo.PROP_TO_DATE, startDate), Restrictions.lt(BookingInfo.PROP_FROM_DATE, endDate)))
					.add(Restrictions.eq(BookingInfo.PROP_STATUS, BookingInfo.STATUS_OPEN));

			List<BookingInfo> list = criteria.list();
			List<ShopTable> bookedTableList = new ArrayList<ShopTable>();
			for (Iterator<BookingInfo> iterator = list.iterator(); iterator.hasNext();) {
				BookingInfo bookingInfo = iterator.next();
				for (ShopTable shopTable : bookingInfo.getTables()) {
					bookedTableList.add(shopTable);
				}
			}
			return bookedTableList;
		} catch (Exception e) {
			PosLog.error(getClass(), e);
		}
		return null;
	}

	public List<BookingInfo> getAllBookingInfoByDate(Date startDate, Date endDate) {
		return getAllBookingInfoByDate(startDate, endDate, null, null);
	}

	@SuppressWarnings("unchecked")
	public List<BookingInfo> getAllBookingInfoByDate(Date startDate, Date endDate, String status, String searchTxt) {
		try (Session session = this.createNewSession()) {
			Criteria criteria = null;
			List<String> customerIds = null;
			if (StringUtils.isNotEmpty(searchTxt)) {
				criteria = session.createCriteria(Customer.class);
				this.addDeletedFilter(criteria);
				// @formatter:off
				criteria.add(
						Restrictions.or(
								Restrictions.ilike(Customer.PROP_NAME, searchTxt, MatchMode.ANYWHERE),
								Restrictions.ilike(Customer.PROP_MOBILE_NO,  searchTxt, MatchMode.START)
								)
						);
				// @formatter:on
				criteria.setProjection(Projections.property(Customer.PROP_ID));
				customerIds = criteria.list();
			}
			criteria = session.createCriteria(BookingInfo.class);
			this.addDeletedFilter(criteria);
			if (status == null) {
				criteria.add(Restrictions.ge(BookingInfo.PROP_FROM_DATE, startDate)).add(Restrictions.lt(BookingInfo.PROP_FROM_DATE, endDate));
			}
			else {
				criteria.add(Restrictions.ge(BookingInfo.PROP_FROM_DATE, startDate)).add(Restrictions.lt(BookingInfo.PROP_FROM_DATE, endDate))
						.add(Restrictions.eq(BookingInfo.PROP_STATUS, status));
			}
			if (customerIds != null && !customerIds.isEmpty()) {
				criteria.add(Restrictions.in(BookingInfo.PROP_CUSTOMER_ID, customerIds));
			}
			return criteria.list();
		} catch (Exception e) {
			PosLog.error(getClass(), e);
		}
		return null;
	}

	public BookingInfo findById(String currentBookingId) {
		if (StringUtils.isEmpty(currentBookingId)) {
			return null;
		}
		try (Session session = this.createNewSession()) {
			Criteria criteria = session.createCriteria(BookingInfo.class);
			this.addDeletedFilter(criteria);
			criteria.add(Restrictions.eq(BookingInfo.PROP_ID, currentBookingId));
			return (BookingInfo) criteria.uniqueResult();
		} catch (Exception e) {
			PosLog.error(getClass(), e);
		}
		return null;
	}

	@SuppressWarnings("unchecked")
	public List<BookingInfo> findByShopTable(ShopTable table, Session session) {
		try {
			Criteria criteria = session.createCriteria(BookingInfo.class);
			this.addDeletedFilter(criteria);
			criteria.createAlias("tables", "tables");
			criteria.add(Restrictions.eq("tables." + ShopTable.PROP_ID, table.getId()));
			return criteria.list();
		} catch (Exception e) {
			PosLog.error(getClass(), e);
		}
		return null;
	}

	public void saveOrUpdateBookingInfos(List<BookingInfo> dataList, boolean updateLastUpdateTime, boolean updateSyncTime) throws Exception {

		if (dataList == null)
			return;

		Transaction tx = null;
		Session session = null;
		try {
			session = createNewSession();
			tx = session.beginTransaction();

			for (Iterator<BookingInfo> iterator = dataList.iterator(); iterator.hasNext();) {
				BookingInfo item = (BookingInfo) iterator.next();
				BookingInfo existingItem = get(item.getId());
				if (existingItem != null) {
					if (!BaseDataServiceDao.get().shouldSave(item.getLastUpdateTime(), existingItem.getLastUpdateTime())) {
						PosLog.info(getClass(), item.getId() + " already updated"); //$NON-NLS-1$
						continue;
					}
					long version = existingItem.getVersion();
					PropertyUtils.copyProperties(existingItem, item);
					existingItem.setVersion(version);
					existingItem.setUpdateLastUpdateTime(updateLastUpdateTime);
					existingItem.setUpdateSyncTime(updateSyncTime);
					update(existingItem, session);
				}
				else {
					item.setUpdateLastUpdateTime(updateLastUpdateTime);
					item.setUpdateSyncTime(updateSyncTime);
					save(item, session);
				}
			}
			tx.commit();
		} catch (Exception e) {
			tx.rollback();
			throw e;
		} finally {
			closeSession(session);
		}
	}

}