package com.floreantpos.model.dao;

import java.io.Serializable;
import java.util.List;

import org.apache.commons.beanutils.PropertyUtils;
import org.apache.commons.lang.StringUtils;
import org.hibernate.Criteria;
import org.hibernate.Hibernate;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.hibernate.criterion.Criterion;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.Projections;
import org.hibernate.criterion.Restrictions;

import com.floreantpos.Messages;
import com.floreantpos.PosException;
import com.floreantpos.PosLog;
import com.floreantpos.model.Department;
import com.floreantpos.model.MenuCategory;
import com.floreantpos.model.OrderType;
import com.floreantpos.model.Outlet;
import com.floreantpos.model.Pagination;
import com.floreantpos.model.PriceRule;
import com.floreantpos.model.SalesArea;
import com.floreantpos.model.Terminal;
import com.orocube.rest.service.server.BaseDataServiceDao;

public class DepartmentDAO extends BaseDepartmentDAO {

	/**
	 * Default constructor.  Can be used in place of getInstance()
	 */
	public DepartmentDAO() {
	}

	@Override
	protected Serializable save(Object obj, Session s) {
		doCheckValidation(obj);
		updateTime(obj);
		return super.save(obj, s);
	}

	@Override
	protected void update(Object obj, Session s) {
		doCheckValidation(obj);
		updateTime(obj);
		super.update(obj, s);
	}

	@Override
	protected void saveOrUpdate(Object obj, Session s) {
		doCheckValidation(obj);
		updateTime(obj);
		super.saveOrUpdate(obj, s);
	}

	protected void doCheckValidation(Object obj) {
		Department department = (Department) obj;

		String name = department.getName();
		if (StringUtils.isBlank(name)) {
			throw new PosException(String.format(Messages.getString("value_cannot_be_empty"), "Name"));
		}
		else if (StringUtils.isNotBlank(name) && name.length() > 60) {
			throw new PosException(String.format(Messages.getString("value_too_long"), "department name", 60)); //$NON-NLS-1$
		}

		String address = department.getAddress();
		if (StringUtils.isNotBlank(address) && address.length() > 220) {
			throw new PosException(String.format(Messages.getString("value_too_long"), "address", 220)); //$NON-NLS-1$
		}

		String description = department.getDescription();
		if (StringUtils.isNotBlank(description) && description.length() > 255) {
			throw new PosException(String.format(Messages.getString("value_too_long"), "description", 255)); //$NON-NLS-1$
		}

	}

	@Override
	protected void delete(Object obj, Session session) {
		Department bean = (Department) obj;
		if (bean == null) {
			throw new PosException(Messages.getString("DepartmentDAO.0")); //$NON-NLS-1$
		}
		checkForeignRelation(bean);
		bean.setDeleted(Boolean.TRUE);
		update(bean, session);
	}

	private void checkForeignRelation(Department bean) throws PosException {
		String id = bean.getId();
		StringBuilder ref = new StringBuilder();
		StringBuilder details = new StringBuilder();
		try (Session ses = createNewSession()) {
			GenericDAO dao = GenericDAO.getInstance();
			append(dao.getForeignDataListNames(ses, MenuCategory.class, "departments", Department.PROP_ID, id), Messages.getString("DepartmentDAO.2"), ref, //$NON-NLS-1$//$NON-NLS-2$
					details);
			append(dao.getForeignDataListNames(ses, OrderType.class, "departments", Department.PROP_ID, id), Messages.getString("DepartmentDAO.4"), ref, //$NON-NLS-1$//$NON-NLS-2$
					details);
			append(dao.getForeignDataListNames(ses, Outlet.class, "departments", Department.PROP_ID, id), Messages.getString("DepartmentDAO.6"), ref, details);//$NON-NLS-1$//$NON-NLS-2$
			append(dao.getForeignDataListNames(ses, SalesArea.class, SalesArea.PROP_DEPARTMENT_ID, id), Messages.getString("DepartmentDAO.7"), ref, details);//$NON-NLS-1$
			append(dao.getForeignDataListNames(ses, PriceRule.class, PriceRule.PROP_DEPARTMENT_ID, id), Messages.getString("DepartmentDAO.8"), ref, details);//$NON-NLS-1$
			append(dao.getForeignDataListNames(ses, Terminal.class, Terminal.PROP_DEPARTMENT_ID, id), Messages.getString("DepartmentDAO.9"), ref, details);//$NON-NLS-1$ 
		}
		String foreignItemDetails = details.toString();
		if (StringUtils.isNotBlank(foreignItemDetails)) {
			String foreignReferences = ref.toString();
			if (foreignReferences.endsWith(", ")) { //$NON-NLS-1$
				foreignReferences = foreignReferences.substring(0, foreignReferences.length() - 2);
			}
			throw new PosException(Messages.getString("DepartmentDAO.11") + foreignReferences + Messages.getString("DepartmentDAO.12") //$NON-NLS-1$ //$NON-NLS-2$
					+ foreignReferences + ".", foreignItemDetails); //$NON-NLS-1$
		}
	}

	private void append(List<String> foreignDataListNames, String ref, StringBuilder refDetails, StringBuilder details) {
		if (foreignDataListNames == null || foreignDataListNames.isEmpty()) {
			return;
		}
		details.append(Messages.getString("DepartmentDAO.14") + ref + ":"); //$NON-NLS-1$ //$NON-NLS-2$
		refDetails.append(ref + ", "); //$NON-NLS-1$
		int count = 1;
		for (String itemName : foreignDataListNames) {
			details.append("\n" + count + ". " + itemName); //$NON-NLS-1$ //$NON-NLS-2$
			count++;
		}
		details.append("\n"); //$NON-NLS-1$
	}

	@Override
	public List<Department> findAll() {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			addDeletedFilter(criteria);

			criteria.addOrder(Order.asc(Department.PROP_NAME).ignoreCase());
			return criteria.list();
		}
	}

	public Department initialize(Department dept) {
		if (dept == null || dept.getId() == null)
			return dept;

		Session session = null;

		try {
			session = createNewSession();
			session.refresh(dept);

			Hibernate.initialize(dept.getOutlets());

			return dept;
		} finally {
			closeSession(session);
		}
	}

	public List<Department> findByOrderType(OrderType orderType) {
		Session session = null;
		Criteria criteria = null;
		try {
			session = getSession();
			criteria = session.createCriteria(getReferenceClass());
			criteria.createAlias("orderTypes", "o"); //$NON-NLS-1$ //$NON-NLS-2$
			criteria.add(Restrictions.eq("o.id", orderType.getId())); //$NON-NLS-1$
			addDeletedFilter(criteria);
			return criteria.list();
		} finally {
			closeSession(session);
		}
	}

	public boolean existsDepartment(OrderType orderType) {
		Session session = null;
		Criteria criteria = null;
		try {
			session = createNewSession();
			criteria = session.createCriteria(getReferenceClass());
			criteria.setProjection(Projections.rowCount());

			if (orderType != null) {
				criteria.createAlias("orderTypes", "o"); //$NON-NLS-1$ //$NON-NLS-2$
				criteria.add(Restrictions.eq("o.id", orderType.getId())); //$NON-NLS-1$
			}
			addDeletedFilter(criteria);
			Number rowCount = (Number) criteria.uniqueResult();
			return rowCount != null && rowCount.intValue() > 0;
		} finally {
			closeSession(session);
		}
	}

	public List<Department> findDepartmentbyOutlet(Outlet outlet) {
		Session session = null;
		try {
			session = getSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			//criteria.add(Restrictions.eq(Department.PROP_OUTLET, outlet));
			addDeletedFilter(criteria);
			List list = criteria.list();
			if (list == null || list.size() == 0) {
				return null;
			}
			return list;

		} catch (Exception e) {
			PosLog.error(DepartmentDAO.class, e);
		} finally {
			if (session != null) {
				session.close();
			}
		}
		return null;

	}

	public Department findById(String departmentId) {
		Session session = null;

		try {
			session = getSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.eq(Department.PROP_ID, departmentId));
			addDeletedFilter(criteria);
			return (Department) criteria.uniqueResult();

		} finally {
			if (session != null) {
				closeSession(session);
			}
		}
	}

	public List<Department> findAllUnSyncDepartment() {
		Session session = null;
		try {
			session = createNewSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			Criterion falseCriteria = Restrictions.eq(Department.PROP_CLOUD_SYNCED, Boolean.FALSE);
			Criterion nullCriteria = Restrictions.isNull(Department.PROP_CLOUD_SYNCED);
			criteria.add(Restrictions.or(falseCriteria, nullCriteria));
			return criteria.list();
		} finally {
			if (session != null) {
				closeSession(session);
			}
		}
	}

	public void saveOrUpdateDepartments(List<Department> departments, boolean updateLastUpdateTime, boolean updateSyncTime) throws Exception {
		if (departments == null)
			return;

		Transaction tx = null;
		Session session = null;
		try {
			session = createNewSession();
			tx = session.beginTransaction();
			for (Department department : departments) {
				Department existingItem = get(department.getId());
				if (existingItem != null) {
					if (!BaseDataServiceDao.get().shouldSave(department.getLastUpdateTime(), existingItem.getLastUpdateTime())) {
						PosLog.info(getClass(), department.getName() + " already updated"); //$NON-NLS-1$
						continue;
					}
					final String id = existingItem.getId();
					long version = existingItem.getVersion();
					PropertyUtils.copyProperties(existingItem, department);
					existingItem.setId(id);
					existingItem.setVersion(version);
					existingItem.setUpdateLastUpdateTime(updateLastUpdateTime);
					existingItem.setUpdateSyncTime(updateSyncTime);
					update(existingItem, session);
				}
				else {
					department.setVersion(0);
					department.setUpdateLastUpdateTime(updateLastUpdateTime);
					department.setUpdateSyncTime(updateSyncTime);
					save(department, session);
				}
			}
			tx.commit();
		} catch (Exception e) {
			tx.rollback();
			throw e;
		} finally {
			closeSession(session);
		}

	}

	public void doLoadDepartment(Pagination pagination) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(Department.class);

			addDeletedFilter(criteria);

			pagination.setNumRows(rowCount(criteria));
			criteria.setFirstResult(pagination.getCurrentRowIndex());
			criteria.setMaxResults(pagination.getPageSize());

			criteria.addOrder(Order.asc(Department.PROP_NAME).ignoreCase());
			pagination.setRows(criteria.list());
		}
	}
}